/*
 * Copyright (C) 2016-2024 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

namespace JSC {

// Are you tired of waiting for all of WebKit to build because you changed the implementation of a
// function in HeapInlines.h?  Does it bother you that you're waiting on rebuilding the JS DOM
// bindings even though your change is in a function called from only 2 .cpp files?  Then HeapUtil.h
// is for you!  Everything in this class should be a static method that takes a Heap& if needed.
// This is a friend of Heap, so you can access all of Heap's privates.
//
// This ends up being an issue because Heap exposes a lot of methods that ought to be inline for
// performance or that must be inline because they are templates.  This class ought to contain
// methods that are used for the implementation of the collector, or for unusual clients that need
// to reach deep into the collector for some reason.  Don't put things in here that would cause you
// to have to include it from more than a handful of places, since that would defeat the purpose.
// This class isn't here to look pretty.  It's to let us hack the GC more easily!

class HeapUtil {
public:
    static bool isPointerGCObjectJSCell(JSC::Heap& heap, TinyBloomFilter<uintptr_t> filter, JSCell* pointer)
    {
        // It could point to a large allocation.
        if (pointer->isPreciseAllocation()) {
            auto* set = heap.objectSpace().preciseAllocationSet();
            ASSERT(set);
            if (set->isEmpty())
                return false;
#if USE(JSVALUE32_64)
            // In 32bit systems a cell pointer can be 0xFFFFFFFF (an entries in the call frame), and this
            // value clashes with the deletedValue in a set<JSCell*>.
            if (!set->isValidValue(pointer))
                return false;
#endif
            return set->contains(pointer);
        }

        const UncheckedKeyHashSet<MarkedBlock*>& set = heap.objectSpace().blocks().set();

        MarkedBlock* candidate = MarkedBlock::blockFor(pointer);
        if (filter.ruleOut(std::bit_cast<uintptr_t>(candidate))) {
            ASSERT(!candidate || !set.contains(candidate));
            return false;
        }

        if (!MarkedBlock::isAtomAligned(pointer))
            return false;

        if (!set.contains(candidate))
            return false;

        if (candidate->handle().cellKind() != HeapCell::JSCell)
            return false;

        if (!candidate->handle().isLiveCell(pointer))
            return false;

        return true;
    }

    // This does not find the cell if the pointer is pointing at the middle of a JSCell.
    static bool isValueGCObject(
        JSC::Heap& heap, TinyBloomFilter<uintptr_t> filter, JSValue value)
    {
        ASSERT(heap.objectSpace().preciseAllocationSet());
        if (!value.isCell())
            return false;
        return isPointerGCObjectJSCell(heap, filter, value.asCell());
    }
};

} // namespace JSC

