/*
 * Copyright (C) 2020 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "TestCommand.h"

#include <wtf/text/StringToIntegerConversion.h>

namespace WTR {

class CommandTokenizer {
public:
    explicit CommandTokenizer(const std::string& input)
        : m_input(input)
        , m_posNextSeparator(0)
    {
        pump();
    }

    bool hasNext() const;
    std::string next();

private:
    void pump();
    static const char kSeparator = '\'';
    const std::string& m_input;
    std::string m_next;
    size_t m_posNextSeparator;
};

void CommandTokenizer::pump()
{
    if (m_posNextSeparator == std::string::npos || m_posNextSeparator == m_input.size()) {
        m_next = std::string();
        return;
    }
    size_t start = m_posNextSeparator ? m_posNextSeparator + 1 : 0;
    m_posNextSeparator = m_input.find(kSeparator, start);
    size_t size = m_posNextSeparator == std::string::npos ? std::string::npos : m_posNextSeparator - start;
    m_next = std::string(m_input, start, size);
}

std::string CommandTokenizer::next()
{
    ASSERT(hasNext());

    std::string oldNext = m_next;
    pump();
    return oldNext;
}

bool CommandTokenizer::hasNext() const
{
    return !m_next.empty();
}

NO_RETURN static void die(const std::string& inputLine)
{
    fprintf(stderr, "Unexpected input line: %s\n", inputLine.c_str());
    exit(1);
}

TestCommand parseInputLine(const std::string& inputLine)
{
    TestCommand result;
    CommandTokenizer tokenizer(inputLine);
    if (!tokenizer.hasNext())
        die(inputLine);

    std::string arg = tokenizer.next();
    result.pathOrURL = arg;
    while (tokenizer.hasNext()) {
        arg = tokenizer.next();
        if (arg == "--timeout") {
            auto timeoutToken = tokenizer.next();
            result.timeout = Seconds::fromMilliseconds(parseInteger<int>(std::span<const char> { timeoutToken }).value_or(0));
        } else if (arg == "-p" || arg == "--pixel-test") {
            result.shouldDumpPixels = true;
            if (tokenizer.hasNext())
                result.expectedPixelHash = tokenizer.next();
        } else if (arg == "--self-compare-with-header") {
            if (tokenizer.hasNext())
                result.selfComparisonHeader = tokenizer.next();
            else
                die(inputLine);
        } else if (arg == "--additional-header") {
            if (tokenizer.hasNext())
                result.additionalHeader = tokenizer.next();
            else
                die(inputLine);
        } else if (arg == std::string("--dump-jsconsolelog-in-stderr"))
            result.dumpJSConsoleLogInStdErr = true;
        else if (arg == std::string("--absolutePath"))
            result.absolutePath = tokenizer.next();
        else if (arg == "--force-dump-pixels")
            result.forceDumpPixels = true;
        else
            die(inputLine);
    }

    if (result.absolutePath.empty())
        result.absolutePath = testPath(result.pathOrURL);

    return result;
}

#if PLATFORM(JAVA)
std::string testPath(const std::string& pathOrURL)
{
    if (pathOrURL.find("http://") == 0 || pathOrURL.find("https://") == 0)
        return { };

    if (pathOrURL.find("file://") == 0)
        return pathOrURL.substr(strlen("file:/"));

    return pathOrURL;
}
#else
std::filesystem::path testPath(const std::string& pathOrURL)
{
    if (pathOrURL.find("http://") == 0 || pathOrURL.find("https://") == 0)
        return { };

    if (pathOrURL.find("file://") == 0)
        return pathOrURL.substr(strlen("file:/"));

    std::error_code ec;
    return std::filesystem::absolute(pathOrURL, ec);
}
#endif

std::string testURLString(const std::string& pathOrURL)
{
    if (pathOrURL.find("http://") == 0 || pathOrURL.find("https://") == 0 || pathOrURL.find("file://") == 0)
        return pathOrURL;

#if PLATFORM(JAVA)
    return "file://" + pathOrURL;
#else
    std::error_code ec;
    return "file://" + std::filesystem::absolute(pathOrURL, ec).generic_string();
#endif
}

}
