/*
 * Copyright (c) 2014, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package test.robot.com.sun.glass.ui.monocle;

import java.util.Collection;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import com.sun.glass.ui.monocle.TestLogShim;
import test.robot.com.sun.glass.ui.monocle.input.devices.TestTouchDevice;
import test.robot.com.sun.glass.ui.monocle.input.devices.TestTouchDevices;

/**
 * Rotate tests generated by two touch points.
 * By default rotation starting form 0 degrees position - on Axis Y,
 * but it can be sent from any other location on the object
 *  */
public final class RotateTest extends ParameterizedTestBase {

    private int newX1;
    private int newY1;
    private static final int ZERO_ANGLE = 0;
    private int centerX;
    private int centerY;
    private int radius;
    private int p1;
    private int p2;

    private static Collection<TestTouchDevice> parameters() {
        return TestTouchDevices.getTouchDeviceParameters(2);
    }

    // @BeforeEach
    // junit5 does not support parameterized class-level tests yet
    public void init(TestTouchDevice device) throws Exception {
        createDevice(device, null);
        //Rotate tests should be run only on platforms that support current feature
        Assumptions.assumeTrue(Boolean.getBoolean("com.sun.javafx.gestures.rotate"));

        centerX = (int) Math.round(width * 0.5);
        centerY = (int) Math.round(height * 0.5);
        radius = (int) Math.round(height * 0.45);
    }

    @AfterEach
    public void releaseAll() throws Exception {
        if ((device != null) && (device.getPressedPoints() == 2)) {
            TestLogShim.reset();
            device.removePoint(p1);
            device.removePoint(p2);
            device.sync();
        }
    }

    private void updateNewTouchPoint(int angle, int radius, int centerX, int centerY) {

        int transformedAngle = 90 - angle;
        newX1 = centerX + (int) Math.round(radius *
                Math.cos(Math.toRadians(transformedAngle)));
        newY1 = centerY - (int) Math.round(radius *
                Math.sin(Math.toRadians(transformedAngle)));
    }

    private int getDistance(int xPoint1, int yPoint1, int xPoint2, int yPoint2) {
        double d = Math.sqrt(Math.pow((xPoint1 - xPoint2), 2)
                + Math.pow((yPoint1 - yPoint2), 2));
        return (int) d;
    }

    private int getRotateThreshold() {
        String s = System.getProperty("com.sun.javafx.gestures.rotate.threshold");
        if (s != null) {
            return Integer.valueOf(s);
        } else {
            return 5;
        }
    }

    private void Rotate(int startAngle, int radius, int x2, int y2, int angleStep,
                        int numOfIterations) throws Exception {

        int totalAngle = angleStep;
        updateNewTouchPoint(startAngle, radius, x2, y2);

        TestLogShim.reset();
        p1 = device.addPoint(newX1, newY1);
        p2 = device.addPoint(x2, y2);
        device.sync();
        //verify pressing two fingers
        TestLogShim.waitForLogContaining("TouchPoint: PRESSED %d, %d", newX1, newY1);
        TestLogShim.waitForLogContaining("TouchPoint: PRESSED %d, %d", x2, y2);

        //saving previous coordinates:
        int previousX = newX1;
        int previousY = newY1;

        updateNewTouchPoint((angleStep + startAngle), radius, x2, y2);

        Assumptions.assumeTrue(getDistance(previousX, previousY, newX1, newY1 )
                > device.getTapRadius());

        //start the rotation
        TestLogShim.reset();
        device.setPoint(p1, newX1, newY1);
        device.sync();
        TestLogShim.waitForLogContaining("TouchPoint: MOVED %d, %d", newX1, newY1);
        TestLogShim.waitForLogContaining("TouchPoint: STATIONARY %d, %d", x2, y2);

        if (Math.abs(angleStep) >= getRotateThreshold()) {
            TestLogShim.waitForLogContaining("Rotation started, angle: " + ZERO_ANGLE
                + ", total angle: " + ZERO_ANGLE + ", inertia value: false");
            TestLogShim.waitForLogContaining("Rotation, angle: " + angleStep
                + ", total angle: " + totalAngle
                + ", inertia value: false");
        } else {
            Assertions.assertEquals(0, TestLogShim.countLogContaining("Rotation started"));
            Assertions.assertEquals(0, TestLogShim.countLogContaining("Rotation, angle"));
        }
        boolean passedTheThreshold =false;
        if (numOfIterations >= 2) {
            for (int i = 2; i <= numOfIterations; i++) {
                updateNewTouchPoint(angleStep * i + startAngle, radius, x2, y2);
                totalAngle += angleStep;
                TestLogShim.reset();
                device.setPoint(p1, newX1, newY1);
                device.sync();

                TestLogShim.waitForLogContaining("TouchPoint: MOVED %d, %d", newX1, newY1);
                TestLogShim.waitForLogContaining("TouchPoint: STATIONARY %d, %d", x2, y2);

                String expectedLog;
                if (Math.abs(angleStep) < getRotateThreshold()) {
                    if(Math.abs(totalAngle) >= getRotateThreshold()) {
                        if (!passedTheThreshold) {
                            expectedLog = "Rotation, angle: " + totalAngle
                                + ", total angle: " + totalAngle
                                + ", inertia value: false";
                            passedTheThreshold = true;
                        } else {
                            expectedLog = "Rotation, angle: " + angleStep
                                + ", total angle: " + totalAngle
                                + ", inertia value: false";
                        }
                    } else {
                        expectedLog = "sync";
                    }
                } else {
                    expectedLog = "Rotation, angle: " + angleStep
                            + ", total angle: " + totalAngle
                            + ", inertia value: false";
                }
                TestLogShim.waitForLogContaining(expectedLog);
            }
        }
        TestLogShim.reset();
        device.removePoint(p1);
        device.removePoint(p2);
        device.sync();
        //verify fingers release
        TestLogShim.waitForLogContaining("TouchPoint: RELEASED %d, %d", newX1, newY1);
        TestLogShim.waitForLogContaining("TouchPoint: RELEASED %d, %d", x2, y2);
        if (Math.abs(totalAngle) >= getRotateThreshold()) {
            TestLogShim.waitForLogContaining("Rotation finished, angle: " + ZERO_ANGLE
                    + ", total angle: " + totalAngle + ", inertia value: false");
            Assertions.assertEquals(1, TestLogShim.countLogContaining("Rotation "
                    + "finished, " + "angle: " + ZERO_ANGLE
                    + ", total angle: " + totalAngle
                    + ", inertia value: false"));
        } else {
            Assertions.assertEquals(0, TestLogShim.countLogContaining("Rotation finished, "
                    + "angle: " + ZERO_ANGLE + ", total angle: " + totalAngle
                    + ", inertia value: false"));
        }
        if (TestLogShim.countLogContaining("Rotation finished") > 0) {
            TestLogShim.waitForLogContainingSubstrings("Rotation", "inertia value: true");
        }
        TestLogShim.reset();
        p2 = device.addPoint(x2, y2);
        device.sync();
        device.removePoint(p2);
        device.sync();
        TestLogShim.waitForLogContaining("TouchPoint: RELEASED %d, %d", x2, y2);
    }

    private void Rotate(int radius, int x2, int y2, int angleStep,
                        int numOfIterations) throws Exception {
        Rotate(0, radius, x2, y2, angleStep, numOfIterations);
    }

    private void Rotate(int startAngle, int angleStep, int numOfIterations) throws Exception {
        Rotate(startAngle, radius, centerX, centerY, angleStep, numOfIterations);
    }

    private void Rotate(int angleStep, int numOfIterations) throws Exception {
        Rotate(0, radius, centerX, centerY, angleStep, numOfIterations);
    }

    /**
     * Tap two fingers, drag a little bit upper finger right in order move,
     * but not enough for rotation.
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testSmallStepRightNoRotateSent(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(4, 1);
    }

    /**
     * Tap two fingers, drag a little bit upper finger right in order move,
     * but not enough for rotation, then make 2 more small moves
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateRightByFewSmallSteps(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(4, 5);
    }

    /**
     * Tap two fingers, drag upper finger right in order to rotate
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateRight(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(15, 6);
    }

    /**
     * Tap two fingers, rotate the object right (by 3 steps, 50 degrees each)
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateRightBigSteps(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(50, 3);
    }

    /**
     * Tap two fingers, rotate the object right by only 1 very big step - 80 degrees
     */
    @Disabled //JDK-8088641
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateRightOneBigStep(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(80, 1);
    }

    /**
     * Tap two fingers, drag a little bit upper finger left in order move,
     * but not enough for rotation.
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testSmallStepLeftNoRotateSent(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(-4, 1);
    }

    /**
     * Tap two fingers, drag a little bit upper finger left in order move,
     * but not enough for rotation, then make 9 more small moves
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateLeftByFewSmallSteps(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(-4, 10);
    }

    /**
     * Tap two fingers, drag upper finger left in order to rotate
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateLeft(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(-10, 4);
    }

    /**
     * Tap two fingers, rotate the object left (by 5 steps, 40 degrees each)
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateLeftBigSteps(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(-40, 5);
    }

    /**
     * Tap two fingers, rotate the object left by only 1 very big step - 70 degrees
     */
    @Disabled //JDK-8088641
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateLeftOneBigStep(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(-70, 1);
    }

    /**
     * Tap two fingers in 45 degrees, rotate the object right
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateRightFrom45Degrees(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(45, 20, 3);
    }

    /**
     * Tap two fingers in 45 degrees, rotate the object left
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateLeftFrom45Degrees(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(45, -20, 3);
    }

    /**
     * Tap two fingers in -45 degrees, rotate the object right
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateRightFromMinus45Degrees(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(-45, 20, 3);
    }

    /**
     * Tap two fingers in -45 degrees, rotate the object left
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateLeftFromMinus45Degrees(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(-45, -20, 3);
    }

    /**
     * Tap two fingers in 140 degrees, rotate the object right
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateRightFrom140Degrees(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(140, 20, 3);
    }

    /**
     * Tap two fingers in 140 degrees, rotate the object left
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateLeftFrom140Degrees(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(140, -20, 3);
    }

    /**
     * Tap two fingers in -140 degrees, rotate the object right
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateRightFromMinus140Degrees(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(-140, 20, 3);
    }

    /**
     * Tap two fingers in -140 degrees, rotate the object left
     */
    @ParameterizedTest
    @MethodSource("parameters")
    public void testRotateLeftFromMinus140Degrees(TestTouchDevice device) throws Exception {
        init(device);
        Rotate(-140, -20, 3);
    }
}
