/*
 * Copyright (c) 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package test.com.sun.javafx.collections;

import com.sun.javafx.collections.ObservableListWrapper;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;

public class ObservableListWrapperTest {

    @Nested
    class RemoveAllTest {
        @Test
        public void testNullArgumentThrowsNPE() {
            var list = new ObservableListWrapper<>(List.of("a", "b", "c"));
            assertThrows(NullPointerException.class, () -> list.removeAll((Collection<String>) null));
        }

        @Test
        public void testEmptyCollectionArgumentDoesNotEnumerateBackingList() {
            var list = new ObservableListWrapper<>(new ArrayList<>(List.of("a", "b", "c")) {
                @Override
                public String get(int index) {
                    throw new AssertionError("get() was not elided");
                }
            });

            list.removeAll(List.of());
        }

        @Test
        public void testEmptyCollectionArgumentWorksCorrectly() {
            var list = new ObservableListWrapper<>(new ArrayList<>(List.of("a", "b", "c")));
            assertFalse(list.removeAll(List.of()));
            assertEquals(List.of("a", "b", "c"), list);
        }
    }

    @Nested
    class RetainAllTest {
        @Test
        public void testNullArgumentThrowsNPE() {
            var list = new ObservableListWrapper<>(List.of("a", "b", "c"));
            assertThrows(NullPointerException.class, () -> list.retainAll((Collection<String>) null));
        }

        @Test
        public void testEmptyCollectionArgumentDoesNotCallContains() {
            var list = new ObservableListWrapper<>(new ArrayList<>(List.of("a", "b", "c")));
            list.retainAll(new ArrayList<String>() {
                @Override
                public boolean contains(Object o) {
                    throw new AssertionError("contains() was not elided");
                }
            });
        }

        @Test
        public void testEmptyCollectionArgumentWorksCorrectly() {
            var list = new ObservableListWrapper<>(new ArrayList<>(List.of("a", "b", "c")));
            assertTrue(list.retainAll(List.of()));
            assertTrue(list.isEmpty());
        }
    }

}
