package com.oracle.objectfile.elf;

import com.oracle.objectfile.dwarf.DwarfARangesSectionImpl;
import com.oracle.objectfile.dwarf.DwarfAbbrevSectionImpl;
import com.oracle.objectfile.dwarf.DwarfDebugInfo;
import com.oracle.objectfile.dwarf.DwarfFrameSectionImpl;
import com.oracle.objectfile.dwarf.DwarfFrameSectionImplAArch64;
import com.oracle.objectfile.dwarf.DwarfFrameSectionImplX86_64;
import com.oracle.objectfile.dwarf.DwarfInfoSectionImpl;
import com.oracle.objectfile.dwarf.DwarfLineSectionImpl;
import com.oracle.objectfile.dwarf.DwarfLocSectionImpl;
import com.oracle.objectfile.dwarf.DwarfStrSectionImpl;

import java.nio.ByteOrder;

public class ELFDebugInfo extends DwarfDebugInfo {

    private final DwarfStrSectionImpl dwarfStrSection;
    private final DwarfAbbrevSectionImpl dwarfAbbrevSection;
    private final DwarfInfoSectionImpl dwarfInfoSection;
    private final DwarfLocSectionImpl dwarfLocSection;
    private final DwarfARangesSectionImpl dwarfARangesSection;
    private final DwarfLineSectionImpl dwarfLineSection;
    private final DwarfFrameSectionImpl dwarfFameSection;
    public final ELFMachine elfMachine;
    /**
     * Register used to hold the heap base.
     */
    private final byte heapbaseRegister;
    /**
     * Register used to hold the current thread.
     */
    private final byte threadRegister;

    public ELFDebugInfo(ELFMachine elfMachine, ByteOrder byteOrder) {
        super(byteOrder);
        this.elfMachine = elfMachine;
        dwarfStrSection = new DwarfStrSectionImpl(this);
        dwarfAbbrevSection = new DwarfAbbrevSectionImpl(this);
        dwarfInfoSection = new DwarfInfoSectionImpl(this);
        dwarfLocSection = new DwarfLocSectionImpl(this);
        dwarfARangesSection = new DwarfARangesSectionImpl(this);
        dwarfLineSection = new DwarfLineSectionImpl(this);

        if (isAarch64()) {
            dwarfFameSection = new DwarfFrameSectionImplAArch64(this);
            this.heapbaseRegister = rheapbase_aarch64;
            this.threadRegister = rthread_aarch64;
        } else {
            dwarfFameSection = new DwarfFrameSectionImplX86_64(this);
            this.heapbaseRegister = rheapbase_x86;
            this.threadRegister = rthread_x86;
        }
    }

    public DwarfStrSectionImpl getStrSectionImpl() {
        return dwarfStrSection;
    }

    public DwarfAbbrevSectionImpl getAbbrevSectionImpl() {
        return dwarfAbbrevSection;
    }

    public DwarfFrameSectionImpl getFrameSectionImpl() {
        return dwarfFameSection;
    }

    public DwarfInfoSectionImpl getInfoSectionImpl() {
        return dwarfInfoSection;
    }

    public DwarfLocSectionImpl getLocSectionImpl() {
        return dwarfLocSection;
    }

    public DwarfARangesSectionImpl getARangesSectionImpl() {
        return dwarfARangesSection;
    }

    public DwarfLineSectionImpl getLineSectionImpl() {
        return dwarfLineSection;
    }

    public byte getHeapbaseRegister() {
        return heapbaseRegister;
    }

    public byte getThreadRegister() {
        return threadRegister;
    }

    /////////////////////////////// EGS ////////////////////////////////
    public boolean isAarch64() {
        return elfMachine == ELFMachine.AArch64;
    }

    public boolean isAMD64() {
        return elfMachine == ELFMachine.X86_64;
    }

    public String textSectionName() {
        return DwarfSectionName.TEXT_SECTION.value;
    }

    public String lineSectionName() {
        return DwarfSectionName.DW_LINE_SECTION.value;
    }

    public String strSectionName() {
        return DwarfSectionName.DW_STR_SECTION.value;
    }

    public String locSectionName() {
        return DwarfSectionName.DW_LOC_SECTION.value;
    }

    public String arangesSectionName() {
        return DwarfSectionName.DW_ARANGES_SECTION.value;
    }

    public String frameSectionName() {
        return DwarfSectionName.DW_FRAME_SECTION.value;
    }

    public String abbrevSectionName() {
        return DwarfSectionName.DW_ABBREV_SECTION.value;
    }

    public String infoSectionName() {
        return DwarfSectionName.DW_INFO_SECTION.value;
    }

    /**
     * Various ELF sections created by GraalVM including all debug info sections. The enum sequence
     * starts with the text section (not defined in the DWARF spec and not created by debug info
     * code).
     */
    enum DwarfSectionName {
        TEXT_SECTION(".text"),
        DW_STR_SECTION(".debug_str"),
        DW_LINE_SECTION(".debug_line"),
        DW_FRAME_SECTION(".debug_frame"),
        DW_ABBREV_SECTION(".debug_abbrev"),
        DW_INFO_SECTION(".debug_info"),
        DW_LOC_SECTION(".debug_loc"),
        DW_ARANGES_SECTION(".debug_aranges");

        private final String value;

        DwarfSectionName(String s) {
            value = s;
        }
    }
}
