/*
 * Copyright (c) 2018, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.svm.hosted.javafx;

import java.lang.reflect.Modifier;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.stream.Stream;

import com.oracle.svm.core.feature.AutomaticallyRegisteredFeature;
import com.oracle.svm.core.feature.InternalFeature;
import com.oracle.svm.core.jdk.JNIRegistrationUtil;
import com.oracle.svm.core.jdk.PlatformNativeLibrarySupport;
import org.graalvm.nativeimage.Platform;
import org.graalvm.nativeimage.Platforms;
import org.graalvm.nativeimage.hosted.RuntimeClassInitialization;
import org.graalvm.nativeimage.hosted.RuntimeReflection;
import org.graalvm.nativeimage.impl.InternalPlatform;

@AutomaticallyRegisteredFeature
@Platforms(InternalPlatform.PLATFORM_JNI.class)
@SuppressWarnings("unused")
final class JavaFXFeature extends JNIRegistrationUtil implements InternalFeature {

    @Override
    public boolean isInConfiguration(IsInConfigurationAccess access) {
        return getJavaFXApplication(access) != null;
    }

    @Override
    public void beforeAnalysis(BeforeAnalysisAccess access) {
        /*
         * Include subclasses of javafx.application.Application for reflection so users don't have
         * to do it every time.
         *
         * See javafx.application.Application#launch(java.lang.String...).
         */
        access.registerSubtypeReachabilityHandler(this::registerApplicationClass, getJavaFXApplication(access));

        /*
         * Static initializers that are not supported in JavaFX.
         */
        Stream.of("com.sun.javafx.tk.quantum.QuantumRenderer", "javafx.stage.Screen", "com.sun.javafx.scene.control.behavior.BehaviorBase", "com.sun.javafx.scene.control.skin.BehaviorSkinBase",
                        "javafx.scene.control.SkinBase", "javafx.scene.control.Control", "javafx.scene.control.PopupControl", "javafx.scene.control.SkinBase$StyleableProperties",
                        "javafx.scene.control.Labeled$StyleableProperties", "javafx.scene.control.SkinBase$StyleableProperties", "com.sun.javafx.tk.quantum.PrismImageLoader2$AsyncImageLoader",
                        "com.sun.javafx.tk.quantum.PrismImageLoader2", "com.sun.prism.PresentableState")
                        .map(access::findClassByName)
                        .filter(Objects::nonNull)
                        .forEach(RuntimeClassInitialization::initializeAtRunTime);

        PlatformNativeLibrarySupport librarySupport = PlatformNativeLibrarySupport.singleton();
        Stream.of("com_sun_glass", "com_sun_javafx_embed", "com_sun_javafx_iio_jpeg", "com_sun_prism_d3d", "com_sun_prism_es2")
                /// uncomment to enable SW pipeline
//                "com_sun_pisces", "com_sun_prism_impl_shape", "com_sun_prism_j2d_print", "com_sun_scenario_effect_impl_sw_sse"
                .forEach(librarySupport::addBuiltinPkgNativePrefix);
        /*
         * Can't just add "com_sun_javafx_font" because some native methods
         * are only defined for one platform and not for others
         */
        Stream<String> fontPackages =
                Platform.includedIn(Platform.WINDOWS.class) ? Stream.of("PrismFontFactory", "directwrite")
                : Platform.includedIn(Platform.LINUX.class) ? Stream.of("FontConfigManager", "freetype")
                : Platform.includedIn(Platform.DARWIN.class) ? Stream.of("DFontDecoder", "MacFontFinder", "coretext")
                : Stream.empty();
        fontPackages.map("com_sun_javafx_font_"::concat)
                    .forEach(librarySupport::addBuiltinPkgNativePrefix);

        access.registerReachabilityHandler(this::enableJavaFX,
                clazz(access, "javafx.scene.Scene"));
        access.registerReachabilityHandler(this::enableIIO,
                clazz(access, "com.sun.javafx.iio.jpeg.JPEGImageLoader"));
        access.registerReachabilityHandler(this::enablePrinting,
                clazz(access, "com.sun.javafx.tk.PrintPipeline"));
        registerIfReachable(access, this::enableControls,"javafx.scene.control.Control");
        registerIfReachable(access, this::enableFXML, "javafx.fxml.FXMLLoader");
        registerIfReachable(access, this::enableSwing, "javafx.embed.swing.SwingNode");
    }

    private static Class<?> getJavaFXApplication(FeatureAccess access) {
        return access.findClassByName("javafx.application.Application");
    }

    private void registerApplicationClass(DuringAnalysisAccess access, Class<?> appClass) {
        if (!Modifier.isAbstract(appClass.getModifiers())) {
            RuntimeReflection.register(appClass);
            RuntimeReflection.registerForReflectiveInstantiation(appClass);
        }
    }

    /**
     * Register reachability callback without loading the trigger class (which may be in an unavailable module)
     */
    private void registerIfReachable(BeforeAnalysisAccess access, Consumer<DuringAnalysisAccess> callback, String className) {
        Class<?> cls = access.findClassByName(className);
        if (cls != null) {
            access.registerReachabilityHandler(callback, cls);
        }
    }

    private void enableJavaFX(DuringAnalysisAccess access) {
        JavaFXReflection.registerJavaFX(access);
        enableGlass(access);
        enableFonts(access);
        if (Platform.includedIn(Platform.LINUX.class)) {
            enableGlassGTK3(access);
            enableFontsLinux(access);
            enablePrismES2(access);
        } else if (Platform.includedIn(Platform.DARWIN.class)) {
            enableGlassMac(access);
            enableFontsMac(access);
            enablePrismES2(access);
        } else if (Platform.includedIn(Platform.WINDOWS.class)) {
            enableGlassWin(access);
            enableFontsWin(access);
            enablePrismD3D(access);
        }
    }

    private void enableGlass(DuringAnalysisAccess access) {
        JavaFXReflection.registerGlass(access);
        JavaFXJNI.registerGlass(access);
    }

    private void enableGlassMac(DuringAnalysisAccess access) {
        JavaFXReflection.registerGlassMac(access);
        JavaFXJNI.registerGlassMac(access);
    }

    private void enableGlassWin(DuringAnalysisAccess access) {
        JavaFXReflection.registerGlassWin(access);
        JavaFXJNI.registerGlassWin(access);
    }

    private void enableGlassGTK3(DuringAnalysisAccess access) {
        JavaFXReflection.registerGlassGTK3(access);
        JavaFXJNI.registerGlassGTK3(access);
    }

    private void enableFonts(DuringAnalysisAccess access) {
        JavaFXJNI.registerFonts(access);
    }

    private void enableFontsMac(DuringAnalysisAccess access) {
        JavaFXReflection.registerFontsMac(access);
        JavaFXJNI.registerFontsMac(access);
    }

    private void enableFontsWin(DuringAnalysisAccess access) {
        JavaFXReflection.registerFontsWin(access);
        JavaFXJNI.registerFontsWin(access);
    }

    private void enableFontsLinux(DuringAnalysisAccess access) {
        JavaFXReflection.registerFontsLinux(access);
        JavaFXJNI.registerFontsLinux(access);
    }

    private void enableIIO(DuringAnalysisAccess access) {
        JavaFXReflection.registerIIO(access);
        JavaFXJNI.registerIIO(access);
    }

    private void enablePrismD3D(DuringAnalysisAccess access) {
        JavaFXReflection.registerPrismD3D(access);
        JavaFXJNI.registerPrismD3D(access);
    }

    private void enablePrismES2(DuringAnalysisAccess access) {
        JavaFXReflection.registerPrismES2(access);
        JavaFXJNI.registerPrismES2(access);
    }

    private void enableControls(DuringAnalysisAccess access) {
        JavaFXReflection.registerControls(access);
    }

    private void enableFXML(DuringAnalysisAccess access) {
        JavaFXReflection.registerFXML(access);
    }

    private void enableSwing(DuringAnalysisAccess access) {
        JavaFXReflection.registerSwing(access);
        JavaFXJNI.registerSwing(access);
    }

    private void enablePrinting(DuringAnalysisAccess access) {
        JavaFXReflection.registerPrinting(access);
    }
}
