package com.oracle.svm.hosted.javafx;

import com.oracle.svm.core.jdk.JNIRegistrationUtil;
import com.oracle.svm.core.jdk.NativeLibrarySupport;
import com.oracle.svm.core.jni.JNIRuntimeAccess;
import com.oracle.svm.hosted.FeatureImpl;
import com.oracle.svm.hosted.c.NativeLibraries;
import org.graalvm.compiler.serviceprovider.GraalServices;
import org.graalvm.compiler.serviceprovider.JavaVersionUtil;
import org.graalvm.nativeimage.Platform;
import org.graalvm.nativeimage.hosted.Feature.DuringAnalysisAccess;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.Vector;
import java.util.stream.Stream;

public class JavaFXJNI extends JNIRegistrationUtil {

    static void registerGlass(DuringAnalysisAccess access) {
        JNIRuntimeAccess.register(IOException.class, Runnable.class, String.class,
                clazz(access, "com.sun.glass.ui.CommonDialogs"),
                clazz(access, "com.sun.glass.ui.CommonDialogs$ExtensionFilter"),
                clazz(access, "com.sun.glass.ui.Screen"),
                clazz(access, "com.sun.glass.ui.Cursor"),
                clazz(access, "com.sun.glass.ui.Size"),
                clazz(access, "com.sun.glass.ui.View"),
                clazz(access, "com.sun.glass.ui.Window"));

        JNIRuntimeAccess.register(fields(access, "com.sun.glass.ui.View", "ptr"));
        JNIRuntimeAccess.register(fields(access, "com.sun.glass.ui.Window", "ptr"));

        JNIRuntimeAccess.register(
                method(access, "java.lang.Runnable", "run"),
                method(access, "java.lang.Throwable", "getMessage"),
                method(access, "java.lang.Class", "forName", String.class, boolean.class, ClassLoader.class),
                method(access, "java.util.List", "add", Object.class),
                method(access, "java.util.Map", "get", Object.class),
                method(access, "com.sun.glass.ui.Application", "getName"),
                method(access, "com.sun.glass.ui.Application", "GetApplication"),
                method(access, "com.sun.glass.ui.Application", "reportException", Throwable.class),
                method(access, "com.sun.glass.ui.CommonDialogs$ExtensionFilter", "getDescription"),
                method(access, "com.sun.glass.ui.CommonDialogs$ExtensionFilter", "extensionsToArray"),
                method(access, "com.sun.glass.ui.Pixels", "attachData", long.class),
                constructor(access, "com.sun.glass.ui.Size", int.class, int.class),
                constructor(access, "com.sun.glass.ui.Screen", long.class, int.class, int.class, int.class, int.class,
                        int.class, int.class, int.class, int.class, int.class, int.class, int.class, int.class, int.class, int.class, int.class,
                        float.class, float.class, float.class, float.class),
                method(access, "com.sun.glass.ui.Screen", "notifySettingsChanged"),
                method(access, "com.sun.glass.ui.View", "notifyResize", int.class, int.class),
                method(access, "com.sun.glass.ui.View", "notifyRepaint", int.class, int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.View", "notifyKey", int.class, int.class, char[].class, int.class),
                method(access, "com.sun.glass.ui.View", "notifyMouse", int.class, int.class, int.class, int.class,
                        int.class, int.class, int.class, boolean.class, boolean.class),
                method(access, "com.sun.glass.ui.View", "notifyScroll",
                        int.class, int.class, int.class, int.class, double.class, double.class,
                        int.class, int.class, int.class, int.class, int.class, double.class, double.class),
                method(access, "com.sun.glass.ui.View", "notifyMenu", int.class, int.class, int.class, int.class, boolean.class),
                method(access, "com.sun.glass.ui.View", "notifyInputMethod",
                        String.class, int[].class, int[].class, byte[].class, int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.View", "notifyView", int.class),
                method(access, "com.sun.glass.ui.View", "notifyDragEnter", int.class, int.class, int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.View", "notifyDragOver", int.class, int.class, int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.View", "notifyDragDrop", int.class, int.class, int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.View", "notifyDragLeave"),
                method(access, "com.sun.glass.ui.View", "notifyInputMethodCandidatePosRequest", int.class),
                method(access, "com.sun.glass.ui.View", "getAccessible"),
                method(access, "com.sun.glass.ui.Window", "notifyClose"),
                method(access, "com.sun.glass.ui.Window", "notifyDestroy"),
                method(access, "com.sun.glass.ui.Window", "notifyDelegatePtr", long.class),
                method(access, "com.sun.glass.ui.Window", "notifyFocus", int.class),
                method(access, "com.sun.glass.ui.Window", "notifyFocusDisabled"),
                method(access, "com.sun.glass.ui.Window", "notifyFocusUngrab"),
                method(access, "com.sun.glass.ui.Window", "notifyMove", int.class, int.class),
                method(access, "com.sun.glass.ui.Window", "notifyMoveToAnotherScreen",
                        clazz(access, "com.sun.glass.ui.Screen")),
                method(access, "com.sun.glass.ui.Window", "notifyResize", int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.Window", "notifyScaleChanged",
                        float.class, float.class, float.class, float.class));
    }

    static void registerGlassGTK3(DuringAnalysisAccess access) {
        registerStaticLib(access, "glass");
        registerStaticLib(access, "glassgtk3",
                "m", "stdc++", "X11", "Xtst", "fontconfig", "freetype", "harfbuzz",
                "gtk-3", "gdk-3", "glib-2.0", "gio-2.0", "gobject-2.0", "gthread-2.0", "gmodule-2.0",
                "gdk_pixbuf-2.0", "cairo", "cairo-gobject", "pango-1.0", "pangocairo-1.0", "pangoft2-1.0", "atk-1.0");

        JNIRuntimeAccess.register(ByteBuffer.class, Iterable.class, Iterator.class, Map.class, Set.class, HashSet.class,
                ArrayList.class, Throwable.class, IllegalStateException.class, UnsupportedOperationException.class,
                clazz(access, "com.sun.glass.ui.CommonDialogs$FileChooserResult"),
                clazz(access, "com.sun.glass.ui.Pixels"),
                clazz(access, "com.sun.glass.ui.gtk.GtkPixels"),
                clazz(access, "com.sun.glass.ui.gtk.GtkView"),
                clazz(access, "com.sun.glass.ui.gtk.GtkWindow"),
                clazz(access, "com.sun.glass.ui.Clipboard"),
                clazz(access, "com.sun.glass.ui.gtk.GtkApplication"));
        JNIRuntimeAccess.register(fields(access, "com.sun.glass.ui.Cursor", "ptr"));
        JNIRuntimeAccess.register(fields(access, "com.sun.glass.ui.gtk.GtkApplication",
                "display", "screen", "visualID"));
        JNIRuntimeAccess.register(
                constructor(access, "java.lang.String", byte[].class),
                constructor(access, "java.lang.String", byte[].class, String.class),
                method(access, "java.lang.String", "getBytes", String.class),
                method(access, "java.nio.ByteBuffer", "array"),
                method(access, "java.nio.ByteBuffer", "wrap", byte[].class),
                method(access, "java.util.Map", "keySet"),
                method(access, "java.util.Map", "containsKey", Object.class),
                method(access, "java.util.Set", "add", Object.class),
                method(access, "java.util.Set", "size"),
                method(access, "java.util.Set", "toArray", Object[].class),
                method(access, "java.lang.Iterable", "iterator"),
                method(access, "java.util.Iterator", "next"),
                method(access, "java.util.Iterator", "hasNext"),
                constructor(access, "java.util.ArrayList"),
                method(access, "java.util.ArrayList", "add", Object.class),
                method(access, "java.util.ArrayList", "get", int.class),
                constructor(access, "java.util.HashSet"),
                method(access, "com.sun.glass.ui.CommonDialogs", "createFileChooserResult",
                        String[].class, clazz(access, "[Lcom.sun.glass.ui.CommonDialogs$ExtensionFilter;"), int.class),
                constructor(access, "com.sun.glass.ui.CommonDialogs$FileChooserResult"),
                constructor(access, "com.sun.glass.ui.gtk.GtkPixels", int.class, int.class, java.nio.ByteBuffer.class),
                method(access, "com.sun.glass.ui.gtk.GtkView", "notifyInputMethodDraw",
                        String.class, int.class, int.class, int.class, byte[].class),
                method(access, "com.sun.glass.ui.gtk.GtkView", "notifyInputMethodCaret", int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.gtk.GtkView", "notifyPreeditMode", boolean.class),
                method(access, "com.sun.glass.ui.Window", "notifyLevelChanged", int.class),
                method(access, "com.sun.glass.ui.Window", "isEnabled"),
                method(access, "com.sun.glass.ui.gtk.GtkWindow", "notifyStateChanged", int.class),
                method(access, "com.sun.glass.ui.Clipboard", "contentChanged"));
    }

    private static void registerMacAccessibleEnum(DuringAnalysisAccess access, String enumName) {
        String className = "com.sun.glass.ui.mac.MacAccessible$" + enumName;
        registerClassFields(access, className, false, "ptr");
        JNIRuntimeAccess.register(method(access, className, "values"));
    }

    static void registerGlassMac(DuringAnalysisAccess access) {
        registerStaticLib(access, "glass", "-framework AppKit", "-framework QuartzCore");
        Stream.of("MacAttribute", "MacAction", "MacRole", "MacSubrole", "MacNotification", "MacOrientation", "MacText")
                .forEach(name -> registerMacAccessibleEnum(access, name));
        registerClassFields(access, "com.sun.glass.ui.mac.MacVariant", true,
                "type", "longArray", "intArray", "stringArray", "variantArray", "int1", "int2", "long1",
                "float1", "float2", "float3", "float4", "double1", "string", "location", "length", "key");
        JNIRuntimeAccess.register(
                clazz(access, "com.sun.glass.ui.EventLoop"),
                clazz(access, "com.sun.glass.ui.Menu"),
                clazz(access, "com.sun.glass.ui.MenuItem$Callback"));
        JNIRuntimeAccess.register(
                fields(access, "com.sun.glass.ui.mac.MacMenuDelegate", "menu"));
        JNIRuntimeAccess.register(
                fields(access, "com.sun.glass.ui.Pixels", "width", "height", "scalex", "scaley"));
        JNIRuntimeAccess.register(
                method(access, "java.lang.Boolean", "booleanValue"),
                method(access, "java.lang.Boolean", "getBoolean", String.class),
                constructor(access, "java.lang.Integer", int.class),
                method(access, "java.lang.Integer", "intValue"),
                method(access, "java.lang.Long", "longValue"),
                constructor(access, "java.lang.String", byte[].class, String.class),
                constructor(access, "java.io.File", String.class),
                constructor(access, "java.io.File", File.class, String.class),
                method(access, "com.sun.glass.ui.Application", "setName", String.class),
                method(access, "com.sun.glass.ui.Application", "getScaleFactor",
                        int.class, int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.Application", "enterNestedEventLoop"),
                method(access, "com.sun.glass.ui.Application", "leaveNestedEventLoop", Object.class),
                method(access, "com.sun.glass.ui.Application", "notifyWillFinishLaunching"),
                method(access, "com.sun.glass.ui.Application", "notifyDidFinishLaunching"),
                method(access, "com.sun.glass.ui.Application", "notifyWillBecomeActive"),
                method(access, "com.sun.glass.ui.Application", "notifyDidBecomeActive"),
                method(access, "com.sun.glass.ui.Application", "notifyWillResignActive"),
                method(access, "com.sun.glass.ui.Application", "notifyDidResignActive"),
                method(access, "com.sun.glass.ui.Application", "notifyWillHide"),
                method(access, "com.sun.glass.ui.Application", "notifyDidHide"),
                method(access, "com.sun.glass.ui.Application", "notifyWillUnhide"),
                method(access, "com.sun.glass.ui.Application", "notifyDidUnhide"),
                method(access, "com.sun.glass.ui.Application", "notifyWillQuit"),
                method(access, "com.sun.glass.ui.Application", "notifyOpenFiles", String[].class),
                method(access, "com.sun.glass.ui.Application", "createPixels",
                        int.class, int.class, int[].class, float.class, float.class),
                method(access, "com.sun.glass.ui.mac.MacApplication", "notifyApplicationDidTerminate"),
                method(access, "com.sun.glass.ui.mac.MacApplication", "setEventThread"),
                method(access, "com.sun.glass.ui.mac.MacCommonDialogs", "isFileNSURLEnabled"),
                constructor(access, "com.sun.glass.ui.EventLoop"),
                method(access, "com.sun.glass.ui.EventLoop", "enter"),
                method(access, "com.sun.glass.ui.EventLoop", "leave", Object.class),
                method(access, "com.sun.glass.ui.mac.MacGestureSupport", "rotateGesturePerformed",
                        clazz(access, "com.sun.glass.ui.View"), int.class, int.class,
                        int.class, int.class, int.class, float.class),
                method(access, "com.sun.glass.ui.mac.MacGestureSupport", "scrollGesturePerformed",
                        clazz(access, "com.sun.glass.ui.View"), int.class, int.class, int.class,
                        int.class, int.class, int.class, float.class, float.class),
                method(access, "com.sun.glass.ui.mac.MacGestureSupport", "swipeGesturePerformed",
                        clazz(access, "com.sun.glass.ui.View"),
                        int.class, int.class, int.class, int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.mac.MacGestureSupport", "magnifyGesturePerformed",
                        clazz(access, "com.sun.glass.ui.View"),
                        int.class, int.class, int.class, int.class, int.class, float.class),
                method(access, "com.sun.glass.ui.mac.MacGestureSupport", "gestureFinished",
                        clazz(access, "com.sun.glass.ui.View"),
                        int.class, int.class, int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.mac.MacGestureSupport", "notifyBeginTouchEvent",
                        clazz(access, "com.sun.glass.ui.View"), int.class, int.class),
                method(access, "com.sun.glass.ui.mac.MacGestureSupport", "notifyNextTouchEvent",
                        clazz(access, "com.sun.glass.ui.View"), int.class, long.class, float.class, float.class),
                method(access, "com.sun.glass.ui.mac.MacGestureSupport", "notifyEndTouchEvent",
                        clazz(access, "com.sun.glass.ui.View")),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityAttributeNames"),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityIsAttributeSettable", long.class),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityAttributeValue", long.class),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityActionNames"),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityActionDescription", long.class),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityIndexOfChild", long.class),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityFocusedUIElement"),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityIsIgnored"),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityHitTest", float.class, float.class),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityPerformAction", long.class),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityParameterizedAttributeNames"),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityAttributeValueForParameter", long.class, long.class),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilitySetValue", long.class, long.class),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityArrayAttributeCount", long.class),
                method(access, "com.sun.glass.ui.mac.MacAccessible", "accessibilityArrayAttributeValues", long.class, int.class, int.class),
                method(access, "com.sun.glass.ui.Menu", "notifyMenuOpening"),
                method(access, "com.sun.glass.ui.Menu", "notifyMenuClosed"),
                method(access, "com.sun.glass.ui.MenuItem$Callback", "action"),
                method(access, "com.sun.glass.ui.MenuItem$Callback", "validate"),
                constructor(access, "com.sun.glass.ui.mac.MacFileNSURL", String.class, long.class),
                constructor(access, "com.sun.glass.ui.CommonDialogs$FileChooserResult", List.class,
                        clazz(access, "com.sun.glass.ui.CommonDialogs$ExtensionFilter")),
                method(access, "com.sun.glass.ui.View", "notifyDragEnd", int.class),
                method(access, "com.sun.glass.ui.mac.MacView", "notifyInputMethodMac",
                        String.class, int.class, int.class, int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.mac.MacView", "notifyResize", int.class, int.class),
                method(access, "com.sun.glass.ui.mac.MacView", "notifyMouse", int.class, int.class,
                        int.class, int.class, int.class, int.class, int.class, boolean.class, boolean.class),
                method(access, "com.sun.glass.ui.mac.MacWindow", "notifyMove", int.class, int.class, boolean.class),
                method(access, "com.sun.glass.ui.mac.MacWindow", "notifyResize", int.class, int.class, int.class));
    }

    static void registerGlassWin(DuringAnalysisAccess access) {
        registerStaticLib(access, "glass", "comdlg32", "dwmapi", "imm32", "uiautomationcore", "urlmon", "winmm");
        JNIRuntimeAccess.register(Class.class, Throwable.class,
                clazz(access, "com.sun.glass.ui.Application"));
        registerClassFields(access, "com.sun.glass.ui.win.WinVariant", false,
                "vt", "lVal", "punkVal", "fltVal", "dblVal", "boolVal", "bstrVal", "pDblVal");
        JNIRuntimeAccess.register(fields(access, "com.sun.glass.ui.Clipboard", "name"));
        JNIRuntimeAccess.register(fields(access, "com.sun.glass.ui.win.WinSystemClipboard", "ptr"));
        JNIRuntimeAccess.register(
                method(access, "java.lang.String", "concat", String.class),
                method(access, "com.sun.glass.ui.Application", "notifyThemeChanged", String.class),
                method(access, "com.sun.glass.ui.CommonDialogs", "createFileChooserResult",
                        String[].class, clazz(access, "[Lcom.sun.glass.ui.CommonDialogs$ExtensionFilter;"), int.class),
                method(access, "com.sun.glass.ui.Cursor", "getType"),
                method(access, "com.sun.glass.ui.Cursor", "getNativeCursor"),
                method(access, "com.sun.glass.ui.View", "getWidth"),
                method(access, "com.sun.glass.ui.View", "getHeight"),
                method(access, "com.sun.glass.ui.Clipboard", "contentChanged"),
                method(access, "com.sun.glass.ui.Clipboard", "actionPerformed", int.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetPatternProvider", int.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_HostRawElementProvider"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetPropertyValue", int.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "Navigate", int.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_BoundingRectangle"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_FragmentRoot"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetEmbeddedFragmentRoots"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetRuntimeId"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetFocus"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "SetFocus"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "Invoke"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "Select"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetSelection"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_IsSelected"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_SelectionContainer"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_CanSelectMultiple"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_IsSelectionRequired"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "AddToSelection"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "RemoveFromSelection"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "AdviseEventAdded", int.class, long.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "AdviseEventRemoved", int.class, long.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_Value"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_IsReadOnly"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_Maximum"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_Minimum"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_SmallChange"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_LargeChange"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "SetValue", double.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "SetValueString", String.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_ValueString"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetVisibleRanges"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "RangeFromChild", long.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "RangeFromPoint", double.class, double.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_DocumentRange"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_SupportedTextSelection"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_ColumnCount"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_RowCount"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetItem", int.class, int.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_Column"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_ColumnSpan"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_ContainingGrid"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_Row"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_RowSpan"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetColumnHeaders"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetRowHeaders"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetRowHeaderItems"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_RowOrColumnMajor"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "GetColumnHeaderItems"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_ToggleState"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "Toggle"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "Expand"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "Collapse"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_ExpandCollapseState"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_CanMove"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_CanResize"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_CanRotate"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "Move", double.class, double.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "Resize", double.class, double.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "Rotate", double.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "Scroll", int.class, int.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "SetScrollPercent", double.class, double.class),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_HorizontallyScrollable"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_HorizontalScrollPercent"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_HorizontalViewSize"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_VerticallyScrollable"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_VerticalScrollPercent"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "get_VerticalViewSize"),
                method(access, "com.sun.glass.ui.win.WinAccessible", "ScrollIntoView"),
                method(access, "com.sun.glass.ui.win.WinDnDClipboard", "getInstance"),
                method(access, "com.sun.glass.ui.win.WinDnDClipboard", "setSourceSupportedActions", int.class),
                method(access, "com.sun.glass.ui.win.WinDnDClipboard", "setDragButton", int.class),
                method(access, "com.sun.glass.ui.win.WinDnDClipboard", "getDragButton"),
                method(access, "com.sun.glass.ui.win.WinSystemClipboard", "fosSerialize", String.class, long.class),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "Clone"),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "Compare",
                        clazz(access, "com.sun.glass.ui.win.WinTextRangeProvider")),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "CompareEndpoints",
                        int.class, clazz(access, "com.sun.glass.ui.win.WinTextRangeProvider"), int.class),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "ExpandToEnclosingUnit", int.class),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "FindAttribute",
                        int.class, clazz(access, "com.sun.glass.ui.win.WinVariant"), boolean.class),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "FindText",
                        String.class, boolean.class, boolean.class),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "GetAttributeValue", int.class),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "GetBoundingRectangles"),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "GetEnclosingElement"),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "GetText", int.class),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "Move", int.class, int.class),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "MoveEndpointByUnit", int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "MoveEndpointByRange",
                        int.class, clazz(access, "com.sun.glass.ui.win.WinTextRangeProvider"), int.class),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "Select"),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "AddToSelection"),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "RemoveFromSelection"),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "GetChildren"),
                method(access, "com.sun.glass.ui.win.WinTextRangeProvider", "ScrollIntoView", boolean.class),
                method(access, "com.sun.glass.ui.win.WinWindow", "notifyResize", int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.win.WinWindow", "notifyMoving",
                        int.class, int.class, int.class, int.class, float.class, float.class,
                        int.class, int.class, int.class, int.class, int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.win.WinMenuImpl", "notifyCommand",
                        clazz(access, "com.sun.glass.ui.Window"), int.class),
                method(access, "com.sun.glass.ui.win.WinGestureSupport", "gesturePerformed",
                        clazz(access, "com.sun.glass.ui.View"), int.class, boolean.class, boolean.class,
                        int.class, int.class, int.class, int.class, float.class, float.class,
                        float.class, float.class, float.class, float.class, float.class),
                method(access, "com.sun.glass.ui.win.WinGestureSupport", "inertiaGestureFinished",
                        clazz(access, "com.sun.glass.ui.View")),
                method(access, "com.sun.glass.ui.win.WinGestureSupport", "notifyBeginTouchEvent",
                        clazz(access, "com.sun.glass.ui.View"), int.class, boolean.class, int.class),
                method(access, "com.sun.glass.ui.win.WinGestureSupport", "notifyNextTouchEvent",
                        clazz(access, "com.sun.glass.ui.View"), int.class, long.class,
                        int.class, int.class, int.class, int.class),
                method(access, "com.sun.glass.ui.win.WinGestureSupport", "notifyEndTouchEvent",
                        clazz(access, "com.sun.glass.ui.View")));

        if (JavaVersionUtil.JAVA_SPEC != 17 || GraalServices.getJavaUpdateVersion() != 1) {
            JNIRuntimeAccess.register(method(access, "com.sun.glass.ui.win.WinView", "notifyResize", int.class, int.class));
        }
    }

    static void registerFonts(DuringAnalysisAccess access) {
        JNIRuntimeAccess.register(
                constructor(access, "com.sun.javafx.geom.Path2D", int.class, byte[].class, int.class, float[].class, int.class),
                constructor(access, "java.util.ArrayList", int.class),
                method(access, "java.util.HashMap", "get", Object.class),
                method(access, "java.util.HashMap", "put", Object.class, Object.class),
                method(access, "java.util.HashMap", "containsKey", Object.class),
                method(access, "java.util.ArrayList", "add", Object.class),
                method(access, "java.lang.String", "toLowerCase", Locale.class));
    }

    static void registerFontsLinux(DuringAnalysisAccess access) {
        registerStaticLib(access, "javafx_font");
        registerStaticLib(access, "javafx_font_freetype");
        registerStaticLib(access, "javafx_font_pango");

        registerClassFields(access, "com.sun.javafx.font.freetype.FT_GlyphSlotRec", true,
                "metrics", "linearHoriAdvance", "linearVertAdvance", "advance_x", "advance_y",
                "format", "bitmap", "bitmap_left", "bitmap_top");
        registerClassFields(access, "com.sun.javafx.font.FontConfigManager$FcCompFont", true,
                "fcName", "firstFont", "allFonts");
        registerClassFields(access, "com.sun.javafx.font.FontConfigManager$FontConfigFont", true,
                "familyName", "fullName", "styleStr", "fontFile");
        registerClassFields(access, "com.sun.javafx.font.freetype.PangoGlyphString", true,
                "offset", "length", "font", "num_chars", "glyphs", "num_glyphs", "widths", "log_clusters");

        JNIRuntimeAccess.register(HashMap.class,
                clazz(access, "com.sun.javafx.geom.Path2D"));
        JNIRuntimeAccess.register(fields(access, "com.sun.javafx.font.freetype.FT_Matrix",
                "xx", "xy", "yx", "yy"));
        JNIRuntimeAccess.register(fields(access, "com.sun.javafx.font.freetype.FT_Bitmap",
                "rows", "width", "pitch", "buffer", "num_grays", "pixel_mode", "palette_mode", "palette"));
        JNIRuntimeAccess.register(fields(access, "com.sun.javafx.font.freetype.FT_Glyph_Metrics",
                "width", "height", "horiAdvance", "vertAdvance", "horiBearingX", "horiBearingY", "vertBearingX", "vertBearingY"));
    }

    static void registerFontsMac(DuringAnalysisAccess access) {
        registerStaticLib(access, "javafx_font");
        registerClassFields(access, "com.sun.javafx.font.coretext.CGAffineTransform", true,
                "a", "b", "c", "d", "tx", "ty");
        registerClassFields(access, "com.sun.javafx.font.coretext.CGPoint", true,
                "x", "y");
        registerClassFields(access, "com.sun.javafx.font.coretext.CGRect", true,
                "origin", "size");
        registerClassFields(access, "com.sun.javafx.font.coretext.CGSize", true,
                "width", "height");
        JNIRuntimeAccess.register(RuntimeException.class, List.class, ArrayList.class, File.class);
    }

    static void registerFontsWin(DuringAnalysisAccess access) {
        registerStaticLib(access, "javafx_font", "gdi32");
        registerClassFields(access, "com.sun.javafx.font.directwrite.DWRITE_GLYPH_METRICS", true,
                "leftSideBearing", "rightSideBearing", "topSideBearing", "bottomSideBearing",
                "advanceWidth", "advanceHeight", "verticalOriginY");
        registerClassFields(access, "com.sun.javafx.font.directwrite.DWRITE_MATRIX", true,
                "m11", "m12", "m21", "m22", "dx", "dy");
        registerClassFields(access, "com.sun.javafx.font.directwrite.DWRITE_GLYPH_RUN", true,
                "fontFace", "fontEmSize", "glyphIndices", "glyphAdvances", "advanceOffset",
                "ascenderOffset", "isSideways", "bidiLevel");
        registerClassFields(access, "com.sun.javafx.font.directwrite.DWRITE_SCRIPT_ANALYSIS", true,
                "script", "shapes");
        registerClassFields(access, "com.sun.javafx.font.directwrite.RECT", true,
                "top", "left", "right", "bottom");
        registerClassFields(access, "com.sun.javafx.font.directwrite.D2D1_PIXEL_FORMAT", true,
                "format", "alphaMode");
        registerClassFields(access, "com.sun.javafx.font.directwrite.D2D1_RENDER_TARGET_PROPERTIES", true,
                "type", "pixelFormat", "dpiX", "dpiY", "usage", "minLevel");
        registerClassFields(access, "com.sun.javafx.font.directwrite.D2D1_COLOR_F", true,
                "r", "g", "b", "a");
        registerClassFields(access, "com.sun.javafx.font.directwrite.D2D1_POINT_2F", true,
                "x", "y");
        registerClassFields(access, "com.sun.javafx.font.directwrite.D2D1_MATRIX_3X2_F", true,
                "_11", "_12", "_21", "_22", "_31", "_32");
    }

    static void registerIIO(DuringAnalysisAccess access) {
        registerStaticLib(access, "javafx_iio");

        JNIRuntimeAccess.register(
                method(access, "java.io.InputStream", "read", byte[].class, int.class, int.class),
                method(access, "java.io.InputStream", "skip", long.class),
                method(access, "com.sun.javafx.iio.common.ImageLoaderImpl", "emitWarning", String.class),
                method(access, "com.sun.javafx.iio.jpeg.JPEGImageLoader", "setInputAttributes",
                        int.class, int.class, int.class, int.class, int.class, byte[].class),
                method(access, "com.sun.javafx.iio.jpeg.JPEGImageLoader", "setOutputAttributes",
                        int.class, int.class),
                method(access, "com.sun.javafx.iio.jpeg.JPEGImageLoader", "updateImageProgress", int.class));
    }

    static void registerPrismD3D(DuringAnalysisAccess access) {
        registerStaticLib(access, "prism_d3d");
        JNIRuntimeAccess.register(fields(access, "com.sun.prism.d3d.D3DDriverInformation",
                "maxSamples", "deviceDescription", "deviceName", "driverName", "warningMessage",
                "vendorID", "deviceID", "subSysId", "product", "version", "subVersion", "buildID",
                "osMajorVersion", "osMinorVersion", "osBuildNumber", "psVersionMajor", "psVersionMinor"));
        JNIRuntimeAccess.register(fields(access, "com.sun.prism.impl.PrismSettings",
                "forceGPU", "isVsyncEnabled", "disableD3D9Ex", "verbose"));
    }

    static void registerPrismES2(DuringAnalysisAccess access) {
        String openGL = Platform.includedIn(Platform.DARWIN.class) ? "-framework OpenGL" : "GL";
        registerStaticLib(access, "prism_es2", openGL);
    }

    static void registerPrismSW(DuringAnalysisAccess access) {
        registerStaticLib(access, "prism_sw");
    }

    static void registerSwing(DuringAnalysisAccess access) {
        registerStaticLib(access, "prism_common");
        JNIRuntimeAccess.register(
                method(access, "jdk.swing.interop.LightweightFrameWrapper", "overrideNativeWindowHandle",
                        long.class, Runnable.class));
    }

    static void registerDecoraSSE(DuringAnalysisAccess access) {
        registerStaticLib(access, "decora_sse");
    }

    private static void registerGlassIos(DuringAnalysisAccess access) {
        JNIRuntimeAccess.register(Object.class, Thread.class, Class.class, Integer.class, Long.class, Boolean.class,
                Map.class, Vector.class, RuntimeException.class,
                clazz(access, "com.sun.glass.ui.Screen"),
                clazz(access, "com.sun.glass.ui.View"),
                clazz(access, "com.sun.glass.ui.ios.IosView"),
                clazz(access, "com.sun.glass.ui.Window"),
                clazz(access, "com.sun.glass.ui.ios.IosWindow"),
                clazz(access, "com.sun.glass.ui.Pixels"),
                clazz(access, "com.sun.glass.ui.ios.IosPixels"),
                clazz(access, "com.sun.glass.ui.ios.IosCursor"),
                clazz(access, "com.sun.glass.ui.ios.IosGestureSupport"));
    }

    private static void registerIioIos(DuringAnalysisAccess access) {
        JNIRuntimeAccess.register(InputStream.class);
    }

    private static void registerFontAndroid(DuringAnalysisAccess access) {
        JNIRuntimeAccess.register(RuntimeException.class,
                clazz(access, "com.sun.glass.ui.monocle.AndroidInputDeviceRegistry"),
                clazz(access, "com.sun.glass.ui.monocle.MonocleWindowManager"),
                clazz(access, ""));
    }

    private static void registerClassFields(DuringAnalysisAccess access, String className, boolean includeNullaryCtor, String... fieldNames) {
        JNIRuntimeAccess.register(clazz(access, className));
        JNIRuntimeAccess.register(fields(access, className, fieldNames));
        if (includeNullaryCtor) {
            JNIRuntimeAccess.register(constructor(access, className));
        }
    }

    private static void registerStaticLib(DuringAnalysisAccess access, String name, String... dynamicDeps) {
        NativeLibrarySupport.singleton().preregisterUninitializedBuiltinLibrary(name);
        NativeLibraries nativeLibraries = ((FeatureImpl.DuringAnalysisAccessImpl) access).getNativeLibraries();
        nativeLibraries.addStaticJniLibrary(name);
        for (String dep : dynamicDeps) {
            nativeLibraries.addDynamicNonJniLibrary(dep);
        }
    }
}
