/*
 * Copyright (c) 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package org.graalvm.compiler.replacements.nodes;

import org.graalvm.compiler.core.common.Stride;
import org.graalvm.compiler.core.common.StrideUtil;
import org.graalvm.compiler.core.common.spi.ForeignCallDescriptor;
import org.graalvm.compiler.debug.GraalError;
import org.graalvm.compiler.nodes.ValueNode;
import org.graalvm.compiler.replacements.NodeStrideUtil;
import org.graalvm.word.Pointer;

import jdk.vm.ci.meta.JavaKind;

public final class ArrayEqualsForeignCalls {
    private static final ForeignCallDescriptor STUB_LONG_ARRAY_EQUALS = foreignCallDescriptorPointers("longArraysEquals");
    private static final ForeignCallDescriptor STUB_FLOAT_ARRAY_EQUALS = foreignCallDescriptorPointers("floatArraysEquals");
    private static final ForeignCallDescriptor STUB_DOUBLE_ARRAY_EQUALS = foreignCallDescriptorPointers("doubleArraysEquals");
    private static final ForeignCallDescriptor STUB_REGION_EQUALS_DYNAMIC_STRIDES = ForeignCalls.pureFunctionForeignCallDescriptor(
                    "arrayRegionEqualsDynamicStrides", boolean.class, Object.class, long.class, Object.class, long.class, int.class, int.class);
    /**
     * CAUTION: the ordering here is important: entries 0-9 must match the indices generated by
     * {@link NodeStrideUtil#getDirectStubCallIndex(ValueNode, Stride, Stride)}.
     *
     * @see #getRegionEqualsStub(ArrayRegionEqualsNode)
     */
    public static final ForeignCallDescriptor[] STUBS = {
                    foreignCallDescriptor("arrayRegionEqualsS1S1"),
                    foreignCallDescriptor("arrayRegionEqualsS1S2"),
                    foreignCallDescriptor("arrayRegionEqualsS1S4"),
                    foreignCallDescriptor("arrayRegionEqualsS2S1"),
                    foreignCallDescriptor("arrayRegionEqualsS2S2"),
                    foreignCallDescriptor("arrayRegionEqualsS2S4"),
                    foreignCallDescriptor("arrayRegionEqualsS4S1"),
                    foreignCallDescriptor("arrayRegionEqualsS4S2"),
                    foreignCallDescriptor("arrayRegionEqualsS4S4"),

                    STUB_REGION_EQUALS_DYNAMIC_STRIDES,

                    STUB_LONG_ARRAY_EQUALS,
    };

    public static final ForeignCallDescriptor[] STUBS_AMD64 = {
                    STUB_FLOAT_ARRAY_EQUALS,
                    STUB_DOUBLE_ARRAY_EQUALS,
    };

    private static ForeignCallDescriptor foreignCallDescriptor(String name) {
        return ForeignCalls.pureFunctionForeignCallDescriptor(name, boolean.class, Object.class, long.class, Object.class, long.class, int.class);
    }

    private static ForeignCallDescriptor foreignCallDescriptorPointers(String name) {
        return ForeignCalls.pureFunctionForeignCallDescriptor(name, boolean.class, Pointer.class, long.class, Pointer.class, long.class, int.class);
    }

    public static ForeignCallDescriptor getArrayEqualsStub(ArrayEqualsNode arrayEqualsNode) {
        JavaKind kind = arrayEqualsNode.getKind();
        switch (kind) {
            case Boolean:
            case Byte:
            case Char:
            case Short:
            case Int:
                return STUBS[StrideUtil.getDirectStubCallIndex(Stride.fromJavaKind(kind).log2, Stride.fromJavaKind(kind).log2)];
            case Long:
                return STUB_LONG_ARRAY_EQUALS;
            case Float:
                return STUB_FLOAT_ARRAY_EQUALS;
            case Double:
                return STUB_DOUBLE_ARRAY_EQUALS;
            default:
                return null;
        }
    }

    public static ForeignCallDescriptor getRegionEqualsStub(ArrayRegionEqualsNode regionEqualsNode) {
        int directStubCallIndex = regionEqualsNode.getDirectStubCallIndex();
        GraalError.guarantee(-1 <= directStubCallIndex && directStubCallIndex < 9, "invalid direct stub call index");
        return directStubCallIndex < 0 ? STUB_REGION_EQUALS_DYNAMIC_STRIDES : STUBS[directStubCallIndex];
    }
}
