/*
 * Copyright (c) 2019, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package org.graalvm.compiler.truffle.runtime.hotspot.libgraal;

/**
 * Encapsulates a handle to an object in the libgraal heap where the handle is only valid within the
 * scope of a {@link TruffleFromLibGraalEntryPoints} method.
 */
class LibGraalScopedHandle implements AutoCloseable {

    /**
     * Handle to a libgraal object.
     */
    private long handle;

    private final Class<?> handleType;

    /**
     * Creates a new {@link LibGraalScopedHandle}.
     *
     * @param handle handle to a libgraal object
     */
    LibGraalScopedHandle(long handle, Class<?> handleType) {
        this.handle = handle;
        this.handleType = handleType;
    }

    long getHandle() {
        if (handle == 0L) {
            throw new IllegalStateException("Using invalid handle to " + handleType.getName() + " object in libgraal heap");
        }
        return handle;
    }

    @Override
    public void close() {
        handle = 0L;
    }

    @Override
    public String toString() {
        return String.format("%s@0x%x", handleType.getName(), handle);
    }
}
