package com.oracle.objectfile.macho;

import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.oracle.objectfile.dwarf.DwarfARangesSectionImpl;
import com.oracle.objectfile.dwarf.DwarfAbbrevSectionImpl;
import com.oracle.objectfile.dwarf.DwarfDebugInfo;
import com.oracle.objectfile.dwarf.DwarfFrameSectionImplAArch64;
import com.oracle.objectfile.dwarf.DwarfFrameSectionImplX86_64;
import com.oracle.objectfile.dwarf.DwarfInfoSectionImpl;
import com.oracle.objectfile.dwarf.DwarfLineSectionImpl;
import com.oracle.objectfile.dwarf.DwarfSectionImpl;
import com.oracle.objectfile.dwarf.DwarfStrSectionImpl;
import com.oracle.objectfile.macho.dsym.DSYMLocSectionImpl;

/**
 * A class that models the debug info in an organization that facilitates generation of the required
 * DWARF sections. It groups common data and behaviours for use by the various subclasses of class
 * DwarfSectionImpl that take responsibility for generating content for a specific section type.
 */
public class DSYMDebugInfo extends DwarfDebugInfo {

    private final MachOCpuType cpuType;

    /**
     * Register used to hold the heap base.
     */
    private final byte heapbaseRegister;

    @SuppressWarnings("this-escape")
    public DSYMDebugInfo(MachOCpuType cpuType, ByteOrder byteOrder, Consumer<DwarfSectionImpl> debugSections) {
        super(byteOrder);
        this.cpuType = cpuType;

        debugSections.accept(new DwarfStrSectionImpl(this));
        debugSections.accept(new DwarfAbbrevSectionImpl(this));
        debugSections.accept(new DwarfInfoSectionImpl(this));
        debugSections.accept(new DwarfARangesSectionImpl(this));
        debugSections.accept(new DwarfLineSectionImpl(this));
        // these sections have some small differences in content generation compared to base
        debugSections.accept(new DSYMLocSectionImpl(this));

        if (cpuType.equals(MachOCpuType.ARM64)) {
            debugSections.accept(new DwarfFrameSectionImplAArch64(this));
            this.heapbaseRegister = rheapbase_aarch64;
        } else {
            debugSections.accept(new DwarfFrameSectionImplX86_64(this));
            this.heapbaseRegister = rheapbase_x86;
        }
    }

    @Override
    public boolean isAarch64() {
        return cpuType.equals(MachOCpuType.ARM64);
    }

    @Override
    public boolean isAMD64() {
        return cpuType.equals(MachOCpuType.X86_64);
    }

    @Override
    public String textSectionName() {
        return DSYMSectionName.TEXT_SECTION.value;
    }

    @Override
    public String lineSectionName() {
        return DSYMSectionName.DW_LINE_SECTION.value;
    }

    @Override
    public String strSectionName() {
        return DSYMSectionName.DW_STR_SECTION.value;
    }

    @Override
    public String  locSectionName() {
        return DSYMSectionName.DW_LOC_SECTION.value;
    }

    @Override
    public String arangesSectionName() {
        return DSYMSectionName.DW_ARANGES_SECTION.value;
    }

    @Override
    public String frameSectionName() {
        return DSYMSectionName.DW_FRAME_SECTION.value;
    }

    @Override
    public String abbrevSectionName() {
        return DSYMSectionName.DW_ABBREV_SECTION.value;
    }

    @Override
    public String infoSectionName() {
        return DSYMSectionName.DW_INFO_SECTION.value;
    }

    @Override
    public boolean layoutDependsOnVaddr() {
        return true;
    }

    @Override
    public boolean isDebugSectionName(String name) {
        return name.startsWith("__debug");
    }

    /**
     * Various ELF sections created by GraalVM including all debug info sections. The enum sequence
     * starts with the text section (not defined in the DWARF spec and not created by debug info
     * code).
     */
    enum DSYMSectionName {
        TEXT_SECTION("__text"),
        DW_STR_SECTION("__debug_str"),
        DW_LINE_SECTION("__debug_line"),
        DW_FRAME_SECTION("__debug_frame"),
        DW_ABBREV_SECTION("__debug_abbrev"),
        DW_INFO_SECTION("__debug_info"),
        DW_LOC_SECTION("__debug_loc"),
        DW_ARANGES_SECTION("__debug_aranges");

        private final String value;

        DSYMSectionName(String s) {
            value = s;
        }

        public String value() {
            return value;
        }

        protected static final List<String> debugSections = Arrays.stream(DSYMSectionName.values()).filter(e -> e != DSYMSectionName.TEXT_SECTION).map(e -> e.value()).collect(Collectors.toList());
    }

    public static final List<String> debugSectionList() {
        return DSYMSectionName.debugSections;
    }

    @Override
    public long relocatableLong(long l) {
        return l;
    }

    @Override
    public int relocatableInt(int i) {
        return i;
    }

    public byte getHeapbaseRegister() {
        return heapbaseRegister;
    }
}
