package com.oracle.svm.hosted.javafx;

import com.oracle.svm.core.jdk.JNIRegistrationUtil;
import org.graalvm.nativeimage.Platform;
import org.graalvm.nativeimage.hosted.Feature.DuringAnalysisAccess;
import org.graalvm.nativeimage.hosted.RuntimeReflection;

import java.lang.reflect.Executable;
import java.util.Arrays;
import java.util.function.Function;
import java.util.stream.Stream;

final class JavaFXReflection extends JNIRegistrationUtil {

    static void registerJavaFX(DuringAnalysisAccess access) {
        registerReflectionClasses(access, Stream.of(
                "Scene", "SubScene", "Node", "Parent", "Group",
                "Camera", "PerspectiveCamera", "ParallelCamera", "LightBase", "AmbientLight", "PointLight",
                "canvas.Canvas", "effect.Effect", "image.Image", "image.ImageView", "input.Clipboard",
                "input.Dragboard", "input.TouchPoint", "layout.Pane", "layout.Region", "paint.Material",
                "shape.Arc", "shape.ArcTo", "shape.Box", "shape.Circle", "shape.ClosePath", "shape.CubicCurve",
                "shape.CubicCurveTo", "shape.Cylinder", "shape.Ellipse", "shape.HLineTo", "shape.Line",
                "shape.LineTo", "shape.Mesh", "shape.MeshView", "shape.MoveTo", "shape.Path", "shape.PathElement",
                "shape.Polygon", "shape.Polyline", "shape.QuadCurve", "shape.QuadCurveTo", "shape.Rectangle",
                "shape.Shape", "shape.Shape3D", "shape.Sphere", "shape.SVGPath", "shape.TriangleMesh",
                "shape.VLineTo", "text.Font", "text.Text", "transform.Transform")
                .map("javafx.scene."::concat));
        registerReflectionClasses(access, Stream.of("Stage", "Window", "PopupWindow")
                .map("javafx.stage."::concat));
        registerReflectionClasses(access,
                "javafx.animation.KeyValue",
                "com.sun.javafx.application.LauncherImpl",
                "com.sun.javafx.stage.EmbeddedWindow",
                "com.sun.javafx.tk.quantum.OverlayWarning");

        Class<?> tkClass = clazz(access, "com.sun.javafx.tk.quantum.QuantumToolkit");
        RuntimeReflection.register(tkClass);
        RuntimeReflection.registerForReflectiveInstantiation(tkClass);

        RuntimeReflection.register(
                method(access, "java.nio.ByteBuffer", "order", java.nio.ByteOrder.class),
                method(access, "java.nio.ByteOrder", "nativeOrder"),
                method(access, "com.sun.javafx.application.LauncherImpl", "launchApplication",
                        String.class, String.class, String[].class));
    }

    static void registerGlass(DuringAnalysisAccess access) {
        registerReflectionClasses(access, Stream.of(
                "Application", "EventLoop", "Menu", "MenuItem$Callback", "Screen", "Size", "View",
                "CommonDialogs$ExtensionFilter", "CommonDialogs$FileChooserResult")
                .map("com.sun.glass.ui."::concat));
    }

    private static void registerPlatformFactory(DuringAnalysisAccess access, String platformName) {
        String className = String.format("com.sun.glass.ui.%s.%sPlatformFactory", platformName.toLowerCase(), platformName);
        Class<?> platformFactoryClass = clazz(access, className);
        RuntimeReflection.register(platformFactoryClass);
        RuntimeReflection.registerForReflectiveInstantiation(platformFactoryClass);
    }

    static void registerGlassGTK3(DuringAnalysisAccess access) {
        registerPlatformFactory(access, "Gtk");
    }

    static void registerGlassMac(DuringAnalysisAccess access) {
        RuntimeReflection.register(Boolean.class, Runnable.class);
        registerPlatformFactory(access, "Mac");
        registerReflectionClasses(access,
                Stream.of("Application", "CommonDialogs", "MenuBarDelegate", "Pixels", "View")
                        .map("com.sun.glass.ui.mac.Mac"::concat));
    }

    static void registerGlassWin(DuringAnalysisAccess access) {
        registerPlatformFactory(access, "Win");
        registerReflectionClasses(access,
                Stream.of("DnDClipboard", "GestureSupport")
                        .map("com.sun.glass.ui.win.Win"::concat));
    }

    private static void registerFontFactory(DuringAnalysisAccess access, String factoryClassName) {
        RuntimeReflection.register(clazz(access, factoryClassName));
        RuntimeReflection.register(method(access, factoryClassName, "getFactory"));
    }

    static void registerFontsLinux(DuringAnalysisAccess access) {
        registerFontFactory(access, "com.sun.javafx.font.freetype.FTFactory");
    }

    static void registerFontsWin(DuringAnalysisAccess access) {
        registerFontFactory(access, "com.sun.javafx.font.directwrite.DWFactory");
    }

    static void registerFontsMac(DuringAnalysisAccess access) {
        registerFontFactory(access, "com.sun.javafx.font.coretext.CTFactory");
    }

    private static void registerPrism(DuringAnalysisAccess access, String pipelineName) {
        String classPrefix = String.format("com.sun.prism.%s.%s", pipelineName.toLowerCase(), pipelineName);

        Class<?> filterContextClass = clazz(access, "com.sun.scenario.effect.FilterContext");
        Class<?> rendererClass = clazz(access, "com.sun.scenario.effect.impl.Renderer");
        Function<String, Executable> peer2ctor = name ->
                constructor(access, name, filterContextClass, rendererClass, String.class);
        registerPrismPeers(access, "com.sun.scenario.effect.impl.prism.Pr", peer2ctor,
                "Crop", "Flood", "Merge", "Reflection");
        registerPrismPeers(access, "com.sun.scenario.effect.impl.prism.ps.PPS", peer2ctor,
                "Blend_ADD", "Blend_BLUE", "Blend_COLOR_BURN", "Blend_COLOR_DODGE", "Blend_DARKEN",
                "Blend_DIFFERENCE", "Blend_EXCLUSION", "Blend_GREEN", "Blend_HARD_LIGHT", "Blend_LIGHTEN", "Blend_MULTIPLY",
                "Blend_OVERLAY", "Blend_RED", "Blend_SCREEN", "Blend_SOFT_LIGHT", "Blend_SRC_ATOP", "Blend_SRC_IN",
                "Blend_SRC_OUT", "Blend_SRC_OVER", "Brightpass", "ColorAdjust", "DisplacementMap", "Effect", "InvertMask",
                "LinearConvolve", "LinearConvolveShadow", "OneSampler", "PerspectiveTransform", "PhongLighting_DISTANT",
                "PhongLighting_POINT", "PhongLighting_SPOT", "SepiaTone", "toPSWDisplacementMap", "TwoSampler", "ZeroSampler");

        registerReflectionClasses(access,
                classPrefix + "Pipeline",
                classPrefix + "ResourceFactory",
                classPrefix + "Shader",
                "com.sun.prism.GraphicsPipeline",
                "com.sun.scenario.effect.impl.prism.PrRenderer",
                "com.sun.scenario.effect.impl.prism.ps.PPSRenderer");
        registerReflectionMethods(
                cls -> method(access, cls, "createRenderer", filterContextClass),
                Stream.of("PrRenderer", "ps.PPSRenderer")
                        .map("com.sun.scenario.effect.impl.prism."::concat)
                        .toArray(String[]::new));
        RuntimeReflection.register(
                method(access, classPrefix + "Pipeline", "getInstance"),
                method(access, "com.sun.prism.GraphicsPipeline", "getFontFactory"),
                method(access, "com.sun.prism.GraphicsPipeline", "getPipeline"));

        String[] shaderNames = Stream.of(
                "AlphaOne_Color", "AlphaOne_ImagePattern", "AlphaOne_LinearGradient", "AlphaOne_RadialGradient",
                "AlphaTexture_Color", "AlphaTexture_ImagePattern", "AlphaTexture_LinearGradient", "AlphaTexture_RadialGradient",
                "AlphaTextureDifference_Color", "AlphaTextureDifference_ImagePattern",
                "AlphaTextureDifference_LinearGradient", "AlphaTextureDifference_RadialGradient",
                "DrawCircle_Color", "DrawCircle_ImagePattern",
                "DrawCircle_LinearGradient_PAD", "DrawCircle_LinearGradient_REFLECT", "DrawCircle_LinearGradient_REPEAT",
                "DrawCircle_RadialGradient_PAD", "DrawCircle_RadialGradient_REFLECT", "DrawCircle_RadialGradient_REPEAT",
                "DrawEllipse_Color", "DrawEllipse_ImagePattern",
                "DrawEllipse_LinearGradient_PAD", "DrawEllipse_LinearGradient_REFLECT", "DrawEllipse_LinearGradient_REPEAT",
                "DrawEllipse_RadialGradient_PAD", "DrawEllipse_RadialGradient_REFLECT", "DrawEllipse_RadialGradient_REPEAT",
                "DrawPgram_Color", "DrawPgram_ImagePattern",
                "DrawPgram_LinearGradient_PAD", "DrawPgram_LinearGradient_REFLECT", "DrawPgram_LinearGradient_REPEAT",
                "DrawPgram_RadialGradient_PAD", "DrawPgram_RadialGradient_REFLECT", "DrawPgram_RadialGradient_REPEAT",
                "DrawRoundRect_Color", "DrawRoundRect_ImagePattern",
                "DrawRoundRect_LinearGradient_PAD", "DrawRoundRect_LinearGradient_REFLECT", "DrawRoundRect_LinearGradient_REPEAT",
                "DrawRoundRect_RadialGradient_PAD", "DrawRoundRect_RadialGradient_REFLECT", "DrawRoundRect_RadialGradient_REPEAT",
                "DrawSemiRoundRect_Color", "DrawSemiRoundRect_ImagePattern",
                "DrawSemiRoundRect_LinearGradient_PAD", "DrawSemiRoundRect_LinearGradient_REFLECT", "DrawSemiRoundRect_LinearGradient_REPEAT",
                "DrawSemiRoundRect_RadialGradient_PAD", "DrawSemiRoundRect_RadialGradient_REFLECT", "DrawSemiRoundRect_RadialGradient_REPEAT",
                "FillCircle_Color", "FillCircle_ImagePattern",
                "FillCircle_LinearGradient_PAD", "FillCircle_LinearGradient_REFLECT", "FillCircle_LinearGradient_REPEAT",
                "FillCircle_RadialGradient_PAD", "FillCircle_RadialGradient_REFLECT", "FillCircle_RadialGradient_REPEAT",
                "FillEllipse_Color", "FillEllipse_ImagePattern",
                "FillEllipse_LinearGradient_PAD", "FillEllipse_LinearGradient_REFLECT", "FillEllipse_LinearGradient_REPEAT",
                "FillEllipse_RadialGradient_PAD", "FillEllipse_RadialGradient_REFLECT", "FillEllipse_RadialGradient_REPEAT",
                "FillPgram_Color", "FillPgram_ImagePattern",
                "FillPgram_LinearGradient_PAD", "FillPgram_LinearGradient_REFLECT", "FillPgram_LinearGradient_REPEAT",
                "FillPgram_RadialGradient_PAD", "FillPgram_RadialGradient_REFLECT", "FillPgram_RadialGradient_REPEAT",
                "FillRoundRect_Color", "FillRoundRect_ImagePattern",
                "FillRoundRect_LinearGradient_PAD", "FillRoundRect_LinearGradient_REFLECT", "FillRoundRect_LinearGradient_REPEAT",
                "FillRoundRect_RadialGradient_PAD", "FillRoundRect_RadialGradient_REFLECT", "FillRoundRect_RadialGradient_REPEAT",
                "Mask_TextureRGB", "Mask_TextureSuper",
                "Solid_Color", "Solid_ImagePattern",
                "Solid_LinearGradient_PAD", "Solid_LinearGradient_REFLECT", "Solid_LinearGradient_REPEAT",
                "Solid_RadialGradient_PAD", "Solid_RadialGradient_REFLECT", "Solid_RadialGradient_REPEAT",
                "Solid_TextureRGB", "Solid_TextureYV12", "Solid_TextureFirstPassLCD", "Solid_TextureSecondPassLCD",
                "Texture_Color", "Texture_ImagePattern",
                "Texture_LinearGradient_PAD", "Texture_LinearGradient_REFLECT", "Texture_LinearGradient_REPEAT",
                "Texture_RadialGradient_PAD", "Texture_RadialGradient_REFLECT", "Texture_RadialGradient_REPEAT")
//                .flatMap(frag -> Stream.of(frag + "_Loader", frag + "_AlphaTest_Loader")) /// do we need AlphaTest?
                .map(frag -> "com.sun.prism.shader." + frag + "_Loader")
                .toArray(String[]::new);
        registerReflectionClasses(access, shaderNames);
        Class<?> shaderFactoryClass = clazz(access, "com.sun.prism.ps.ShaderFactory");
        registerReflectionMethods(
                cls -> method(access, cls, "loadShader", shaderFactoryClass, java.io.InputStream.class),
                shaderNames);
    }

    static void registerPrismD3D(DuringAnalysisAccess access) {
        registerPrism(access, "D3D");

        Class<?> shaderSourceClass = clazz(access, "com.sun.scenario.effect.impl.hw.d3d.D3DShaderSource");
        RuntimeReflection.register(shaderSourceClass);
        RuntimeReflection.registerForReflectiveInstantiation(shaderSourceClass);
    }

    static void registerPrismES2(DuringAnalysisAccess access) {
        registerPrism(access, "ES2");

        String factory = Platform.includedIn(Platform.LINUX.class) ? "X11GLFactory" : "MacGLFactory";
        Class<?>[] classesWithNullaryCtors = toClasses(access,
                "com.sun.prism.es2." + factory,
                "com.sun.prism.es2.ES2PhongShader",
                "com.sun.scenario.effect.impl.es2.ES2ShaderSource");
        RuntimeReflection.register(classesWithNullaryCtors);
        RuntimeReflection.registerForReflectiveInstantiation(classesWithNullaryCtors);
    }

    static void registerControls(DuringAnalysisAccess access) {
        registerReflectionClasses(access,
                "javafx.scene.control.Control",
                "javafx.scene.control.TableColumnBase",
                "com.sun.javafx.scene.control.skin.Utils");
        RuntimeReflection.register(
                method(access, "com.sun.javafx.scene.control.skin.Utils", "getResource", String.class));
    }

    static void registerIIO(DuringAnalysisAccess access) {
        registerReflectionClasses(access, "javax.imageio.ImageIO");
    }

    static void registerFXML(DuringAnalysisAccess access) {
        RuntimeReflection.register(clazz(access, "javafx.fxml.FXMLLoader"));
    }

    static void registerSwing(DuringAnalysisAccess access) {
        registerReflectionClasses(access,
                "com.sun.javafx.embed.swing.SwingFXUtilsImpl",
                "javafx.embed.swing.SwingNode",
                "jdk.swing.interop.LightweightFrameWrapper");
    }

    static void registerPrinting(DuringAnalysisAccess access) {
        registerReflectionClasses(access,
                "com.sun.prism.j2d.PrismPrintPipeline",
                "javafx.print.Printer",
                "javax.print.attribute.standard.DialogOwner",
                "sun.font.FontUtilities");
    }


    private static Class<?>[] toClasses(
            DuringAnalysisAccess access, Function<String, String> nameMapper, String... classNames)
    {
        return Arrays.stream(classNames)
                .map(name -> clazz(access, nameMapper.apply(name)))
                .toArray(Class[]::new);
    }

    private static Class<?>[] toClasses(DuringAnalysisAccess access, String... classNames) {
        return toClasses(access, Function.identity(), classNames);
    }

    private static void registerReflectionClasses(DuringAnalysisAccess access, Stream<String> classNames) {
        RuntimeReflection.register(classNames
                .map(name -> clazz(access, name))
                .toArray(Class[]::new));
    }

    private static void registerReflectionClasses(DuringAnalysisAccess access, String... classNames) {
        RuntimeReflection.register(toClasses(access, classNames));
    }

    private static void registerReflectionMethods(Function<String, Executable> mapper, String... classNames) {
        RuntimeReflection.register(
                Arrays.stream(classNames)
                        .map(mapper)
                        .toArray(Executable[]::new));
    }

    private static void registerPrismPeers(
            DuringAnalysisAccess access, String prefix, Function<String, Executable> ctorMapper, String... peerNames)
    {
        String[] peerClassNames = Arrays.stream(peerNames)
                .map(name -> prefix + name + "Peer")
                .toArray(String[]::new);
        registerReflectionClasses(access, peerClassNames);
        registerReflectionMethods(ctorMapper, peerClassNames);
    }
}
