/*
 * Copyright (c) 2024, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.graal.pointsto.flow;

import com.oracle.graal.pointsto.PointsToAnalysis;
import com.oracle.graal.pointsto.meta.AnalysisType;
import com.oracle.graal.pointsto.typestate.TypeState;
import com.oracle.graal.pointsto.util.AnalysisError;

import jdk.vm.ci.code.BytecodePosition;

/**
 * This flow represents a boolean check, i.e. a null check, type check, or primitive check that
 * converts its inputs into boolean values. Instances of this class are used as conditions for
 * {@link ConditionalFlow}.
 */
public abstract class BooleanCheckTypeFlow extends TypeFlow<BytecodePosition> {

    public BooleanCheckTypeFlow(BytecodePosition position, AnalysisType declaredType) {
        super(position, declaredType);
    }

    protected BooleanCheckTypeFlow(BooleanCheckTypeFlow original, MethodFlowsGraph methodFlows) {
        super(original, methodFlows);
    }

    protected static TypeState convertToBoolean(PointsToAnalysis bb, boolean canBeTrue, boolean canBeFalse) {
        if (canBeTrue && canBeFalse) {
            return TypeState.anyPrimitiveState();
        } else if (canBeTrue) {
            return TypeState.forBoolean(bb, true);
        } else if (canBeFalse) {
            return TypeState.forBoolean(bb, false);
        }
        return TypeState.forEmpty();
    }

    protected static TypeState convertToBoolean(PointsToAnalysis bb, TypeState trueState, TypeState falseState) {
        return convertToBoolean(bb, trueState.isNotEmpty(), falseState.isNotEmpty());
    }

    @Override
    public void addPredicated(PointsToAnalysis bb, TypeFlow<?> predicatedFlow) {
        AnalysisError.shouldNotReachHere(getClass() + " shouldn't act as a predicate.");
    }
}
