/*
 * Copyright (c) 2015, 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package jdk.graal.compiler.api.directives.test;

import jdk.graal.compiler.api.directives.GraalDirectives;
import jdk.graal.compiler.core.test.GraalCompilerTest;
import jdk.graal.compiler.graph.iterators.NodeIterable;
import jdk.graal.compiler.nodes.LoopBeginNode;
import jdk.graal.compiler.nodes.StructuredGraph;
import jdk.graal.compiler.nodes.cfg.ControlFlowGraph;
import org.junit.Assert;
import org.junit.Test;

public class IterationDirectiveTest extends GraalCompilerTest {

    public static int loopFrequencySnippet(int arg) {
        int x = arg;
        while (GraalDirectives.injectIterationCount(128, x > 1)) {
            GraalDirectives.controlFlowAnchor(); // prevent loop peeling or unrolling
            if (x % 2 == 0) {
                x /= 2;
            } else {
                x = 3 * x + 1;
            }
        }
        return x;
    }

    @Test
    public void testLoopFrequency() {
        test("loopFrequencySnippet", 7);
    }

    @Override
    protected void checkLowTierGraph(StructuredGraph graph) {
        NodeIterable<LoopBeginNode> loopBeginNodes = graph.getNodes(LoopBeginNode.TYPE);
        Assert.assertEquals("LoopBeginNode count", 1, loopBeginNodes.count());

        LoopBeginNode loopBeginNode = loopBeginNodes.first();
        ControlFlowGraph cfg = ControlFlowGraph.newBuilder(graph).computeFrequency(true).build();
        Assert.assertEquals("loop frequency of " + loopBeginNode, 128, cfg.localLoopFrequency(loopBeginNode), 0);
    }
}
