/*
 * Copyright (c) 2019, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.api.test.polyglot;

import java.util.List;

import org.graalvm.nativeimage.hosted.Feature;
import org.graalvm.nativeimage.hosted.RuntimeReflection;

import com.oracle.truffle.tck.tests.ValueAssert;

/**
 * Register all <i>named</i> inner classes under {@link #TEST_CLASSES} for reflection, so that they
 * do not need to be listed in {@code reflect-config.json}. Anonymous inner classes however must be
 * listed in {@code reflect-config.json}, as Java provides no way to list those.
 */
public class RegisterTestClassesForReflectionFeature implements Feature {

    private static final List<Class<?>> TEST_CLASSES = List.of(
                    ValueHostInteropTest.class,
                    ValueHostConversionTest.class,
                    ValueAssert.class,
                    ValueAPITest.class,
                    ValueScopingTest.class,
                    PolyglotExceptionTest.class,
                    LanguageSPIHostInteropTest.class,
                    HostAccessTest.class,
                    HostRecordAccessTest.class,
                    ExposeToGuestTest.class,
                    ContextAPITest.class,
                    GR40903Outer.class,
                    GR40903Outer.Inner.class,
                    HostObjectToStringTest.class);

    protected static void registerClass(Class<?> clazz) {
        RuntimeReflection.register(clazz);
        RuntimeReflection.register(clazz.getConstructors());
        RuntimeReflection.register(clazz.getDeclaredConstructors());
        RuntimeReflection.register(clazz.getMethods());
        RuntimeReflection.register(clazz.getDeclaredMethods());
        RuntimeReflection.register(clazz.getFields());
        RuntimeReflection.register(clazz.getDeclaredFields());
    }

    public void beforeAnalysis(BeforeAnalysisAccess access) {
        for (Class<?> testClass : TEST_CLASSES) {
            for (Class<?> innerClass : testClass.getDeclaredClasses()) {
                registerClass(innerClass);
            }
        }
    }

}
