/*
 * Copyright (c) 2013, 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package org.graalvm.visualizer.source.java.ui;

import org.graalvm.visualizer.source.Location;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;

import javax.swing.JFileChooser;
import java.awt.CardLayout;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.util.concurrent.atomic.AtomicReference;

/**
 * @author sdedic
 */
public class ProjectContentPanel extends javax.swing.JPanel implements PropertyChangeListener {
    private final Location loc;
    private final JFileChooser chooser;

    /**
     * Creates new form ProjectContentPanel
     */
    public ProjectContentPanel(Location loc, JFileChooser chooser) {
        this.loc = loc;
        this.chooser = chooser;

        initComponents();

        String spec = loc.getFile().getFileSpec();
        String simple;
        int last = spec.lastIndexOf("/");
        if (last == -1) {
            simple = spec;
        } else {
            simple = spec.substring(last + 1);
        }
        fileName.setText(simple);
        chooser.addPropertyChangeListener(this);
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        if (JFileChooser.SELECTED_FILE_CHANGED_PROPERTY.equals(evt.getPropertyName())) {
            refresh();
        }
    }


    private void notFound() {
        ((CardLayout) outcomeHolder.getLayout()).show(outcomeHolder, "missing");
    }

    private void found(String where) {
        foundIn.setText(where);
        ((CardLayout) outcomeHolder.getLayout()).show(outcomeHolder, "found");
    }

    private void refresh() {
        File f = chooser.getSelectedFile();
        if (f == null) {
            notFound();
            return;
        }
        FileObject fo = FileUtil.toFileObject(f);
        Project p = FileOwnerQuery.getOwner(fo);
        if (p == null) {
            notFound();
            return;
        }
        AtomicReference<FileObject> fRoot = new AtomicReference<>();
        if (JavaProjectLocatorUI.findSourceFile(p, loc.getFileName(), fRoot) != null) {
            FileObject root = fRoot.get();
            FileObject prjDir = p.getProjectDirectory();
            String rel = FileUtil.getRelativePath(prjDir, root);
            found(rel);
        } else {
            notFound();
        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jLabel3 = new javax.swing.JLabel();
        outcomeHolder = new javax.swing.JPanel();
        foundPanel = new javax.swing.JPanel();
        jLabel2 = new javax.swing.JLabel();
        foundIn = new javax.swing.JLabel();
        missingPanel = new javax.swing.JPanel();
        jLabel4 = new javax.swing.JLabel();
        fileName = new javax.swing.JTextField();
        jLabel1 = new javax.swing.JLabel();

        org.openide.awt.Mnemonics.setLocalizedText(jLabel3, org.openide.util.NbBundle.getMessage(ProjectContentPanel.class, "ProjectContentPanel.jLabel3.text")); // NOI18N

        outcomeHolder.setLayout(new java.awt.CardLayout());

        org.openide.awt.Mnemonics.setLocalizedText(jLabel2, org.openide.util.NbBundle.getMessage(ProjectContentPanel.class, "ProjectContentPanel.jLabel2.text")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(foundIn, org.openide.util.NbBundle.getMessage(ProjectContentPanel.class, "ProjectContentPanel.foundIn.text")); // NOI18N

        javax.swing.GroupLayout foundPanelLayout = new javax.swing.GroupLayout(foundPanel);
        foundPanel.setLayout(foundPanelLayout);
        foundPanelLayout.setHorizontalGroup(
                foundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addGroup(foundPanelLayout.createSequentialGroup()
                                .addComponent(jLabel2)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(foundIn, javax.swing.GroupLayout.DEFAULT_SIZE, 106, Short.MAX_VALUE)
                                .addContainerGap())
        );
        foundPanelLayout.setVerticalGroup(
                foundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addGroup(foundPanelLayout.createSequentialGroup()
                                .addContainerGap()
                                .addGroup(foundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                                        .addComponent(jLabel2, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                                        .addComponent(foundIn, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                                .addContainerGap(50, Short.MAX_VALUE))
        );

        outcomeHolder.add(foundPanel, "found");

        org.openide.awt.Mnemonics.setLocalizedText(jLabel4, org.openide.util.NbBundle.getMessage(ProjectContentPanel.class, "ProjectContentPanel.jLabel4.text")); // NOI18N

        javax.swing.GroupLayout missingPanelLayout = new javax.swing.GroupLayout(missingPanel);
        missingPanel.setLayout(missingPanelLayout);
        missingPanelLayout.setHorizontalGroup(
                missingPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addGroup(missingPanelLayout.createSequentialGroup()
                                .addComponent(jLabel4)
                                .addGap(0, 120, Short.MAX_VALUE))
        );
        missingPanelLayout.setVerticalGroup(
                missingPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addGroup(missingPanelLayout.createSequentialGroup()
                                .addComponent(jLabel4)
                                .addGap(0, 62, Short.MAX_VALUE))
        );

        outcomeHolder.add(missingPanel, "missing");

        fileName.setEditable(false);
        fileName.setText(org.openide.util.NbBundle.getMessage(ProjectContentPanel.class, "ProjectContentPanel.fileName.text")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(jLabel1, org.openide.util.NbBundle.getMessage(ProjectContentPanel.class, "ProjectContentPanel.jLabel1.text")); // NOI18N

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
                layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addGroup(layout.createSequentialGroup()
                                .addContainerGap()
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                        .addComponent(jLabel1)
                                        .addComponent(fileName)
                                        .addComponent(outcomeHolder, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                                .addContainerGap())
        );
        layout.setVerticalGroup(
                layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addGroup(layout.createSequentialGroup()
                                .addGap(2, 2, 2)
                                .addComponent(jLabel1)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(fileName, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(outcomeHolder, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );
    }// </editor-fold>//GEN-END:initComponents


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JTextField fileName;
    private javax.swing.JLabel foundIn;
    private javax.swing.JPanel foundPanel;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JPanel missingPanel;
    private javax.swing.JPanel outcomeHolder;
    // End of variables declaration//GEN-END:variables
}
