/*
 * Copyright (c) 2013, 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package org.graalvm.visualizer.shell.ui;

import org.graalvm.visualizer.shell.ShellUtils;
import org.graalvm.visualizer.shell.list.ScriptFolderNode;
import org.netbeans.api.settings.ConvertAsProperties;
import org.openide.awt.ActionID;
import org.openide.awt.ActionReference;
import org.openide.explorer.ExplorerManager;
import org.openide.explorer.ExplorerUtils;
import org.openide.explorer.view.BeanTreeView;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.nodes.Node;
import org.openide.util.NbBundle.Messages;
import org.openide.windows.TopComponent;

import javax.swing.ActionMap;
import javax.swing.text.DefaultEditorKit;
import java.awt.BorderLayout;
import java.io.IOException;

/**
 * Top component which displays something.
 */
@ConvertAsProperties(
        dtd = "-//org.graalvm.visualizer.shell//ScriptNavigator//EN",
        autostore = false
)
@TopComponent.Description(
        preferredID = ScriptNavigatorTopComponent.ID,
        persistenceType = TopComponent.PERSISTENCE_ALWAYS
)
@TopComponent.Registration(mode = "navigator", openAtStartup = false, position = 200)
@ActionID(category = "Window", id = "org.graalvm.visualizer.shell.ScriptNavigatorTopComponent")
@ActionReference(path = "Menu/Window", position = 97)
@TopComponent.OpenActionRegistration(
        displayName = "#CTL_ScriptNavigatorAction",
        preferredID = "ScriptNavigatorTopComponent"
)
@Messages({
        "CTL_ScriptNavigatorAction=Script List",
        "CTL_ScriptNavigatorTopComponent=Scripts Window",
        "HINT_ScriptNavigatorTopComponent=Displays user scripts"
})
public final class ScriptNavigatorTopComponent extends TopComponent
        implements ExplorerManager.Provider {
    public static final String ID = "ScriptNavigatorTopComponent"; // NOI18N

    private final ExplorerManager listMan = new ExplorerManager();
    private final BeanTreeView contentView = new BeanTreeView();

    public ScriptNavigatorTopComponent() {
        initComponents();
        setName(Bundle.CTL_ScriptNavigatorTopComponent());
        setToolTipText(Bundle.HINT_ScriptNavigatorTopComponent());
        putClientProperty(TopComponent.PROP_KEEP_PREFERRED_SIZE_WHEN_SLIDED_IN, Boolean.TRUE);

        ActionMap map = this.getActionMap();
        map.put(DefaultEditorKit.copyAction, ExplorerUtils.actionCopy(listMan));
        map.put(DefaultEditorKit.cutAction, ExplorerUtils.actionCut(listMan));
        map.put(DefaultEditorKit.pasteAction, ExplorerUtils.actionPaste(listMan));
        map.put("delete", ExplorerUtils.actionDelete(listMan, true)); // NOI18N

        associateLookup(ExplorerUtils.createLookup(listMan, map));

        ExplorerUtils.activateActions(listMan, true);
        listMan.addPropertyChangeListener((e) -> {
            final String n = e.getPropertyName();
            if (ExplorerManager.PROP_SELECTED_NODES.equals(n)) {
                setActivatedNodes((Node[]) e.getNewValue());
            }
        });

        contentView.setRootVisible(false);

        try {
            FileObject root = ShellUtils.getScriptRoot();
            DataObject droot = DataObject.find(root);
            Node orig = droot.getNodeDelegate();
            Node n = new ScriptFolderNode(orig);
            listMan.setRootContext(n);
        } catch (IOException ex) {
            // PENDING: put some error node into the view
        }
        contentPanel.add(contentView, BorderLayout.CENTER);
    }

    @Override
    public ExplorerManager getExplorerManager() {
        return listMan;
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        contentPanel = new javax.swing.JPanel();

        contentPanel.setLayout(new java.awt.BorderLayout());

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
                layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addComponent(contentPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
                layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addComponent(contentPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );
    }// </editor-fold>//GEN-END:initComponents

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel contentPanel;

    // End of variables declaration//GEN-END:variables
    @Override
    public void componentOpened() {
    }

    @Override
    public void componentClosed() {
        // TODO add custom code on component closing
    }

    void writeProperties(java.util.Properties p) {
        p.setProperty("version", "1.0");
    }

    void readProperties(java.util.Properties p) {
        String version = p.getProperty("version");
    }
}
