/*
 * Copyright (c) 2013, 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package org.graalvm.visualizer.source.impl.ui;

import org.graalvm.visualizer.source.impl.SourceRepositoryImpl;
import org.graalvm.visualizer.source.impl.SourceRepositoryNode;
import org.graalvm.visualizer.source.impl.actions.RemoveSourceRootAction;
import org.netbeans.api.settings.ConvertAsProperties;
import org.openide.awt.ActionID;
import org.openide.awt.ActionReference;
import org.openide.explorer.ExplorerManager;
import org.openide.explorer.view.BeanTreeView;
import org.openide.nodes.Node;
import org.openide.util.NbBundle.Messages;
import org.openide.windows.TopComponent;

import javax.swing.tree.TreeSelectionModel;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

/**
 * Top component which displays something.
 */
@ConvertAsProperties(
        dtd = "-//org.graalvm.visualizer.source.impl//SourceCollection//EN",
        autostore = false
)
@TopComponent.Description(
        preferredID = "SourceCollectionTopComponent",
        //iconBase="SET/PATH/TO/ICON/HERE",
        persistenceType = TopComponent.PERSISTENCE_ALWAYS
)
@TopComponent.Registration(mode = "customLeft", openAtStartup = true, position = 3000)
@ActionID(category = "Window", id = "org.graalvm.visualizer.source.impl.SourceCollectionTopComponent")
@ActionReference(path = "Menu/Window", position = 95)
@TopComponent.OpenActionRegistration(
        displayName = "#CTL_SourceCollectionAction",
        preferredID = "SourceCollectionTopComponent"
)
@Messages({
        "CTL_SourceCollectionAction=Source Collections",
        "CTL_SourceCollectionTopComponent=Source Collections",
        "HINT_SourceCollectionTopComponent=Locations of source files for navigation"
})
public final class SourceCollectionTopComponent extends TopComponent implements ExplorerManager.Provider, PropertyChangeListener {
    private BeanTreeView tree;
    private ExplorerManager mgr;

    public SourceCollectionTopComponent() {
        mgr = new ExplorerManager();
        initComponents();
        setName(Bundle.CTL_SourceCollectionTopComponent());
        setToolTipText(Bundle.HINT_SourceCollectionTopComponent());
        putClientProperty(TopComponent.PROP_KEEP_PREFERRED_SIZE_WHEN_SLIDED_IN, Boolean.TRUE);

        tree = new BeanTreeView();
        tree.setRootVisible(false);
        tree.setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        add(tree);

        SourceRepositoryImpl repo = SourceRepositoryImpl.getInstance();
        mgr.setRootContext(new SourceRepositoryNode(repo));
        mgr.addPropertyChangeListener(this);

        getActionMap().put("delete", new RemoveSourceRootAction.Perfomer(mgr, true));
    }

    private void refreshActivatedNodes() {
        Node[] nodes = mgr.getSelectedNodes();
        setActivatedNodes(nodes);
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */

    @Override
    public ExplorerManager getExplorerManager() {
        return mgr;
    }

    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        setLayout(new java.awt.BorderLayout());
    }// </editor-fold>//GEN-END:initComponents

    // Variables declaration - do not modify//GEN-BEGIN:variables
    // End of variables declaration//GEN-END:variables
    @Override
    public void componentOpened() {
        // TODO add custom code on component opening
    }

    @Override
    public void componentClosed() {
        // TODO add custom code on component closing
    }

    void writeProperties(java.util.Properties p) {
        // better to version settings since initial version as advocated at
        // http://wiki.apidesign.org/wiki/PropertyFiles
        p.setProperty("version", "1.0");
        // TODO store your settings
    }

    void readProperties(java.util.Properties p) {
        String version = p.getProperty("version");
        // TODO read your settings according to their version
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        if (ExplorerManager.PROP_SELECTED_NODES.equals(evt.getPropertyName())) {
            refreshActivatedNodes();
        }
    }
}
