/*
 * Copyright (c) 2019, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.js.nodes.interop;

import com.oracle.truffle.api.dsl.GenerateUncached;
import com.oracle.truffle.api.dsl.ImportStatic;
import com.oracle.truffle.api.dsl.NeverDefault;
import com.oracle.truffle.api.dsl.Specialization;
import com.oracle.truffle.api.interop.InteropLibrary;
import com.oracle.truffle.api.library.CachedLibrary;
import com.oracle.truffle.js.nodes.JavaScriptBaseNode;
import com.oracle.truffle.js.runtime.JSConfig;
import com.oracle.truffle.js.runtime.JSRealm;
import com.oracle.truffle.js.runtime.objects.JSDynamicObject;

/**
 * Node that returns a suitable prototype for a foreign object.
 */
@GenerateUncached
@ImportStatic({JSConfig.class})
public abstract class ForeignObjectPrototypeNode extends JavaScriptBaseNode {

    public abstract JSDynamicObject execute(Object truffleObject);

    @Specialization(limit = "InteropLibraryLimit")
    public JSDynamicObject doTruffleObject(Object truffleObject,
                    @CachedLibrary("truffleObject") InteropLibrary interop) {
        JSRealm realm = getRealm();
        if (interop.hasArrayElements(truffleObject)) {
            return realm.getForeignArrayPrototype();
        } else if (interop.isInstant(truffleObject)) {
            return realm.getForeignDatePrototype();
        } else if (interop.hasHashEntries(truffleObject)) {
            return realm.getForeignMapPrototype();
        } else if (interop.hasIterator(truffleObject)) {
            return realm.getForeignIterablePrototype();
        } else if (interop.isIterator(truffleObject)) {
            return realm.getForeignIteratorPrototype();
        } else if (interop.isString(truffleObject)) {
            return realm.getForeignStringPrototype();
        } else if (interop.isNumber(truffleObject)) {
            return realm.getForeignNumberPrototype();
        } else if (interop.isBoolean(truffleObject)) {
            return realm.getForeignBooleanPrototype();
        } else if (interop.isException(truffleObject)) {
            return realm.getForeignErrorPrototype();
        } else if (interop.isExecutable(truffleObject) || interop.isInstantiable(truffleObject)) {
            return realm.getForeignFunctionPrototype();
        } else {
            return realm.getForeignObjectPrototype();
        }
    }

    @NeverDefault
    public static ForeignObjectPrototypeNode create() {
        return ForeignObjectPrototypeNodeGen.create();
    }

    public static ForeignObjectPrototypeNode getUncached() {
        return ForeignObjectPrototypeNodeGen.getUncached();
    }
}
