/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.jdwp.server.impl;

import com.oracle.svm.jdwp.bridge.EventKind;
import com.oracle.svm.jdwp.server.api.BreakpointInfo;
import com.oracle.svm.jdwp.server.impl.EventInfo;
import com.oracle.svm.jdwp.server.impl.StepInfo;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.IntUnaryOperator;
import jdk.vm.ci.meta.ResolvedJavaType;

public final class RequestFilter {
    private final int requestId;
    private final EventKind eventKind;
    private final List<Filter> filters = new ArrayList<Filter>();
    private List<BreakpointInfo> breakpointInfos;
    private StepInfo stepInfo;

    public RequestFilter(int requestId, EventKind eventKind) {
        this.requestId = requestId;
        this.eventKind = eventKind;
    }

    public int getRequestId() {
        return this.requestId;
    }

    public EventKind getEventKind() {
        return this.eventKind;
    }

    void setStepInfo(StepInfo info) {
        this.stepInfo = info;
    }

    public StepInfo getStepInfo() {
        return this.stepInfo;
    }

    void addBreakpointInfo(BreakpointInfo info) {
        if (this.breakpointInfos == null) {
            this.breakpointInfos = new ArrayList<BreakpointInfo>(1);
        }
        this.breakpointInfos.add(info);
    }

    public List<BreakpointInfo> getBreakpointInfos() {
        return this.breakpointInfos;
    }

    void addCount(int count) {
        this.filters.add(new Filter.Count(count));
    }

    void addThread(long threadId) {
        this.filters.add(new Filter.ThreadId(threadId));
    }

    void addClassOnly(ResolvedJavaType type) {
        this.filters.add(new Filter.ClassOnly(type));
    }

    void addClassNameMatch(String classNamePattern) {
        this.filters.add(new Filter.ClassNameMatch(classNamePattern));
    }

    void addClassNameExclude(String classNamePattern) {
        this.filters.add(new Filter.ClassNameExclude(classNamePattern));
    }

    void addLocation(long classId, long methodId, long bci) {
        this.filters.add(new Filter.Location(classId, methodId, bci));
    }

    void addThis(long objectId) {
        this.filters.add(new Filter.This(objectId));
    }

    void addPlatformThreadsOnly() {
        this.filters.add(new Filter.PlatformThreadsOnly());
    }

    public boolean isHit(EventInfo event) {
        for (Filter filter : this.filters) {
            if (filter.isHit(event)) continue;
            return false;
        }
        return true;
    }

    public boolean matchesType(ResolvedJavaType eventType) {
        for (Filter filter : this.filters) {
            if (filter.matchesType(eventType)) continue;
            return false;
        }
        return true;
    }

    private static abstract class Filter {
        private Filter() {
        }

        abstract boolean isHit(EventInfo var1);

        boolean matchesType(ResolvedJavaType eventType) {
            return true;
        }

        static final class PlatformThreadsOnly
        extends Filter {
            PlatformThreadsOnly() {
            }

            @Override
            boolean isHit(EventInfo event) {
                return !EventInfo.isThreadVirtual();
            }
        }

        static final class This
        extends Filter {
            private final long objectId;

            This(long objectId) {
                this.objectId = objectId;
            }

            @Override
            boolean isHit(EventInfo event) {
                return EventInfo.thisId() == this.objectId;
            }
        }

        static final class Location
        extends Filter {
            private final long classId;
            private final long methodId;
            private final long bci;

            Location(long classId, long methodId, long bci) {
                this.classId = classId;
                this.methodId = methodId;
                this.bci = bci;
            }

            @Override
            boolean isHit(EventInfo event) {
                return event.classId() == this.classId && event.methodId() == this.methodId && event.bci() == this.bci;
            }
        }

        static final class ClassNameExclude
        extends ClassNameMatch {
            ClassNameExclude(String classNamePattern) {
                super(classNamePattern);
            }

            @Override
            boolean isHit(EventInfo event) {
                return !super.isHit(event);
            }

            @Override
            boolean matchesType(ResolvedJavaType eventType) {
                return !super.matchesType(eventType);
            }
        }

        static class ClassNameMatch
        extends Filter {
            private final String pattern;
            private final ClassNameMatchKind matchKind;

            ClassNameMatch(String classNamePattern) {
                int length = classNamePattern.length();
                if (length > 0 && classNamePattern.charAt(0) == '*') {
                    this.pattern = classNamePattern.substring(1);
                    this.matchKind = ClassNameMatchKind.ENDS;
                } else if (length > 0 && classNamePattern.charAt(length - 1) == '*') {
                    this.pattern = classNamePattern.substring(0, length - 1);
                    this.matchKind = ClassNameMatchKind.STARTS;
                } else {
                    this.pattern = classNamePattern;
                    this.matchKind = ClassNameMatchKind.EXACT;
                }
            }

            @Override
            boolean isHit(EventInfo event) {
                String name = event.className();
                return this.matchesTypeName(name);
            }

            @Override
            boolean matchesType(ResolvedJavaType eventType) {
                String name = eventType.toClassName();
                return this.matchesTypeName(name);
            }

            private boolean matchesTypeName(String name) {
                return switch (this.matchKind.ordinal()) {
                    case 0 -> name.equals(this.pattern);
                    case 1 -> name.startsWith(this.pattern);
                    case 2 -> name.endsWith(this.pattern);
                    default -> throw new IllegalStateException("Unknown match kind " + String.valueOf((Object)this.matchKind));
                };
            }

            static enum ClassNameMatchKind {
                EXACT,
                STARTS,
                ENDS;

            }
        }

        static final class ClassOnly
        extends Filter {
            private final ResolvedJavaType type;

            ClassOnly(ResolvedJavaType type) {
                this.type = type;
            }

            @Override
            boolean isHit(EventInfo event) {
                return this.type.isAssignableFrom(event.type());
            }

            @Override
            boolean matchesType(ResolvedJavaType eventType) {
                return this.type.isAssignableFrom(eventType);
            }
        }

        static final class ThreadId
        extends Filter {
            private final long threadId;

            ThreadId(long threadId) {
                this.threadId = threadId;
            }

            @Override
            boolean isHit(EventInfo event) {
                return event.threadId() == this.threadId;
            }
        }

        static final class Count
        extends Filter
        implements IntUnaryOperator {
            private final AtomicInteger hitCount;

            Count(int count) {
                this.hitCount = new AtomicInteger(count);
            }

            @Override
            boolean isHit(EventInfo event) {
                int count = this.hitCount.updateAndGet(this);
                if (count <= 0) {
                    event.disableNextEvents();
                }
                return count == 0;
            }

            @Override
            public int applyAsInt(int lastHitCount) {
                if (lastHitCount > 0) {
                    return lastHitCount - 1;
                }
                return -1;
            }
        }
    }
}

