/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.thread;

import com.oracle.svm.core.Uninterruptible;
import com.oracle.svm.core.feature.AutomaticallyRegisteredImageSingleton;
import com.oracle.svm.core.heap.Heap;
import com.oracle.svm.core.heap.RestrictHeapAccess;
import com.oracle.svm.core.jfr.JfrTicks;
import com.oracle.svm.core.jfr.events.SafepointBeginEvent;
import com.oracle.svm.core.jfr.events.SafepointEndEvent;
import com.oracle.svm.core.log.Log;
import com.oracle.svm.core.option.RuntimeOptionKey;
import com.oracle.svm.core.thread.RecurringCallbackSupport;
import com.oracle.svm.core.thread.SafepointCheckCounter;
import com.oracle.svm.core.thread.ThreadSuspendSupport;
import com.oracle.svm.core.thread.VMOperationControl;
import com.oracle.svm.core.thread.VMThreads;
import com.oracle.svm.core.util.TimeUtils;
import com.oracle.svm.core.util.VMError;
import jdk.graal.compiler.api.replacements.Fold;
import jdk.graal.compiler.nodes.PauseNode;
import jdk.graal.compiler.word.Word;
import org.graalvm.nativeimage.CurrentIsolate;
import org.graalvm.nativeimage.ImageSingletons;
import org.graalvm.nativeimage.IsolateThread;
import org.graalvm.nativeimage.Platform;
import org.graalvm.nativeimage.Platforms;
import org.graalvm.word.UnsignedWord;

@AutomaticallyRegisteredImageSingleton
public final class Safepoint {
    private static final int NOT_AT_SAFEPOINT = 0;
    private static final int SYNCHRONIZING = 1;
    private static final int AT_SAFEPOINT = 2;
    private volatile int safepointState = 0;
    private volatile UnsignedWord safepointId;
    private volatile IsolateThread requestingThread;

    @Platforms(value={Platform.HOSTED_ONLY.class})
    Safepoint() {
    }

    @Fold
    public static Safepoint singleton() {
        return (Safepoint)ImageSingletons.lookup(Safepoint.class);
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    boolean isInProgress() {
        return this.safepointState == 2;
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    public boolean isPendingOrInProgress() {
        int state = this.safepointState;
        return state == 1 || state == 2;
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    public UnsignedWord getSafepointId() {
        return this.safepointId;
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    public boolean isMasterThread() {
        return this.requestingThread == CurrentIsolate.getCurrentThread();
    }

    @RestrictHeapAccess(access=RestrictHeapAccess.Access.NO_ALLOCATION, reason="The safepoint logic must not allocate.")
    boolean startSafepoint(String reason) {
        boolean lock;
        assert (VMOperationControl.mayExecuteVmOperations());
        long startTicks = JfrTicks.elapsedTicks();
        boolean bl = lock = !VMThreads.THREAD_MUTEX.isOwner();
        if (lock) {
            VMThreads.THREAD_MUTEX.lock();
        }
        assert (this.requestingThread.isNull());
        this.requestingThread = CurrentIsolate.getCurrentThread();
        ((Heap)ImageSingletons.lookup(Heap.class)).prepareForSafepoint();
        this.safepointState = 1;
        int numJavaThreads = Safepoint.requestThreadsEnterSafepoint(reason);
        this.safepointState = 2;
        this.safepointId = this.safepointId.add(1);
        SafepointBeginEvent.emit(this.getSafepointId(), numJavaThreads, startTicks);
        return lock;
    }

    @RestrictHeapAccess(access=RestrictHeapAccess.Access.NO_ALLOCATION, reason="The safepoint logic must not allocate.")
    void endSafepoint(boolean unlock) {
        assert (VMOperationControl.mayExecuteVmOperations());
        long startTicks = JfrTicks.elapsedTicks();
        this.safepointState = 0;
        Safepoint.releaseThreadsFromSafepoint();
        if (unlock) {
            VMThreads.THREAD_MUTEX.unlock();
        }
        this.requestingThread = (IsolateThread)Word.nullPointer();
        ((Heap)ImageSingletons.lookup(Heap.class)).endSafepoint();
        SafepointEndEvent.emit(this.getSafepointId(), startTicks);
        VMThreads.singleton().cleanupExitedOsThreads();
    }

    private static int requestThreadsEnterSafepoint(String reason) {
        long startNanos;
        assert (VMThreads.THREAD_MUTEX.isOwner()) : "must hold mutex while waiting for safepoints";
        long loopNanos = startNanos = System.nanoTime();
        long warningNanos = -1L;
        long failureNanos = -1L;
        int loopCount = 1;
        while (true) {
            int numThreads = 0;
            int atSafepoint = 0;
            int ignoreSafepoints = 0;
            int notAtSafepoint = 0;
            IsolateThread thread = VMThreads.firstThread();
            while (thread.isNonNull()) {
                ++numThreads;
                if (thread != CurrentIsolate.getCurrentThread()) {
                    int status = VMThreads.StatusSupport.getStatusVolatile(thread);
                    int safepointBehavior = VMThreads.SafepointBehavior.getSafepointBehaviorVolatile(thread);
                    if (safepointBehavior == 1) {
                        ++notAtSafepoint;
                    } else if (safepointBehavior == 2) {
                        ++ignoreSafepoints;
                    } else {
                        assert (safepointBehavior == 0);
                        switch (status) {
                            case 1: 
                            case 4: {
                                if (SafepointCheckCounter.getVolatile(thread) > 0) {
                                    Safepoint.requestEnterSafepoint(thread);
                                }
                                ++notAtSafepoint;
                                break;
                            }
                            case 2: {
                                ++atSafepoint;
                                break;
                            }
                            case 3: {
                                if (VMThreads.StatusSupport.compareAndSetNativeToSafepoint(thread)) {
                                    ++atSafepoint;
                                    break;
                                }
                                ++notAtSafepoint;
                                break;
                            }
                            default: {
                                throw VMError.shouldNotReachHere("Unexpected thread status");
                            }
                        }
                    }
                }
                thread = VMThreads.nextThread(thread);
            }
            if (notAtSafepoint == 0) {
                return numThreads;
            }
            if (warningNanos == -1L || failureNanos == -1L) {
                warningNanos = Options.SafepointPromptnessWarningNanos.getValue();
                failureNanos = Options.SafepointPromptnessFailureNanos.getValue();
            }
            long nanosSinceStart = TimeUtils.nanoSecondsSince(startNanos);
            if (warningNanos > 0L || failureNanos > 0L) {
                boolean fatalError;
                long nanosSinceLastWarning = TimeUtils.nanoSecondsSince(loopNanos);
                boolean printWarning = warningNanos > 0L && TimeUtils.nanoTimeLessThan(warningNanos, nanosSinceLastWarning);
                boolean bl = fatalError = failureNanos > 0L && TimeUtils.nanoTimeLessThan(failureNanos, nanosSinceStart);
                if (printWarning || fatalError) {
                    Log.log().string("[Safepoint: not all threads reached a safepoint (").string(reason).string(") within ").signed(warningNanos).string(" ns. ").string("Total wait time so far: ").signed(nanosSinceStart).string(" ns.").newline();
                    Log.log().string("  loopCount: ").signed(loopCount).string("  atSafepoint: ").signed(atSafepoint).string("  ignoreSafepoints: ").signed(ignoreSafepoints).string("  notAtSafepoint: ").signed(notAtSafepoint).string("]").newline();
                    loopNanos = System.nanoTime();
                    if (fatalError) {
                        VMError.guarantee(false, "Safepoint promptness failure.");
                    }
                }
            }
            Safepoint.yieldOrSleepOrPause(nanosSinceStart);
            ++loopCount;
        }
    }

    private static void yieldOrSleepOrPause(long nanosSinceStart) {
        if (VMThreads.singleton().supportsNativeYieldAndSleep()) {
            if (nanosSinceStart < 1000000L) {
                VMThreads.singleton().yield();
            } else {
                VMThreads.singleton().nativeSleep(1);
            }
        } else {
            PauseNode.pause();
        }
    }

    private static void releaseThreadsFromSafepoint() {
        assert (VMThreads.THREAD_MUTEX.isOwner()) : "must hold mutex when releasing safepoints.";
        IsolateThread thread = VMThreads.firstThread();
        while (thread.isNonNull()) {
            if (thread != CurrentIsolate.getCurrentThread() && !VMThreads.SafepointBehavior.ignoresSafepoints(thread)) {
                assert (VMThreads.StatusSupport.getStatusVolatile(thread) == 2);
                Safepoint.restoreCounter(thread);
                if (!ThreadSuspendSupport.isSuspended(thread)) {
                    VMThreads.StatusSupport.setStatusNative(thread);
                }
            }
            thread = VMThreads.nextThread(thread);
        }
    }

    private static void requestEnterSafepoint(IsolateThread thread) {
        if (RecurringCallbackSupport.isEnabled()) {
            int value;
            do {
                value = SafepointCheckCounter.getVolatile(thread);
                assert (value >= 0) : "the value can only be negative if a safepoint was requested";
            } while (!SafepointCheckCounter.compareAndSet(thread, value, -value));
        } else {
            SafepointCheckCounter.setVolatile(thread, 0);
        }
    }

    private static void restoreCounter(IsolateThread thread) {
        int value = SafepointCheckCounter.getVolatile(thread);
        if (value < 0) {
            int newValue = -(value + 2);
            assert (newValue >= -2 && newValue < Integer.MAX_VALUE) : "overflow";
            newValue = newValue <= 0 ? 1 : newValue;
            SafepointCheckCounter.setVolatile(thread, newValue);
        }
    }

    public static class Options {
        public static final RuntimeOptionKey<Long> SafepointPromptnessWarningNanos = new RuntimeOptionKey<Long>(Long.valueOf(TimeUtils.millisToNanos(0L)), RuntimeOptionKey.RuntimeOptionKeyFlag.RelevantForCompilationIsolates);
        public static final RuntimeOptionKey<Long> SafepointPromptnessFailureNanos = new RuntimeOptionKey<Long>(Long.valueOf(TimeUtils.millisToNanos(0L)), RuntimeOptionKey.RuntimeOptionKeyFlag.RelevantForCompilationIsolates);
    }
}

