/*
 * Copyright (c) 2025, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.svm.hosted.jdk;

import java.awt.GraphicsEnvironment;

import com.oracle.svm.hosted.jdk.jniRegistration.BaseClassMemberRegistration;
import com.oracle.svm.hosted.jdk.jniRegistration.ClassJavaAwtWindowRegistration;
import com.oracle.svm.hosted.jdk.jniRegistration.ClassSunFont2DRegistration;
import com.oracle.svm.hosted.jdk.jniRegistration.ClassSunJava2dWindowsWindowsFlags;
import org.graalvm.nativeimage.ImageSingletons;
import org.graalvm.nativeimage.Platforms;
import org.graalvm.nativeimage.hosted.Feature;
import org.graalvm.nativeimage.hosted.RuntimeJNIAccess;
import org.graalvm.nativeimage.hosted.RuntimeReflection;
import org.graalvm.nativeimage.impl.ConfigurationCondition;
import org.graalvm.nativeimage.impl.InternalPlatform;
import org.graalvm.nativeimage.impl.RuntimeResourceSupport;

import com.oracle.svm.core.SubstrateOptions;
import com.oracle.svm.core.feature.AutomaticallyRegisteredFeature;
import com.oracle.svm.core.feature.InternalFeature;
import com.oracle.svm.core.jdk.JNIRegistrationUtil;
import com.oracle.svm.core.jdk.NativeLibrarySupport;
import com.oracle.svm.core.jdk.PlatformNativeLibrarySupport;
import com.oracle.svm.hosted.FeatureImpl;
import com.oracle.svm.hosted.c.NativeLibraries;

import java.lang.constant.ClassDesc;
import java.lang.constant.MethodTypeDesc;
import java.lang.invoke.MethodHandles;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Stream;

@Platforms({InternalPlatform.PLATFORM_JNI.class})
@AutomaticallyRegisteredFeature
@SuppressWarnings({"unused"})
public class JNIRegistrationAwt extends JNIRegistrationUtil implements InternalFeature {

    private boolean isSupportedPlatform() {
        return isLinux() || isWindows() || isDarwin();
    }

    @Override
    public void duringSetup(DuringSetupAccess access) {
        // register resources here as they must be registered before `beforeAnalysis()`
        RuntimeResourceSupport registry = ImageSingletons.lookup(RuntimeResourceSupport.class);

        // AWT
        ConfigurationCondition awtReachable = ConfigurationCondition.create(access.findClassByName("java.awt.Toolkit"), false);
        registry.addResources(awtReachable, "sun.awt.resources.awt", "");
        if (isDarwin()) {
            registry.addResourceBundles(awtReachable, "sun.util.resources.CurrencyNames");
            registry.addResourceBundles(awtReachable, "sun.util.resources.CalendarData");
            registry.addResourceBundles(awtReachable, "sun.util.resources.LocaleNames");
        }
        if (!isHeadless()) {
            registry.addResources(ConfigurationCondition.create(access.findClassByName("java.awt.dnd.DragSource"), false),
                    "sun.awt.*", "");
            registry.addResources(ConfigurationCondition.create(access.findClassByName("java.awt.datatransfer.SystemFlavorMap"), false),
                    "sun.datatransfer.resources.flavormap.properties", "");
        }
        registry.addResources(ConfigurationCondition.create(access.findClassByName("java.awt.color.ICC_Profile"), false),
                "sun.java2d.cmm.profiles.*", "");

        // LAF
        ConfigurationCondition swingReachable = ConfigurationCondition.create(access.findClassByName("javax.swing.JComponent"), false);
        registry.addResourceBundles(swingReachable, "com.sun.swing.internal.plaf.basic.resources.basic");
        registry.addResourceBundles(swingReachable, "com.sun.swing.internal.plaf.metal.resources.metal");
        registry.addResourceBundles(swingReachable, "com.sun.swing.internal.plaf.synth.resources.synth");
        registry.addResourceBundles(swingReachable, "com.sun.java.swing.plaf.motif.resources.motif");
        registry.addResources(swingReachable, "com.sun.java.swing.plaf.motif.icons.*", "");
        registry.addResourceBundles(swingReachable, "com.sun.java.swing.plaf.windows.resources.windows");
        registry.addResources(swingReachable, "com.sun.java.swing.plaf.windows.icons.*", "");
        registry.addResources(swingReachable, "com.sun.java.swing.plaf.gtk.resources.metacity.*", "");

        ConfigurationCondition oceanThemeReachable = ConfigurationCondition.create(access.findClassByName("javax.swing.plaf.metal.OceanTheme"), false);
        registry.addResources(oceanThemeReachable, "javax.swing.plaf.basic.icons.*", "");
        registry.addResources(oceanThemeReachable, "javax.swing.plaf.metal.icons.*", "");

        // HTML
        ConfigurationCondition htmlReachable = ConfigurationCondition.create(access.findClassByName("javax.swing.text.html.HTMLEditorKit"), false);
        registry.addResources(htmlReachable, "javax.swing.text.html.parser.html32.bdtd", "");
        registry.addResources(htmlReachable, "javax.swing.text.html.default.css", "");

        registry.addResources(ConfigurationCondition.create(access.findClassByName("javax.swing.text.rtf.RTFReader"), false),
                "javax.swing.text.rtf.charsets.*", "");
    }

    @Override
    public void beforeAnalysis(BeforeAnalysisAccess access) {
        if (isSupportedPlatform()) {
            access.registerReachabilityHandler(JNIRegistrationAwt::handlePreferencesClassReachable,
                            clazz(access, "java.awt.Toolkit"),
                            clazz(access, "sun.java2d.cmm.lcms.LCMS"),
                            clazz(access, "java.awt.event.NativeLibLoader"),
                            clazz(access, "sun.awt.NativeLibLoader"),
                            clazz(access, "sun.awt.image.NativeLibLoader"),
                            clazz(access, "java.awt.image.ColorModel"),
                            clazz(access, "java.awt.GraphicsEnvironment"),
                            clazz(access, "sun.font.FontManagerNativeLibrary"),
                            clazz(access, "javax.print.PrintServiceLookup"),
                            clazz(access, "sun.java2d.Disposer"));

            access.registerReachabilityHandler(JNIRegistrationAwt::registerComponent,
                    clazz(access, "java.awt.Component"));

            if (isPosix()) {
                if (isDarwin()) {
                    PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("java_awt");
                    PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("sun_awt");
                    PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("sun_java2d");
                    PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("sun_print");

                    PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("sun_font");
                    PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("com_sun_imageio_plugins_jpeg");
                }
                access.registerReachabilityHandler(JNIRegistrationAwt::registerFreeType,
                        clazz(access, "sun.font.FontManagerNativeLibrary"));

                access.registerReachabilityHandler(JNIRegistrationAwt::registerLCMS,
                        clazz(access, "sun.java2d.cmm.lcms.LCMS"));

                access.registerReachabilityHandler(JNIRegistrationAwt::registerImagingLib,
                        clazz(access, "sun.awt.image.ImagingLib"));

                access.registerReachabilityHandler(JNIRegistrationAwt::registerJPEG,
                        clazz(access, "sun.awt.image.JPEGImageDecoder"),
                        clazz(access, "com.sun.imageio.plugins.jpeg.JPEGImageReader"),
                        clazz(access, "com.sun.imageio.plugins.jpeg.JPEGImageWriter"));
            }

            if (isDarwin()) {
                PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("awt_lwawt");
                PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("sun_lwawt_macosx");
                PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("apple_laf");
                PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("com_apple_eio");
                PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("com_apple_eawt");
                PlatformNativeLibrarySupport.singleton().addBuiltinPkgNativePrefix("com_apple_laf");
            }

            access.registerReachabilityHandler(JNIRegistrationAwt::registerFontManager,
                    clazz(access, "sun.font.SunFontManager"));

            if (!isHeadless()) {
                access.registerReachabilityHandler(JNIRegistrationAwt::registerKeyCodes,
                        clazz(access, "java.awt.event.KeyEvent"));

                access.registerReachabilityHandler(JNIRegistrationAwt::registerDataTransferer,
                        clazz(access, "sun.awt.datatransfer.DataTransferer"));

                access.registerReachabilityHandler(JNIRegistrationAwt::registerShellFolderManager,
                        clazz(access, "sun.swing.FilePane"));

                if (isLinux()) {
                    access.registerReachabilityHandler(JNIRegistrationAwt::registerGtkFileDialog,
                            clazz(access, "sun.awt.X11.GtkFileDialogPeer"));
                }
            }
        }
    }

    private static void handlePreferencesClassReachable(DuringAnalysisAccess access) {

        RuntimeJNIAccess.register(java.lang.System.class);
        RuntimeJNIAccess.register(method(access, "java.lang.System", "load", java.lang.String.class));
        RuntimeJNIAccess.register(method(access, "java.lang.System", "loadLibrary", String.class));
        RuntimeJNIAccess.register(method(access, "java.lang.System", "getProperty", java.lang.String.class));
        RuntimeJNIAccess.register(method(access, "java.lang.System", "setProperty", String.class, String.class));


        RuntimeJNIAccess.register(GraphicsEnvironment.class);
        RuntimeJNIAccess.register(method(access, "java.awt.GraphicsEnvironment", "isHeadless"));

        NativeLibraries nativeLibraries = getNativeLibraries(access);

        if (isDarwin()) {
            NativeLibrarySupport.singleton().preregisterUninitializedBuiltinLibrary("awt");
            nativeLibraries.addStaticJniLibrary("awt");

            nativeLibraries.addStaticJniLibrary("osx");
            nativeLibraries.addStaticJniLibrary("osxui");
            nativeLibraries.addStaticJniLibrary("osxapp");
            nativeLibraries.addStaticJniLibrary("jawt");

            NativeLibrarySupport.singleton().preregisterUninitializedBuiltinLibrary("osx");
            NativeLibrarySupport.singleton().preregisterUninitializedBuiltinLibrary("osxui");
            NativeLibrarySupport.singleton().preregisterUninitializedBuiltinLibrary("osxapp");
            NativeLibrarySupport.singleton().preregisterUninitializedBuiltinLibrary("jawt");


            // JLI for JLI_MemAlloc (JavaRuntimeSupport)
            NativeLibrarySupport.singleton().preregisterUninitializedBuiltinLibrary("jli");
            nativeLibraries.addStaticJniLibrary("jli");

            Stream.of(
                    "Cocoa",
                    "Carbon",
                    "OpenGL",
                    "CoreServices",
                    "Accelerate",
                    "QuartzCore",
                    "Security",
                    "JavaRuntimeSupport"
            )
                    .forEach(framework -> {
                        nativeLibraries.addDynamicNonJniLibrary("-framework " + framework);
                    });

            // 8260931: Implement JEP 382: New macOS Rendering Pipeline
            nativeLibraries.addDynamicNonJniLibrary("-framework Metal");

            nativeLibraries.addDynamicNonJniLibrary("stdc++");
            nativeLibraries.addDynamicNonJniLibrary("m");
        }

        registerCommonClasses(access);

        if (!isHeadless() || isDarwin()) {
            registerHeadfullClasses(access);
        }

        access.registerReachabilityHandler(JNIRegistrationAwt::registerHtml32bdtd,
                        clazz(access, "javax.swing.text.html.HTMLEditorKit"));
    }

    private static void registerComponent(DuringAnalysisAccess access) {
        RuntimeJNIAccess.register(java.awt.Component.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.Component",
                "appContext", "background", "cursor", "enabled", "focusable",
                "foreground", "graphicsConfig", "height", "parent",
                "isPacked", "name", "peer", "visible", "width", "x", "y"));
        RuntimeJNIAccess.register(method(access, "java.awt.Component", "getLocationOnScreen_NoTreeLock"));
        RuntimeJNIAccess.register(method(access, "java.awt.Component", "getParent_NoClientCode"));

        RuntimeJNIAccess.register(java.awt.event.KeyEvent.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.event.KeyEvent",
                "isProxyActive", "keyChar", "keyCode"));
    }

    private static void registerJPEG(DuringAnalysisAccess access) {
        if (isDarwin()) {
            NativeLibraries nativeLibraries = getNativeLibraries(access);

            NativeLibrarySupport.singleton().preregisterUninitializedBuiltinLibrary("javajpeg");
            nativeLibraries.addStaticJniLibrary("javajpeg");
        }
        registerJPEGClasses(access);
    }

    private static void registerJPEGClasses(DuringAnalysisAccess access) {
        RuntimeJNIAccess.register(clazz(access, "sun.awt.image.JPEGImageDecoder"));
        RuntimeJNIAccess.register(method(access, "sun.awt.image.JPEGImageDecoder", "sendHeaderInfo",
                int.class, int.class, boolean.class, boolean.class, boolean.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.image.JPEGImageDecoder", "sendPixels",
                byte[].class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.image.JPEGImageDecoder", "sendPixels",
                int[].class, int.class));
    }

    private static void registerSystemColor(DuringAnalysisAccess access) {

        RuntimeReflection.register(java.awt.SystemColor.class);

        RuntimeReflection.register(fields(access, "java.awt.SystemColor",
                "activeCaption",
                "activeCaptionBorder",
                "activeCaptionText",
                "control",
                "controlDkShadow",
                "controlHighlight",
                "controlLtHighlight",
                "controlShadow",
                "controlText",
                "desktop",
                "inactiveCaption",
                "inactiveCaptionBorder",
                "inactiveCaptionText",
                "info",
                "infoText",
                "menu",
                "menuText",
                "scrollbar",
                "text",
                "textHighlight",
                "textHighlightText",
                "textInactiveText",
                "textText",
                "window",
                "windowBorder",
                "windowText"));
    }

    private static void registerEvent(DuringAnalysisAccess access) {
        RuntimeJNIAccess.register(fields(access, "java.awt.Event", "target", "x", "y"));
    }

    private static void registerWindowsMenu(DuringAnalysisAccess access) {

        RuntimeJNIAccess.register(java.awt.Menu.class);
        RuntimeJNIAccess.register(method(access, "java.awt.Menu", "countItemsImpl"));
        RuntimeJNIAccess.register(method(access, "java.awt.Menu", "getItemImpl",
                int.class));

        RuntimeJNIAccess.register(java.awt.MenuComponent.class);
        RuntimeJNIAccess.register(method(access, "java.awt.MenuComponent", "getFont_NoClientCode"));

        RuntimeJNIAccess.register(java.awt.MenuItem.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.MenuItem", "enabled", "label"));

        RuntimeJNIAccess.register(java.awt.CheckboxMenuItem.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.CheckboxMenuItem", "state"));

        RuntimeJNIAccess.register(java.awt.PopupMenu.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.PopupMenu", "isTrayIconPopup"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WMenuItemPeer"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.windows.WMenuItemPeer", "isCheckbox", "shortcutLabel"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WMenuItemPeer", "getDefaultFont"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WMenuItemPeer", "handleAction",
                long.class, int.class));

        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WCheckboxMenuItemPeer", "handleAction",
                boolean.class));
    }

    private static void registerTrayIcon(DuringAnalysisAccess access) {
        RuntimeJNIAccess.register(java.awt.TrayIcon.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.TrayIcon", "actionCommand", "id"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WTrayIconPeer"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WTrayIconPeer", "postEvent",
                java.awt.AWTEvent.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WTrayIconPeer", "showPopupMenu",
                int.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WTrayIconPeer", "updateImage"));

        RuntimeJNIAccess.register(clazz(access, "java.awt.event.ActionEvent"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.event.ActionEvent",
                Object.class, int.class, String.class, long.class, int.class));
    }

    private static void registerImagingLib(DuringAnalysisAccess access) {
        if (isDarwin()) {
            NativeLibraries nativeLibraries = getNativeLibraries(access);

            NativeLibrarySupport.singleton().preregisterUninitializedBuiltinLibrary("mlib_image");
            nativeLibraries.addStaticJniLibrary("mlib_image");
        }
    }

    private static void registerLCMS(DuringAnalysisAccess access) {
        if (isDarwin()) {
            NativeLibraries nativeLibraries = getNativeLibraries(access);

            NativeLibrarySupport.singleton().preregisterUninitializedBuiltinLibrary("lcms");
            nativeLibraries.addStaticJniLibrary("lcms");
        }
    }

    private static void registerFreeType(DuringAnalysisAccess access) {
        if (SubstrateOptions.StaticExecutable.getValue()) {
            /*
             * Freetype uses fontconfig through dlsym. This may not work in a statically linked
             * executable
             */
            return;
        }

        if (isDarwin()) {
            NativeLibraries nativeLibraries = getNativeLibraries(access);
            NativeLibrarySupport.singleton().preregisterUninitializedBuiltinLibrary("fontmanager");
            nativeLibraries.addStaticJniLibrary("fontmanager", "awt_lwawt", "freetype");
        }

        if (isLinux()) {
            RuntimeJNIAccess.register(clazz(access, "sun.font.FontConfigManager$FontConfigInfo"));
            RuntimeJNIAccess.register(fields(access, "sun.font.FontConfigManager$FontConfigInfo", "fcVersion", "cacheDirs"));
            RuntimeJNIAccess.register(clazz(access, "sun.font.FontConfigManager$FcCompFont"));
            RuntimeJNIAccess.register(fields(access, "sun.font.FontConfigManager$FcCompFont", "fcName", "firstFont", "allFonts"));
            RuntimeJNIAccess.register(clazz(access, "sun.font.FontConfigManager$FontConfigFont"));
            RuntimeJNIAccess.register(constructor(access, "sun.font.FontConfigManager$FontConfigFont"));
            RuntimeJNIAccess.register(fields(access, "sun.font.FontConfigManager$FontConfigFont", "familyName", "styleStr", "fullName", "fontFile"));
        }
    }

    private static void registerErrorClasses(DuringAnalysisAccess access) {

        RuntimeJNIAccess.register(clazz(access, "java.lang.Exception"));
        RuntimeJNIAccess.register(constructor(access, "java.lang.Exception", java.lang.String.class));

        RuntimeJNIAccess.register(clazz(access, "java.awt.AWTError"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.AWTError", java.lang.String.class));
    }

    private static void registerKeyCodes(DuringAnalysisAccess access) {

        RuntimeReflection.register(java.awt.event.KeyEvent.class);

        String[] keys = Arrays.stream(java.awt.event.KeyEvent.class
                .getDeclaredFields())
                .filter(f -> f.getType() == Integer.TYPE && f.getName().startsWith("VK_"))
                .map(f -> f.getName())
                .toArray(size -> new String[size]);

        RuntimeReflection.register(fields(access, "java.awt.event.KeyEvent", keys));
    }

    private static void registerDataTransferer(DuringAnalysisAccess access) {

        RuntimeReflection.register(java.lang.String.class);
        RuntimeReflection.register(java.util.List.class);

        RuntimeReflection.register(java.io.Reader.class);
        RuntimeReflection.register(java.io.InputStream.class);
        RuntimeReflection.register(java.io.Serializable.class);
        RuntimeReflection.register(java.nio.CharBuffer.class);
        RuntimeReflection.register(java.nio.ByteBuffer.class);

        RuntimeReflection.register(java.awt.Image.class);

        RuntimeReflection.register(byte[].class);
        RuntimeReflection.register(char[].class);

        RuntimeReflection.register(constructor(access, "java.rmi.MarshalledObject",
                java.lang.Object.class));
        RuntimeReflection.register(method(access, "java.rmi.MarshalledObject", "get"));

        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageReader",
                "readInputData", byte[].class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageReader",
                "skipInputBytes", long.class));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageReader",
                "warningOccurred", int.class));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageReader",
                "warningWithMessage", String.class));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageReader",
                "setImageData", int.class, int.class, int.class, int.class, int.class, byte[].class));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageReader",
                "acceptPixels", int.class, boolean.class));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageReader",
                "passStarted", int.class));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageReader",
                "passComplete"));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageReader",
                "pushBack", int.class));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageReader",
                "skipPastImage", int.class));

        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageWriter",
                "writeOutputData", byte[].class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageWriter",
                "warningOccurred", int.class));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageWriter",
                "warningWithMessage", String.class));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageWriter",
                "writeMetadata"));
        RuntimeJNIAccess.register(method(access, "com.sun.imageio.plugins.jpeg.JPEGImageWriter",
                "grabPixels", int.class));

        RuntimeJNIAccess.register(fields(access, "javax.imageio.plugins.jpeg.JPEGQTable", "qTable"));
        RuntimeJNIAccess.register(fields(access, "javax.imageio.plugins.jpeg.JPEGHuffmanTable",
                "lengths", "values"));
    }

    private static void registerShellFolderManager(DuringAnalysisAccess access) {

        RuntimeReflection.register(clazz(access, "sun.awt.shell.ShellFolderManager"));
        RuntimeReflection.register(constructor(access, "sun.awt.shell.ShellFolderManager"));

        if (isWindows()) {
            RuntimeReflection.register(clazz(access, "sun.awt.shell.Win32ShellFolderManager2"));
            RuntimeReflection.register(constructor(access, "sun.awt.shell.Win32ShellFolderManager2"));

            RuntimeJNIAccess.register(clazz(access, "sun.awt.shell.Win32ShellFolder2"));
            RuntimeJNIAccess.register(fields(access, "sun.awt.shell.Win32ShellFolder2", "FDATE", "FNAME", "FSIZE", "FTYPE", "displayName", "folderType", "pIShellIcon"));
            RuntimeJNIAccess.register(method(access, "sun.awt.shell.Win32ShellFolder2", "setRelativePIDL", long.class));
            RuntimeJNIAccess.register(method(access, "sun.awt.shell.Win32ShellFolder2", "setIShellFolder", long.class));

            RuntimeJNIAccess.register(clazz(access, "sun.awt.shell.Win32ShellFolder2$KnownFolderDefinition"));
            RuntimeJNIAccess.register(constructor(access, "sun.awt.shell.Win32ShellFolder2$KnownFolderDefinition"));
            RuntimeJNIAccess.register(fields(access, "sun.awt.shell.Win32ShellFolder2$KnownFolderDefinition", "attributes", "category", "defenitionFlags", "description", "ftidType", "guid", "icon", "localizedName", "name", "parent", "parsingName", "path", "relativePath", "saveLocation", "security", "tooltip"));

            RuntimeJNIAccess.register(clazz(access, "sun.awt.shell.ShellFolderColumnInfo"));
            RuntimeJNIAccess.register(constructor(access, "sun.awt.shell.ShellFolderColumnInfo", java.lang.String.class, int.class, int.class, boolean.class));
        }
    }

    private static void registerFontManager(DuringAnalysisAccess access) {
        RuntimeJNIAccess.register(clazz(access, "sun.font.GlyphLayout$GVData"));
        RuntimeJNIAccess.register(fields(access, "sun.font.GlyphLayout$GVData",
                "_count", "_flags", "_glyphs", "_indices", "_positions"));
        RuntimeJNIAccess.register(method(access, "sun.font.GlyphLayout$GVData",
                "grow"));

        if (isLinux()) {
            RuntimeJNIAccess.register(method(access, "java.awt.GraphicsEnvironment",
                    "getLocalGraphicsEnvironment"));
            RuntimeJNIAccess.register(clazz(access, "sun.java2d.SunGraphicsEnvironment"));
            RuntimeJNIAccess.register(method(access, "sun.java2d.SunGraphicsEnvironment",
                    "isDisplayLocal"));
        }

        RuntimeJNIAccess.register(clazz(access, "sun.font.FontUtilities"));
        RuntimeJNIAccess.register(method(access, "sun.font.FontUtilities", "debugFonts"));
    }

    private static void registerGtkFileDialog(DuringAnalysisAccess access) {
        RuntimeJNIAccess.register(fields(access, "sun.awt.X11.GtkFileDialogPeer", "widget"));
        RuntimeJNIAccess.register(method(access, "sun.awt.X11.GtkFileDialogPeer",
                "filenameFilterCallback", java.lang.String.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.X11.GtkFileDialogPeer",
                "setFileInternal", java.lang.String.class, java.lang.String[].class));
        RuntimeJNIAccess.register(method(access, "sun.awt.X11.GtkFileDialogPeer",
                "setWindow", long.class));
    }

    private static void registerHtml32bdtd(DuringAnalysisAccess duringAnalysisAccess) {

        RuntimeReflection.register(clazz(duringAnalysisAccess, "javax.swing.text.html.HTMLEditorKit"));
        RuntimeReflection.register(constructor(duringAnalysisAccess, "javax.swing.text.html.HTMLEditorKit"));

        RuntimeReflection.register(clazz(duringAnalysisAccess, "javax.swing.text.rtf.RTFEditorKit"));
        RuntimeReflection.register(constructor(duringAnalysisAccess, "javax.swing.text.rtf.RTFEditorKit"));

        RuntimeReflection.register(clazz(duringAnalysisAccess, "javax.swing.JEditorPane$PlainEditorKit"));
        RuntimeReflection.register(constructor(duringAnalysisAccess, "javax.swing.JEditorPane$PlainEditorKit"));
    }

    private static NativeLibraries getNativeLibraries(DuringAnalysisAccess access) {
        FeatureImpl.DuringAnalysisAccessImpl a = (FeatureImpl.DuringAnalysisAccessImpl) access;
        return a.getNativeLibraries();
    }

    private static boolean isHeadless() {
        return GraphicsEnvironment.isHeadless();
    }

    private static void registerCommonClasses(DuringAnalysisAccess access) {

        if (isLinux()) {
            registerErrorClasses(access);
        } else if (isWindows()) {
            registerErrorClasses(access);
            registerCommonWindowsClasses(access);
        }

        RuntimeJNIAccess.register(java.awt.AlphaComposite.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.AlphaComposite", "extraAlpha", "rule"));

        RuntimeJNIAccess.register(java.awt.Color.class);
        RuntimeJNIAccess.register(constructor(access, "java.awt.Color", int.class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "java.awt.Color", "getRGB"));

        RuntimeJNIAccess.register(java.awt.geom.AffineTransform.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.geom.AffineTransform",
                "m00", "m01", "m02", "m10", "m11", "m12"));

        RuntimeJNIAccess.register(java.awt.geom.GeneralPath.class);
        RuntimeJNIAccess.register(constructor(access, "java.awt.geom.GeneralPath"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.geom.GeneralPath",
                int.class, byte[].class, int.class, float[].class, int.class));

        RuntimeJNIAccess.register(java.awt.geom.Path2D.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.geom.Path2D", "numTypes", "pointTypes", "windingRule"));

        RuntimeJNIAccess.register(java.awt.geom.Path2D.Float.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.geom.Path2D$Float", "floatCoords"));

        RuntimeJNIAccess.register(java.awt.geom.Point2D.Float.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.geom.Point2D$Float", "x", "y"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.geom.Point2D$Float", float.class, float.class));

        RuntimeJNIAccess.register(java.awt.geom.Rectangle2D.Float.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.geom.Rectangle2D$Float", "height", "width", "x", "y"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.geom.Rectangle2D$Float"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.geom.Rectangle2D$Float",
                float.class, float.class, float.class, float.class));


        RuntimeJNIAccess.register(java.awt.image.ColorModel.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.image.ColorModel",
                "colorSpace", "colorSpaceType", "isAlphaPremultiplied", "is_sRGB", "nBits",
                "numComponents", "supportsAlpha", "transparency"));
        RuntimeJNIAccess.register(method(access, "java.awt.image.ColorModel", "getRGBdefault"));

        RuntimeJNIAccess.register(java.awt.image.IndexColorModel.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.image.IndexColorModel",
                "allgrayopaque", "colorData", "map_size", "rgb", "transparent_index"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.SunHints"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.SunHints", "INTVAL_STROKE_PURE"));

        RuntimeJNIAccess.register(clazz(access, "sun.font.CharToGlyphMapper"));
        RuntimeJNIAccess.register(method(access, "sun.font.CharToGlyphMapper", "charToGlyph", int.class));

        registerJNIAccessForClassMembers(access, new ClassSunFont2DRegistration());

        RuntimeJNIAccess.register(clazz(access, "sun.font.FontStrike"));
        RuntimeJNIAccess.register(method(access, "sun.font.FontStrike", "getGlyphMetrics", int.class));

        RuntimeJNIAccess.register(clazz(access, "sun.font.FreetypeFontScaler"));
        RuntimeJNIAccess.register(method(access, "sun.font.FreetypeFontScaler", "invalidateScaler"));

        RuntimeJNIAccess.register(clazz(access, "sun.font.GlyphList"));
        RuntimeJNIAccess.register(fields(access, "sun.font.GlyphList",
                "images", "lcdRGBOrder", "lcdSubPixPos", "len", "positions", "usePositions", "x", "y", "gposx", "gposy"));

        RuntimeJNIAccess.register(clazz(access, "sun.font.PhysicalStrike"));
        RuntimeJNIAccess.register(fields(access, "sun.font.PhysicalStrike", "pScalerContext"));
        RuntimeJNIAccess.register(method(access, "sun.font.PhysicalStrike", "adjustPoint",
                java.awt.geom.Point2D.Float.class));
        RuntimeJNIAccess.register(method(access, "sun.font.PhysicalStrike",
                "getGlyphPoint", int.class, int.class));

        RuntimeJNIAccess.register(clazz(access, "sun.font.StrikeMetrics"));
        RuntimeJNIAccess.register(constructor(access, "sun.font.StrikeMetrics",
                float.class, float.class, float.class, float.class, float.class,
                float.class, float.class, float.class, float.class, float.class));

        RuntimeJNIAccess.register(clazz(access, "sun.font.TrueTypeFont"));
        RuntimeJNIAccess.register(method(access, "sun.font.TrueTypeFont", "readBlock",
                java.nio.ByteBuffer.class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.font.TrueTypeFont", "readBytes", int.class, int.class));

        RuntimeJNIAccess.register(clazz(access, "sun.font.Type1Font"));
        RuntimeJNIAccess.register(method(access, "sun.font.Type1Font", "readFile", java.nio.ByteBuffer.class));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.Disposer"));
        RuntimeJNIAccess.register(method(access, "sun.java2d.Disposer", "addRecord",
                java.lang.Object.class, long.class, long.class));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.InvalidPipeException"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.InvalidPipeException", String.class));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.NullSurfaceData"));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.SunGraphics2D"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.SunGraphics2D",
                "clipRegion", "composite", "eargb", "lcdTextContrast", "pixel", "strokeHint"));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.SurfaceData"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.SurfaceData", "pData", "valid"));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.Blit"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.Blit", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.BlitBg"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.BlitBg", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.CompositeType"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.loops.CompositeType",
                "AnyAlpha", "Src", "SrcNoEa", "SrcOver", "SrcOverNoEa", "Xor"));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.DrawGlyphList"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.DrawGlyphList", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.DrawGlyphListAA"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.DrawGlyphListAA", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.DrawGlyphListLCD"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.DrawGlyphListLCD", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.DrawLine"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.DrawLine", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.DrawParallelogram"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.DrawParallelogram", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.DrawPath"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.DrawPath", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.DrawPolygons"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.DrawPolygons", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.DrawRect"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.DrawRect", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.FillParallelogram"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.FillParallelogram",
                long.class, clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.FillPath"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.FillPath", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.FillRect"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.FillRect", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.FillSpans"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.FillSpans", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.GraphicsPrimitive"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.loops.GraphicsPrimitive", "pNativePrim"));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.GraphicsPrimitiveMgr"));
        RuntimeJNIAccess.register(method(access, "sun.java2d.loops.GraphicsPrimitiveMgr",
                "register", clazz(access, "[Lsun.java2d.loops.GraphicsPrimitive;")));

        RuntimeJNIAccess.register(clazz(access, "[Lsun.java2d.loops.GraphicsPrimitive;"));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.MaskBlit"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.MaskBlit", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.MaskFill"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.MaskFill", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.ScaledBlit"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.ScaledBlit", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.SurfaceType"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.loops.SurfaceType",
                "Any3Byte", "Any4Byte", "AnyByte", "AnyColor", "AnyInt", "AnyShort",
                "ByteBinary1Bit", "ByteBinary2Bit", "ByteBinary4Bit", "ByteGray", "ByteIndexed",
                "ByteIndexedBm", "FourByteAbgr", "FourByteAbgrPre", "Index12Gray", "Index8Gray",
                "IntArgb", "IntArgbBm", "IntArgbPre", "IntBgr", "IntRgb", "IntRgbx", "OpaqueColor",
                "ThreeByteBgr", "Ushort4444Argb", "Ushort555Rgb", "Ushort555Rgbx", "Ushort565Rgb",
                "UshortGray", "UshortIndexed"));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.TransformHelper"));
        RuntimeJNIAccess.register(constructor(access, "sun.java2d.loops.TransformHelper", long.class,
                clazz(access, "sun.java2d.loops.SurfaceType"),
                clazz(access, "sun.java2d.loops.CompositeType"),
                clazz(access, "sun.java2d.loops.SurfaceType")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.loops.XORComposite"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.loops.XORComposite",
                "alphaMask", "xorColor", "xorPixel"));


        List.of("sun.java2d.loops.OpaqueCopyAnyToArgb",
                "sun.java2d.loops.OpaqueCopyArgbToAny",
                "sun.java2d.loops.XorCopyArgbToAny",
                "sun.java2d.loops.SetFillRectANY",
                "sun.java2d.loops.SetFillPathANY",
                "sun.java2d.loops.SetFillSpansANY",
                "sun.java2d.loops.SetDrawLineANY",
                "sun.java2d.loops.SetDrawPolygonsANY",
                "sun.java2d.loops.SetDrawPathANY",
                "sun.java2d.loops.SetDrawRectANY",
                "sun.java2d.loops.XorFillRectANY",
                "sun.java2d.loops.XorFillPathANY",
                "sun.java2d.loops.XorFillSpansANY",
                "sun.java2d.loops.XorDrawLineANY",
                "sun.java2d.loops.XorDrawPolygonsANY",
                "sun.java2d.loops.XorDrawPathANY",
                "sun.java2d.loops.XorDrawRectANY",
                "sun.java2d.loops.XorDrawGlyphListANY",
                "sun.java2d.loops.XorDrawGlyphListAAANY")
                .forEach(graphicsPrimitive -> {
                    RuntimeReflection.register(clazz(access, graphicsPrimitive));
                    RuntimeReflection.register(constructor(access, graphicsPrimitive));
                });

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.pipe.Region"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.pipe.Region",
                "bands", "endIndex", "hix", "hiy", "lox", "loy"));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.pipe.RegionIterator"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.pipe.RegionIterator",
                "curIndex", "numXbands", "region"));

        RuntimeJNIAccess.register(java.awt.image.BufferedImage.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.image.BufferedImage",
                "colorModel", "imageType", "raster"));
        RuntimeJNIAccess.register(method(access, "java.awt.image.BufferedImage",
                "getRGB", int.class, int.class, int.class, int.class, int[].class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "java.awt.image.BufferedImage",
                "setRGB", int.class, int.class, int.class, int.class, int[].class, int.class, int.class));

        RuntimeJNIAccess.register(java.awt.image.Raster.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.image.Raster",
                "dataBuffer", "height", "minX", "minY", "numBands", "numDataElements",
                "sampleModel", "sampleModelTranslateX", "sampleModelTranslateY", "width"));

        RuntimeJNIAccess.register(java.awt.image.SampleModel.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.image.SampleModel", "height", "width"));
        RuntimeJNIAccess.register(method(access, "java.awt.image.SampleModel",
                "getPixels", int.class, int.class, int.class, int.class, int[].class,
                java.awt.image.DataBuffer.class));
        RuntimeJNIAccess.register(method(access, "java.awt.image.SampleModel",
                "setPixels", int.class, int.class, int.class, int.class, int[].class,
                java.awt.image.DataBuffer.class));

        RuntimeJNIAccess.register(java.awt.image.SinglePixelPackedSampleModel.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.image.SinglePixelPackedSampleModel",
                "bitMasks", "bitOffsets", "bitSizes", "maxBitSize"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.image.BufImgSurfaceData$ICMColorData"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.image.BufImgSurfaceData$ICMColorData", "pData"));
        RuntimeJNIAccess.register(constructor(access, "sun.awt.image.BufImgSurfaceData$ICMColorData", long.class));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.image.IntegerComponentRaster"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.image.IntegerComponentRaster",
                "data", "dataOffsets", "pixelStride", "scanlineStride", "type"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.AWTAutoShutdown"));
        RuntimeJNIAccess.register(method(access, "sun.awt.AWTAutoShutdown", "notifyToolkitThreadBusy"));
        RuntimeJNIAccess.register(method(access, "sun.awt.AWTAutoShutdown", "notifyToolkitThreadFree"));
    }

    private static void registerCommonWindowsClasses(DuringAnalysisAccess access) {

        RuntimeJNIAccess.register(java.lang.Boolean.class);
        RuntimeJNIAccess.register(method(access, "java.lang.Boolean", "getBoolean", java.lang.String.class));

        // Toolkit
        RuntimeJNIAccess.register(java.awt.Toolkit.class);
        RuntimeJNIAccess.register(method(access, "java.awt.Toolkit", "getDefaultToolkit"));
        RuntimeJNIAccess.register(method(access, "java.awt.Toolkit", "getFontMetrics", java.awt.Font.class));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.SunToolkit"));
        RuntimeJNIAccess.register(method(access, "sun.awt.SunToolkit", "isTouchKeyboardAutoShowEnabled"));

        // Headless Toolkit creates platform toolkit first
        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WToolkit"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WToolkit", "windowsSettingChange"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WToolkit", "displayChanged"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WToolkit", "paletteChanged"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.Win32GraphicsEnvironment"));
        RuntimeJNIAccess.register(method(access, "sun.awt.Win32GraphicsEnvironment", "dwmCompositionChanged", boolean.class));

        RuntimeJNIAccess.register(java.awt.Component.class);

        RuntimeJNIAccess.register(java.awt.Insets.class);
        RuntimeJNIAccess.register(constructor(access, "java.awt.Insets", int.class, int.class, int.class, int.class));
        RuntimeJNIAccess.register(fields(access, "java.awt.Insets", "bottom", "left", "right", "top"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.image.SunVolatileImage"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.image.SunVolatileImage", "volSurfaceManager"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.image.VolatileSurfaceManager"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.image.VolatileSurfaceManager", "sdCurrent"));

        RuntimeJNIAccess.register(java.awt.desktop.UserSessionEvent.Reason.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.desktop.UserSessionEvent$Reason", "CONSOLE", "LOCK", "REMOTE", "UNSPECIFIED"));

        registerJNIAccessForClassMembers(access, new ClassSunJava2dWindowsWindowsFlags());

        RuntimeJNIAccess.register(java.awt.Font.class);
        RuntimeJNIAccess.register(method(access, "java.awt.Font", "getFontPeer"));
        RuntimeJNIAccess.register(method(access, "java.awt.Font", "getFont", java.lang.String.class));
        RuntimeJNIAccess.register(fields(access, "java.awt.Font", "name", "pData", "size", "style"));

        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDesktopPeer", "systemSleepCallback", boolean.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDesktopPeer", "userSessionCallback", boolean.class, java.awt.desktop.UserSessionEvent.Reason.class));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WDesktopPeer"));
    }

    private static void registerHeadfullClasses(DuringAnalysisAccess access) {

        if (isLinux()) {
            registerLinuxClasses(access);
            registerLinuxX11Classes(access);
        } else if (isDarwin()) {
            registerDarwinClasses(access);
        } else if (isWindows()) {
            registerWindowsClasses(access);
        }

        registerLookAndFeelClasses(access);

        RuntimeJNIAccess.register(java.lang.String.class);
        RuntimeJNIAccess.register(java.util.Locale.class);
        RuntimeJNIAccess.register(method(access, "java.lang.String", "toLowerCase", java.util.Locale.class));

        RuntimeJNIAccess.register(java.util.ArrayList.class);
        RuntimeJNIAccess.register(constructor(access, "java.util.ArrayList"));
        RuntimeJNIAccess.register(constructor(access, "java.util.ArrayList", int.class));
        RuntimeJNIAccess.register(method(access, "java.util.ArrayList", "add", java.lang.Object.class));
        RuntimeJNIAccess.register(method(access, "java.util.ArrayList", "contains", java.lang.Object.class));

        RuntimeJNIAccess.register(java.awt.Dimension.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.Dimension", "height", "width"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.Dimension", int.class, int.class));

        RuntimeJNIAccess.register(java.awt.AWTEvent.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.AWTEvent", "bdata", "consumed", "id"));

        RuntimeReflection.register(clazz(access, "java.awt.SequencedEvent"));
        RuntimeJNIAccess.register(clazz(access, "java.awt.SequencedEvent"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.SequencedEvent", java.awt.AWTEvent.class));

        RuntimeJNIAccess.register(fields(access, "java.awt.Color", "value"));

        RuntimeJNIAccess.register(java.awt.DisplayMode.class);
        RuntimeJNIAccess.register(constructor(access, "java.awt.DisplayMode", int.class, int.class, int.class, int.class));

        RuntimeJNIAccess.register(java.awt.Font.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.Font", "pData", "size", "style"));
        RuntimeJNIAccess.register(method(access, "java.awt.Font", "getFamily_NoClientCode"));
        RuntimeJNIAccess.register(method(access, "java.awt.Font", "getFontPeer"));
        RuntimeJNIAccess.register(method(access, "java.awt.Font", "getName"));

        RuntimeJNIAccess.register(java.awt.Insets.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.Insets", "bottom", "left", "right", "top"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.Insets",
                int.class, int.class, int.class, int.class));

        RuntimeJNIAccess.register(java.awt.Rectangle.class);
        RuntimeJNIAccess.register(constructor(access, "java.awt.Rectangle",
                int.class, int.class, int.class, int.class));

        RuntimeJNIAccess.register(java.awt.event.InputEvent.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.event.InputEvent", "modifiers"));

        RuntimeJNIAccess.register(java.awt.image.DirectColorModel.class);
        RuntimeJNIAccess.register(constructor(access, "java.awt.image.DirectColorModel",
                int.class, int.class, int.class, int.class, int.class));

        RuntimeJNIAccess.register(java.lang.ClassLoader.class);
        RuntimeJNIAccess.register(method(access, "java.lang.ClassLoader", "getPlatformClassLoader"));
        RuntimeJNIAccess.register(method(access, "java.lang.ClassLoader", "loadClass", java.lang.String.class));

        RuntimeJNIAccess.register(java.lang.Thread.class);
        RuntimeJNIAccess.register(method(access, "java.lang.Thread", "yield"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.SunToolkit"));
        RuntimeJNIAccess.register(method(access, "sun.awt.SunToolkit", "awtLock"));
        RuntimeJNIAccess.register(method(access, "sun.awt.SunToolkit", "awtLockNotify"));
        RuntimeJNIAccess.register(method(access, "sun.awt.SunToolkit", "awtLockNotifyAll"));
        RuntimeJNIAccess.register(method(access, "sun.awt.SunToolkit", "awtLockWait", long.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.SunToolkit", "awtUnlock"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.image.ByteComponentRaster"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.image.ByteComponentRaster",
                "data", "dataOffsets", "pixelStride", "scanlineStride", "type"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.image.GifImageDecoder"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.image.GifImageDecoder",
                "outCode", "prefix", "suffix"));
        RuntimeJNIAccess.register(method(access, "sun.awt.image.GifImageDecoder",
                "readBytes", byte[].class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.image.GifImageDecoder", "sendPixels",
                int.class, int.class, int.class, int.class,
                byte[].class, java.awt.image.ColorModel.class));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.image.ImageRepresentation"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.image.ImageRepresentation",
                "numSrcLUT", "srcLUTtransIndex"));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.pipe.ShapeSpanIterator"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.pipe.ShapeSpanIterator", "pData"));

        RuntimeJNIAccess.register(java.io.InputStream.class);
        RuntimeJNIAccess.register(method(access, "java.io.InputStream", "available"));
        RuntimeJNIAccess.register(method(access, "java.io.InputStream", "read",
                byte[].class, int.class, int.class));

        RuntimeReflection.register(clazz(access, "sun.java2d.marlin.DMarlinRenderingEngine"));
        RuntimeReflection.register(constructor(access, "sun.java2d.marlin.DMarlinRenderingEngine"));

        RuntimeJNIAccess.register(java.lang.Boolean.class);
        RuntimeJNIAccess.register(constructor(access, "java.lang.Boolean", boolean.class));
        RuntimeJNIAccess.register(java.lang.Float.class);
        RuntimeJNIAccess.register(constructor(access, "java.lang.Float", float.class));
        RuntimeJNIAccess.register(java.lang.Integer.class);
        RuntimeJNIAccess.register(constructor(access, "java.lang.Integer", int.class));
    }

    private static void registerComponentUIs(DuringAnalysisAccess access, String packageName, String... componentUIs) {
        Stream.of(componentUIs)
                .map(name -> String.format("%s.%s", packageName, name))
                .forEach(componentUI -> {
                    RuntimeReflection.register(clazz(access, componentUI));
                    RuntimeReflection.register(method(access, componentUI, "createUI",
                            clazz(access, "javax.swing.JComponent")));
                });
    }

    private static void registerDNDClasses(DuringAnalysisAccess access) {

        RuntimeJNIAccess.register(clazz(access, "sun.awt.dnd.SunDragSourceContextPeer"));
        RuntimeJNIAccess.register(method(access, "sun.awt.dnd.SunDragSourceContextPeer", "dragEnter", int.class, int.class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.dnd.SunDragSourceContextPeer", "dragExit", int.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.dnd.SunDragSourceContextPeer", "operationChanged", int.class, int.class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.dnd.SunDragSourceContextPeer", "dragDropFinished", boolean.class, int.class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.dnd.SunDragSourceContextPeer", "dragMotion", int.class, int.class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.dnd.SunDragSourceContextPeer", "dragMouseMoved", int.class, int.class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.dnd.SunDropTargetContextPeer", "handleExitMessage", java.awt.Component.class, long.class));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.dnd.SunDropTargetContextPeer"));
        RuntimeJNIAccess.register(method(access, "sun.awt.dnd.SunDropTargetContextPeer", "handleMotionMessage", java.awt.Component.class, int.class, int.class, int.class, int.class, long[].class, long.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.dnd.SunDropTargetContextPeer", "handleDropMessage", java.awt.Component.class, int.class, int.class, int.class, int.class, long[].class, long.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.dnd.SunDropTargetContextPeer", "handleEnterMessage", java.awt.Component.class, int.class, int.class, int.class, int.class, long[].class, long.class));
    }

    private static void registerLookAndFeelClasses(DuringAnalysisAccess access) {

        // Basic L&F
        registerComponentUIs(access, "javax.swing.plaf.basic",
                "BasicButtonUI",
                "BasicCheckBoxUI",
                "BasicColorChooserUI",
                "BasicFormattedTextFieldUI",
                "BasicMenuBarUI",
                "BasicMenuUI",
                "BasicMenuItemUI",
                "BasicCheckBoxMenuItemUI",
                "BasicRadioButtonMenuItemUI",
                "BasicRadioButtonUI",
                "BasicToggleButtonUI",
                "BasicPopupMenuUI",
                "BasicProgressBarUI",
                "BasicScrollBarUI",
                "BasicScrollPaneUI",
                "BasicSplitPaneUI",
                "BasicSliderUI",
                "BasicSeparatorUI",
                "BasicSpinnerUI",
                "BasicToolBarSeparatorUI",
                "BasicPopupMenuSeparatorUI",
                "BasicTabbedPaneUI",
                "BasicTextAreaUI",
                "BasicTextFieldUI",
                "BasicPasswordFieldUI",
                "BasicTextPaneUI",
                "BasicEditorPaneUI",
                "BasicTreeUI",
                "BasicLabelUI",
                "BasicListUI",
                "BasicToolBarUI",
                "BasicToolTipUI",
                "BasicComboBoxUI",
                "BasicTableUI",
                "BasicTableHeaderUI",
                "BasicInternalFrameUI",
                "BasicDesktopPaneUI",
                "BasicDesktopIconUI",
                "BasicFileChooserUI",
                "BasicOptionPaneUI",
                "BasicPanelUI",
                "BasicViewportUI",
                "BasicRootPaneUI");

        Class<?> lazyActionMapClass = clazz(access, "javax.swing.plaf.basic.LazyActionMap");
        Stream.of("ButtonListener", "ComboBoxUI", "DesktopPaneUI", "InternalFrameUI", "LabelUI", "ListUI",
                        "MenuBarUI", "MenuItemUI", "MenuUI", "OptionPaneUI", "PopupMenuUI", "RootPaneUI",
                        "ScrollBarUI", "ScrollPaneUI", "SliderUI", "SpinnerUI", "SplitPaneUI", "TabbedPaneUI",
                        "TableHeaderUI", "TableUI", "ToolBarUI", "TreeUI")
                .map("javax.swing.plaf.basic.Basic"::concat)
                .map(className -> method(access, className, "loadActionMap", lazyActionMapClass))
                .forEach(RuntimeReflection::register);

        // Metal L&F
        RuntimeReflection.register(clazz(access, "javax.swing.plaf.metal.MetalLookAndFeel"));
        RuntimeReflection.register(constructor(access, "javax.swing.plaf.metal.MetalLookAndFeel"));

        registerComponentUIs(access, "javax.swing.plaf.metal",

                "MetalButtonUI",
                "MetalCheckBoxUI",
                "MetalComboBoxUI",
                "MetalDesktopIconUI",
                "MetalFileChooserUI",
                "MetalInternalFrameUI",
                "MetalLabelUI",
                "MetalPopupMenuSeparatorUI",
                "MetalProgressBarUI",
                "MetalRadioButtonUI",
                "MetalScrollBarUI",
                "MetalScrollPaneUI",
                "MetalSeparatorUI",
                "MetalSliderUI",
                "MetalSplitPaneUI",
                "MetalTabbedPaneUI",
                "MetalTextFieldUI",
                "MetalToggleButtonUI",
                "MetalToolBarUI",
                "MetalToolTipUI",
                "MetalTreeUI",
                "MetalRootPaneUI",
                "MetalMenuBarUI");

        // Synth L&F
        registerComponentUIs(access, "javax.swing.plaf.synth", "SynthLookAndFeel");

        // Nimbus L&F
        RuntimeReflection.register(clazz(access, "javax.swing.plaf.nimbus.NimbusLookAndFeel"));
        RuntimeReflection.register(constructor(access, "javax.swing.plaf.nimbus.NimbusLookAndFeel"));

        Stream.of(
                "MenuBarPainter",
                "SpinnerNextButtonPainter",
                "RadioButtonPainter",
                "TextAreaPainter",
                "CheckBoxMenuItemPainter",
                "FormattedTextFieldPainter",
                "ToggleButtonPainter",
                "InternalFrameTitlePaneMaximizeButtonPainter",
                "MenuPainter",
                "SliderThumbPainter",
                "DesktopIconPainter",
                "MenuBarMenuPainter",
                "TableEditorPainter",
                "ComboBoxTextFieldPainter",
                "RadioButtonMenuItemPainter",
                "InternalFrameTitlePaneMenuButtonPainter",
                "TextFieldPainter",
                "TableHeaderPainter",
                "InternalFrameTitlePaneIconifyButtonPainter",
                "DesktopPanePainter",
                "PopupMenuSeparatorPainter",
                "TreeCellPainter",
                "OptionPanePainter",
                "TextPanePainter",
                "ScrollPanePainter",
                "ToolBarToggleButtonPainter",
                "ProgressBarPainter",
                "TabbedPaneTabAreaPainter",
                "TableHeaderRendererPainter",
                "InternalFramePainter",
                "ComboBoxArrowButtonPainter",
                "SeparatorPainter",
                "ScrollBarButtonPainter",
                "SpinnerPanelSpinnerFormattedTextFieldPainter",
                "EditorPanePainter",
                "OptionPaneMessageAreaOptionPaneLabelPainter",
                "PopupMenuPainter",
                "TreeCellEditorPainter",
                "ButtonPainter",
                "CheckBoxPainter",
                "SliderTrackPainter",
                "SpinnerPreviousButtonPainter",
                "FileChooserPainter",
                "TabbedPaneTabPainter",
                "TreePainter",
                "InternalFrameTitlePaneCloseButtonPainter",
                "ToolBarPainter",
                "ToolTipPainter",
                "ToolBarButtonPainter",
                "ComboBoxPainter",
                "ScrollBarThumbPainter",
                "PasswordFieldPainter",
                "ArrowButtonPainter",
                "ScrollBarTrackPainter",
                "MenuItemPainter",
                "SplitPaneDividerPainter")
                .map("javax.swing.plaf.nimbus."::concat)
                .forEach(cls -> {
                    RuntimeReflection.register(clazz(access, cls));
                    RuntimeReflection.register(
                            constructor(access, cls,
                                    clazz(access, "javax.swing.plaf.nimbus.AbstractRegionPainter$PaintContext"),
                                    int.class));
                });

        // Motif L&F

        RuntimeReflection.register(clazz(access, "com.sun.java.swing.plaf.motif.MotifLookAndFeel"));
        RuntimeReflection.register(constructor(access, "com.sun.java.swing.plaf.motif.MotifLookAndFeel"));

        registerComponentUIs(access, "com.sun.java.swing.plaf.motif",
                "MotifButtonUI",
                "MotifCheckBoxUI",
                "MotifFileChooserUI",
                "MotifLabelUI",
                "MotifMenuBarUI",
                "MotifMenuUI",
                "MotifMenuItemUI",
                "MotifCheckBoxMenuItemUI",
                "MotifRadioButtonMenuItemUI",
                "MotifRadioButtonUI",
                "MotifToggleButtonUI",
                "MotifPopupMenuUI",
                "MotifProgressBarUI",
                "MotifScrollBarUI",
                "MotifScrollPaneUI",
                "MotifSliderUI",
                "MotifSplitPaneUI",
                "MotifTabbedPaneUI",
                "MotifTextAreaUI",
                "MotifTextFieldUI",
                "MotifPasswordFieldUI",
                "MotifTextPaneUI",
                "MotifEditorPaneUI",
                "MotifTreeUI",
                "MotifInternalFrameUI",
                "MotifDesktopPaneUI",
                "MotifSeparatorUI",
                "MotifPopupMenuSeparatorUI",
                "MotifOptionPaneUI",
                "MotifComboBoxUI",
                "MotifDesktopIconUI");

        if (isLinux()) {
            RuntimeReflection.register(clazz(access, "com.sun.java.swing.plaf.gtk.GTKLookAndFeel"));
            RuntimeReflection.register(constructor(access, "com.sun.java.swing.plaf.gtk.GTKLookAndFeel"));
            RuntimeReflection.register(method(access, "com.sun.java.swing.plaf.gtk.GTKLookAndFeel", "createUI",
                    javax.swing.JComponent.class));

            RuntimeReflection.register(clazz(access, "com.sun.java.swing.plaf.gtk.GTKIconFactory"));
            RuntimeReflection.register(method(access, "com.sun.java.swing.plaf.gtk.GTKIconFactory", "getCheckBoxMenuItemCheckIcon"));

            List.of("paintTreeExpandedIcon",
                    "paintTreeCollapsedIcon",
                    "paintCheckBoxIcon",
                    "paintRadioButtonIcon",
                    "paintCheckBoxMenuItemCheckIcon",
                    "paintRadioButtonMenuItemCheckIcon",
                    "paintAscendingSortIcon",
                    "paintDescendingSortIcon")
                    .forEach(paintIconMethod -> {
                        RuntimeReflection.register(method(access, "com.sun.java.swing.plaf.gtk.GTKPainter", paintIconMethod,
                                javax.swing.plaf.synth.SynthContext.class, java.awt.Graphics.class,
                                int.class, int.class, int.class, int.class, int.class));
                    });

            RuntimeReflection.register(method(access, "com.sun.java.swing.plaf.gtk.GTKPainter", "paintMenuArrowIcon",
                    javax.swing.plaf.synth.SynthContext.class, java.awt.Graphics.class,
                    int.class, int.class, int.class, int.class, int.class,
                    clazz(access, "com.sun.java.swing.plaf.gtk.GTKConstants$ArrowType")));

            RuntimeReflection.register(method(access, "com.sun.java.swing.plaf.gtk.GTKPainter", "paintToolBarHandleIcon",
                    javax.swing.plaf.synth.SynthContext.class, java.awt.Graphics.class,
                    int.class, int.class, int.class, int.class, int.class,
                    clazz(access, "com.sun.java.swing.plaf.gtk.GTKConstants$Orientation")));

            List.of("getAscendingSortIcon",
                    "getDescendingSortIcon",
                    "getTreeExpandedIcon",
                    "getTreeCollapsedIcon",
                    "getRadioButtonIcon",
                    "getCheckBoxIcon",
                    "getMenuArrowIcon",
                    "getCheckBoxMenuItemCheckIcon",
                    "getRadioButtonMenuItemCheckIcon",
                    "getToolBarHandleIcon")
                    .forEach(getIconMethod -> {
                        RuntimeReflection.register(method(access, "com.sun.java.swing.plaf.gtk.GTKIconFactory", getIconMethod));
                    });

            RuntimeReflection.register(method(access, "com.sun.java.swing.plaf.gtk.GTKPainter$ListTableFocusBorder",
                    "getSelectedCellBorder"));
        } else if (isDarwin()) {

            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaLookAndFeel"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaLookAndFeel"));

            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaMenuPainter"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaMenuPainter"));
            RuntimeReflection.register(clazz(access, "sun.swing.AltProcessor"));
            RuntimeReflection.register(constructor(access, "sun.swing.AltProcessor"));
            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaScrollRegionBorder"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaScrollRegionBorder"));
            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaTextFieldBorder"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaTextFieldBorder"));
            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaUtilControlSize$PropertySizeListener"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaUtilControlSize$PropertySizeListener"));

            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaKeyBindings"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaKeyBindings"));

            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaButtonBorder$Dynamic"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaButtonBorder$Dynamic"));
            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaButtonBorder$Named"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaButtonBorder$Named", clazz(access, "com.apple.laf.AquaButtonBorder$Named")));
            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaButtonBorder$Toggle"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaButtonBorder$Toggle"));
            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaButtonBorder$Toolbar"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaButtonBorder$Toolbar"));
            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaGroupBorder$TabbedPane"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaGroupBorder$TabbedPane"));
            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaGroupBorder$Titleless"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaGroupBorder$Titleless"));

            registerComponentUIs(access,"com.apple.laf",
                    "AquaButtonUI",
                    "AquaButtonCheckBoxUI",
                    "AquaMenuItemUI",
                    "AquaLabelUI",
                    "AquaListUI",
                    "AquaMenuUI",
                    "AquaMenuItemUI",
                    "AquaOptionPaneUI",
                    "AquaPanelUI",
                    "AquaMenuItemUI",
                    "AquaButtonRadioUI",
                    "AquaProgressBarUI",
                    "AquaRootPaneUI",
                    "AquaSliderUI",
                    "AquaScrollBarUI",
                    "AquaTabbedPaneContrastUI",
                    "AquaTabbedPaneUI",
                    "AquaTableUI",
                    "AquaButtonToggleUI",
                    "AquaToolBarUI",
                    "AquaToolTipUI",
                    "AquaTreeUI",
                    "AquaInternalFrameUI",
                    "AquaInternalFrameDockIconUI",
                    "AquaInternalFramePaneUI",
                    "AquaEditorPaneUI",
                    "AquaTextFieldUI",
                    "AquaTextPaneUI",
                    "AquaComboBoxUI",
                    "AquaPopupMenuUI",
                    "AquaTextAreaUI",
                    "AquaMenuBarUI",
                    "AquaFileChooserUI",
                    "AquaTextPasswordFieldUI",
                    "AquaTableHeaderUI",
                    "AquaTextFieldFormattedUI",
                    "AquaSpinnerUI",
                    "AquaSplitPaneUI",
                    "AquaScrollPaneUI",
                    "AquaPopupMenuSeparatorUI",
                    "AquaPopupMenuSeparatorUI",
                    "AquaToolBarSeparatorUI"
            );

            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaButtonCheckBoxUI"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaButtonRadioUI"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaButtonToggleUI"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaButtonUI"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaLabelUI"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaPanelUI"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaPopupMenuSeparatorUI"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaRootPaneUI"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaToolTipUI"));

            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaToolBarUI$ToolBarBorder"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaToolBarUI$ToolBarBorder"));
            RuntimeReflection.register(clazz(access, "com.apple.laf.AquaButtonUI$AquaHierarchyButtonListener"));
            RuntimeReflection.register(constructor(access, "com.apple.laf.AquaButtonUI$AquaHierarchyButtonListener"));

        } else if (isWindows()) {

            RuntimeReflection.register(clazz(access, "com.sun.java.swing.plaf.windows.WindowsLookAndFeel"));
            RuntimeReflection.register(constructor(access, "com.sun.java.swing.plaf.windows.WindowsLookAndFeel"));

            RuntimeReflection.register(clazz(access, "com.sun.java.swing.plaf.windows.WindowsClassicLookAndFeel"));
            RuntimeReflection.register(constructor(access, "com.sun.java.swing.plaf.windows.WindowsClassicLookAndFeel"));

            // exception from componentUI class
            // WindowsSeparatorUI does not contain static createUI(JComponent) method
            RuntimeReflection.register(clazz(access, "com.sun.java.swing.plaf.windows.WindowsSeparatorUI"));

            registerComponentUIs(access, "com.sun.java.swing.plaf.windows",
                    "WindowsButtonUI",
                    "WindowsCheckBoxUI",
                    "WindowsCheckBoxMenuItemUI",
                    "WindowsLabelUI",
                    "WindowsRadioButtonUI",
                    "WindowsRadioButtonMenuItemUI",
                    "WindowsToggleButtonUI",
                    "WindowsProgressBarUI",
                    "WindowsSliderUI",
                    // "WindowsSeparatorUI", // does not contain static createUI(JComponent) method
                    "WindowsSplitPaneUI",
                    "WindowsSpinnerUI",
                    "WindowsTabbedPaneUI",
                    "WindowsTextAreaUI",
                    "WindowsTextFieldUI",
                    "WindowsPasswordFieldUI",
                    "WindowsTextPaneUI",
                    "WindowsEditorPaneUI",
                    "WindowsTreeUI",
                    "WindowsToolBarUI",
                    "WindowsToolBarSeparatorUI",
                    "WindowsComboBoxUI",
                    "WindowsTableHeaderUI",
                    "WindowsInternalFrameUI",
                    "WindowsDesktopPaneUI",
                    "WindowsDesktopIconUI",
                    "WindowsFileChooserUI",
                    "WindowsMenuUI",
                    "WindowsMenuItemUI",
                    "WindowsMenuBarUI",
                    "WindowsPopupMenuUI",
                    "WindowsPopupMenuSeparatorUI",
                    "WindowsScrollBarUI",
                    "WindowsRootPaneUI");
        }
    }

    private static void registerLinuxClasses(DuringAnalysisAccess access) {
        RuntimeJNIAccess.register(clazz(access, "sun.java2d.xr.XRBackendNative"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.xr.XRBackendNative",
                "FMTPTR_A8", "FMTPTR_ARGB32", "MASK_XIMG"));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.xr.XRSurfaceData"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.xr.XRSurfaceData", "picture", "xid"));
    }

    private static void registerLinuxX11Classes(DuringAnalysisAccess access) {

        RuntimeJNIAccess.register(clazz(access, "sun.awt.UNIXToolkit"));
        RuntimeJNIAccess.register(method(access, "sun.awt.UNIXToolkit", "loadIconCallback",
                byte[].class, int.class, int.class, int.class, int.class, int.class, boolean.class));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11GraphicsEnvironment"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11.XWindow"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.X11.XWindow", "drawState",
                "graphicsConfig", "target"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11.XFramePeer"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11.XDialogPeer"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11.XRootWindow"));
        RuntimeJNIAccess.register(method(access, "sun.awt.X11.XRootWindow", "getXRootWindow"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11.XBaseWindow"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.X11.XBaseWindow", "window"));
        RuntimeJNIAccess.register(method(access, "sun.awt.X11.XBaseWindow", "getWindow"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11.XContentWindow"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11.XErrorHandlerUtil"));
        RuntimeJNIAccess.register(method(access, "sun.awt.X11.XErrorHandlerUtil", "init", long.class));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11.XToolkit"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.X11.XToolkit", "modLockIsShiftLock", "numLockMask"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11GraphicsConfig"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.X11GraphicsConfig", "aData", "bitsPerPixel"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11GraphicsDevice"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.X11GraphicsDevice", "screen"));
        RuntimeJNIAccess.register(method(access, "sun.awt.X11GraphicsDevice", "addDoubleBufferVisual", int.class));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.X11InputMethodBase"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.X11InputMethodBase", "pData"));

        RuntimeJNIAccess.register(method(access, "sun.awt.X11.XErrorHandlerUtil", "globalErrorHandler",
                long.class, long.class));
    }

    private static void registerDarwinClasses(DuringAnalysisAccess access) {

        registerKeyCodes(access);
        registerDNDClasses(access);
        registerJPEGClasses(access);
        registerSystemColor(access);
        registerDarwinMenu(access);

        RuntimeJNIAccess.register(constructor(access, "java.util.Locale", java.lang.String.class, java.lang.String.class, java.lang.String.class));

        RuntimeReflection.register(java.awt.Toolkit.class);
        RuntimeReflection.register(fields(access, "java.awt.Toolkit", "eventListener"));

        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.LWCToolkit"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.LWCToolkit", "installToolkitThreadInJava"));

        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.NSEvent"));
        RuntimeJNIAccess.register(constructor(access, "sun.lwawt.macosx.NSEvent", int.class, int.class, int.class, int.class, int.class, int.class, int.class, int.class, double.class, double.class, int.class));
        RuntimeJNIAccess.register(constructor(access, "sun.lwawt.macosx.NSEvent", int.class, int.class, clazz(access, "short"), java.lang.String.class, java.lang.String.class));

        RuntimeJNIAccess.register(method(access, "java.awt.event.InputEvent", "getButtonDownMasks"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.CGraphicsEnvironment"));
        RuntimeJNIAccess.register(method(access, "sun.awt.CGraphicsEnvironment", "_displayReconfiguration", int.class, boolean.class));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.datatransfer.DataTransferer"));
        RuntimeJNIAccess.register(method(access, "sun.awt.datatransfer.DataTransferer", "convertData", java.lang.Object.class, java.awt.datatransfer.Transferable.class, long.class, java.util.Map.class, boolean.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.datatransfer.DataTransferer", "getInstance"));

        RuntimeJNIAccess.register(clazz(access, "com.apple.eawt.FullScreenHandler"));
        RuntimeJNIAccess.register(method(access, "com.apple.eawt.FullScreenHandler", "handleFullScreenEventFromNative", java.awt.Window.class, int.class));
        RuntimeJNIAccess.register(clazz(access, "com.apple.eawt._AppEventHandler"));
        RuntimeJNIAccess.register(method(access, "com.apple.eawt._AppEventHandler", "handleNativeNotification", int.class));
        RuntimeJNIAccess.register(clazz(access, "com.apple.eawt._AppMenuBarHandler"));
        RuntimeJNIAccess.register(method(access, "com.apple.eawt._AppMenuBarHandler", "initMenuStates", boolean.class, boolean.class, boolean.class, boolean.class));

        RuntimeJNIAccess.register(java.awt.geom.Point2D.Double.class);
        RuntimeJNIAccess.register(constructor(access, "java.awt.geom.Point2D$Double", double.class, double.class));
        RuntimeJNIAccess.register(java.awt.geom.Rectangle2D.Double.class);
        RuntimeJNIAccess.register(constructor(access, "java.awt.geom.Rectangle2D$Double", double.class, double.class, double.class, double.class));
        RuntimeJNIAccess.register(method(access, "java.awt.geom.Rectangle2D$Float", "setRect", float.class, float.class, float.class, float.class));

        RuntimeJNIAccess.register(java.awt.image.Kernel.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.image.Kernel", "data", "height", "width"));
        RuntimeJNIAccess.register(java.awt.image.PackedColorModel.class);

        RuntimeJNIAccess.register(clazz(access, "javax.accessibility.Accessible"));
        RuntimeJNIAccess.register(clazz(access, "javax.accessibility.AccessibleBundle"));
        RuntimeJNIAccess.register(fields(access, "javax.accessibility.AccessibleBundle", "key"));
        RuntimeJNIAccess.register(clazz(access, "javax.accessibility.AccessibleRole"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.image.BytePackedRaster"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.image.BytePackedRaster", "data", "dataBitOffset", "pixelBitStride", "scanlineStride", "type"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.image.ShortComponentRaster"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.image.ShortComponentRaster", "data", "dataOffsets", "pixelStride", "scanlineStride", "type"));
        RuntimeJNIAccess.register(clazz(access, "sun.font.CFontManager"));
        RuntimeJNIAccess.register(method(access, "sun.font.CFontManager", "registerFont", java.lang.String.class, java.lang.String.class));
        RuntimeJNIAccess.register(fields(access, "sun.font.GlyphList", "images", "lcdRGBOrder", "lcdSubPixPos", "len", "positions", "usePositions", "x", "y"));
        //RuntimeJNIAccess.register(method(access, "sun.java2d.loops.GraphicsPrimitiveMgr", "register", clazz(access, "sun.java2d.loops.GraphicsPrimitive[]")));

        RuntimeJNIAccess.register(clazz(access, "sun.java2d.opengl.CGLLayer"));
        RuntimeJNIAccess.register(method(access, "sun.java2d.opengl.CGLLayer", "drawInCGLContext"));
        RuntimeJNIAccess.register(clazz(access, "sun.java2d.opengl.CGLSurfaceData$CGLLayerSurfaceData"));
        RuntimeJNIAccess.register(clazz(access, "sun.java2d.opengl.CGLSurfaceData$CGLOffScreenSurfaceData"));
        RuntimeJNIAccess.register(clazz(access, "sun.java2d.opengl.OGLSurfaceData"));
        RuntimeJNIAccess.register(fields(access, "sun.java2d.opengl.OGLSurfaceData", "isBIOpShaderEnabled", "isFBObjectEnabled", "isGradShaderEnabled", "isLCDShaderEnabled", "nativeHeight", "nativeWidth"));
        RuntimeJNIAccess.register(method(access, "sun.java2d.opengl.OGLSurfaceData", "dispose", long.class, clazz(access, "sun.java2d.opengl.OGLGraphicsConfig")));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.LWComponentPeer"));
        RuntimeJNIAccess.register(fields(access, "sun.lwawt.LWComponentPeer", "target"));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.LWWindowPeer"));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.CAccessibility"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CAccessibility", "getAccessibility", java.lang.String[].class));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CAccessibility", "getAccessibleIndexInParent", clazz(access, "javax.accessibility.Accessible"), java.awt.Component.class));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CAccessibility", "getAccessibleRole", clazz(access, "javax.accessibility.Accessible"), java.awt.Component.class));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CAccessibility", "getFocusOwner", java.awt.Component.class));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.CAccessible"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CAccessible", "getCAccessible", clazz(access, "javax.accessibility.Accessible")));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.CDragSourceContextPeer"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CDragSourceContextPeer", "dragMouseMoved", int.class, int.class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CDragSourceContextPeer", "resetHovering"));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.CDropTargetContextPeer"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CDropTargetContextPeer", "getDropTargetContextPeer"));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.CFRetainedResource"));
        RuntimeJNIAccess.register(fields(access, "sun.lwawt.macosx.CFRetainedResource", "ptr"));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.CInputMethod"));
        RuntimeJNIAccess.register(fields(access, "sun.lwawt.macosx.CInputMethod", "fCurrentText", "fCurrentTextLength"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CInputMethod", "unmarkText"));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.CPlatformView"));
        RuntimeJNIAccess.register(fields(access, "sun.lwawt.macosx.CPlatformView", "peer"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformView", "deliverKeyEvent", clazz(access, "sun.lwawt.macosx.NSEvent")));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformView", "deliverMouseEvent", clazz(access, "sun.lwawt.macosx.NSEvent")));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.CPlatformWindow"));
        RuntimeJNIAccess.register(fields(access, "sun.lwawt.macosx.CPlatformWindow", "target"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "deliverIconify", boolean.class));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "deliverMoveResizeEvent", int.class, int.class, int.class, int.class, boolean.class));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "deliverNCMouseDown"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "deliverWindowClosingEvent"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "deliverWindowFocusEvent", boolean.class, clazz(access, "sun.lwawt.macosx.CPlatformWindow")));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "isBlocked"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "isSimpleWindowOwnedByEmbeddedFrame"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "isVisible"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "windowDidBecomeMain"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "windowDidEnterFullScreen"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "windowDidExitFullScreen"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "windowWillEnterFullScreen"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "windowWillExitFullScreen"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "windowWillMiniaturize"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CPlatformWindow", "checkBlockingAndOrder"));

        RuntimeReflection.register(java.awt.event.MouseMotionListener.class);
    }

    private static void registerDarwinMenu(DuringAnalysisAccess access) {
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.CCheckboxMenuItem"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CCheckboxMenuItem", "handleAction", boolean.class));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.CMenuItem"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CMenuItem", "handleAction", long.class, int.class));
        RuntimeJNIAccess.register(clazz(access, "sun.lwawt.macosx.CTrayIcon"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CTrayIcon", "getPopupMenuModel"));
        RuntimeJNIAccess.register(method(access, "sun.lwawt.macosx.CTrayIcon", "handleMouseEvent", clazz(access, "sun.lwawt.macosx.NSEvent")));
    }

    private static void registerWindowsClasses(DuringAnalysisAccess access) {

        registerKeyCodes(access);
        registerJPEGClasses(access);
        registerSystemColor(access);

        registerEvent(access);
        registerWindowsMenu(access);
        registerTrayIcon(access);

        RuntimeJNIAccess.register(java.util.HashMap.class);
        RuntimeJNIAccess.register(method(access, "java.util.HashMap", "containsKey", java.lang.Object.class));
        RuntimeJNIAccess.register(method(access, "java.util.HashMap", "put", java.lang.Object.class, java.lang.Object.class));

        // Events
        RuntimeJNIAccess.register(java.awt.event.MouseEvent.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.event.MouseEvent",
                "x", "y", "button", "causedByTouchEvent"));

        RuntimeJNIAccess.register(constructor(access, "java.awt.event.MouseEvent",
                clazz(access, "java.awt.Component"),
                int.class, long.class, int.class, int.class,
                int.class, int.class, int.class, int.class, boolean.class, int.class
        ));

        RuntimeJNIAccess.register(constructor(access, "java.awt.event.MouseWheelEvent",
                java.awt.Component.class,
                int.class, long.class, int.class, int.class,
                int.class, int.class, int.class, int.class, boolean.class, int.class,
                int.class, int.class, double.class
        ));

        RuntimeJNIAccess.register(clazz(access, "java.awt.event.FocusEvent"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.event.FocusEvent", java.awt.Component.class, int.class, boolean.class,java.awt.Component.class));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.UngrabEvent"));
        RuntimeJNIAccess.register(constructor(access, "sun.awt.UngrabEvent", java.awt.Component.class));


        RuntimeJNIAccess.register(method(access, "java.awt.Component", "getFont_NoClientCode"));
        RuntimeJNIAccess.register(method(access, "java.awt.Component", "getToolkitImpl"));
        RuntimeJNIAccess.register(method(access, "java.awt.Component", "isEnabledImpl"));

        RuntimeJNIAccess.register(fields(access, "java.awt.Container", "layoutMgr"));
        RuntimeJNIAccess.register(fields(access, "java.awt.Cursor",
                "pData", "type"));

        RuntimeJNIAccess.register(method(access, "java.awt.Cursor", "setPData", long.class));

        RuntimeJNIAccess.register(java.awt.Dialog.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.Dialog", "title", "undecorated"));

        RuntimeJNIAccess.register(fields(access, "java.awt.Point", "x", "y"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.Point", int.class, int.class));

        RuntimeReflection.register(java.awt.event.MouseMotionListener.class);

        // Popup
        RuntimeJNIAccess.register(clazz(access, "com.sun.java.swing.plaf.windows.WindowsPopupWindow"));
        RuntimeJNIAccess.register(clazz(access, "javax.swing.Popup$HeavyWeightWindow"));

        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDesktopProperties", "setColorProperty", java.lang.String.class, int.class, int.class, int.class));

        RuntimeJNIAccess.register(java.awt.FontMetrics.class);
        RuntimeJNIAccess.register(clazz(access, "sun.awt.image.IntegerComponentRaster"));

        registerDNDClasses(access);

        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WDragSourceContextPeer"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WDropTargetContextPeer"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDropTargetContextPeer", "getWDropTargetContextPeer"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDropTargetContextPeer", "getFileStream", java.lang.String.class, long.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDropTargetContextPeer", "getIStream", long.class));


        RuntimeJNIAccess.register(clazz(access, "sun.awt.im.InputMethodWindow"));
        RuntimeJNIAccess.register(java.awt.desktop.UserSessionEvent.Reason.class);
        RuntimeJNIAccess.register(clazz(access, "sun.awt.FontDescriptor"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDesktopProperties", "setFontProperty", java.lang.String.class, java.lang.String.class, int.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDesktopProperties", "setIntegerProperty", java.lang.String.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WComponentPeer", "disposeLater"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WComponentPeer", "handleExpose", int.class, int.class, int.class, int.class));
        RuntimeJNIAccess.register(fields(access, "java.awt.Component", "appContext", "background", "cursor", "enabled", "focusable", "foreground", "graphicsConfig", "height", "parent", "peer", "visible", "width", "x", "y"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WComponentPeer"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.PlatformFont", "componentFonts", "fontConfig"));
        RuntimeJNIAccess.register(method(access, "sun.awt.ExtendedKeyCodes", "getExtendedKeyCodeForChar", int.class));
        RuntimeJNIAccess.register(method(access, "java.awt.event.InputEvent", "getButtonDownMasks"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.Win32GraphicsConfig"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.Win32GraphicsConfig", "visual"));

        RuntimeJNIAccess.register(fields(access, "sun.awt.Win32GraphicsDevice", "dynamicColorModel"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.Win32GraphicsDevice"));

        RuntimeJNIAccess.register(constructor(access, "sun.awt.TimedWindowEvent", java.awt.Window.class, int.class, java.awt.Window.class, int.class, int.class, long.class));
        RuntimeJNIAccess.register(method(access, "java.awt.Font", "getFont", java.lang.String.class));
        RuntimeJNIAccess.register(fields(access, "java.awt.Font", "name", "pData", "size", "style"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WWindowPeer", "setBackground", java.awt.Color.class));
        RuntimeJNIAccess.register(fields(access, "java.awt.event.KeyEvent", "extendedKeyCode", "keyChar", "keyCode", "primaryLevelUnicode", "rawCode", "scancode"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.event.MouseEvent", java.awt.Component.class, int.class, long.class, int.class, int.class, int.class, int.class, int.class, int.class, boolean.class, int.class));
        RuntimeJNIAccess.register(fields(access, "java.awt.desktop.UserSessionEvent$Reason", "CONSOLE", "LOCK", "REMOTE", "UNSPECIFIED"));
        RuntimeJNIAccess.register(method(access, "java.lang.Boolean", "getBoolean", java.lang.String.class));
        RuntimeJNIAccess.register(java.awt.event.MouseWheelEvent.class);
        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WFramePeer"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.event.MouseWheelEvent", java.awt.Component.class, int.class, long.class, int.class, int.class, int.class, int.class, int.class, int.class, boolean.class, int.class, int.class, int.class, double.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDesktopPeer", "userSessionCallback", boolean.class, java.awt.desktop.UserSessionEvent.Reason.class));
        RuntimeJNIAccess.register(fields(access, "sun.awt.windows.WDesktopProperties", "pData"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.image.DirectColorModel", int.class, int.class, int.class, int.class));
        RuntimeJNIAccess.register(fields(access, "java.awt.event.MouseEvent", "button", "causedByTouchEvent", "x", "y"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.TimedWindowEvent"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDesktopProperties", "setStringProperty", java.lang.String.class, java.lang.String.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WComponentPeer", "postEvent", java.awt.AWTEvent.class));

        registerJNIAccessForClassMembers(access, new ClassJavaAwtWindowRegistration());

        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WGlobalCursorManager"));
        RuntimeJNIAccess.register(java.awt.Cursor.class);
        RuntimeJNIAccess.register(java.awt.Frame.class);
        RuntimeJNIAccess.register(java.awt.Window.Type.class);
        RuntimeJNIAccess.register(fields(access, "sun.awt.windows.WPanelPeer", "insets_"));
        RuntimeJNIAccess.register(method(access, "sun.awt.PlatformFont", "makeConvertedMultiFontString", java.lang.String.class));

        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WFramePeer", "getExtendedState"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.windows.WFramePeer", "keepOnMinimize"));

        RuntimeJNIAccess.register(fields(access, "sun.awt.FontDescriptor", "nativeName", "useUnicode"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.PlatformFont"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WComponentPeer", "replaceSurfaceData"));
        RuntimeJNIAccess.register(fields(access, "java.awt.Frame", "undecorated"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WFontPeer"));

        RuntimeJNIAccess.register(fields(access, "sun.awt.windows.WComponentPeer", "hwnd", "winGraphicsConfig"));
        RuntimeJNIAccess.register(method(access, "sun.awt.SunToolkit", "isTouchKeyboardAutoShowEnabled"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.windows.WFontPeer", "textComponentFontName"));

        RuntimeJNIAccess.register(clazz(access, "sun.font.FontDesignMetrics"));
        RuntimeJNIAccess.register(method(access, "sun.font.FontDesignMetrics", "getHeight"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WDesktopProperties"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.windows.WObjectPeer", "createError", "destroyed", "pData", "target"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WComponentPeer", "replaceSurfaceDataLater"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WWindowPeer", "draggedToNewScreen"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.EmbeddedFrame"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WObjectPeer", "getPeerForTarget", java.lang.Object.class));
        RuntimeJNIAccess.register(method(access, "java.lang.Thread", "currentThread"));
        RuntimeJNIAccess.register(fields(access, "java.awt.FontMetrics", "font"));
        RuntimeJNIAccess.register(method(access, "java.awt.FontMetrics", "getHeight"));
        RuntimeJNIAccess.register(fields(access, "java.awt.image.IndexColorModel", "allgrayopaque", "colorData", "lookupcache", "map_size", "rgb", "transparent_index"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.image.IndexColorModel", int.class, int.class, int[].class, int.class, int.class, java.math.BigInteger.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDesktopProperties", "setBooleanProperty", java.lang.String.class, boolean.class));
        RuntimeJNIAccess.register(method(access, "java.lang.Enum", "name"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WPanelPeer"));
        RuntimeJNIAccess.register(fields(access, "sun.awt.windows.WWindowPeer", "windowType"));
        RuntimeJNIAccess.register(java.awt.event.ComponentEvent.class);
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WGlobalCursorManager", "nativeUpdateCursor", java.awt.Component.class));
        RuntimeJNIAccess.register(java.awt.Container.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.Rectangle", "height", "width", "x", "y"));
        RuntimeJNIAccess.register(constructor(access, "java.awt.event.ComponentEvent", java.awt.Component.class, int.class));
        RuntimeJNIAccess.register(java.awt.Point.class);
        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WObjectPeer"));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.ExtendedKeyCodes"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WDesktopProperties", "setSoundProperty", java.lang.String.class, java.lang.String.class));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.LightweightFrame"));
        RuntimeJNIAccess.register(method(access, "java.util.Locale", "forLanguageTag", java.lang.String.class));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WWindowPeer"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WWindowPeer", "notifyWindowStateChanged", int.class, int.class));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WDesktopPeer"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WFileDialogPeer"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WWindowPeer", "getActiveWindowHandles", java.awt.Component.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WFileDialogPeer", "checkFilenameFilter", java.lang.String.class));
        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WClipboard"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WFileDialogPeer", "handleSelected", char[].class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WClipboard", "lostSelectionOwnershipImpl"));

        RuntimeJNIAccess.register(constructor(access, "java.awt.event.KeyEvent", java.awt.Component.class, int.class, long.class, int.class, int.class, char.class, int.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WFileDialogPeer", "isMultipleMode"));
        RuntimeJNIAccess.register(java.awt.FileDialog.class);
        RuntimeJNIAccess.register(fields(access, "java.awt.FileDialog", "dir", "file", "filter", "mode"));
        RuntimeJNIAccess.register(java.awt.Toolkit.class);
        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WInputMethod"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WFileDialogPeer", "setHWnd", long.class));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WFileDialogPeer", "handleCancel"));
        RuntimeJNIAccess.register(method(access, "sun.awt.windows.WInputMethod", "inquireCandidatePosition"));
        RuntimeJNIAccess.register(method(access, "java.awt.Toolkit", "getFontMetrics", java.awt.Font.class));
        RuntimeJNIAccess.register(fields(access, "sun.awt.windows.WFileDialogPeer", "fileFilter", "parent"));

        RuntimeJNIAccess.register(clazz(access, "sun.awt.windows.WDialogPeer"));

        RuntimeReflection.register(clazz(access, "sun.awt.Symbol"));
        RuntimeReflection.register(constructor(access, "sun.awt.windows.WingDings"));
    }

    private static void registerJNIAccessForClassMembers(Feature.DuringAnalysisAccess access,
                    BaseClassMemberRegistration memberRegistration) {
        String className = memberRegistration.getClassName();
        // Register class name
        RuntimeJNIAccess.register(clazz(access, className));
        // Register class fields
        RuntimeJNIAccess.register(JNIRegistrationUtil.fields(access, className, memberRegistration.getFields()));
        // Register class methods
        MethodHandles.Lookup lookup = MethodHandles.lookup();
        for (BaseClassMemberRegistration.MethodSignature m : memberRegistration.getMethods()) {
            String methodName = m.name();
            MethodTypeDesc methodTypeDesc = m.methodTypeDesc();
            RuntimeJNIAccess.register(method(access, className, methodName, readMethodSignature(methodTypeDesc, lookup)));
        }
    }

    private static Class<?>[] readMethodSignature(MethodTypeDesc methodTypeDesc, MethodHandles.Lookup lookup) {
        List<ClassDesc> paramTYpeDescriptorList = methodTypeDesc.parameterList();
        Class<?>[] paramTypes = new Class<?>[paramTYpeDescriptorList.size()];
        for (int i = 0; i < paramTYpeDescriptorList.size(); i++) {
            ClassDesc classDesc = paramTYpeDescriptorList.get(i);
            try {
                paramTypes[i] = classDesc.resolveConstantDesc(lookup);
            } catch (ReflectiveOperationException e) {
                throw new RuntimeException(e);
            }
        }
        return paramTypes;
    }
}
