/*
 * Copyright (c) 2015, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 3 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.truffle.r.test.access.vector;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.Assume.assumeFalse;
import static org.junit.Assume.assumeTrue;

import org.junit.Assert;
import org.junit.Test;
import org.junit.experimental.theories.DataPoints;
import org.junit.experimental.theories.Theories;
import org.junit.experimental.theories.Theory;
import org.junit.runner.RunWith;

import com.oracle.truffle.r.nodes.access.vector.SearchFirstStringNode.CompareStringNode;
import com.oracle.truffle.r.nodes.access.vector.SearchFirstStringNode.CompareStringNode.StringEqualsNode;
import com.oracle.truffle.r.runtime.RRuntime;
import com.oracle.truffle.r.test.test.TestBase;
import com.oracle.truffle.r.test.test.TestUtilities;
import com.oracle.truffle.r.test.test.TestUtilities.NodeHandle;

@RunWith(Theories.class)
public class StringCompareNodeTest extends TestBase {
    @Test
    public void dummy() {
        // to make sure this file is recognized as a test
    }

    // Please note that "FB" and "Ea" produce a hash collision. Thats why its tested here.
    @DataPoints public static String[] TEST = {RRuntime.STRING_NA, "a", "abc", "bc".intern(), "bc".intern(), "FB", "Ea"};

    @Theory
    public void testExactNA(String a, String b) {
        execInContext(() -> {
            assumeTrue(a == RRuntime.STRING_NA || b == RRuntime.STRING_NA);
            try {
                executeCompare(true, a, b);
                Assert.fail();
            } catch (AssertionError e) {
            }
            return null;
        });
    }

    @Theory
    public void testNonExactNA(String a, String b) {
        execInContext(() -> {
            assumeTrue(a == RRuntime.STRING_NA || b == RRuntime.STRING_NA);
            try {
                executeCompare(false, a, b);
                Assert.fail();
            } catch (AssertionError e) {
            }
            return null;
        });
    }

    @Theory
    public void testExact(String a, String b) {
        execInContext(() -> {
            assumeFalse(a == RRuntime.STRING_NA);
            assumeFalse(b == RRuntime.STRING_NA);
            assertThat(executeCompare(true, a, b), is(a.equals(b)));
            assertThat(executeHashCompare(a, b), is(a.equals(b)));
            return null;
        });
    }

    @Theory
    public void testNonExact(String a, String b) {
        execInContext(() -> {
            assumeFalse(a == RRuntime.STRING_NA);
            assumeFalse(b == RRuntime.STRING_NA);
            assertThat(executeCompare(false, a, b), is(a.startsWith(b)));
            return null;
        });
    }

    private static boolean executeCompare(boolean exact, String a, String b) {
        NodeHandle<CompareStringNode> handle = TestUtilities.createHandle(exact ? CompareStringNode.createEquals() : CompareStringNode.createStartsWith(),
                        (node, args) -> node.executeCompare((String) args[0], (String) args[1]));
        return (Boolean) handle.call(a, b);
    }

    private static boolean executeHashCompare(String a, String b) {
        NodeHandle<StringEqualsNode> handle = TestUtilities.createHandle(CompareStringNode.createEquals(),
                        (node, args) -> node.executeCompare((String) args[0], ((String) args[0]).hashCode(), (String) args[1]));
        return (Boolean) handle.call(a, b);
    }
}
