/*
 * Copyright (c) 2013, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 3 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.truffle.r.test;

import java.util.HashMap;
import java.util.Map;

/**
 * A {@link IncludeList} instance is a map of test inputs to a pairs:
 * <ol>
 * <li>the expected test output, i.e. from GnuR, for a test input</li>
 * <li>the output actually generated by FastR for the input</li>
 * </ol>
 * When the FastR output fails to match the expected GnuR output, the {@link IncludeList} instance
 * can be consulted for a match for the input. If one is found, the FastR output recorded in the the
 * {@link IncludeList} instance can then be used in the match. Evidently care must be taken to
 * ensure that the FastR output is in fact acceptable for the test input.
 *
 */
public final class IncludeList implements TestTrait {
    private static final Map<String, IncludeList> all = new HashMap<>();

    public static class Results {

        public final String fastR;
        public final String expected;
        private boolean used;

        public Results(String fastR, String expected) {
            this.fastR = fastR;
            this.expected = expected;
        }

        @Override
        public String toString() {
            return "FastR: " + fastR + ", Expected: " + expected;
        }
    }

    private final Map<String, Results> map = new HashMap<>();
    private final String name;

    private IncludeList(String name) {
        this.name = name;
    }

    public static IncludeList create(String name) {
        IncludeList result = all.get(name);
        if (result == null) {
            result = new IncludeList(name);
            all.put(name, result);
        }
        return result;
    }

    public void add(String input, String actual, String expected) {
        map.put(input, new Results(actual, expected));
    }

    Results get(String expression) {
        return map.get(expression);
    }

    void markUsed(String expression) {
        map.get(expression).used = true;
    }

    public static void report() {
        for (Map.Entry<String, IncludeList> listEntry : all.entrySet()) {
            Map<String, Results> map = listEntry.getValue().map;
            int unusedCount = map.size();
            for (Map.Entry<String, Results> entry : map.entrySet()) {
                if (entry.getValue().used) {
                    unusedCount--;
                } else {
                    System.out.println("unused entry: " + entry.getKey());
                }
            }
            if (unusedCount > 0) {
                System.out.printf("%n%d unused entries in include-list (%s)%n", unusedCount, listEntry.getKey());
            }
        }
    }

    @Override
    public String getName() {
        return name;
    }
}
