/*
 * Copyright (c) 2016, 2019, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package jdk.graal.compiler.api.test;

import java.io.IOException;
import java.lang.module.ModuleDescriptor.Requires;
import java.net.URI;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import jdk.graal.compiler.debug.DebugOptions;

import jdk.internal.module.Modules;

public class ModuleSupport {

    /**
     * Exports the package named {@code packageName} declared in {@code moduleMember}'s module to
     * {@code requestor}'s module.
     */
    public static void exportPackageTo(Class<?> moduleMember, String packageName, Class<?> requestor) {
        Module moduleToExport = moduleMember.getModule();
        Module requestorModule = requestor.getModule();
        if (moduleToExport != requestorModule) {
            Modules.addExports(moduleToExport, packageName, requestorModule);
        }
    }

    /**
     * Exports all packages declared in {@code moduleMember}'s module to the unnamed module
     * associated with {@code cl}.
     */
    public static void exportAllPackagesTo(Class<?> moduleMember, ClassLoader cl) {
        Module moduleToExport = moduleMember.getModule();
        Module unnamedModule = cl.getUnnamedModule();
        for (String packageName : moduleToExport.getPackages()) {
            Modules.addExports(moduleToExport, packageName, unnamedModule);
        }
    }

    /**
     * Exports and opens all packages in the module named {@code name} to all unnamed modules.
     */
    public static void exportAndOpenAllPackagesToUnnamed(String name) {
        exportAndOpenAllPackagesToUnnamed(name, true);
    }

    /**
     * Exports and opens all packages in the module named {@code name} to all unnamed modules.
     */
    public static void exportAndOpenAllPackagesToUnnamed(String name, boolean required) {
        Optional<Module> maybeModule = ModuleLayer.boot().findModule(name);
        Module module = required ? maybeModule.orElseThrow() : maybeModule.orElse(null);
        if (module == null) {
            return;
        }
        Set<String> packages = module.getPackages();
        for (String pkg : packages) {
            Modules.addExportsToAllUnnamed(module, pkg);
            Modules.addOpensToAllUnnamed(module, pkg);
        }
    }

    /**
     * Gets the names of the classes in the JDK image (i.e. lib/modules) that implement Graal.
     *
     * @throws IOException
     */
    public static List<String> getJRTGraalClassNames() throws IOException {
        List<String> classNames = new ArrayList<>();
        FileSystem fs = FileSystems.newFileSystem(URI.create("jrt:/"), Collections.emptyMap());
        Module graalModule = DebugOptions.class.getModule();
        Set<String> graalModuleSet = new HashSet<>();
        graalModuleSet.add(graalModule.getName());
        for (Module module : graalModule.getLayer().modules()) {
            if (requires(module, graalModule)) {
                graalModuleSet.add(module.getName());
            }
        }

        Path top = fs.getPath("/modules/");
        Files.find(top, Integer.MAX_VALUE,
                        (path, attrs) -> attrs.isRegularFile()).forEach(p -> {
                            int nameCount = p.getNameCount();
                            if (nameCount > 2) {
                                String base = p.getName(nameCount - 1).toString();
                                if (base.endsWith(".class") && !base.equals("module-info.class")) {
                                    String module = p.getName(1).toString();
                                    if (graalModuleSet.contains(module)) {
                                        // Strip module prefix and convert to dotted
                                        // form
                                        String className = p.subpath(2, nameCount).toString().replace('/', '.');
                                        // Strip ".class" suffix
                                        className = className.replace('/', '.').substring(0, className.length() - ".class".length());
                                        classNames.add(className);
                                    }
                                }
                            }
                        });
        return classNames;
    }

    private static boolean requires(Module module, Module graalModule) {
        ModuleLayer graalLayer = graalModule.getLayer();
        for (Requires r : module.getDescriptor().requires()) {
            if (r.name().equals(graalModule.getName())) {
                return true;
            }
            Module dep = graalLayer.findModule(r.name()).get();
            if (requires(dep, graalModule)) {
                return true;
            }
        }
        return false;
    }
}
