/*
 * Copyright (c) 2024, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.truffle.espresso.substitutions;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import com.oracle.truffle.espresso.classfile.descriptors.ByteSequence;
import com.oracle.truffle.espresso.meta.EspressoError;
import com.oracle.truffle.espresso.runtime.Classpath;

public class JImageExtensions {
    private final ZipFile zipFile;
    private final Map<String, Set<String>> extensions;

    public JImageExtensions(Path jarPath, Map<String, Set<String>> extensions) {
        this.extensions = extensions;
        try {
            zipFile = new ZipFile(jarPath.toFile());
        } catch (IOException e) {
            throw EspressoError.shouldNotReachHere(e);
        }
    }

    public byte[] getClassBytes(ByteSequence name) {
        ZipEntry entry = zipFile.getEntry(name.toString());
        if (entry == null) {
            return null;
        }
        try {
            return Classpath.readZipEntry(zipFile, entry);
        } catch (IOException e) {
            throw EspressoError.shouldNotReachHere(e);
        }
    }

    public Map<String, Set<String>> getExtensions() {
        return extensions;
    }
}
