/*
 * Copyright (c) 2018, 2022, Oracle and/or its affiliates.
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of
 * conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other materials provided
 * with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.oracle.truffle.llvm.runtime.nodes.intrinsics.interop;

import com.oracle.truffle.api.dsl.Fallback;
import com.oracle.truffle.api.dsl.GenerateAOT;
import com.oracle.truffle.api.dsl.NodeChild;
import com.oracle.truffle.api.dsl.Specialization;
import com.oracle.truffle.api.interop.InteropLibrary;
import com.oracle.truffle.api.library.CachedLibrary;
import com.oracle.truffle.llvm.runtime.library.internal.LLVMAsForeignLibrary;
import com.oracle.truffle.llvm.runtime.nodes.api.LLVMExpressionNode;
import com.oracle.truffle.llvm.runtime.nodes.intrinsics.llvm.LLVMIntrinsic;
import com.oracle.truffle.llvm.runtime.pointer.LLVMManagedPointer;
import com.oracle.truffle.llvm.runtime.pointer.LLVMNativePointer;

@NodeChild(value = "object", type = LLVMExpressionNode.class)
public abstract class LLVMPolyglotBoxedPredicate extends LLVMIntrinsic {

    @FunctionalInterface
    public interface Predicate {

        boolean match(InteropLibrary interop, Object obj);
    }

    final Predicate predicate;
    @Child InteropLibrary stringInterop = InteropLibrary.getFactory().create("");

    protected LLVMPolyglotBoxedPredicate(Predicate predicate) {
        this.predicate = predicate;
    }

    @Specialization(guards = "!foreigns.isForeign(pointer)")
    @GenerateAOT.Exclude
    boolean matchNonForeignManaged(@SuppressWarnings("unused") LLVMManagedPointer pointer,
                    @SuppressWarnings("unused") @CachedLibrary(limit = "3") LLVMAsForeignLibrary foreigns,
                    @CachedLibrary(limit = "3") InteropLibrary interop) {
        return predicate.match(interop, pointer);
    }

    @Specialization(guards = "foreigns.isForeign(pointer)")
    @GenerateAOT.Exclude
    boolean matchForeignManaged(LLVMManagedPointer pointer,
                    @CachedLibrary(limit = "3") LLVMAsForeignLibrary foreigns,
                    @CachedLibrary(limit = "3") InteropLibrary interop) {
        Object foreign = foreigns.asForeign(pointer.getObject());
        assert foreign != null;
        return predicate.match(interop, foreign);
    }

    @Specialization
    @GenerateAOT.Exclude
    boolean matchNative(LLVMNativePointer pointer,
                    @CachedLibrary(limit = "3") InteropLibrary interop) {
        return predicate.match(interop, pointer);
    }

    @Specialization
    boolean matchString(String str) {
        return predicate.match(stringInterop, str);
    }

    @Fallback
    public boolean fallback(@SuppressWarnings("unused") Object object) {
        return false;
    }
}
