/*
 * Copyright (c) 2020, 2024, Oracle and/or its affiliates.
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of
 * conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other materials provided
 * with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.oracle.truffle.llvm.tests.internal.bitcode;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.Assert.assertThrows;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.regex.Pattern;

import org.graalvm.polyglot.PolyglotException;
import org.graalvm.polyglot.Value;
import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.hamcrest.core.StringContains;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;

import com.oracle.truffle.api.CallTarget;
import com.oracle.truffle.api.TruffleFile;
import com.oracle.truffle.llvm.tests.Platform;
import com.oracle.truffle.llvm.tests.options.TestOptions;
import com.oracle.truffle.tck.TruffleRunner;

public class AllocationLimitsTest {

    private static final Path TEST_DIR = new File(TestOptions.getTestDistribution("SULONG_EMBEDDED_TEST_SUITES"), "other").toPath();
    private static final String FILENAME = "bitcode-O0.bc";
    public static final BaseMatcher<String> EXCEEDS_LIMIT = new BaseMatcher<>() {
        private final Pattern compile = Pattern.compile(".*exceeds.*limit.*");

        @Override
        public boolean matches(Object item) {
            return compile.matcher(item.toString()).matches();
        }

        @Override
        public void describeTo(Description description) {
            description.appendText(" matching '" + compile.pattern() + "'");
        }
    };

    @ClassRule public static TruffleRunner.RunWithPolyglotRule runWithPolyglot = new TruffleRunner.RunWithPolyglotRule();

    protected static Object loadTestBitcodeInternal(String name) {
        File file = TEST_DIR.resolve(name).resolve(FILENAME).toFile();
        TruffleFile tf = runWithPolyglot.getTruffleTestEnv().getPublicTruffleFile(file.toURI());
        com.oracle.truffle.api.source.Source source;
        try {
            source = com.oracle.truffle.api.source.Source.newBuilder("llvm", tf).build();
        } catch (IOException ex) {
            throw new AssertionError(ex);
        }
        CallTarget target = runWithPolyglot.getTruffleTestEnv().parsePublic(source);
        return target.call();
    }

    protected static Value loadTestBitcodeValue(String name) {
        File file = TEST_DIR.resolve(name).resolve(FILENAME).toFile();
        org.graalvm.polyglot.Source source;
        try {
            source = org.graalvm.polyglot.Source.newBuilder("llvm", file).build();
        } catch (IOException ex) {
            throw new AssertionError(ex);
        }
        return runWithPolyglot.getPolyglotContext().eval(source);
    }

    public static Value library;

    @Before
    public void setup() {
        TestOptions.assumeBundledLLVM();
        Assume.assumeTrue("Skipping linux/amd64 only test", Platform.isLinux() && Platform.isAMD64());
        library = loadTestBitcodeValue("allocation_limits.ll.dir");
    }

    @Test
    public void allocaMaxSize() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_max_size").execute());
        assertThat(exception.getMessage(), StringContains.containsString("unsupported value range"));
    }

    @Test
    public void allocaMaxSizeI1() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_max_size_i1").execute());
        assertThat(exception.getMessage(), EXCEEDS_LIMIT);
    }

    @Test
    public void allocaMaxSizeI64() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_max_size_i64").execute());
        assertThat(exception.getMessage(), EXCEEDS_LIMIT);
    }

    @Test
    public void allocaParameter() {
        Value v = library.getMember("alloca_parameter").execute(16L);
        Assert.assertNotNull(v);
    }

    @Test
    public void allocaParameterMaxSize() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_parameter").execute(0xFFFF_FFFF_FFFF_FFFFL));
        assertThat(exception.getMessage(), EXCEEDS_LIMIT);
    }

    @Test
    public void allocaParameterOverflowInt() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_parameter").execute(0xFFFF_FFFF_0000_0010L));
        assertThat(exception.getMessage(), EXCEEDS_LIMIT);
    }

    @Test
    public void allocaOverflowInt() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_overflow_int").execute());
        assertThat(exception.getMessage(), StringContains.containsString("unsupported value range"));
    }

    @Test
    public void allocaOverflowIntI1() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_overflow_int_i1").execute());
        assertThat(exception.getMessage(), EXCEEDS_LIMIT);
    }

    @Test
    public void allocaOverflowIntI64() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_overflow_int_i64").execute());
        assertThat(exception.getMessage(), EXCEEDS_LIMIT);
    }

    /**
     * Checks whether we can create a ArrayType of maximum size (0xFFFF_FFFF_FFFF_FFFF) without
     * allocating it.
     */
    @Test
    public void arrayMaxSizePtr() {
        Value v = library.getMember("array_max_size_ptr").execute(new Object[]{null});
        Assert.assertTrue(v.asBoolean());
    }

    @Test
    public void arrayNegativeOffset() {
        Value v = library.getMember("alloca_array_negative_offset").execute();
        Assert.assertEquals(2, v.asLong());
    }

    @Test
    public void allocaArrayExceedSize() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_array_exceed_size").execute());
        assertThat(exception.getMessage(), EXCEEDS_LIMIT);
    }

    @Test
    public void allocaArrayOverflowInt() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_array_overflow_int").execute());
        assertThat(exception.getMessage(), EXCEEDS_LIMIT);
    }

    /**
     * Checks whether we can create a VectorType of maximum size (0xFFFFFFFF) without allocating it.
     */
    @Test
    public void vectorMaxSizePtr() {
        Value v = library.getMember("vector_max_size_ptr").execute(new Object[]{null});
        Assert.assertTrue(v.asBoolean());
    }

    @Test
    public void allocaVectorIntMinValue() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_vector_int_min_value").execute());
        assertThat(exception.getMessage(), EXCEEDS_LIMIT);
    }

    @Test
    public void allocaVectorIntMinusOne() {
        PolyglotException exception = assertThrows(PolyglotException.class, () -> library.getMember("alloca_vector_int_minus_one").execute());
        assertThat(exception.getMessage(), EXCEEDS_LIMIT);
    }

    @Test
    public void allocaVarWidthMinIntBits() {
        Value v = library.getMember("alloca_varwidth_min_int_bits").execute();
        Assert.assertTrue(v.asBoolean());
    }

    @Test
    public void allocaVarWidthMaxIntBits() {
        Value v = library.getMember("alloca_varwidth_max_int_bits").execute();
        long bytesAllocated = v.asLong();
        Assert.assertTrue(bytesAllocated != 0);
    }

    /**
     * Checks whether we can create a VariableBitWidthType of VariableBitWidthType#MAX_INT_BITS size
     * without allocating it.
     */
    @Test
    public void varWidthMaxIntBitsPtr() {
        Value v = library.getMember("varwidth_max_int_bits_ptr").execute(new Object[]{null});
        Assert.assertTrue(v.asBoolean());
    }
}
