/*
 * Copyright (c) 2016, 2022, Oracle and/or its affiliates.
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of
 * conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other materials provided
 * with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.oracle.truffle.llvm.tests.internal.types.floating;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import org.junit.Test;

import com.oracle.truffle.llvm.runtime.floating.LLVM80BitFloat;

public class LLVM80BitFromIntTest extends LLVM80BitTest {

    @Test
    public void testZero() {
        LLVM80BitFloat val = LLVM80BitFloat.fromInt(0);
        LLVM80BitFloat expected = LLVM80BitFloat.fromRawValues(false, 0, 0);
        assertEquals(expected, val);
    }

    @Test
    public void testOne() {
        LLVM80BitFloat val = LLVM80BitFloat.fromInt(1);
        LLVM80BitFloat expected = LLVM80BitFloat.fromRawValues(false, 0x3fff, Long.reverse(1));
        assertEquals(expected, val);
    }

    @Test
    public void testPositiveInfinity() {
        LLVM80BitFloat test = LLVM80BitFloat.fromRawValues(false, 0b111111111111111, 1L << 63);
        assertTrue(test.isPositiveInfinity());
        assertTrue(test.isInfinity());
        assertFalse(test.isNegativeInfinity());
    }

    @Test
    public void testNegativeInfinity() {
        LLVM80BitFloat test = LLVM80BitFloat.fromRawValues(true, 0b111111111111111, 1L << 63);
        assertTrue(test.isNegativeInfinity());
        assertTrue(test.isInfinity());
        assertFalse(test.isPositiveInfinity());
    }

    @Test
    public void testQNan() {
        LLVM80BitFloat test = nan();
        assertTrue(!test.isInfinity());
        assertFalse(test.isOrdered());
    }

    @Test
    public void testTen() {
        LLVM80BitFloat val = LLVM80BitFloat.fromInt(10);
        LLVM80BitFloat expected = LLVM80BitFloat.fromRawValues(false, 0x4002, Long.reverse(5));
        assertEquals(expected, val);
    }

    @Test
    public void testMaxInt() {
        LLVM80BitFloat val = LLVM80BitFloat.fromInt(Integer.MAX_VALUE);
        LLVM80BitFloat expected = LLVM80BitFloat.fromRawValues(false, 0x401d, 0xfffffffe00000000L);
        assertEquals(expected, val);
    }

    @Test
    public void testMinInt() {
        LLVM80BitFloat val = LLVM80BitFloat.fromInt(Integer.MIN_VALUE);
        LLVM80BitFloat expected = LLVM80BitFloat.fromRawValues(true, 0x401e, 0x8000000000000000L);
        assertEquals(expected, val);
    }

    @Test
    public void testMinusOne() {
        LLVM80BitFloat val = LLVM80BitFloat.fromInt(-1);
        LLVM80BitFloat expected = LLVM80BitFloat.fromRawValues(true, 0x3fff, Long.reverse(1));
        assertEquals(expected, val);
    }

    @Test
    public void testNegative() {
        LLVM80BitFloat val = LLVM80BitFloat.fromInt(-1234);
        LLVM80BitFloat expected = LLVM80BitFloat.fromRawValues(true, 0x4009, 0x9a40000000000000L);
        assertEquals(expected, val);
    }

    @Test
    public void testHighValue() {
        LLVM80BitFloat val = LLVM80BitFloat.fromInt(1234567);
        LLVM80BitFloat expected = LLVM80BitFloat.fromRawValues(false, 0b100000000010011, 0x96b4380000000000L);
        assertEquals(expected, val);
    }
}
