/*
 * Copyright (c) 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.oracle.truffle.api.dsl.test;

import static org.junit.Assert.assertSame;

import com.oracle.truffle.api.CallTarget;
import com.oracle.truffle.api.dsl.NodeChild;
import com.oracle.truffle.api.dsl.Specialization;
import com.oracle.truffle.api.dsl.test.ExportMessageTestFactory.MyNodeFactory;
import com.oracle.truffle.api.dsl.test.TypeSystemTest.ValueNode;
import com.oracle.truffle.api.dsl.test.examples.Interop.TruffleObject;
import com.oracle.truffle.api.library.CachedLibrary;
import com.oracle.truffle.api.library.ExportLibrary;
import com.oracle.truffle.api.library.ExportMessage;
import com.oracle.truffle.api.library.GenerateLibrary;
import com.oracle.truffle.api.library.Library;

import org.junit.Test;

public class ExportMessageTest {
    @GenerateLibrary
    public abstract static class MyLibrary extends Library {
        public abstract int messageA(Object receiver);

        public abstract int messageB(Object receiver);
    }

    /**
     * This test class there are no overlapping names for boundary class methods generated by the
     * exported library messages.
     */
    @ExportLibrary(value = MyLibrary.class)
    public static class MyObject extends TruffleObject {
        public Object o;

        public Object getObject() {
            return o == null ? this : o;
        }

        @ExportMessage
        static class MessageA {
            @Specialization(limit = "1")
            public static int fn(MyObject o, @CachedLibrary("o.getObject()") MyLibrary lib) {
                if (o.o != null) {
                    lib.messageA(o.getObject());
                }
                return 5;
            }
        }

        @ExportMessage
        static class MessageB {
            @Specialization(limit = "1")
            public static int fn(MyObject o, @CachedLibrary("o.getObject()") MyLibrary lib) {
                if (o.o != null) {
                    lib.messageA(o.getObject());
                }
                return 7;
            }
        }
    }

    @NodeChild
    @NodeChild
    public static class MyNode extends ValueNode {

        @Specialization(limit = "3")
        public int fn(boolean b, Object o, @CachedLibrary("o") MyLibrary lib) {
            return b ? lib.messageA(o) : lib.messageB(o);
        }
    }

    @Test
    public void testMyObject() {
        MyObject object = new MyObject();
        CallTarget root = TestHelper.createCallTarget(MyNodeFactory.getInstance());

        assertSame(5, root.call(true, object));
        assertSame(7, root.call(false, object));
    }
}
