/*
 * Copyright (c) 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.js.test.builtins;

import org.graalvm.polyglot.Context;
import org.graalvm.polyglot.PolyglotException;
import org.graalvm.polyglot.Value;
import org.junit.Assert;
import org.junit.Test;

import com.oracle.truffle.js.lang.JavaScriptLanguage;
import com.oracle.truffle.js.runtime.JSContextOptions;
import com.oracle.truffle.js.test.JSTest;

public class IteratorHelperPrototypeBuiltinsTest {
    @Test
    public void testPrototype() {
        Context.Builder builder = JSTest.newContextBuilder();
        builder.option(JSContextOptions.ITERATOR_HELPERS_NAME, "true");
        try (Context context = builder.build()) {
            Value result = context.eval(JavaScriptLanguage.ID, "Object.getPrototypeOf(Object.getPrototypeOf([].values().drop(0))) === Iterator.prototype");
            Assert.assertTrue(result.asBoolean());

            result = context.eval(JavaScriptLanguage.ID, "typeof Object.getPrototypeOf([].values().drop(0))");
            Assert.assertEquals("object", result.asString());
        }
    }

    @Test
    public void testNext() {
        Context.Builder builder = JSTest.newContextBuilder();
        builder.option(JSContextOptions.ITERATOR_HELPERS_NAME, "true");
        try (Context context = builder.build()) {
            Value result = context.eval(JavaScriptLanguage.ID, "[1].values().drop(0).next()");
            Assert.assertTrue(result.hasMembers());
            Assert.assertTrue(result.hasMember("done"));
            Assert.assertTrue(result.hasMember("value"));
            Assert.assertFalse(result.getMember("done").asBoolean());
            Assert.assertEquals(1, result.getMember("value").asInt());

            try {
                context.eval(JavaScriptLanguage.ID, "[1].values().drop(0).next.call([2].values())");
                Assert.fail("No exception thrown");
            } catch (PolyglotException e) {
                Assert.assertTrue(e.getMessage().startsWith("TypeError: "));
            }
        }
    }

    @Test
    public void testReturn() {
        Context.Builder builder = JSTest.newContextBuilder();
        builder.option(JSContextOptions.ITERATOR_HELPERS_NAME, "true");
        try (Context context = builder.build()) {
            Value result = context.eval(JavaScriptLanguage.ID, "function* test() {yield 1; yield 2;}; var x = test(); var y = x.drop(0); y.next(); y.return(); x.next()");
            Assert.assertTrue(result.hasMembers());
            Assert.assertTrue(result.hasMember("done"));
            Assert.assertTrue(result.hasMember("value"));
            Assert.assertTrue(result.getMember("done").asBoolean());
            Assert.assertTrue(result.getMember("value").isNull());

            result = context.eval(JavaScriptLanguage.ID, "function* test() {yield 1; yield 2;}; var x = test(); var y = [1].values().flatMap(() => x); y.next(); y.return(); x.next()");
            Assert.assertTrue(result.hasMembers());
            Assert.assertTrue(result.hasMember("done"));
            Assert.assertTrue(result.hasMember("value"));
            Assert.assertTrue(result.getMember("done").asBoolean());
            Assert.assertTrue(result.getMember("value").isNull());

            try {
                context.eval(JavaScriptLanguage.ID, "[1].values().drop(0).return.call([2].values())");
                Assert.fail("No exception thrown");
            } catch (PolyglotException e) {
                Assert.assertTrue(e.getMessage().startsWith("TypeError: "));
            }
        }
    }

    @Test
    public void testGeneratorStates() {
        Context.Builder builder = JSTest.newContextBuilder();
        builder.option(JSContextOptions.ITERATOR_HELPERS_NAME, "true");
        try (Context context = builder.build()) {
            try {
                context.eval(JavaScriptLanguage.ID, "var obj = Iterator.from({next(){return obj.next()}}).map(v => v); obj.next()");
                Assert.fail("No exception thrown");
            } catch (PolyglotException e) {
                Assert.assertTrue(e.getMessage().startsWith("TypeError: "));
            }

            Value result = context.eval(JavaScriptLanguage.ID, "Iterator.from({next(){return obj.next()}}).map(v => v).return()");
            Assert.assertTrue(result.hasMembers());
            Assert.assertTrue(result.hasMember("done"));
            Assert.assertTrue(result.hasMember("value"));
            Assert.assertTrue(result.getMember("done").asBoolean());
            Assert.assertTrue(result.getMember("value").isNull());

            result = context.eval(JavaScriptLanguage.ID, "var obj = Iterator.from({next(){throw 'X'}}).map(v => v); try {obj.next()} catch {}; obj.next()");
            Assert.assertTrue(result.hasMembers());
            Assert.assertTrue(result.hasMember("done"));
            Assert.assertTrue(result.hasMember("value"));
            Assert.assertTrue(result.getMember("done").asBoolean());
            Assert.assertTrue(result.getMember("value").isNull());

            result = context.eval(JavaScriptLanguage.ID, "function* test() {try {yield 1} finally {yield 2}}; test().map(x => x).return()");
            Assert.assertTrue(result.hasMembers());
            Assert.assertTrue(result.hasMember("done"));
            Assert.assertTrue(result.hasMember("value"));
            Assert.assertTrue(result.getMember("done").asBoolean());
            Assert.assertTrue(result.getMember("value").isNull());

            result = context.eval(JavaScriptLanguage.ID, "var called = false;" +
                            "function* test() {try {yield 1} finally {called = true; yield 2}}; test().map(x => x).return();" +
                            "called");
            Assert.assertTrue(result.isBoolean());
            Assert.assertFalse(result.asBoolean());

            result = context.eval(JavaScriptLanguage.ID, "function* test() {try {yield 1} finally {yield 2}}; var obj = test().map(x => x); obj.next(); obj.return()");
            Assert.assertTrue(result.hasMembers());
            Assert.assertTrue(result.hasMember("done"));
            Assert.assertTrue(result.hasMember("value"));
            Assert.assertTrue(result.getMember("done").asBoolean());
            Assert.assertTrue(result.getMember("value").isNull());

            result = context.eval(JavaScriptLanguage.ID, "var called = false;" +
                            "function* test() {try {yield 1} finally {called = true; yield 2}}; var obj = test().map(x => x); obj.next(); obj.return();" +
                            "called");
            Assert.assertTrue(result.isBoolean());
            Assert.assertTrue(result.asBoolean());
        }
    }
}
