package com.oracle.objectfile.elf;

import com.oracle.objectfile.dwarf.DwarfARangesSectionImpl;
import com.oracle.objectfile.dwarf.DwarfAbbrevSectionImpl;
import com.oracle.objectfile.dwarf.DwarfDebugInfo;
import com.oracle.objectfile.dwarf.DwarfFrameSectionImpl;
import com.oracle.objectfile.dwarf.DwarfFrameSectionImplAArch64;
import com.oracle.objectfile.dwarf.DwarfFrameSectionImplX86_64;
import com.oracle.objectfile.dwarf.DwarfInfoSectionImpl;
import com.oracle.objectfile.dwarf.DwarfLineSectionImpl;
import com.oracle.objectfile.dwarf.DwarfLocSectionImpl;
import com.oracle.objectfile.dwarf.DwarfRangesSectionImpl;
import com.oracle.objectfile.dwarf.DwarfStrSectionImpl;
import com.oracle.objectfile.dwarf.constants.DwarfSectionName;

import java.nio.ByteOrder;

public class ELFDebugInfo extends DwarfDebugInfo {

    private final DwarfStrSectionImpl dwarfStrSection;
    private final DwarfAbbrevSectionImpl dwarfAbbrevSection;
    private final DwarfInfoSectionImpl dwarfInfoSection;
    private final DwarfLocSectionImpl dwarfLocSection;
    private final DwarfARangesSectionImpl dwarfARangesSection;
    private final DwarfRangesSectionImpl dwarfRangesSection;
    private final DwarfLineSectionImpl dwarfLineSection;
    private final DwarfFrameSectionImpl dwarfFameSection;
    public final ELFMachine elfMachine;
    /**
     * Register used to hold the heap base.
     */
    private final byte heapbaseRegister;
    /**
     * Register used to hold the current thread.
     */
    private final byte threadRegister;

    @SuppressWarnings("this-escape")
    public ELFDebugInfo(ELFMachine elfMachine, ByteOrder byteOrder) {
        super(byteOrder);
        this.elfMachine = elfMachine;
        dwarfStrSection = new DwarfStrSectionImpl(this);
        dwarfAbbrevSection = new DwarfAbbrevSectionImpl(this);
        dwarfInfoSection = new DwarfInfoSectionImpl(this);
        dwarfLocSection = new DwarfLocSectionImpl(this);
        dwarfARangesSection = new DwarfARangesSectionImpl(this);
        dwarfRangesSection = new DwarfRangesSectionImpl(this);
        dwarfLineSection = new DwarfLineSectionImpl(this);

        if (elfMachine == ELFMachine.AArch64) {
            dwarfFameSection = new DwarfFrameSectionImplAArch64(this);
            this.heapbaseRegister = rheapbase_aarch64;
            this.threadRegister = rthread_aarch64;
        } else {
            dwarfFameSection = new DwarfFrameSectionImplX86_64(this);
            this.heapbaseRegister = rheapbase_x86;
            this.threadRegister = rthread_x86;
        }
    }

    public DwarfStrSectionImpl getStrSectionImpl() {
        return dwarfStrSection;
    }

    public DwarfAbbrevSectionImpl getAbbrevSectionImpl() {
        return dwarfAbbrevSection;
    }

    public DwarfFrameSectionImpl getFrameSectionImpl() {
        return dwarfFameSection;
    }

    public DwarfInfoSectionImpl getInfoSectionImpl() {
        return dwarfInfoSection;
    }

    public DwarfLocSectionImpl getLocSectionImpl() {
        return dwarfLocSection;
    }

    public DwarfARangesSectionImpl getARangesSectionImpl() {
        return dwarfARangesSection;
    }

    public DwarfRangesSectionImpl getRangesSectionImpl() {
        return dwarfRangesSection;
    }

    public DwarfLineSectionImpl getLineSectionImpl() {
        return dwarfLineSection;
    }

    public byte getHeapbaseRegister() {
        return heapbaseRegister;
    }

    public byte getThreadRegister() {
        return threadRegister;
    }

    /////////////////////////////// EGS ////////////////////////////////
    @Override
    public boolean isAarch64() {
        return elfMachine == ELFMachine.AArch64;
    }

    @Override
    public boolean isAMD64() {
        return elfMachine == ELFMachine.X86_64;
    }

    @Override
    public DwarfSectionName textSectionName() {
        return ELFSectionName.TEXT_SECTION;
    }

    @Override
    public DwarfSectionName lineSectionName() {
        return ELFSectionName.DW_LINE_SECTION;
    }

    @Override
    public DwarfSectionName strSectionName() {
        return ELFSectionName.DW_STR_SECTION;
    }

    @Override
    public DwarfSectionName locSectionName() {
        return ELFSectionName.DW_LOC_SECTION;
    }

    @Override
    public DwarfSectionName rangesSectionName() {
        return ELFSectionName.DW_RANGES_SECTION;
    }

    @Override
    public DwarfSectionName arangesSectionName() {
        return ELFSectionName.DW_ARANGES_SECTION;
    }

    @Override
    public DwarfSectionName frameSectionName() {
        return ELFSectionName.DW_FRAME_SECTION;
    }

    @Override
    public DwarfSectionName abbrevSectionName() {
        return ELFSectionName.DW_ABBREV_SECTION;
    }

    @Override
    public DwarfSectionName infoSectionName() {
        return ELFSectionName.DW_INFO_SECTION;
    }

    /**
     * Various ELF sections created by GraalVM including all debug info sections. The enum sequence
     * starts with the text section (not defined in the DWARF spec and not created by debug info
     * code).
     */
    enum ELFSectionName implements DwarfSectionName {
        TEXT_SECTION(".text"),
        DW_STR_SECTION(".debug_str"),
        DW_LINE_SECTION(".debug_line"),
        DW_FRAME_SECTION(".debug_frame"),
        DW_ABBREV_SECTION(".debug_abbrev"),
        DW_INFO_SECTION(".debug_info"),
        DW_LOC_SECTION(".debug_loc"),
        DW_ARANGES_SECTION(".debug_aranges"),
        DW_RANGES_SECTION(".debug_ranges");

        private final String value;

        ELFSectionName(String s) {
            value = s;
        }

        public String value() {
            return value;
        }
    }
}
