/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.thread;

import com.oracle.svm.core.Uninterruptible;
import com.oracle.svm.core.heap.RestrictHeapAccess;
import com.oracle.svm.core.jdk.UninterruptibleUtils;
import com.oracle.svm.core.jfr.sampler.JfrRecurringCallbackExecutionSampler;
import com.oracle.svm.core.option.HostedOptionKey;
import com.oracle.svm.core.thread.SafepointCheckCounter;
import com.oracle.svm.core.thread.VMOperation;
import com.oracle.svm.core.thread.VMThreads;
import com.oracle.svm.core.threadlocal.FastThreadLocalFactory;
import com.oracle.svm.core.threadlocal.FastThreadLocalInt;
import com.oracle.svm.core.threadlocal.FastThreadLocalObject;
import com.oracle.svm.core.util.VMError;
import jdk.graal.compiler.api.replacements.Fold;
import org.graalvm.nativeimage.CurrentIsolate;
import org.graalvm.nativeimage.IsolateThread;
import org.graalvm.nativeimage.Threading;

public class RecurringCallbackSupport {
    private static final FastThreadLocalObject<RecurringCallbackTimer> timerTL = FastThreadLocalFactory.createObject(RecurringCallbackTimer.class, "RecurringCallbackSupport.timer");
    private static final FastThreadLocalInt suspendedTL = FastThreadLocalFactory.createInt("RecurringCallbackSupport.suspended");

    @Fold
    public static boolean isEnabled() {
        return ConcealedOptions.SupportRecurringCallback.getValue() != false || JfrRecurringCallbackExecutionSampler.isPresent();
    }

    public static RecurringCallbackTimer createCallbackTimer(long intervalNanos, Threading.RecurringCallback callback) {
        assert (RecurringCallbackSupport.isEnabled());
        assert (callback != null);
        return new RecurringCallbackTimer(intervalNanos, callback);
    }

    @Uninterruptible(reason="Prevent VM operations that modify the recurring callbacks.")
    public static void installCallback(IsolateThread thread, RecurringCallbackTimer timer, boolean overwriteExisting) {
        if (overwriteExisting) {
            RecurringCallbackSupport.uninstallCallback(thread);
        }
        RecurringCallbackSupport.installCallback(thread, timer);
    }

    @Uninterruptible(reason="Prevent VM operations that modify the recurring callbacks.")
    public static void installCallback(IsolateThread thread, RecurringCallbackTimer timer) {
        assert (RecurringCallbackSupport.isEnabled());
        assert (timer != null);
        assert (timer.targetIntervalNanos > 0L);
        assert (thread == CurrentIsolate.getCurrentThread() || VMOperation.isInProgressAtSafepoint());
        assert (timerTL.get(thread) == null) : "only one callback can be installed at a time";
        timerTL.set(thread, timer);
        SafepointCheckCounter.setVolatile(thread, timer.requestedChecks);
    }

    @Uninterruptible(reason="Prevent VM operations that modify the recurring callbacks.")
    public static void uninstallCallback(IsolateThread thread) {
        assert (RecurringCallbackSupport.isEnabled());
        assert (thread == CurrentIsolate.getCurrentThread() || VMOperation.isInProgressAtSafepoint());
        timerTL.set(thread, null);
    }

    @Uninterruptible(reason="Prevent VM operations that modify the recurring callbacks.", callerMustBe=true)
    public static Threading.RecurringCallback getCallback(IsolateThread thread) {
        assert (RecurringCallbackSupport.isEnabled());
        assert (thread == CurrentIsolate.getCurrentThread() || VMOperation.isInProgressAtSafepoint());
        RecurringCallbackTimer value = timerTL.get(thread);
        if (value != null) {
            return value.callback;
        }
        return null;
    }

    @Uninterruptible(reason="Must not contain safepoint checks.")
    static void maybeExecuteCallback() {
        RecurringCallbackTimer timer;
        assert (VMThreads.StatusSupport.isStatusJava()) : "must only be executed when the thread is in Java state";
        RecurringCallbackTimer recurringCallbackTimer = timer = RecurringCallbackSupport.isEnabled() ? timerTL.get() : null;
        if (timer != null) {
            timer.evaluate();
        } else {
            SafepointCheckCounter.setVolatile(Integer.MAX_VALUE);
        }
    }

    @Uninterruptible(reason="Prevent VM operations that modify the recurring callbacks.", callerMustBe=true)
    static boolean isCallbackInstalled(IsolateThread thread) {
        return RecurringCallbackSupport.isEnabled() && timerTL.get(thread) != null;
    }

    static boolean needsNativeToJavaSlowpath() {
        return RecurringCallbackSupport.isEnabled() && ConcealedOptions.CheckRecurringCallbackOnNativeToJavaTransition.getValue() != false && timerTL.get() != null && !RecurringCallbackSupport.isCallbackTimerSuspended();
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    public static void suspendCallbackTimer(String reason) {
        if (!RecurringCallbackSupport.isEnabled()) {
            return;
        }
        RecurringCallbackSupport.incrementSuspended();
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    public static void resumeCallbackTimerAtNextSafepointCheck() {
        RecurringCallbackTimer timer;
        if (!RecurringCallbackSupport.isEnabled()) {
            return;
        }
        RecurringCallbackSupport.decrementSuspended();
        if (!RecurringCallbackSupport.isCallbackTimerSuspended() && (timer = timerTL.get()) != null) {
            timer.updateStatistics();
            timer.setCounter(1);
        }
    }

    public static void resumeCallbackTimer() {
        if (!RecurringCallbackSupport.isEnabled()) {
            return;
        }
        RecurringCallbackSupport.decrementSuspended();
        if (!RecurringCallbackSupport.isCallbackTimerSuspended()) {
            RecurringCallbackSupport.resumeCallbackTimer0();
        }
    }

    @Uninterruptible(reason="Prevent unexpected recurring callback execution (pending exception must not be destroyed).")
    private static void resumeCallbackTimer0() {
        try {
            RecurringCallbackSupport.maybeExecuteCallback();
        }
        catch (SafepointException e) {
            RecurringCallbackSupport.throwUnchecked(RecurringCallbackTimer.getAndClearPendingException());
        }
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    public static boolean isCallbackUnsupportedOrTimerSuspended() {
        return !RecurringCallbackSupport.isEnabled() || RecurringCallbackSupport.isCallbackTimerSuspended();
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    public static boolean isCallbackTimerSuspended() {
        assert (RecurringCallbackSupport.isEnabled());
        return suspendedTL.get() != 0;
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    private static void incrementSuspended() {
        assert (RecurringCallbackSupport.isEnabled());
        assert (suspendedTL.get() >= 0);
        suspendedTL.set(suspendedTL.get() + 1);
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    private static void decrementSuspended() {
        assert (RecurringCallbackSupport.isEnabled());
        int newValue = suspendedTL.get() - 1;
        assert (newValue >= 0);
        suspendedTL.set(newValue);
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    private static <T extends Throwable> void throwUnchecked(Throwable exception) throws T {
        throw exception;
    }

    public static class ConcealedOptions {
        public static final HostedOptionKey<Boolean> SupportRecurringCallback = new HostedOptionKey<Boolean>(false);
        static final HostedOptionKey<Boolean> CheckRecurringCallbackOnNativeToJavaTransition = new HostedOptionKey<Boolean>(false);
    }

    public static class RecurringCallbackTimer {
        private static final FastThreadLocalObject<Throwable> EXCEPTION_TL = FastThreadLocalFactory.createObject(Throwable.class, "RecurringCallbackTimer.exception");
        private static final Threading.RecurringCallbackAccess CALLBACK_ACCESS = new RecurringCallbackAccessImpl();
        private static final double EWMA_LAMBDA = 0.3;
        private static final double TARGET_INTERVAL_FLEXIBILITY = 0.95;
        private static final int INITIAL_CHECKS = 100;
        private static final long MINIMUM_INTERVAL_NANOS = 1000L;
        private final long targetIntervalNanos;
        private final long flexibleTargetIntervalNanos;
        private final Threading.RecurringCallback callback;
        private int requestedChecks;
        private double ewmaChecksPerNano;
        private long lastCapture;
        private long lastCallbackExecution;
        private volatile boolean isExecuting = false;

        RecurringCallbackTimer(long targetIntervalNanos, Threading.RecurringCallback callback) {
            long now;
            this.targetIntervalNanos = Math.max(1000L, targetIntervalNanos);
            this.flexibleTargetIntervalNanos = (long)((double)targetIntervalNanos * 0.95);
            this.callback = callback;
            this.lastCapture = now = System.nanoTime();
            this.lastCallbackExecution = now;
            this.requestedChecks = 100;
        }

        @Uninterruptible(reason="Prevent recurring callback execution.", callerMustBe=true)
        public static Throwable getAndClearPendingException() {
            Throwable t = EXCEPTION_TL.get();
            VMError.guarantee(t != null, "There must be a recurring callback exception pending.");
            EXCEPTION_TL.set(null);
            return t;
        }

        @Uninterruptible(reason="Must not contain safepoint checks.")
        void evaluate() {
            this.updateStatistics();
            try {
                this.maybeExecuteCallback();
            }
            finally {
                this.updateCounter();
            }
        }

        @Uninterruptible(reason="Must be uninterruptible to avoid races with the safepoint code.")
        void updateStatistics() {
            long now = System.nanoTime();
            long elapsedNanos = now - this.lastCapture;
            int skippedChecks = RecurringCallbackTimer.getSkippedChecks(CurrentIsolate.getCurrentThread());
            int executedChecks = this.requestedChecks - skippedChecks;
            assert (executedChecks >= 0);
            if (elapsedNanos > 0L && executedChecks > 0) {
                double checksPerNano = (double)executedChecks / (double)elapsedNanos;
                this.ewmaChecksPerNano = this.ewmaChecksPerNano == 0.0 ? checksPerNano : 0.3 * checksPerNano + 0.7 * this.ewmaChecksPerNano;
                this.lastCapture = now;
            }
        }

        @Uninterruptible(reason="Must be uninterruptible to avoid races with the safepoint code.")
        private static int getSkippedChecks(IsolateThread thread) {
            int rawValue = SafepointCheckCounter.getVolatile(thread);
            return rawValue >= 0 ? rawValue : -rawValue;
        }

        @Uninterruptible(reason="Must not contain safepoint checks.")
        private void maybeExecuteCallback() {
            block7: {
                if (this.isCallbackDisabled()) {
                    return;
                }
                this.isExecuting = true;
                try {
                    if (System.nanoTime() < this.lastCallbackExecution + this.flexibleTargetIntervalNanos) break block7;
                    this.setCounter(Integer.MAX_VALUE);
                    try {
                        this.invokeCallback();
                    }
                    finally {
                        this.lastCallbackExecution = System.nanoTime();
                        this.updateStatistics();
                    }
                }
                finally {
                    this.isExecuting = false;
                }
            }
        }

        @Uninterruptible(reason="Must not contain safepoint checks.")
        private void updateCounter() {
            long nextDeadline = this.lastCallbackExecution + this.targetIntervalNanos;
            long remainingNanos = nextDeadline - System.nanoTime();
            if (remainingNanos < 0L && this.isCallbackDisabled()) {
                this.setCounter(Integer.MAX_VALUE);
            } else {
                double checks = this.ewmaChecksPerNano * (double)(remainingNanos = UninterruptibleUtils.Math.max(remainingNanos, 1000L));
                this.setCounter(checks > 2.147483647E9 ? Integer.MAX_VALUE : (checks < 1.0 ? 1 : (int)checks));
            }
        }

        @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
        void setCounter(int value) {
            this.requestedChecks = value;
            SafepointCheckCounter.setVolatile(value);
        }

        @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
        private boolean isCallbackDisabled() {
            return this.isExecuting || RecurringCallbackSupport.isCallbackTimerSuspended();
        }

        @Uninterruptible(reason="Required by caller, but does not apply to callee.", calleeMustBe=false)
        @RestrictHeapAccess(reason="Recurring callbacks must not allocate.", access=RestrictHeapAccess.Access.NO_ALLOCATION)
        private void invokeCallback() {
            try {
                this.callback.run(CALLBACK_ACCESS);
            }
            catch (SafepointException e) {
                throw e;
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        }

        private static final class RecurringCallbackAccessImpl
        implements Threading.RecurringCallbackAccess {
            private RecurringCallbackAccessImpl() {
            }

            public void throwException(Throwable t) {
                EXCEPTION_TL.set(t);
                throw SafepointException.SINGLETON;
            }
        }
    }

    static final class SafepointException
    extends RuntimeException {
        public static final SafepointException SINGLETON = new SafepointException();
        private static final long serialVersionUID = 1L;

        private SafepointException() {
        }
    }
}

