/*
 * Copyright (c) 2022, 2022, Oracle and/or its affiliates. All rights reserved.
 * Copyright (c) 2022, 2022, BELLSOFT. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.oracle.svm.core.genscavenge.parallel;

import java.util.function.BooleanSupplier;

import jdk.graal.compiler.api.replacements.Fold;
import org.graalvm.nativeimage.CurrentIsolate;
import org.graalvm.nativeimage.ImageSingletons;
import org.graalvm.nativeimage.Platform;
import org.graalvm.nativeimage.Platforms;
import org.graalvm.nativeimage.c.function.CEntryPoint;
import org.graalvm.nativeimage.c.function.CEntryPointLiteral;
import org.graalvm.nativeimage.c.function.CFunctionPointer;
import org.graalvm.nativeimage.c.struct.RawPointerTo;
import org.graalvm.nativeimage.c.struct.RawStructure;
import org.graalvm.nativeimage.c.struct.SizeOf;
import org.graalvm.word.Pointer;
import org.graalvm.word.PointerBase;
import org.graalvm.word.UnsignedWord;
import org.graalvm.word.WordFactory;

import com.oracle.svm.core.NeverInline;
import com.oracle.svm.core.SubstrateGCOptions;
import com.oracle.svm.core.SubstrateOptions;
import com.oracle.svm.core.Uninterruptible;
import com.oracle.svm.core.UnmanagedMemoryUtil;
import com.oracle.svm.core.c.function.CEntryPointOptions;
import com.oracle.svm.core.config.ConfigurationValues;
import com.oracle.svm.core.feature.AutomaticallyRegisteredFeature;
import com.oracle.svm.core.feature.InternalFeature;
import com.oracle.svm.core.genscavenge.AlignedHeapChunk;
import com.oracle.svm.core.genscavenge.GCImpl;
import com.oracle.svm.core.genscavenge.GreyToBlackObjectVisitor;
import com.oracle.svm.core.genscavenge.GreyToBlackObjRefVisitor;
import com.oracle.svm.core.genscavenge.HeapChunk;
import com.oracle.svm.core.genscavenge.HeapParameters;
import com.oracle.svm.core.genscavenge.UnalignedHeapChunk;
import com.oracle.svm.core.genscavenge.remset.RememberedSet;
import com.oracle.svm.core.graal.nodes.WriteCurrentVMThreadNode;
import com.oracle.svm.core.graal.snippets.CEntryPointSnippets;
import com.oracle.svm.core.jdk.Jvm;
import com.oracle.svm.core.jdk.UninterruptibleUtils;
import com.oracle.svm.core.locks.VMCondition;
import com.oracle.svm.core.locks.VMMutex;
import com.oracle.svm.core.log.Log;
import com.oracle.svm.core.option.SubstrateOptionKey;
import com.oracle.svm.core.os.CommittedMemoryProvider;
import com.oracle.svm.core.os.ChunkBasedCommittedMemoryProvider;
import com.oracle.svm.core.thread.PlatformThreads;
import com.oracle.svm.core.thread.PlatformThreads.ThreadLocalKey;
import com.oracle.svm.core.thread.VMThreads.OSThreadHandle;
import com.oracle.svm.core.thread.VMThreads.OSThreadHandlePointer;
import com.oracle.svm.core.util.UserError;
import com.oracle.svm.core.util.VMError;

/**
 * A garbage collector that tries to shorten GC pauses by using multiple worker threads. Currently,
 * the only phase supported is scanning grey objects during a full GC. The number of worker threads
 * can be set with a runtime option (see {@link SubstrateOptions#ParallelGCThreads}).
 * <p>
 * The GC worker threads are unattached threads that are started lazily and that call AOT-compiled
 * code. So, they don't have an {@link org.graalvm.nativeimage.IsolateThread} data structure and
 * don't participate in the safepoint handling.
 * <p>
 * Worker threads use heap chunks as the unit of work. Chunks to be scanned are stored in the
 * {@link ChunkQueue}. Worker threads pop chunks from the queue and scan them for references to live
 * objects to be promoted. When promoting an aligned chunk object, they speculatively allocate
 * memory for its copy in the to-space, then compete to install forwarding pointer in the original
 * object. The winning thread proceeds to copy object data, losing threads retract the speculatively
 * allocated memory.
 * <p>
 * Each worker thread allocates memory in its own thread local allocation chunk for speed. As
 * allocation chunks become filled up, they are pushed to {@link ChunkQueue}. This pop-scan-push
 * cycle continues until the chunk buffer becomes empty. At this point, worker threads are parked
 * and the GC routine continues on the main GC thread.
 */
public class ParallelGC {
    private enum Phase {
        SEQUENTIAL,
        PARALLEL,
        CLEANUP,
        SHUTDOWN,
    }

    public static final int SCAN_CARD_TABLE = 0b000;
    public static final int SCAN_GREY_OBJECTS = 0b010;

    private static final int SCAN_OP_MASK = 0b010;
    private static final int UNALIGNED_BIT = 0b001;
    private static final UnsignedWord POINTER_MASK = WordFactory.unsigned(0b111).not();

    private static final int MAX_WORKER_THREADS = 8;

    /**
     * Worker thread states occupy separate cache lines to avoid false sharing.
     * We assume 128 bytes (16 words) cache line here.
     */
    private static final int CACHE_LINE_WORDS = 16;

    private final VMMutex mutex = new VMMutex("parallelGC");
    private final VMCondition seqPhase = new VMCondition(mutex);
    private final VMCondition parPhase = new VMCondition(mutex);
    private final ChunkQueue chunkQueue = new ChunkQueue();
    private final CEntryPointLiteral<CFunctionPointer> gcWorkerRunFunc = CEntryPointLiteral.create(ParallelGC.class, "gcWorkerRun", GCWorkerThreadState.class);

    private boolean initialized;
    private ThreadLocalKey workerStateTL;
    private GCWorkerThreadState workerStates;
    private UnsignedWord workerStatesSize;
    private OSThreadHandlePointer workerThreads;
    private int numWorkerThreads;
    private int busyWorkerThreads;
    private volatile Phase phase;

    @Platforms(Platform.HOSTED_ONLY.class)
    public ParallelGC() {
    }

    @Fold
    public static ParallelGC singleton() {
        return ImageSingletons.lookup(ParallelGC.class);
    }

    @Fold
    public static boolean isEnabled() {
        return SubstrateOptions.useParallelGC();
    }

    @Fold
    static int wordSize() {
        return ConfigurationValues.getTarget().wordSize;
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    public boolean isInParallelPhase() {
        return phase == Phase.PARALLEL;
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    public VMMutex getMutex() {
        return mutex;
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    public Pointer getAllocChunkScanPointer(int age, boolean clear) {
        GCWorkerThreadState state = getWorkerThreadState();
        Pointer chunk = (Pointer) state.read(age);
        if (clear) {
            state.write(age, WordFactory.nullPointer());
        }
        return chunk;
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    public void setAllocChunk(int age, AlignedHeapChunk.AlignedHeader allocChunk) {
        Pointer scanPtr = WordFactory.nullPointer();
        if (allocChunk.isNonNull()) {
            Pointer top = HeapChunk.getTopPointer(allocChunk);
            // Use chunk as allocation chunk unless it is full (top == end)
            if (top.belowThan(HeapChunk.getEndPointer(allocChunk))) {
                scanPtr = top;
            }
        }
        getWorkerThreadState().write(age, scanPtr);
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    public void push(AlignedHeapChunk.AlignedHeader aChunk, int scanOp) {
        Pointer ptr = scanOp == SCAN_GREY_OBJECTS ? AlignedHeapChunk.getObjectsStart(aChunk) : HeapChunk.asPointer(aChunk);
        push(ptr, scanOp);
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    public void push(UnalignedHeapChunk.UnalignedHeader uChunk, int scanOp) {
        push(HeapChunk.asPointer(uChunk).or(ParallelGC.UNALIGNED_BIT), scanOp);
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    private void push(Pointer ptr, int scanOp) {
        assert ptr.isNonNull();
        ptr = ptr.or(scanOp);
        chunkQueue.push(ptr);
        if (phase == Phase.PARALLEL) {
            assert mutex.isOwner(true);
            parPhase.signal();
        }
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    public void pushAllocChunk(Pointer ptr) {
        /*
         * Scanning (and therefore enqueueing) is only necessary if there are any not yet scanned
         * objects in the chunk.
         */
        assert isEnabled() && ptr.isNonNull();
        GCWorkerThreadState state = getWorkerThreadState();
        AlignedHeapChunk.AlignedHeader chunk = AlignedHeapChunk.getEnclosingChunkFromObjectPointer(ptr);
        if (chunk.notEqual(getScannedChunk(state)) && HeapChunk.getTopPointer(chunk).aboveThan(ptr)) {
            push(ptr, SCAN_GREY_OBJECTS);
        }
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    private GCWorkerThreadState getWorkerThreadState() {
        if (CurrentIsolate.getCurrentThread().isNull()) {
            return PlatformThreads.singleton().getUnmanagedThreadLocalValue(workerStateTL);
        }
        return workerStates;
    }

    public void initialize() {
        if (initialized) {
            return;
        }

        initialized = true;
        phase = Phase.PARALLEL;

        chunkQueue.initialize();
        workerStateTL = PlatformThreads.singleton().createUnmanagedThreadLocal();
        numWorkerThreads = busyWorkerThreads = getWorkerCount();

        /* Round worker thread state size up to CACHE_LINE_WORDS */
        int workerStateWords = (getStateWords() + CACHE_LINE_WORDS - 1) / CACHE_LINE_WORDS * CACHE_LINE_WORDS;
        /* Allocate one struct per worker thread and one struct for the main GC thread. */
        int numWorkerStates = numWorkerThreads + 1;
        workerStatesSize = WordFactory.unsigned(workerStateWords * numWorkerStates * wordSize());
        workerStates = (GCWorkerThreadState) ChunkBasedCommittedMemoryProvider.get()
                .allocateAlignedChunk(workerStatesSize, WordFactory.unsigned(CACHE_LINE_WORDS * wordSize()));
        VMError.guarantee(workerStates.isNonNull());

        /* Start the worker threads and wait until they are in a well-defined state. */
        workerThreads = (OSThreadHandlePointer) ChunkBasedCommittedMemoryProvider.get()
                .allocateUnalignedChunk(SizeOf.unsigned(OSThreadHandlePointer.class).multiply(numWorkerThreads));
        VMError.guarantee(workerThreads.isNonNull());
        for (int i = 0; i < numWorkerThreads; i++) {
            GCWorkerThreadState workerState = workerStates.addressOf(workerStateWords * (i + 1));
            /* Reuse scanned chunk slot for the isolate since it is read just once at thread start */
            setScannedChunk(workerState, CurrentIsolate.getIsolate());
            OSThreadHandle thread = PlatformThreads.singleton().startThreadUnmanaged(gcWorkerRunFunc.getFunctionPointer(), workerState, 0);
            workerThreads.write(i, thread);
        }
        waitUntilWorkerThreadsFinish();
    }

    @Uninterruptible(reason = "Tear-down in progress.")
    public void tearDown() {
        if (!initialized) {
            return;
        }

        initialized = false;
        chunkQueue.teardown();

        /* Signal the worker threads so that they can shut down. */
        phase = Phase.SHUTDOWN;
        parPhase.broadcast();

        for (int i = 0; i < numWorkerThreads; i++) {
            OSThreadHandle thread = workerThreads.read(i);
            PlatformThreads.singleton().joinThreadUnmanaged(thread);
        }
        busyWorkerThreads = 0;

        ChunkBasedCommittedMemoryProvider.get().freeAlignedChunk(workerStates, workerStatesSize, WordFactory.unsigned(CACHE_LINE_WORDS * wordSize()));
        ChunkBasedCommittedMemoryProvider.get().freeUnalignedChunk(workerThreads, SizeOf.unsigned(OSThreadHandlePointer.class).multiply(numWorkerThreads));
        workerThreads = WordFactory.nullPointer();

        PlatformThreads.singleton().deleteUnmanagedThreadLocal(workerStateTL);
        workerStateTL = WordFactory.nullPointer();
        numWorkerThreads = 0;

        /*
         * Free any chunks left in the chunk releaser. This needs locking because a worker thread
         * might be doing this in parallel. */
        mutex.lockNoTransitionUnspecifiedOwner();
        try {
            doCleanup();
        } finally {
            mutex.unlockNoTransitionUnspecifiedOwner();
        }
    }

    private static int getWorkerCount() {
        int setting = SubstrateOptions.DeprecatedOptions.ParallelGCThreads.getValue();
        int workerCount = setting > 0 ? setting : getDefaultWorkerCount();
        verboseGCLog().string("[Number of ParallelGC threads: ").unsigned(workerCount).string("]").newline();
        return workerCount;
    }

    private static int getDefaultWorkerCount() {
        /* This does not take the container support into account. */
        int cpus = Jvm.JVM_ActiveProcessorCount();
        return UninterruptibleUtils.Math.min(cpus, MAX_WORKER_THREADS);
    }

    @Uninterruptible(reason = "Heap base is not set up yet.")
    @CEntryPoint(include = UseParallelGC.class, publishAs = CEntryPoint.Publish.NotPublished)
    @CEntryPointOptions(prologue = GCWorkerThreadPrologue.class, epilogue = CEntryPointOptions.NoEpilogue.class)
    private static void gcWorkerRun(GCWorkerThreadState state) {
        try {
            ParallelGC.singleton().work(state);
        } catch (Throwable e) {
            throw VMError.shouldNotReachHere(e);
        }
    }

    @NeverInline("Prevent reads from floating up.")
    @Uninterruptible(reason = "Called from a GC worker thread.")
    private void work(GCWorkerThreadState state) {
        PlatformThreads.singleton().setUnmanagedThreadLocalValue(workerStateTL, state);
        try {
            work0(state);
        } catch (Throwable e) {
            VMError.shouldNotReachHere(e);
        }
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    private void work0(GCWorkerThreadState state) {
        while (phase != Phase.SHUTDOWN) {
            Pointer ptr;
            mutex.lockNoTransitionUnspecifiedOwner();
            try {
                ptr = chunkQueue.pop();
                /* Block if there is no local/global work. */
                if (ptr.isNull() && getNextAllocChunkScanPointer(false).isNull()) {
                    decrementBusyWorkers();
                    do {
                        parPhase.blockNoTransitionUnspecifiedOwner();
                        attemptCleanup();
                    } while (phase == Phase.SEQUENTIAL);
                    incrementBusyWorkers();
                }
            } finally {
                mutex.unlockNoTransitionUnspecifiedOwner();
            }

            if (ptr.isNonNull()) {
                scanChunk(ptr);
            } else {
                scanAllocChunk(state);
            }
        }
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    private static void scanChunk(Pointer ptr) {
        assert ptr.isNonNull();
        boolean aligned = ptr.and(UNALIGNED_BIT).notEqual(UNALIGNED_BIT);
        UnsignedWord op = ptr.and(SCAN_OP_MASK);
        ptr = ptr.and(POINTER_MASK);
        GreyToBlackObjectVisitor visitor = GCImpl.getGCImpl().getGreyToBlackObjectVisitor();
        if (op.equal(SCAN_CARD_TABLE)) {
            AlignedHeapChunk.AlignedHeader aChunk = WordFactory.nullPointer();
            UnalignedHeapChunk.UnalignedHeader uChunk = WordFactory.nullPointer();
            if (aligned) {
                aChunk = (AlignedHeapChunk.AlignedHeader) ptr;
            } else {
                uChunk = (UnalignedHeapChunk.UnalignedHeader) ptr;
            }
            GreyToBlackObjRefVisitor refVisitor = GCImpl.getGCImpl().getGreyToBlackObjRefVisitor();
            RememberedSet.get().walkDirtyObjects(aChunk, uChunk, WordFactory.nullPointer(), visitor, refVisitor, true);
        } else if (op.equal(SCAN_GREY_OBJECTS)) {
            if (aligned) {
                AlignedHeapChunk.AlignedHeader aChunk = AlignedHeapChunk.getEnclosingChunkFromObjectPointer(ptr);
                HeapChunk.walkObjectsFromInline(aChunk, ptr, visitor);
            } else {
                UnalignedHeapChunk.UnalignedHeader uChunk = (UnalignedHeapChunk.UnalignedHeader) ptr;
                UnalignedHeapChunk.walkObjectsInline(uChunk, visitor);
            }
        } else {
            VMError.shouldNotReachHere("Unknown opcode");
        }
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    private void scanAllocChunk(GCWorkerThreadState state) {
        Pointer scanPtr = getNextAllocChunkScanPointer(false);
        if (scanPtr.isNonNull()) {
            AlignedHeapChunk.AlignedHeader allocChunk = AlignedHeapChunk.getEnclosingChunkFromObjectPointer(scanPtr);
            setScannedChunk(state, allocChunk);
            allocChunk.getSpace().walkAllocChunk(allocChunk, scanPtr, GCImpl.getGCImpl().getGreyToBlackObjectVisitor());
            setScannedChunk(state, WordFactory.nullPointer());
        }
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    private Pointer getNextAllocChunkScanPointer(boolean clear) {
        for (int i = 1; i < getStateWords(); i++) {
            Pointer scanPtr = getAllocChunkScanPointer(i, clear);
            if (scanPtr.isNonNull()) {
                AlignedHeapChunk.AlignedHeader allocChunk = AlignedHeapChunk.getEnclosingChunkFromObjectPointer(scanPtr);
                if (HeapChunk.getTopPointer(allocChunk).aboveThan(scanPtr)) {
                    return scanPtr;
                }
            }
        }
        return WordFactory.nullPointer();
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    private void incrementBusyWorkers() {
        assert mutex.isOwner(true);
        ++busyWorkerThreads;
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    private void decrementBusyWorkers() {
        assert mutex.isOwner(true);
        if (--busyWorkerThreads == 0) {
            phase = Phase.SEQUENTIAL;
            seqPhase.signal();
        }
    }

    /**
     * Start parallel phase and wait until all chunks have been processed.
     */
    @Uninterruptible(reason = "Called from uninterruptible code.", mayBeInlined = true)
    public void scheduleScan() {
        /* Push all alloc chunks filled during sequential phase */
        Pointer allocChunk;
        while ((allocChunk = getNextAllocChunkScanPointer(true)).isNonNull()) {
            push(allocChunk, SCAN_GREY_OBJECTS);
        }

        /* Reset all thread local states. */
        UnmanagedMemoryUtil.fillLongs((Pointer) workerStates, workerStatesSize, 0L);

        mutex.lockNoTransitionUnspecifiedOwner();
        try {
            /* A cleanup might have been scheduled. Wait for it to finish. */
            waitUntilWorkerThreadsFinish0();

            /* Let worker threads run. */
            phase = Phase.PARALLEL;
            parPhase.broadcast();

            waitUntilWorkerThreadsFinish0();
        } finally {
            mutex.unlockNoTransitionUnspecifiedOwner();
        }

        assert chunkQueue.isEmpty();
        assert phase != Phase.PARALLEL;
        assert busyWorkerThreads == 0;
    }

    @Uninterruptible(reason = "Called from uninterruptible code.", mayBeInlined = true)
    private void waitUntilWorkerThreadsFinish() {
        mutex.lockNoTransitionUnspecifiedOwner();
        try {
            waitUntilWorkerThreadsFinish0();
        } finally {
            mutex.unlockNoTransitionUnspecifiedOwner();
        }
    }

    @Uninterruptible(reason = "Called from uninterruptible code.", mayBeInlined = true)
    private void waitUntilWorkerThreadsFinish0() {
        while (phase != Phase.SEQUENTIAL) {
            seqPhase.blockNoTransitionUnspecifiedOwner();
        }
    }

    public void scheduleCleanup() {
        mutex.lock();
        try {
            phase = Phase.CLEANUP;
            parPhase.signal();
        } finally {
            mutex.unlock();
        }
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    private void attemptCleanup() {
        if (phase == Phase.CLEANUP) {
            doCleanup();
            phase = Phase.SEQUENTIAL;
            seqPhase.signal();
        }
    }

    @Uninterruptible(reason = "Called from a GC worker thread.")
    private void doCleanup() {
        assert mutex.isOwner(true) || phase == Phase.SHUTDOWN;
        GCImpl.getGCImpl().freeChunks();
    }

    private static Log verboseGCLog() {
        return SubstrateGCOptions.VerboseGC.getValue() ? Log.log() : Log.noopLog();
    }

    /*
     * Worker thread state is a bunch of chunk pointers laid out in the following manner:
     * word 0   :  scanned chunk pointer, also reused for isolate pointer at the very start of a thread
     *      1   :
     *  ...     :  survivor states' allocation pointers
     *      N   :
     *      N+1 :  old gen allocation pointer
     */
    @RawPointerTo(GCWorkerThreadState.AlignedChunkPointer.class)
    interface GCWorkerThreadState extends PointerBase {
        @RawStructure
        interface AlignedChunkPointer extends PointerBase {}

        PointerBase read(int age);
        void write(int age, PointerBase value);
        GCWorkerThreadState addressOf(int index);
    }

    @Fold
    static int getStateWords() {
        return HeapParameters.getMaxSurvivorSpaces() + 2;
    }

    @Uninterruptible(reason = "Called from uninterruptible code.", mayBeInlined = true)
    private static PointerBase getScannedChunk(GCWorkerThreadState state) {
        return state.read(0);
    }

    @Uninterruptible(reason = "Called from uninterruptible code.", mayBeInlined = true)
    private static void setScannedChunk(GCWorkerThreadState state, PointerBase chunkPointer) {
        state.write(0, chunkPointer);
    }

    private static class GCWorkerThreadPrologue implements CEntryPointOptions.Prologue {
        @Uninterruptible(reason = "prologue")
        @SuppressWarnings("unused")
        public static void enter(GCWorkerThreadState state) {
            CEntryPointSnippets.setHeapBase(getScannedChunk(state));
            WriteCurrentVMThreadNode.writeCurrentVMThread(WordFactory.nullPointer());
        }
    }

    private static class UseParallelGC implements BooleanSupplier {
        @Override
        public boolean getAsBoolean() {
            return ParallelGC.isEnabled();
        }
    }
}

@Platforms(Platform.HOSTED_ONLY.class)
@AutomaticallyRegisteredFeature()
@SuppressWarnings("unused")
class ParallelGCFeature implements InternalFeature {
    @Override
    public boolean isInConfiguration(IsInConfigurationAccess access) {
        return ParallelGC.isEnabled();
    }

    @Override
    public void afterRegistration(AfterRegistrationAccess access) {
        verifyOptionEnabled(SubstrateOptions.SpawnIsolates);

        ImageSingletons.add(ParallelGC.class, new ParallelGC());
    }

    private static void verifyOptionEnabled(SubstrateOptionKey<Boolean> option) {
        String optionMustBeEnabledFmt = "When using the parallel garbage collector ('--gc=parallel'), please note that option '%s' must be enabled.";
        UserError.guarantee(option.getValue(), optionMustBeEnabledFmt, option.getName());
    }
}
