/*
 * Copyright (c) 2017, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.svm.truffle.nfi;

import static com.oracle.svm.truffle.nfi.NativeSignature.ExecuteHelper;

import jdk.graal.compiler.word.Word;
import org.graalvm.nativeimage.ImageSingletons;
import org.graalvm.nativeimage.StackValue;
import org.graalvm.nativeimage.UnmanagedMemory;
import org.graalvm.nativeimage.c.struct.SizeOf;
import org.graalvm.nativeimage.c.type.CLongPointer;
import org.graalvm.nativeimage.c.type.WordPointer;

import com.oracle.svm.core.annotate.Alias;
import com.oracle.svm.core.annotate.RecomputeFieldValue;
import com.oracle.svm.core.annotate.RecomputeFieldValue.Kind;
import com.oracle.svm.core.annotate.Substitute;
import com.oracle.svm.core.annotate.TargetClass;
import com.oracle.svm.core.fieldvaluetransformer.NewEmptyArrayFieldValueTransformer;
import com.oracle.svm.core.handles.PrimitiveArrayView;
import com.oracle.svm.truffle.nfi.NativeAPI.NativeTruffleContext;
import com.oracle.svm.truffle.nfi.NativeAPI.NativeTruffleEnv;
import com.oracle.svm.truffle.nfi.NativeSignature.CifData;
import com.oracle.svm.truffle.nfi.NativeSignature.PrepareHelper;
import com.oracle.svm.truffle.nfi.libffi.LibFFI;
import com.oracle.svm.truffle.nfi.libffi.LibFFI.ffi_cif;
import com.oracle.truffle.api.CallTarget;
import com.oracle.truffle.api.CompilerDirectives.TruffleBoundary;

@TargetClass(className = "com.oracle.truffle.nfi.backend.libffi.LibFFIContext", onlyWith = TruffleNFIFeature.IsEnabled.class)
final class Target_com_oracle_truffle_nfi_backend_libffi_LibFFIContext {

    // clear these fields, they will be re-filled by patchContext
    @Alias @RecomputeFieldValue(kind = Kind.Reset) private long nativeContext;
    @Alias @RecomputeFieldValue(kind = Kind.Custom, declClass = NewEmptyArrayFieldValueTransformer.class) Target_com_oracle_truffle_nfi_backend_libffi_LibFFIType[] simpleTypeMap;
    @Alias @RecomputeFieldValue(kind = Kind.Custom, declClass = NewEmptyArrayFieldValueTransformer.class) Target_com_oracle_truffle_nfi_backend_libffi_LibFFIType[] arrayTypeMap;
    @Alias @RecomputeFieldValue(kind = Kind.Reset) Target_com_oracle_truffle_nfi_backend_libffi_LibFFIType cachedEnvType;

    @Alias Target_com_oracle_truffle_nfi_backend_libffi_LibFFILanguage language;

    @Alias
    native long getNativeEnv();

    @Alias
    native boolean attachThread();

    @Alias
    native void detachThread();

    @Alias
    native Target_com_oracle_truffle_nfi_backend_libffi_ClosureNativePointer createClosureNativePointer(long nativeClosure, long codePointer, CallTarget callTarget,
                    Target_com_oracle_truffle_nfi_backend_libffi_LibFFISignature signature, Object receiver);

    @Alias
    native void newClosureRef(long codePointer);

    @Alias
    native void releaseClosureRef(long codePointer);

    @Alias
    native Object getClosureObject(long codePointer);

    @Alias
    protected native void initializeSimpleType(Target_com_oracle_truffle_nfi_backend_spi_types_NativeSimpleType simpleType, int size, int alignment, long ffiType);

    @Substitute
    private long initializeNativeContext() {
        TruffleNFISupport support = ImageSingletons.lookup(TruffleNFISupport.class);

        NativeTruffleContext ret = UnmanagedMemory.malloc(SizeOf.get(NativeTruffleContext.class));
        ret.setContextHandle(support.createContextHandle(this));

        NFIInitialization.initializeContext(ret);
        NFIInitialization.initializeSimpleTypes(this);

        return ret.rawValue();
    }

    @Substitute
    private static void disposeNativeContext(long context) {
        TruffleNFISupport support = ImageSingletons.lookup(TruffleNFISupport.class);
        NativeTruffleContext ctx = Word.pointer(context);
        support.destroyContextHandle(ctx.contextHandle());
        UnmanagedMemory.free(ctx);
    }

    @Substitute
    private static long initializeNativeEnvV2(long context, @SuppressWarnings("unused") Target_com_oracle_truffle_nfi_backend_spi_NFIState state) {
        /*
         * No need to store the state, on SVM thread-local lookups are reasonably fast from the
         * native side.
         */
        return initializeNativeEnv(context);
    }

    @Substitute
    private static void disposeNativeEnvV2(long env) {
        UnmanagedMemory.free(Word.pointer(env));
    }

    @Substitute
    private static long initializeNativeEnv(long context) {
        NativeTruffleContext ctx = Word.pointer(context);
        NativeTruffleEnv env = UnmanagedMemory.malloc(SizeOf.get(NativeTruffleEnv.class));
        NFIInitialization.initializeEnv(env, ctx);
        return env.rawValue();
    }

    @Substitute
    Target_com_oracle_truffle_nfi_backend_libffi_ClosureNativePointer allocateClosureObjectRet(Target_com_oracle_truffle_nfi_backend_libffi_LibFFISignature signature, CallTarget callTarget,
                    Object receiver) {
        return NativeClosure.prepareClosure(this, signature, callTarget, receiver, NativeClosure.INVOKE_CLOSURE_OBJECT_RET.getFunctionPointer());
    }

    @Substitute
    Target_com_oracle_truffle_nfi_backend_libffi_ClosureNativePointer allocateClosureStringRet(Target_com_oracle_truffle_nfi_backend_libffi_LibFFISignature signature, CallTarget callTarget,
                    Object receiver) {
        return NativeClosure.prepareClosure(this, signature, callTarget, receiver, NativeClosure.INVOKE_CLOSURE_STRING_RET.getFunctionPointer());
    }

    @Substitute
    Target_com_oracle_truffle_nfi_backend_libffi_ClosureNativePointer allocateClosureBufferRet(Target_com_oracle_truffle_nfi_backend_libffi_LibFFISignature signature, CallTarget callTarget,
                    Object receiver) {
        return NativeClosure.prepareClosure(this, signature, callTarget, receiver, NativeClosure.INVOKE_CLOSURE_BUFFER_RET.getFunctionPointer());
    }

    @Substitute
    Target_com_oracle_truffle_nfi_backend_libffi_ClosureNativePointer allocateClosureVoidRet(Target_com_oracle_truffle_nfi_backend_libffi_LibFFISignature signature, CallTarget callTarget,
                    Object receiver) {
        return NativeClosure.prepareClosure(this, signature, callTarget, receiver, NativeClosure.INVOKE_CLOSURE_VOID_RET.getFunctionPointer());
    }

    @Substitute
    @SuppressWarnings("static-method")
    long prepareSignature(Target_com_oracle_truffle_nfi_backend_libffi_LibFFIType retType, int argCount, Target_com_oracle_truffle_nfi_backend_libffi_LibFFIType... args) {
        CifData data = PrepareHelper.prepareArgs(argCount, args);
        int ret = LibFFI.ffi_prep_cif(data.cif(), LibFFI.FFI_DEFAULT_ABI(), Word.unsigned(argCount), Word.pointer(retType.type), data.args());
        return PrepareHelper.checkRet(data, ret);
    }

    @Substitute
    @SuppressWarnings("static-method")
    long prepareSignatureVarargs(Target_com_oracle_truffle_nfi_backend_libffi_LibFFIType retType, int argCount, int nFixedArgs, Target_com_oracle_truffle_nfi_backend_libffi_LibFFIType... args) {
        CifData data = PrepareHelper.prepareArgs(argCount, args);
        int ret = LibFFI.ffi_prep_cif_var(data.cif(), LibFFI.FFI_DEFAULT_ABI(), Word.unsigned(nFixedArgs), Word.unsigned(argCount), Word.pointer(retType.type), data.args());
        return PrepareHelper.checkRet(data, ret);
    }

    @Substitute
    @TruffleBoundary
    void executeNative(long cif, long functionPointer, byte[] primArgs, int patchCount, int[] patchOffsets, Object[] objArgs, byte[] ret) {
        try (LocalNativeScope scope = TruffleNFISupport.createLocalScope(patchCount);
                        PrimitiveArrayView retBuffer = PrimitiveArrayView.createForReadingAndWriting(ret)) {
            NativeTruffleContext ctx = Word.pointer(nativeContext);
            LibFFI.ffi_cif ffiCif = Word.pointer(cif);
            ExecuteHelper.execute(ctx, ffiCif, retBuffer.addressOfArrayElement(0), functionPointer, primArgs, patchCount, patchOffsets, objArgs, scope);
        }
    }

    @Substitute
    @TruffleBoundary
    long executePrimitive(long cif, long functionPointer, byte[] primArgs, int patchCount, int[] patchOffsets, Object[] objArgs) {
        try (LocalNativeScope scope = TruffleNFISupport.createLocalScope(patchCount)) {
            NativeTruffleContext ctx = Word.pointer(nativeContext);
            ffi_cif ffiCif = Word.pointer(cif);
            CLongPointer retPtr = StackValue.get(8);
            ExecuteHelper.execute(ctx, ffiCif, retPtr, functionPointer, primArgs, patchCount, patchOffsets, objArgs, scope);
            return retPtr.read();
        }
    }

    @Substitute
    @TruffleBoundary
    Object executeObject(long cif, long functionPointer, byte[] primArgs, int patchCount, int[] patchOffsets, Object[] objArgs) {
        try (LocalNativeScope scope = TruffleNFISupport.createLocalScope(patchCount)) {
            NativeTruffleContext ctx = Word.pointer(nativeContext);
            ffi_cif ffiCif = Word.pointer(cif);
            WordPointer retPtr = StackValue.get(8);
            ExecuteHelper.execute(ctx, ffiCif, retPtr, functionPointer, primArgs, patchCount, patchOffsets, objArgs, scope);
            return ImageSingletons.lookup(TruffleNFISupport.class).resolveHandle(retPtr.read());
        }
    }

    @Substitute
    private void loadNFILib() {
        // do nothing, the NFI library is statically linked to the SVM image
    }

    @Substitute
    @TruffleBoundary
    static long loadLibrary(@SuppressWarnings("unused") long nativeContext, String name, int flags) {
        return TruffleNFISupport.loadLibrary(nativeContext, name, flags);
    }

    @Substitute
    @TruffleBoundary
    static void freeLibrary(long library) {
        TruffleNFISupport.freeLibrary(library);
    }

    @Substitute
    @TruffleBoundary
    Object lookupSymbol(Target_com_oracle_truffle_nfi_backend_libffi_LibFFILibrary library, String name) {
        if (ImageSingletons.lookup(TruffleNFISupport.class).errnoGetterFunctionName.equals(name)) {
            return new ErrnoMirror();
        } else {
            return Target_com_oracle_truffle_nfi_backend_libffi_LibFFISymbol.create(library, name, lookup(nativeContext, library.handle, name));
        }
    }

    @Substitute
    @TruffleBoundary
    static long lookup(@SuppressWarnings("unused") long nativeContext, long library, String name) {
        return TruffleNFISupport.lookup(nativeContext, library, name);
    }
}
