/*
 * Copyright (c) 2014, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 3 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.truffle.r.nodes.builtin.base.printer;

import java.io.IOException;
import java.io.PrintWriter;

import com.oracle.truffle.api.CompilerDirectives.TruffleBoundary;
import com.oracle.truffle.r.runtime.data.RExpression;
import com.oracle.truffle.r.runtime.data.RRaw;
import com.oracle.truffle.r.runtime.data.RStringVector;

final class ExpressionPrinter extends AbstractValuePrinter<RExpression> {

    static final ExpressionPrinter INSTANCE = new ExpressionPrinter();

    private ExpressionPrinter() {
        // singleton
    }

    @Override
    @TruffleBoundary
    protected void printValue(RExpression expr, PrintContext printCtx) throws IOException {
        final PrintWriter out = printCtx.output();
        final PrintContext valPrintCtx = printCtx.cloneContext();
        valPrintCtx.parameters().setSuppressIndexLabels(true);

        out.print("expression(");
        RStringVector names = expr.getNames();
        for (int i = 0; i < expr.getLength(); i++) {
            if (i != 0) {
                out.print(", ");
            }
            if (names != null && names.getDataAt(i) != null) {
                out.print(names.getDataAt(i));
                out.print(" = ");
            }
            Object data = expr.getDataAt(i);
            if (data instanceof RRaw) {
                // when in expression, raw is printed differently
                // than in ValuePrinters.INSTANCE.print() -> RawVectorPrinter.print()
                out.print("as.raw(0x");
                ValuePrinters.INSTANCE.print(data, valPrintCtx);
                out.print(")");
            } else {
                ValuePrinters.INSTANCE.print(data, valPrintCtx);
            }
        }
        out.print(')');
    }
}
