/*
 * Copyright (c) 2015, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package jdk.graal.compiler.core.test;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.junit.Test;

import jdk.graal.compiler.core.common.GraalOptions;
import jdk.graal.compiler.java.BytecodeParserOptions;
import jdk.graal.compiler.nodes.spi.ProfileProvider;
import jdk.graal.compiler.options.OptionValues;
import jdk.vm.ci.meta.ResolvedJavaMethod;

public class ArrayLengthProviderTest extends GraalCompilerTest {

    public static Object test0Snippet(ArrayList<?> list, boolean a) {
        while (true) {
            Object[] array = toArray(list);
            if (array.length < 1) {
                return null;
            }
            if (array[0] instanceof String || a) {
                /*
                 * This code is outside of the loop. Accessing the array reqires a ValueProxyNode.
                 * When the simplification of the ArrayLengthNode replaces the length access with
                 * the ArrayList.size used to create the array, then the value needs to have a
                 * ValueProxyNode too. In addition, the two parts of the if-condition actually lead
                 * to two separate loop exits, with two separate proxy nodes. A ValuePhiNode is
                 * present originally for the array, and the array length simplification needs to
                 * create a new ValuePhiNode for the two newly introduced ValueProxyNode.
                 */
                if (array.length < 1) {
                    return null;
                }
                return array[0];
            }
        }
    }

    public static Object test1Snippet(List<?> list, boolean a, boolean b) {
        while (true) {
            Object[] array = toArray(list);
            if (a || b) {
                if (array.length < 1) {
                    return null;
                }
                return array[0];
            }
        }
    }

    public static Object[] toArray(List<?> list) {
        return new Object[list.size()];
    }

    @Override
    protected ProfileProvider getProfileProvider(ResolvedJavaMethod method) {
        if (profile) {
            return null;
        }
        return NO_PROFILE_PROVIDER;
    }

    boolean profile = true;

    @Test
    public void test0() {
        test("test0Snippet", new ArrayList<>(Arrays.asList("a", "b")), true);
    }

    @Test
    public void test1() {
        test("test1Snippet", new ArrayList<>(Arrays.asList("a", "b")), true, true);

    }

    @Test
    public void test2() {
        resetCache();
        profile = false;
        OptionValues o = new OptionValues(getInitialOptions(), GraalOptions.LoopUnswitch, false, BytecodeParserOptions.InlineDuringParsing, false);
        test(o, "test1Snippet", new ArrayList<>(Arrays.asList("a", "b")), true, true);
        profile = true;
    }
}
