/*
 * Copyright (c) 2016, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
/*
 */
package jdk.graal.compiler.core.test;

import org.junit.Test;

import jdk.graal.compiler.code.CompilationResult;
import jdk.graal.compiler.core.phases.HighTier;
import jdk.graal.compiler.debug.DebugContext;
import jdk.graal.compiler.nodes.StructuredGraph;
import jdk.graal.compiler.nodes.StructuredGraph.AllowAssumptions;
import jdk.graal.compiler.options.OptionValues;
import jdk.graal.compiler.phases.tiers.Suites;

import jdk.vm.ci.meta.ResolvedJavaMethod;

public final class DontReuseArgumentSpaceTest extends GraalCompilerTest {

    @Override
    @SuppressWarnings("try")
    protected Suites createSuites(OptionValues options) {
        return super.createSuites(new OptionValues(options, HighTier.Options.Inline, false));
    }

    @BytecodeParserNeverInline
    public static int killArguments(int a, int b, int c, int d, int e, int f, int g, int h, int i, int j) {
        return a + b + c + d + e + f + g + h + i + j;
    }

    @BytecodeParserNeverInline
    public static int callTwice(int a, int b, int c, int d, int e, int f, int g, int h, int i, int j) {
        /*
         * Call the same method twice so the arguments are in the same place each time and might
         * appear to be redundant moves.
         */
        killArguments(a, b, c, d, e, f, g, h, i, j);
        return killArguments(a, b, c, d, e, f, g, h, i, j);
    }

    @Test
    public void run0() throws Throwable {
        /*
         * Exercise the methods once so everything is resolved
         */
        callTwice(1, 2, 3, 4, 5, 6, 7, 8, 9, 10);

        /*
         * Create a standalone compile of killArguments. This test assumes that zapping of argument
         * space is being performed by the backend.
         */
        ResolvedJavaMethod javaMethod = getResolvedJavaMethod("killArguments");
        StructuredGraph graph = parseEager(javaMethod, AllowAssumptions.YES);
        CompilationResult compilationResult = compile(javaMethod, graph);
        DebugContext debug = getDebugContext();
        getBackend().createDefaultInstalledCode(debug, javaMethod, compilationResult);

        test("callTwice", 1, 2, 3, 4, 5, 6, 7, 8, 9, 10);
    }
}
