/*
 * Copyright (c) 2019, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.truffle.espresso.nodes;

import com.oracle.truffle.api.CompilerAsserts;
import com.oracle.truffle.api.CompilerDirectives;
import com.oracle.truffle.api.CompilerDirectives.TruffleBoundary;
import com.oracle.truffle.api.frame.Frame;
import com.oracle.truffle.api.frame.MaterializedFrame;
import com.oracle.truffle.api.instrumentation.ProbeNode;
import com.oracle.truffle.api.instrumentation.StandardTags;
import com.oracle.truffle.api.instrumentation.Tag;
import com.oracle.truffle.api.interop.NodeLibrary;
import com.oracle.truffle.api.library.ExportLibrary;
import com.oracle.truffle.api.library.ExportMessage;
import com.oracle.truffle.api.nodes.Node;
import com.oracle.truffle.api.source.SourceSection;

/**
 * Node that simulates espresso statements for debugging support.
 */
@ExportLibrary(NodeLibrary.class)
public final class EspressoStatementNode extends EspressoInstrumentableNode implements BciProvider {

    private final int startBci;
    @Child private volatile ProbeNode eagerProbe;
    private final SourceSection sourceSection;

    EspressoStatementNode(int startBci, SourceSection section) {
        this.startBci = startBci;
        this.sourceSection = section;
    }

    @Override
    public SourceSection getSourceSection() {
        return sourceSection;
    }

    @Override
    public int getBci(@SuppressWarnings("unused") Frame frame) {
        return startBci;
    }

    @Override
    public ProbeNode findProbe() {
        ProbeNode p = this.eagerProbe;
        if (p == null) {
            CompilerDirectives.transferToInterpreterAndInvalidate();
            this.eagerProbe = p = insert(createProbe(getSourceSection()));
        }
        CompilerAsserts.partialEvaluationConstant(p);
        return p;
    }

    @Override
    public boolean hasTag(Class<? extends Tag> tag) {
        return tag == StandardTags.StatementTag.class;
    }

    @Override
    public WrapperNode createWrapper(ProbeNode probe) {
        throw new UnsupportedOperationException();
    }

    private BytecodeNode getBytecodeNode() {
        Node parent = getParent();
        while (!(parent instanceof BytecodeNode) && parent != null) {
            parent = parent.getParent();
        }
        return (BytecodeNode) parent;
    }

    @ExportMessage
    @SuppressWarnings("static-method")
    public boolean hasScope(@SuppressWarnings("unused") Frame frame) {
        return true;
    }

    @ExportMessage
    public Object getScope(Frame frame, boolean nodeEnter) {
        return getScopeSlowPath(frame != null ? frame.materialize() : null, nodeEnter);
    }

    @TruffleBoundary
    private Object getScopeSlowPath(MaterializedFrame frame, boolean nodeEnter) {
        return getBytecodeNode().getScope(frame, nodeEnter);
    }
}
