/*
 * Copyright (c) 2021, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.regex.tregex.test;

import java.util.Collections;
import java.util.Map;

import org.junit.Test;

import com.oracle.truffle.regex.RegexSyntaxException.ErrorCode;
import com.oracle.truffle.regex.tregex.string.Encodings;

public class OracleDBTests extends RegexTestBase {

    private static final Map<String, String> ENGINE_OPTIONS = Map.of("regexDummyLang.Flavor", "OracleDB");

    @Override
    Map<String, String> getEngineOptions() {
        return ENGINE_OPTIONS;
    }

    @Override
    Encodings.Encoding getTRegexEncoding() {
        return Encodings.UTF_8;
    }

    @Test
    public void generatedTests() {
        /* GENERATED CODE BEGIN - KEEP THIS MARKER FOR AUTOMATIC UPDATES */

        test("\\0", "", "\\0", 0, true, 1, 2);
        test("\\b", "", "\\b", 0, true, 1, 2);
        test("\\B", "", "\\B", 0, true, 1, 2);
        test("\\f", "", "\\f", 0, true, 1, 2);
        test("\\n", "", "\\n", 0, true, 1, 2);
        test("\\r", "", "\\r", 0, true, 1, 2);
        test("\\t", "", "\\t", 0, true, 1, 2);
        test("\\v", "", "\\v", 0, true, 1, 2);
        test("\\x", "", "\\x", 0, true, 1, 2);
        test("\\u", "", "\\u", 0, true, 1, 2);
        test("\\p", "", "\\p", 0, true, 1, 2);
        test("\\P", "", "\\P", 0, true, 1, 2);
        test("(:)", "", ":", 0, true, 0, 1, 0, 1);
        test("(?:)", "", "?:", 0, true, 1, 2, 1, 2);
        test("(=)", "", "=", 0, true, 0, 1, 0, 1);
        test("(?=)", "", "?=", 0, true, 1, 2, 1, 2);
        test("(!)", "", "!", 0, true, 0, 1, 0, 1);
        test("(?!)", "", "?!", 0, true, 1, 2, 1, 2);
        test("(<=)", "", "<=", 0, true, 0, 2, 0, 2);
        test("(?<=)", "", "?<=", 0, true, 1, 3, 1, 3);
        test("(<!)", "", "<!", 0, true, 0, 2, 0, 2);
        test("(?<!)", "", "?<!", 0, true, 1, 3, 1, 3);
        test("(<a>)", "", "<a>", 0, true, 0, 3, 0, 3);
        test("(?<a>)", "", "?<a>", 0, true, 1, 4, 1, 4);
        test("(P<a>)", "", "P<a>", 0, true, 0, 4, 0, 4);
        test("(?P<a>)", "", "?P<a>", 0, true, 1, 5, 1, 5);
        test("(P=a)", "", "P=a", 0, true, 0, 3, 0, 3);
        test("(?P=a)", "", "?P=a", 0, true, 1, 4, 1, 4);
        test("\\Ax", "", "yx", 0, false);
        test("\\s\\Z", "", "x \n", 0, true, 1, 2);
        test("\\s\\Z", "", "x \n", 2, true, 2, 3);
        test("\\s\\Z", "", "x \n \n", 0, true, 3, 4);
        test("\\s\\Z", "m", "x \n \n", 0, true, 3, 4);
        test("\\s$", "", "x \n", 0, true, 1, 2);
        test("\\s$", "", "x \n", 2, true, 2, 3);
        test("\\s$", "", "x \n \n", 0, true, 3, 4);
        test("\\s$", "m", "x \n \n", 0, true, 1, 2);
        test("\\s\\z", "", "x \n", 0, true, 2, 3);
        test("\\s\\z", "", "x \n", 2, true, 2, 3);
        test("\\s\\z", "", "x \n \n", 0, true, 4, 5);
        test("\\s\\z", "m", "x \n \n", 0, true, 4, 5);
        test("}", "", "}", 0, true, 0, 1);
        test("{", "", "{", 0, true, 0, 1);
        test("]", "", "]", 0, true, 0, 1);
        expectSyntaxError("[", "", "_", 0, ErrorCode.UnmatchedBracket);
        expectSyntaxError(")", "", "_", 0, ErrorCode.UnmatchedParenthesis);
        test("x{,1}", "", "x", 0, false);
        test("x{,1}?", "", "x", 0, false);
        test("x{,1}", "", "x{,1}", 0, true, 0, 5);
        test("x{,1}?", "", "x{,1}", 0, true, 0, 5);
        test("x{,1}?", "", "x{,1", 0, true, 0, 4);
        test("x{1,}", "", "x{1,}", 0, true, 0, 1);
        expectSyntaxError("x{2,1}", "", "_", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{2147483649}", "", "x{2147483649}", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{2147483649,}", "", "x{2147483649}", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{2147483649,2}", "", "_", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{2147483649,2147483650}", "", "xxx", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{0,4294967295}", "", "x{4294967295}", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{1,4294967295}", "", "x{4294967295}", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{2147483648,4294967295}", "", "xx", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{1,2147483648}", "", "xx", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{2,2147483648}", "", "xx", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{2,2147483649}", "", "xx", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{4294967296}", "", "x{4294967296}", 0, ErrorCode.InvalidQuantifier);
        expectSyntaxError("x{4294967297}", "", "x{4294967297}", 0, ErrorCode.InvalidQuantifier);
        test("x??", "", "x", 0, true, 0, 0);
        test("\\A*x\\Z+", "", "x", 0, true, 0, 1);
        test("\\A*x\\Z+", "", "xx", 0, true, 1, 2);
        test("\\A+x\\Z+", "", "xx", 0, false);
        test("x{2}?", "", "xxxxx", 0, true, 0, 2);
        test("(a{0,1})*b\\1", "", "aab", 0, true, 0, 3, 2, 2);
        test("(a{0,1})*b\\1", "", "aaba", 0, true, 0, 3, 2, 2);
        test("(a{0,1})*b\\1", "", "aabaa", 0, true, 0, 3, 2, 2);
        test("[]]", "", "]", 0, true, 0, 1);
        test("[][]+", "", "[]", 0, true, 0, 2);
        expectSyntaxError("[]", "", "_", 0, ErrorCode.UnmatchedBracket);
        expectSyntaxError("[b-a]", "", "_", 0, ErrorCode.InvalidCharacterClass);
        test("[-a]", "", "-a", 0, true, 0, 1);
        test("[a-]", "", "-a", 0, true, 0, 1);
        test("[[ab]]", "", "[]ab]", 0, true, 0, 2);
        test("[[ab]]", "", "[]ab]", 1, true, 3, 5);
        test("[[::]]", "", "[]:", 0, true, 0, 1);
        test("[[::]]", "", "[]:", 1, true, 1, 2);
        test("[[::]]", "", "[]:", 2, true, 2, 3);
        expectSyntaxError("[[:ab:]]", "", "_", 0, ErrorCode.InvalidCharacterClass);
        test("[[:upper:]]+", "", ":upper:", 0, false);
        test("[[:upper:]]+", "", ":UPPER:", 0, true, 1, 6);
        test("[:upper:]+", "", ":upper:", 0, true, 0, 7);
        test("[[:lower:]]+", "", ":LOWER:", 0, false);
        test("[[:lower:]]+", "", ":lower:", 0, true, 1, 6);
        test("[:lower:]+", "", ":lower:", 0, true, 0, 7);
        test("[x[ab]y]", "", "ay]", 0, true, 0, 3);
        test("[a\\]+", "", "a\\", 0, true, 0, 2);
        test("[\\a]+", "", "a\\", 0, true, 0, 2);
        test("[[:lower:]-[:upper:]]+", "", "a-A", 0, true, 0, 3);
        test("[a-z]+", "", "abc-", 0, true, 0, 3);
        test("[[.a.]-[.z.]]+", "", "abc-", 0, true, 0, 3);
        test("[[=a=]-[=z=]]+", "", "abc-", 0, true, 0, 3);
        test("[[.a.]-[=z=]]+", "", "abc-", 0, true, 0, 3);
        test("[[=a=]-[.z.]]+", "", "abc-", 0, true, 0, 3);
        expectSyntaxError("[[.a.]-[.ch.]]+", "", "_", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[.a.]-[:lower:]]+", "", "_", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[=a=]-[:lower:]]+", "", "_", 0, ErrorCode.InvalidCharacterClass);
        test("[[:upper:]-[.a.]]+", "", "a-A", 0, true, 0, 3);
        test("[[=c=]-c]", "", "\u010d-=c", 0, true, 4, 5);
        test("[[=c=]-]+", "", "\u010d-=c", 0, true, 0, 3);
        expectSyntaxError("(\\2())", "", "_", 0, ErrorCode.InvalidBackReference);
        test("(\\1a)", "", "aa", 0, false);
        test("(\\1a|){2}", "", "aa", 0, true, 0, 0, 0, 0);
        test("(\\1a|)*", "", "aa", 0, true, 0, 0, 0, 0);
        test("(()b|\\2a){2}", "", "ba", 0, true, 0, 2, 1, 2, 0, 0);
        test("(a\\1)", "", "aa", 0, false);
        test("(a|b\\1){2}", "", "aba", 0, true, 0, 3, 1, 3);
        test("(a|(b\\1)){2}", "", "aba", 0, true, 0, 3, 1, 3, 1, 3);
        test("((a)|b\\1){2}", "", "aba", 0, true, 0, 3, 1, 3, 0, 1);
        test("((a|b\\1)){2}", "", "aba", 0, true, 0, 3, 1, 3, 1, 3);
        test("((a|b\\2)){2}", "", "aba", 0, true, 0, 3, 1, 3, 1, 3);
        test("((a)|b\\2){2}", "", "aba", 0, true, 0, 3, 1, 3, 0, 1);
        test("((a)|b\\2)*", "", "aba", 0, true, 0, 3, 1, 3, 0, 1);
        test("(a|b\\1*){2}", "", "abaaaa", 0, true, 0, 6, 1, 6);
        test("^(a|\\1b)+$", "", "ab", 0, false);
        test("(a)\\10", "", "aa0", 0, true, 0, 3, 0, 1);
        test("(a)\\100", "", "aa00", 0, true, 0, 4, 0, 1);
        test("a a", "x", "aa", 0, true, 0, 2);
        test("a\na", "x", "aa", 0, true, 0, 2);
        test("a\u0009a", "x", "aa", 0, false);
        test("a\u000ba", "x", "aa", 0, false);
        test("a\u000ca", "x", "aa", 0, false);
        test("a\ra", "x", "aa", 0, false);
        test("\\", "", "aa", 0, true, 0, 0);
        test("\\077", "", "?", 0, false);
        test("\\077", "", "077", 0, true, 0, 3);
        test("\\x61", "", "a", 0, false);
        test("\\x61", "", "x61", 0, true, 0, 3);
        test("\\u0061", "", "a", 0, false);
        test("\\u0061", "", "u0061", 0, true, 0, 5);
        test("(a*|b)*", "", "aaaaaabaaaaaaaaaaaaaabb", 0, true, 0, 6, 6, 6);
        test("a((b?)*)*", "", "ab", 0, true, 0, 2, 2, 2, 2, 2);
        test("(()|()|a)+b", "", "ab", 0, true, 0, 2, 1, 1, 1, 1, -1, -1);
        test("a(\\z|())", "", "a", 0, true, 0, 1, 1, 1, -1, -1);
        test("abracadabra$", "", "abracadabracadabra", 0, true, 7, 18);
        test("a...b", "", "abababbb", 0, true, 2, 7);
        test("XXXXXX", "", "..XXXXXX", 0, true, 2, 8);
        test("\\)", "", "()", 0, true, 1, 2);
        test("a]", "", "a]a", 0, true, 0, 2);
        test("}", "", "}", 0, true, 0, 1);
        test("\\}", "", "}", 0, true, 0, 1);
        test("\\]", "", "]", 0, true, 0, 1);
        test("]", "", "]", 0, true, 0, 1);
        test("]", "", "]", 0, true, 0, 1);
        test("{", "", "{", 0, true, 0, 1);
        test("}", "", "}", 0, true, 0, 1);
        test("^a", "", "ax", 0, true, 0, 1);
        test("\\^a", "", "a^a", 0, true, 1, 3);
        test("a\\^", "", "a^", 0, true, 0, 2);
        test("a$", "", "aa", 0, true, 1, 2);
        test("a\\$", "", "a$", 0, true, 0, 2);
        test("a($)", "", "aa", 0, true, 1, 2, 2, 2);
        test("a*(^a)", "", "aa", 0, true, 0, 1, 0, 1);
        test("(..)*(...)*", "", "a", 0, true, 0, 0, -1, -1, -1, -1);
        test("(..)*(...)*", "", "abcd", 0, true, 0, 4, 2, 4, -1, -1);
        test("(ab|a)(bc|c)", "", "abc", 0, true, 0, 3, 0, 2, 2, 3);
        test("(ab)c|abc", "", "abc", 0, true, 0, 3, 0, 2);
        test("a{0}b", "", "ab", 0, true, 1, 2);
        test("(a*)(b?)(b+)b{3}", "", "aaabbbbbbb", 0, true, 0, 10, 0, 3, 3, 4, 4, 7);
        test("(a*)(b{0,1})(b{1,})b{3}", "", "aaabbbbbbb", 0, true, 0, 10, 0, 3, 3, 4, 4, 7);
        expectSyntaxError("a{9876543210}", "", "a", 0, ErrorCode.InvalidQuantifier);
        test("((a|a)|a)", "", "a", 0, true, 0, 1, 0, 1, 0, 1);
        test("(a*)(a|aa)", "", "aaaa", 0, true, 0, 4, 0, 3, 3, 4);
        test("a*(a.|aa)", "", "aaaa", 0, true, 0, 4, 2, 4);
        test("a(b)|c(d)|a(e)f", "", "aef", 0, true, 0, 3, -1, -1, -1, -1, 1, 2);
        test("(a|b)?.*", "", "b", 0, true, 0, 1, 0, 1);
        test("(a|b)c|a(b|c)", "", "ac", 0, true, 0, 2, 0, 1, -1, -1);
        test("(a|b)c|a(b|c)", "", "ab", 0, true, 0, 2, -1, -1, 1, 2);
        test("(a|b)*c|(a|ab)*c", "", "abc", 0, true, 0, 3, 1, 2, -1, -1);
        test("(a|b)*c|(a|ab)*c", "", "xc", 0, true, 1, 2, -1, -1, -1, -1);
        test("(.a|.b).*|.*(.a|.b)", "", "xa", 0, true, 0, 2, 0, 2, -1, -1);
        test("a?(ab|ba)ab", "", "abab", 0, true, 0, 4, 0, 2);
        test("a?(ac{0}b|ba)ab", "", "abab", 0, true, 0, 4, 0, 2);
        test("ab|abab", "", "abbabab", 0, true, 0, 2);
        test("aba|bab|bba", "", "baaabbbaba", 0, true, 5, 8);
        test("aba|bab", "", "baaabbbaba", 0, true, 6, 9);
        test("(aa|aaa)*|(a|aaaaa)", "", "aa", 0, true, 0, 2, 0, 2, -1, -1);
        test("(a.|.a.)*|(a|.a...)", "", "aa", 0, true, 0, 2, 0, 2, -1, -1);
        test("ab|a", "", "xabc", 0, true, 1, 3);
        test("ab|a", "", "xxabc", 0, true, 2, 4);
        test("(Ab|cD)*", "", "aBcD", 0, true, 0, 0, -1, -1);
        test("[^-]", "", "--a", 0, true, 2, 3);
        test("[a-]*", "", "--a", 0, true, 0, 3);
        test("[a-m-]*", "", "--amoma--", 0, true, 0, 4);
        test(":::1:::0:|:::1:1:0:", "", ":::0:::1:::1:::0:", 0, true, 8, 17);
        test(":::1:::0:|:::1:1:1:", "", ":::0:::1:::1:::0:", 0, true, 8, 17);
        test("[[:upper:]]", "", "A", 0, true, 0, 1);
        test("[[:lower:]]+", "", "`az{", 0, true, 1, 3);
        test("[[:upper:]]+", "", "@AZ[", 0, true, 1, 3);
        test("[[-]]", "", "[[-]]", 0, true, 2, 4);
        test("\\n", "", "\\n", 0, true, 1, 2);
        test("\\n", "", "\\n", 0, true, 1, 2);
        test("[^a]", "", "\\n", 0, true, 0, 1);
        test("\\na", "", "\\na", 0, true, 1, 3);
        test("(a)(b)(c)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("xxx", "", "xxx", 0, true, 0, 3);
        test("(^|[ (,;])((([Ff]eb[^ ]* *|0*2/|\\* */?)0*[6-7]))([^0-9]|$)", "", "feb 6,", 0, true, 0, 6, 0, 0, 0, 5, 0, 5, 0, 4, 5, 6);
        test("(^|[ (,;])((([Ff]eb[^ ]* *|0*2/|\\* */?)0*[6-7]))([^0-9]|$)", "", "2/7", 0, true, 0, 3, 0, 0, 0, 3, 0, 3, 0, 2, 3, 3);
        test("(^|[ (,;])((([Ff]eb[^ ]* *|0*2/|\\* */?)0*[6-7]))([^0-9]|$)", "", "feb 1,Feb 6", 0, true, 5, 11, 5, 6, 6, 11, 6, 11, 6, 10, 11, 11);
        test("((((((((((((((((((((((((((((((x))))))))))))))))))))))))))))))", "", "x", 0, true, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1);
        test("((((((((((((((((((((((((((((((x))))))))))))))))))))))))))))))*", "", "xx", 0, true, 0, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2);
        test("a?(ab|ba)*", "", "ababababababababababababababababababababababababababababababababababababababababa", 0, true, 0, 81, 79, 81);
        test("abaa|abbaa|abbbaa|abbbbaa", "", "ababbabbbabbbabbbbabbbbaa", 0, true, 18, 25);
        test("abaa|abbaa|abbbaa|abbbbaa", "", "ababbabbbabbbabbbbabaa", 0, true, 18, 22);
        test("aaac|aabc|abac|abbc|baac|babc|bbac|bbbc", "", "baaabbbabac", 0, true, 7, 11);
        test(".*", "", "\\x01\\xff", 0, true, 0, 8);
        test("aaaa|bbbb|cccc|ddddd|eeeeee|fffffff|gggg|hhhh|iiiii|jjjjj|kkkkk|llll", "", "XaaaXbbbXcccXdddXeeeXfffXgggXhhhXiiiXjjjXkkkXlllXcbaXaaaa", 0, true, 53, 57);
        test("aaaa\\nbbbb\\ncccc\\nddddd\\neeeeee\\nfffffff\\ngggg\\nhhhh\\niiiii\\njjjjj\\nkkkkk\\nllll", "", "XaaaXbbbXcccXdddXeeeXfffXgggXhhhXiiiXjjjXkkkXlllXcbaXaaaa", 0, false);
        test("a*a*a*a*a*b", "", "aaaaaaaaab", 0, true, 0, 10);
        test("^", "", "a", 0, true, 0, 0);
        test("$", "", "a", 0, true, 1, 1);
        test("^$", "", "a", 0, false);
        test("^a$", "", "a", 0, true, 0, 1);
        test("abc", "", "abc", 0, true, 0, 3);
        test("abc", "", "xabcy", 0, true, 1, 4);
        test("abc", "", "ababc", 0, true, 2, 5);
        test("ab*c", "", "abc", 0, true, 0, 3);
        test("ab*bc", "", "abc", 0, true, 0, 3);
        test("ab*bc", "", "abbc", 0, true, 0, 4);
        test("ab*bc", "", "abbbbc", 0, true, 0, 6);
        test("ab+bc", "", "abbc", 0, true, 0, 4);
        test("ab+bc", "", "abbbbc", 0, true, 0, 6);
        test("ab?bc", "", "abbc", 0, true, 0, 4);
        test("ab?bc", "", "abc", 0, true, 0, 3);
        test("ab?c", "", "abc", 0, true, 0, 3);
        test("^abc$", "", "abc", 0, true, 0, 3);
        test("^abc", "", "abcc", 0, true, 0, 3);
        test("abc$", "", "aabc", 0, true, 1, 4);
        test("^", "", "abc", 0, true, 0, 0);
        test("$", "", "abc", 0, true, 3, 3);
        test("a.c", "", "abc", 0, true, 0, 3);
        test("a.c", "", "axc", 0, true, 0, 3);
        test("a.*c", "", "axyzc", 0, true, 0, 5);
        test("a[bc]d", "", "abd", 0, true, 0, 3);
        test("a[b-d]e", "", "ace", 0, true, 0, 3);
        test("a[b-d]", "", "aac", 0, true, 1, 3);
        test("a[-b]", "", "a-", 0, true, 0, 2);
        test("a[b-]", "", "a-", 0, true, 0, 2);
        test("a]", "", "a]", 0, true, 0, 2);
        test("a[]]b", "", "a]b", 0, true, 0, 3);
        test("a[^bc]d", "", "aed", 0, true, 0, 3);
        test("a[^-b]c", "", "adc", 0, true, 0, 3);
        test("a[^]b]c", "", "adc", 0, true, 0, 3);
        test("ab|cd", "", "abc", 0, true, 0, 2);
        test("ab|cd", "", "abcd", 0, true, 0, 2);
        test("a\\(b", "", "a(b", 0, true, 0, 3);
        test("a\\(*b", "", "ab", 0, true, 0, 2);
        test("a\\(*b", "", "a((b", 0, true, 0, 4);
        test("((a))", "", "abc", 0, true, 0, 1, 0, 1, 0, 1);
        test("(a)b(c)", "", "abc", 0, true, 0, 3, 0, 1, 2, 3);
        test("a+b+c", "", "aabbabc", 0, true, 4, 7);
        test("a*", "", "aaa", 0, true, 0, 3);
        test("(a*)*", "", "-", 0, true, 0, 0, 0, 0);
        test("(a*)+", "", "-", 0, true, 0, 0, 0, 0);
        test("(a*|b)*", "", "-", 0, true, 0, 0, 0, 0);
        test("(a+|b)*", "", "ab", 0, true, 0, 2, 1, 2);
        test("(a+|b)+", "", "ab", 0, true, 0, 2, 1, 2);
        test("(a+|b)?", "", "ab", 0, true, 0, 1, 0, 1);
        test("[^ab]*", "", "cde", 0, true, 0, 3);
        test("(^)*", "", "-", 0, true, 0, 0, 0, 0);
        test("a*", "", "a", 0, true, 0, 1);
        test("([abc])*d", "", "abbbcd", 0, true, 0, 6, 4, 5);
        test("([abc])*bcd", "", "abcd", 0, true, 0, 4, 0, 1);
        test("a|b|c|d|e", "", "e", 0, true, 0, 1);
        test("(a|b|c|d|e)f", "", "ef", 0, true, 0, 2, 0, 1);
        test("((a*|b))*", "", "-", 0, true, 0, 0, 0, 0, 0, 0);
        test("abcd*efg", "", "abcdefg", 0, true, 0, 7);
        test("ab*", "", "xabyabbbz", 0, true, 1, 3);
        test("ab*", "", "xayabbbz", 0, true, 1, 2);
        test("(ab|cd)e", "", "abcde", 0, true, 2, 5, 2, 4);
        test("[abhgefdc]ij", "", "hij", 0, true, 0, 3);
        test("(a|b)c*d", "", "abcd", 0, true, 1, 4, 1, 2);
        test("(ab|ab*)bc", "", "abc", 0, true, 0, 3, 0, 1);
        test("a([bc]*)c*", "", "abc", 0, true, 0, 3, 1, 3);
        test("a([bc]*)(c*d)", "", "abcd", 0, true, 0, 4, 1, 3, 3, 4);
        test("a([bc]+)(c*d)", "", "abcd", 0, true, 0, 4, 1, 3, 3, 4);
        test("a([bc]*)(c+d)", "", "abcd", 0, true, 0, 4, 1, 2, 2, 4);
        test("a[bcd]*dcdcde", "", "adcdcde", 0, true, 0, 7);
        test("(ab|a)b*c", "", "abc", 0, true, 0, 3, 0, 2);
        test("((a)(b)c)(d)", "", "abcd", 0, true, 0, 4, 0, 3, 0, 1, 1, 2, 3, 4);
        test("[A-Za-z_][A-Za-z0-9_]*", "", "alpha", 0, true, 0, 5);
        test("^a(bc+|b[eh])g|.h$", "", "abh", 0, true, 1, 3, -1, -1);
        test("(bc+d$|ef*g.|h?i(j|k))", "", "effgz", 0, true, 0, 5, 0, 5, -1, -1);
        test("(bc+d$|ef*g.|h?i(j|k))", "", "ij", 0, true, 0, 2, 0, 2, 1, 2);
        test("(bc+d$|ef*g.|h?i(j|k))", "", "reffgz", 0, true, 1, 6, 1, 6, -1, -1);
        test("(((((((((a)))))))))", "", "a", 0, true, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1);
        test("multiple words", "", "multiple words yeah", 0, true, 0, 14);
        test("(.*)c(.*)", "", "abcde", 0, true, 0, 5, 0, 2, 3, 5);
        test("abcd", "", "abcd", 0, true, 0, 4);
        test("a(bc)d", "", "abcd", 0, true, 0, 4, 1, 3);
        test("a[\u0001-\u0003]?c", "", "a\u0002c", 0, true, 0, 3);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Muammar Qaddafi", 0, true, 0, 15, -1, -1, 10, 12);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Mo'ammar Gadhafi", 0, true, 0, 16, -1, -1, 11, 13);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Muammar Kaddafi", 0, true, 0, 15, -1, -1, 10, 12);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Muammar Qadhafi", 0, true, 0, 15, -1, -1, 10, 12);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Muammar Gadafi", 0, true, 0, 14, -1, -1, 10, 11);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Mu'ammar Qadafi", 0, true, 0, 15, -1, -1, 11, 12);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Moamar Gaddafi", 0, true, 0, 14, -1, -1, 9, 11);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Mu'ammar Qadhdhafi", 0, true, 0, 18, -1, -1, 13, 15);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Muammar Khaddafi", 0, true, 0, 16, -1, -1, 11, 13);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Muammar Ghaddafy", 0, true, 0, 16, -1, -1, 11, 13);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Muammar Ghadafi", 0, true, 0, 15, -1, -1, 11, 12);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Muammar Ghaddafi", 0, true, 0, 16, -1, -1, 11, 13);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Muamar Kaddafi", 0, true, 0, 14, -1, -1, 9, 11);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Muammar Quathafi", 0, true, 0, 16, -1, -1, 11, 13);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Muammar Gheddafi", 0, true, 0, 16, -1, -1, 11, 13);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Moammar Khadafy", 0, true, 0, 15, -1, -1, 11, 12);
        test("M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]", "", "Moammar Qudhafi", 0, true, 0, 15, -1, -1, 10, 12);
        test("a+(b|c)*d+", "", "aabcdd", 0, true, 0, 6, 3, 4);
        test("^.+$", "", "vivi", 0, true, 0, 4);
        test("^(.+)$", "", "vivi", 0, true, 0, 4, 0, 4);
        test("^([^!.]+).att.com!(.+)$", "", "gryphon.att.com!eby", 0, true, 0, 19, 0, 7, 16, 19);
        test("^([^!]+!)?([^!]+)$", "", "bas", 0, true, 0, 3, -1, -1, 0, 3);
        test("^([^!]+!)?([^!]+)$", "", "bar!bas", 0, true, 0, 7, 0, 4, 4, 7);
        test("^([^!]+!)?([^!]+)$", "", "foo!bas", 0, true, 0, 7, 0, 4, 4, 7);
        test("^.+!([^!]+!)([^!]+)$", "", "foo!bar!bas", 0, true, 0, 11, 4, 8, 8, 11);
        test("((foo)|(bar))!bas", "", "bar!bas", 0, true, 0, 7, 0, 3, -1, -1, 0, 3);
        test("((foo)|(bar))!bas", "", "foo!bar!bas", 0, true, 4, 11, 4, 7, -1, -1, 4, 7);
        test("((foo)|(bar))!bas", "", "foo!bas", 0, true, 0, 7, 0, 3, 0, 3, -1, -1);
        test("((foo)|bar)!bas", "", "bar!bas", 0, true, 0, 7, 0, 3, -1, -1);
        test("((foo)|bar)!bas", "", "foo!bar!bas", 0, true, 4, 11, 4, 7, -1, -1);
        test("((foo)|bar)!bas", "", "foo!bas", 0, true, 0, 7, 0, 3, 0, 3);
        test("(foo|(bar))!bas", "", "bar!bas", 0, true, 0, 7, 0, 3, 0, 3);
        test("(foo|(bar))!bas", "", "foo!bar!bas", 0, true, 4, 11, 4, 7, 4, 7);
        test("(foo|(bar))!bas", "", "foo!bas", 0, true, 0, 7, 0, 3, -1, -1);
        test("(foo|bar)!bas", "", "bar!bas", 0, true, 0, 7, 0, 3);
        test("(foo|bar)!bas", "", "foo!bar!bas", 0, true, 4, 11, 4, 7);
        test("(foo|bar)!bas", "", "foo!bas", 0, true, 0, 7, 0, 3);
        test("^(([^!]+!)?([^!]+)|.+!([^!]+!)([^!]+))$", "", "foo!bar!bas", 0, true, 0, 11, 0, 11, -1, -1, -1, -1, 4, 8, 8, 11);
        test("^([^!]+!)?([^!]+)$|^.+!([^!]+!)([^!]+)$", "", "bas", 0, true, 0, 3, -1, -1, 0, 3, -1, -1, -1, -1);
        test("^([^!]+!)?([^!]+)$|^.+!([^!]+!)([^!]+)$", "", "bar!bas", 0, true, 0, 7, 0, 4, 4, 7, -1, -1, -1, -1);
        test("^([^!]+!)?([^!]+)$|^.+!([^!]+!)([^!]+)$", "", "foo!bar!bas", 0, true, 0, 11, -1, -1, -1, -1, 4, 8, 8, 11);
        test("^([^!]+!)?([^!]+)$|^.+!([^!]+!)([^!]+)$", "", "foo!bas", 0, true, 0, 7, 0, 4, 4, 7, -1, -1, -1, -1);
        test("^(([^!]+!)?([^!]+)|.+!([^!]+!)([^!]+))$", "", "bas", 0, true, 0, 3, 0, 3, -1, -1, 0, 3, -1, -1, -1, -1);
        test("^(([^!]+!)?([^!]+)|.+!([^!]+!)([^!]+))$", "", "bar!bas", 0, true, 0, 7, 0, 7, 0, 4, 4, 7, -1, -1, -1, -1);
        test("^(([^!]+!)?([^!]+)|.+!([^!]+!)([^!]+))$", "", "foo!bar!bas", 0, true, 0, 11, 0, 11, -1, -1, -1, -1, 4, 8, 8, 11);
        test("^(([^!]+!)?([^!]+)|.+!([^!]+!)([^!]+))$", "", "foo!bas", 0, true, 0, 7, 0, 7, 0, 4, 4, 7, -1, -1, -1, -1);
        test(".*(/XXX).*", "", "/XXX", 0, true, 0, 4, 0, 4);
        test(".*(\\\\XXX).*", "", "\\XXX", 0, true, 0, 4, 0, 4);
        test("\\\\XXX", "", "\\XXX", 0, true, 0, 4);
        test(".*(/000).*", "", "/000", 0, true, 0, 4, 0, 4);
        test(".*(\\\\000).*", "", "\\000", 0, true, 0, 4, 0, 4);
        test("\\\\000", "", "\\000", 0, true, 0, 4);
        test("aa*", "", "xaxaax", 0, true, 1, 2);
        test("(a*)(ab)*(b*)", "", "abc", 0, true, 0, 2, 0, 1, -1, -1, 1, 2);
        test("(a*)(ab)*(b*)", "", "abc", 0, true, 0, 2, 0, 1, -1, -1, 1, 2);
        test("((a*)(ab)*)((b*)(a*))", "", "aba", 0, true, 0, 3, 0, 1, 0, 1, -1, -1, 1, 3, 1, 2, 2, 3);
        test("((a*)(ab)*)((b*)(a*))", "", "aba", 0, true, 0, 3, 0, 1, 0, 1, -1, -1, 1, 3, 1, 2, 2, 3);
        test("(...?.?)*", "", "xxxxxx", 0, true, 0, 6, 4, 6);
        test("(...?.?)*", "", "xxxxxx", 0, true, 0, 6, 4, 6);
        test("(...?.?)*", "", "xxxxxx", 0, true, 0, 6, 4, 6);
        test("(a|ab)(bc|c)", "", "abcabc", 0, true, 0, 3, 0, 1, 1, 3);
        test("(a|ab)(bc|c)", "", "abcabc", 0, true, 0, 3, 0, 1, 1, 3);
        test("(aba|a*b)(aba|a*b)", "", "ababa", 0, true, 0, 4, 0, 3, 3, 4);
        test("(aba|a*b)(aba|a*b)", "", "ababa", 0, true, 0, 4, 0, 3, 3, 4);
        test("a(b)*\\1", "", "a", 0, false);
        test("a(b)*\\1", "", "a", 0, false);
        test("a(b)*\\1", "", "abab", 0, false);
        test("(a*){2}", "", "xxxxx", 0, true, 0, 0, 0, 0);
        test("(a*){2}", "", "xxxxx", 0, true, 0, 0, 0, 0);
        test("a(b)*\\1", "", "abab", 0, false);
        test("a(b)*\\1", "", "abab", 0, false);
        test("a(b)*\\1", "", "abab", 0, false);
        test("(a*)*", "", "a", 0, true, 0, 1, 1, 1);
        test("(a*)*", "", "ax", 0, true, 0, 1, 1, 1);
        test("(a*)*", "", "a", 0, true, 0, 1, 1, 1);
        test("(aba|a*b)*", "", "ababa", 0, true, 0, 4, 3, 4);
        test("(aba|a*b)*", "", "ababa", 0, true, 0, 4, 3, 4);
        test("(aba|a*b)*", "", "ababa", 0, true, 0, 4, 3, 4);
        test("(a(b)?)+", "", "aba", 0, true, 0, 3, 2, 3, 1, 2);
        test("(a(b)?)+", "", "aba", 0, true, 0, 3, 2, 3, 1, 2);
        test("(a(b)*)*\\2", "", "abab", 0, true, 0, 4, 2, 3, 1, 2);
        test("(a(b)*)*\\2", "", "abab", 0, true, 0, 4, 2, 3, 1, 2);
        test("(a?)((ab)?)(b?)a?(ab)?b?", "", "abab", 0, true, 0, 4, 0, 1, 1, 1, -1, -1, 1, 2, -1, -1);
        test(".*(.*)", "", "ab", 0, true, 0, 2, 2, 2);
        test(".*(.*)", "", "ab", 0, true, 0, 2, 2, 2);
        test("(a|ab)(c|bcd)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4);
        test("(a|ab)(bcd|c)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4);
        test("(ab|a)(c|bcd)", "", "abcd", 0, true, 0, 3, 0, 2, 2, 3);
        test("(ab|a)(bcd|c)", "", "abcd", 0, true, 0, 3, 0, 2, 2, 3);
        test("((a|ab)(c|bcd))(d*)", "", "abcd", 0, true, 0, 4, 0, 4, 0, 1, 1, 4, 4, 4);
        test("((a|ab)(bcd|c))(d*)", "", "abcd", 0, true, 0, 4, 0, 4, 0, 1, 1, 4, 4, 4);
        test("((ab|a)(c|bcd))(d*)", "", "abcd", 0, true, 0, 4, 0, 3, 0, 2, 2, 3, 3, 4);
        test("((ab|a)(bcd|c))(d*)", "", "abcd", 0, true, 0, 4, 0, 3, 0, 2, 2, 3, 3, 4);
        test("(a|ab)((c|bcd)(d*))", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 1, 4, 4, 4);
        test("(a|ab)((bcd|c)(d*))", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 1, 4, 4, 4);
        test("(ab|a)((c|bcd)(d*))", "", "abcd", 0, true, 0, 4, 0, 2, 2, 4, 2, 3, 3, 4);
        test("(ab|a)((bcd|c)(d*))", "", "abcd", 0, true, 0, 4, 0, 2, 2, 4, 2, 3, 3, 4);
        test("(a*)(b|abc)", "", "abc", 0, true, 0, 2, 0, 1, 1, 2);
        test("(a*)(abc|b)", "", "abc", 0, true, 0, 2, 0, 1, 1, 2);
        test("((a*)(b|abc))(c*)", "", "abc", 0, true, 0, 3, 0, 2, 0, 1, 1, 2, 2, 3);
        test("((a*)(abc|b))(c*)", "", "abc", 0, true, 0, 3, 0, 2, 0, 1, 1, 2, 2, 3);
        test("(a*)((b|abc)(c*))", "", "abc", 0, true, 0, 3, 0, 1, 1, 3, 1, 2, 2, 3);
        test("(a*)((abc|b)(c*))", "", "abc", 0, true, 0, 3, 0, 1, 1, 3, 1, 2, 2, 3);
        test("(a*)(b|abc)", "", "abc", 0, true, 0, 2, 0, 1, 1, 2);
        test("(a*)(abc|b)", "", "abc", 0, true, 0, 2, 0, 1, 1, 2);
        test("((a*)(b|abc))(c*)", "", "abc", 0, true, 0, 3, 0, 2, 0, 1, 1, 2, 2, 3);
        test("((a*)(abc|b))(c*)", "", "abc", 0, true, 0, 3, 0, 2, 0, 1, 1, 2, 2, 3);
        test("(a*)((b|abc)(c*))", "", "abc", 0, true, 0, 3, 0, 1, 1, 3, 1, 2, 2, 3);
        test("(a*)((abc|b)(c*))", "", "abc", 0, true, 0, 3, 0, 1, 1, 3, 1, 2, 2, 3);
        test("(a|ab)", "", "ab", 0, true, 0, 1, 0, 1);
        test("(ab|a)", "", "ab", 0, true, 0, 2, 0, 2);
        test("(a|ab)(b*)", "", "ab", 0, true, 0, 2, 0, 1, 1, 2);
        test("(ab|a)(b*)", "", "ab", 0, true, 0, 2, 0, 2, 2, 2);
        test("a+", "", "xaax", 0, true, 1, 3);
        test(".(a*).", "", "xaax", 0, true, 0, 4, 1, 3);
        test("(a?)((ab)?)", "", "ab", 0, true, 0, 1, 0, 1, 1, 1, -1, -1);
        test("(a?)((ab)?)(b?)", "", "ab", 0, true, 0, 2, 0, 1, 1, 1, -1, -1, 1, 2);
        test("((a?)((ab)?))(b?)", "", "ab", 0, true, 0, 2, 0, 1, 0, 1, 1, 1, -1, -1, 1, 2);
        test("(a?)(((ab)?)(b?))", "", "ab", 0, true, 0, 2, 0, 1, 1, 2, 1, 1, -1, -1, 1, 2);
        test("(.?)", "", "x", 0, true, 0, 1, 0, 1);
        test("(.?){1}", "", "x", 0, true, 0, 1, 0, 1);
        test("(.?)(.?)", "", "x", 0, true, 0, 1, 0, 1, 1, 1);
        test("(.?){2}", "", "x", 0, true, 0, 1, 1, 1);
        test("(.?)*", "", "x", 0, true, 0, 1, 1, 1);
        test("(.?.?)", "", "xxx", 0, true, 0, 2, 0, 2);
        test("(.?.?){1}", "", "xxx", 0, true, 0, 2, 0, 2);
        test("(.?.?)(.?.?)", "", "xxx", 0, true, 0, 3, 0, 2, 2, 3);
        test("(.?.?){2}", "", "xxx", 0, true, 0, 3, 2, 3);
        test("(.?.?)(.?.?)(.?.?)", "", "xxx", 0, true, 0, 3, 0, 2, 2, 3, 3, 3);
        test("(.?.?){3}", "", "xxx", 0, true, 0, 3, 3, 3);
        test("(.?.?)*", "", "xxx", 0, true, 0, 3, 3, 3);
        test("a?((ab)?)(b?)", "", "ab", 0, true, 0, 2, 1, 1, -1, -1, 1, 2);
        test("(a?)((ab)?)b?", "", "ab", 0, true, 0, 2, 0, 1, 1, 1, -1, -1);
        test("a?((ab)?)b?", "", "ab", 0, true, 0, 2, 1, 1, -1, -1);
        test("(a*){2}", "", "xxxxx", 0, true, 0, 0, 0, 0);
        test("(ab?)(b?a)", "", "aba", 0, true, 0, 3, 0, 2, 2, 3);
        test("(a|ab)(ba|a)", "", "aba", 0, true, 0, 3, 0, 1, 1, 3);
        test("(a|ab|ba)", "", "aba", 0, true, 0, 1, 0, 1);
        test("(a|ab|ba)(a|ab|ba)", "", "aba", 0, true, 0, 3, 0, 1, 1, 3);
        test("(a|ab|ba)*", "", "aba", 0, true, 0, 3, 1, 3);
        test("(aba|a*b)", "", "ababa", 0, true, 0, 3, 0, 3);
        test("(aba|a*b)(aba|a*b)", "", "ababa", 0, true, 0, 4, 0, 3, 3, 4);
        test("(aba|a*b)*", "", "ababa", 0, true, 0, 4, 3, 4);
        test("(aba|ab|a)", "", "ababa", 0, true, 0, 3, 0, 3);
        test("(aba|ab|a)(aba|ab|a)", "", "ababa", 0, true, 0, 5, 0, 2, 2, 5);
        test("(aba|ab|a)*", "", "ababa", 0, true, 0, 3, 0, 3);
        test("(a(b)?)", "", "aba", 0, true, 0, 2, 0, 2, 1, 2);
        test("(a(b)?)(a(b)?)", "", "aba", 0, true, 0, 3, 0, 2, 1, 2, 2, 3, -1, -1);
        test("(a(b)?)+", "", "aba", 0, true, 0, 3, 2, 3, 1, 2);
        test("(.*)(.*)", "", "xx", 0, true, 0, 2, 0, 2, 2, 2);
        test(".*(.*)", "", "xx", 0, true, 0, 2, 2, 2);
        test("(a.*z|b.*y)", "", "azbazby", 0, true, 0, 5, 0, 5);
        test("(a.*z|b.*y)(a.*z|b.*y)", "", "azbazby", 0, true, 0, 7, 0, 5, 5, 7);
        test("(a.*z|b.*y)*", "", "azbazby", 0, true, 0, 7, 5, 7);
        test("(.|..)(.*)", "", "ab", 0, true, 0, 2, 0, 1, 1, 2);
        test("((..)*(...)*)", "", "xxx", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)*(...)*)((..)*(...)*)", "", "xxx", 0, true, 0, 2, 0, 2, 0, 2, -1, -1, 2, 2, -1, -1, -1, -1);
        test("((..)*(...)*)*", "", "xxx", 0, true, 0, 2, 2, 2, 0, 2, -1, -1);
        test("(a{0,1})*b\\1", "", "ab", 0, true, 0, 2, 1, 1);
        test("(a*)*b\\1", "", "ab", 0, true, 0, 2, 1, 1);
        test("(a*)b\\1*", "", "ab", 0, true, 0, 2, 0, 1);
        test("(a*)*b\\1*", "", "ab", 0, true, 0, 2, 1, 1);
        test("(a{0,1})*b(\\1)", "", "ab", 0, true, 0, 2, 1, 1, 2, 2);
        test("(a*)*b(\\1)", "", "ab", 0, true, 0, 2, 1, 1, 2, 2);
        test("(a*)b(\\1)*", "", "ab", 0, true, 0, 2, 0, 1, -1, -1);
        test("(a*)*b(\\1)*", "", "ab", 0, true, 0, 2, 1, 1, 2, 2);
        test("(a{0,1})*b\\1", "", "aba", 0, true, 0, 2, 1, 1);
        test("(a*)*b\\1", "", "aba", 0, true, 0, 2, 1, 1);
        test("(a*)b\\1*", "", "aba", 0, true, 0, 3, 0, 1);
        test("(a*)*b\\1*", "", "aba", 0, true, 0, 2, 1, 1);
        test("(a*)*b(\\1)*", "", "aba", 0, true, 0, 2, 1, 1, 2, 2);
        test("(a{0,1})*b\\1", "", "abaa", 0, true, 0, 2, 1, 1);
        test("(a*)*b\\1", "", "abaa", 0, true, 0, 2, 1, 1);
        test("(a*)b\\1*", "", "abaa", 0, true, 0, 4, 0, 1);
        test("(a*)*b\\1*", "", "abaa", 0, true, 0, 2, 1, 1);
        test("(a*)*b(\\1)*", "", "abaa", 0, true, 0, 2, 1, 1, 2, 2);
        test("(a{0,1})*b\\1", "", "aab", 0, true, 0, 3, 2, 2);
        test("(a*)*b\\1", "", "aab", 0, true, 0, 3, 2, 2);
        test("(a*)b\\1*", "", "aab", 0, true, 0, 3, 0, 2);
        test("(a*)*b\\1*", "", "aab", 0, true, 0, 3, 2, 2);
        test("(a*)*b(\\1)*", "", "aab", 0, true, 0, 3, 2, 2, 3, 3);
        test("(a{0,1})*b\\1", "", "aaba", 0, true, 0, 3, 2, 2);
        test("(a*)*b\\1", "", "aaba", 0, true, 0, 3, 2, 2);
        test("(a*)b\\1*", "", "aaba", 0, true, 0, 3, 0, 2);
        test("(a*)*b\\1*", "", "aaba", 0, true, 0, 3, 2, 2);
        test("(a*)*b(\\1)*", "", "aaba", 0, true, 0, 3, 2, 2, 3, 3);
        test("(a{0,1})*b\\1", "", "aabaa", 0, true, 0, 3, 2, 2);
        test("(a*)*b\\1", "", "aabaa", 0, true, 0, 3, 2, 2);
        test("(a*)b\\1*", "", "aabaa", 0, true, 0, 5, 0, 2);
        test("(a*)*b\\1*", "", "aabaa", 0, true, 0, 3, 2, 2);
        test("(a*)*b(\\1)*", "", "aabaa", 0, true, 0, 3, 2, 2, 3, 3);
        test("(x)*a\\1", "", "a", 0, false);
        test("(x)*a\\1*", "", "a", 0, true, 0, 1, -1, -1);
        test("(x)*a(\\1)", "", "a", 0, false);
        test("(x)*a(\\1)*", "", "a", 0, true, 0, 1, -1, -1, -1, -1);
        test("(aa(b(b))?)+", "", "aabbaa", 0, true, 0, 6, 4, 6, 2, 4, 3, 4);
        test("(a(b)?)+", "", "aba", 0, true, 0, 3, 2, 3, 1, 2);
        test("([ab]+)([bc]+)([cd]*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("([ab]*)([bc]*)([cd]*)\\1", "", "abcdaa", 0, true, 0, 5, 0, 1, 1, 3, 3, 4);
        test("([ab]*)([bc]*)([cd]*)\\1", "", "abcdab", 0, true, 0, 6, 0, 2, 2, 3, 3, 4);
        test("([ab]*)([bc]*)([cd]*)\\1*", "", "abcdaa", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("([ab]*)([bc]*)([cd]*)\\1*", "", "abcdab", 0, true, 0, 6, 0, 2, 2, 3, 3, 4);
        test("^(A([^B]*))?(B(.*))?", "", "Aa", 0, true, 0, 2, 0, 2, 1, 2, -1, -1, -1, -1);
        test("^(A([^B]*))?(B(.*))?", "", "Bb", 0, true, 0, 2, -1, -1, -1, -1, 0, 2, 1, 2);
        test(".*([AB]).*\\1", "", "ABA", 0, true, 0, 3, 0, 1);
        test("[^A]*A", "", "\\nA", 0, true, 0, 3);
        test("(a|ab)(c|bcd)(d*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(a|ab)(bcd|c)(d*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(ab|a)(c|bcd)(d*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("(ab|a)(bcd|c)(d*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("(a*)(b|abc)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a*)(abc|b)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a*)(b|abc)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a*)(abc|b)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a|ab)(c|bcd)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(a|ab)(bcd|c)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(ab|a)(c|bcd)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("(ab|a)(bcd|c)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("(a*)*", "", "a", 0, true, 0, 1, 1, 1);
        test("(a*)*", "", "x", 0, true, 0, 0, 0, 0);
        test("(a*)*", "", "aaaaaa", 0, true, 0, 6, 6, 6);
        test("(a*)*", "", "aaaaaax", 0, true, 0, 6, 6, 6);
        test("(a*)+", "", "a", 0, true, 0, 1, 1, 1);
        test("(a*)+", "", "x", 0, true, 0, 0, 0, 0);
        test("(a*)+", "", "aaaaaa", 0, true, 0, 6, 6, 6);
        test("(a*)+", "", "aaaaaax", 0, true, 0, 6, 6, 6);
        test("(a+)*", "", "a", 0, true, 0, 1, 0, 1);
        test("(a+)*", "", "x", 0, true, 0, 0, -1, -1);
        test("(a+)*", "", "aaaaaa", 0, true, 0, 6, 0, 6);
        test("(a+)*", "", "aaaaaax", 0, true, 0, 6, 0, 6);
        test("(a+)+", "", "a", 0, true, 0, 1, 0, 1);
        test("(a+)+", "", "x", 0, false);
        test("(a+)+", "", "aaaaaa", 0, true, 0, 6, 0, 6);
        test("(a+)+", "", "aaaaaax", 0, true, 0, 6, 0, 6);
        test("([a]*)*", "", "a", 0, true, 0, 1, 1, 1);
        test("([a]*)*", "", "x", 0, true, 0, 0, 0, 0);
        test("([a]*)*", "", "aaaaaa", 0, true, 0, 6, 6, 6);
        test("([a]*)*", "", "aaaaaax", 0, true, 0, 6, 6, 6);
        test("([a]*)+", "", "a", 0, true, 0, 1, 1, 1);
        test("([a]*)+", "", "x", 0, true, 0, 0, 0, 0);
        test("([a]*)+", "", "aaaaaa", 0, true, 0, 6, 6, 6);
        test("([a]*)+", "", "aaaaaax", 0, true, 0, 6, 6, 6);
        test("([^b]*)*", "", "a", 0, true, 0, 1, 1, 1);
        test("([^b]*)*", "", "b", 0, true, 0, 0, 0, 0);
        test("([^b]*)*", "", "aaaaaa", 0, true, 0, 6, 6, 6);
        test("([^b]*)*", "", "aaaaaab", 0, true, 0, 6, 6, 6);
        test("([ab]*)*", "", "a", 0, true, 0, 1, 1, 1);
        test("([ab]*)*", "", "aaaaaa", 0, true, 0, 6, 6, 6);
        test("([ab]*)*", "", "ababab", 0, true, 0, 6, 6, 6);
        test("([ab]*)*", "", "bababa", 0, true, 0, 6, 6, 6);
        test("([ab]*)*", "", "b", 0, true, 0, 1, 1, 1);
        test("([ab]*)*", "", "bbbbbb", 0, true, 0, 6, 6, 6);
        test("([ab]*)*", "", "aaaabcde", 0, true, 0, 5, 5, 5);
        test("([^a]*)*", "", "b", 0, true, 0, 1, 1, 1);
        test("([^a]*)*", "", "bbbbbb", 0, true, 0, 6, 6, 6);
        test("([^a]*)*", "", "aaaaaa", 0, true, 0, 0, 0, 0);
        test("([^ab]*)*", "", "ccccxx", 0, true, 0, 6, 6, 6);
        test("([^ab]*)*", "", "ababab", 0, true, 0, 0, 0, 0);
        test("((z)+|a)*", "", "zabcde", 0, true, 0, 2, 1, 2, 0, 1);
        test("a+?", "", "aaaaaa", 0, true, 0, 1);
        test("(a)", "", "aaa", 0, true, 0, 1, 0, 1);
        test("(a*?)", "", "aaa", 0, true, 0, 0, 0, 0);
        test("(a)*?", "", "aaa", 0, true, 0, 0, -1, -1);
        test("(a*?)*?", "", "aaa", 0, true, 0, 0, -1, -1);
        test("(a*)*(x)", "", "x", 0, true, 0, 1, 0, 0, 0, 1);
        test("(a*)*(x)", "", "ax", 0, true, 0, 2, 1, 1, 1, 2);
        test("(a*)*(x)", "", "axa", 0, true, 0, 2, 1, 1, 1, 2);
        test("(a*)*(x)(\\1)", "", "x", 0, true, 0, 1, 0, 0, 0, 1, 1, 1);
        test("(a*)*(x)(\\1)", "", "ax", 0, true, 0, 2, 1, 1, 1, 2, 2, 2);
        test("(a*)*(x)(\\1)", "", "axa", 0, true, 0, 2, 1, 1, 1, 2, 2, 2);
        test("(a*)*(x)(\\1)(x)", "", "axax", 0, true, 0, 4, 0, 1, 1, 2, 2, 3, 3, 4);
        test("(a*)*(x)(\\1)(x)", "", "axxa", 0, true, 0, 3, 1, 1, 1, 2, 2, 2, 2, 3);
        test("(a*)*(x)", "", "x", 0, true, 0, 1, 0, 0, 0, 1);
        test("(a*)*(x)", "", "ax", 0, true, 0, 2, 1, 1, 1, 2);
        test("(a*)*(x)", "", "axa", 0, true, 0, 2, 1, 1, 1, 2);
        test("(a*)+(x)", "", "x", 0, true, 0, 1, 0, 0, 0, 1);
        test("(a*)+(x)", "", "ax", 0, true, 0, 2, 1, 1, 1, 2);
        test("(a*)+(x)", "", "axa", 0, true, 0, 2, 1, 1, 1, 2);
        test("(a*){2}(x)", "", "x", 0, true, 0, 1, 0, 0, 0, 1);
        test("(a*){2}(x)", "", "ax", 0, true, 0, 2, 1, 1, 1, 2);
        test("(a*){2}(x)", "", "axa", 0, true, 0, 2, 1, 1, 1, 2);
        test("((..)|(.))", "", "a", 0, true, 0, 1, 0, 1, -1, -1, 0, 1);
        test("((..)|(.))((..)|(.))", "", "a", 0, false);
        test("((..)|(.))((..)|(.))((..)|(.))", "", "a", 0, false);
        test("((..)|(.)){1}", "", "a", 0, true, 0, 1, 0, 1, -1, -1, 0, 1);
        test("((..)|(.)){2}", "", "a", 0, false);
        test("((..)|(.)){3}", "", "a", 0, false);
        test("((..)|(.))*", "", "a", 0, true, 0, 1, 0, 1, -1, -1, 0, 1);
        test("((..)|(.))", "", "aa", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)|(.))((..)|(.))", "", "aa", 0, true, 0, 2, 0, 1, -1, -1, 0, 1, 1, 2, -1, -1, 1, 2);
        test("((..)|(.))((..)|(.))((..)|(.))", "", "aa", 0, false);
        test("((..)|(.)){1}", "", "aa", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)|(.)){2}", "", "aa", 0, true, 0, 2, 1, 2, -1, -1, 1, 2);
        test("((..)|(.)){3}", "", "aa", 0, false);
        test("((..)|(.))*", "", "aa", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)|(.))", "", "aaa", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)|(.))((..)|(.))", "", "aaa", 0, true, 0, 3, 0, 2, 0, 2, -1, -1, 2, 3, -1, -1, 2, 3);
        test("((..)|(.))((..)|(.))((..)|(.))", "", "aaa", 0, true, 0, 3, 0, 1, -1, -1, 0, 1, 1, 2, -1, -1, 1, 2, 2, 3, -1, -1, 2, 3);
        test("((..)|(.)){1}", "", "aaa", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)|(.)){2}", "", "aaa", 0, true, 0, 3, 2, 3, 0, 2, 2, 3);
        test("((..)|(.)){3}", "", "aaa", 0, true, 0, 3, 2, 3, -1, -1, 2, 3);
        test("((..)|(.))*", "", "aaa", 0, true, 0, 3, 2, 3, 0, 2, 2, 3);
        test("((..)|(.))", "", "aaaa", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)|(.))((..)|(.))", "", "aaaa", 0, true, 0, 4, 0, 2, 0, 2, -1, -1, 2, 4, 2, 4, -1, -1);
        test("((..)|(.))((..)|(.))((..)|(.))", "", "aaaa", 0, true, 0, 4, 0, 2, 0, 2, -1, -1, 2, 3, -1, -1, 2, 3, 3, 4, -1, -1, 3, 4);
        test("((..)|(.)){1}", "", "aaaa", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)|(.)){2}", "", "aaaa", 0, true, 0, 4, 2, 4, 2, 4, -1, -1);
        test("((..)|(.)){3}", "", "aaaa", 0, true, 0, 4, 3, 4, 0, 2, 3, 4);
        test("((..)|(.))*", "", "aaaa", 0, true, 0, 4, 2, 4, 2, 4, -1, -1);
        test("((..)|(.))", "", "aaaaa", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)|(.))((..)|(.))", "", "aaaaa", 0, true, 0, 4, 0, 2, 0, 2, -1, -1, 2, 4, 2, 4, -1, -1);
        test("((..)|(.))((..)|(.))((..)|(.))", "", "aaaaa", 0, true, 0, 5, 0, 2, 0, 2, -1, -1, 2, 4, 2, 4, -1, -1, 4, 5, -1, -1, 4, 5);
        test("((..)|(.)){1}", "", "aaaaa", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)|(.)){2}", "", "aaaaa", 0, true, 0, 4, 2, 4, 2, 4, -1, -1);
        test("((..)|(.)){3}", "", "aaaaa", 0, true, 0, 5, 4, 5, 2, 4, 4, 5);
        test("((..)|(.))*", "", "aaaaa", 0, true, 0, 5, 4, 5, 2, 4, 4, 5);
        test("((..)|(.))", "", "aaaaaa", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)|(.))((..)|(.))", "", "aaaaaa", 0, true, 0, 4, 0, 2, 0, 2, -1, -1, 2, 4, 2, 4, -1, -1);
        test("((..)|(.))((..)|(.))((..)|(.))", "", "aaaaaa", 0, true, 0, 6, 0, 2, 0, 2, -1, -1, 2, 4, 2, 4, -1, -1, 4, 6, 4, 6, -1, -1);
        test("((..)|(.)){1}", "", "aaaaaa", 0, true, 0, 2, 0, 2, 0, 2, -1, -1);
        test("((..)|(.)){2}", "", "aaaaaa", 0, true, 0, 4, 2, 4, 2, 4, -1, -1);
        test("((..)|(.)){3}", "", "aaaaaa", 0, true, 0, 6, 4, 6, 4, 6, -1, -1);
        test("((..)|(.))*", "", "aaaaaa", 0, true, 0, 6, 4, 6, 4, 6, -1, -1);
        test("X(.?){0,}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){1,}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){2,}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){3,}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){4,}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){5,}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){6,}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){7,}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){8,}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){0,8}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){1,8}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){2,8}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){3,8}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){4,8}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){5,8}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){6,8}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){7,8}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("X(.?){8,8}Y", "", "X1234567Y", 0, true, 0, 9, 8, 8);
        test("(a|ab|c|bcd){0,}(d*)", "", "ababcd", 0, true, 0, 1, 0, 1, 1, 1);
        test("(a|ab|c|bcd){1,}(d*)", "", "ababcd", 0, true, 0, 1, 0, 1, 1, 1);
        test("(a|ab|c|bcd){2,}(d*)", "", "ababcd", 0, true, 0, 6, 3, 6, 6, 6);
        test("(a|ab|c|bcd){3,}(d*)", "", "ababcd", 0, true, 0, 6, 3, 6, 6, 6);
        test("(a|ab|c|bcd){4,}(d*)", "", "ababcd", 0, false);
        test("(a|ab|c|bcd){0,10}(d*)", "", "ababcd", 0, true, 0, 1, 0, 1, 1, 1);
        test("(a|ab|c|bcd){1,10}(d*)", "", "ababcd", 0, true, 0, 1, 0, 1, 1, 1);
        test("(a|ab|c|bcd){2,10}(d*)", "", "ababcd", 0, true, 0, 6, 3, 6, 6, 6);
        test("(a|ab|c|bcd){3,10}(d*)", "", "ababcd", 0, true, 0, 6, 3, 6, 6, 6);
        test("(a|ab|c|bcd){4,10}(d*)", "", "ababcd", 0, false);
        test("(a|ab|c|bcd)*(d*)", "", "ababcd", 0, true, 0, 1, 0, 1, 1, 1);
        test("(a|ab|c|bcd)+(d*)", "", "ababcd", 0, true, 0, 1, 0, 1, 1, 1);
        test("(ab|a|c|bcd){0,}(d*)", "", "ababcd", 0, true, 0, 6, 4, 5, 5, 6);
        test("(ab|a|c|bcd){1,}(d*)", "", "ababcd", 0, true, 0, 6, 4, 5, 5, 6);
        test("(ab|a|c|bcd){2,}(d*)", "", "ababcd", 0, true, 0, 6, 4, 5, 5, 6);
        test("(ab|a|c|bcd){3,}(d*)", "", "ababcd", 0, true, 0, 6, 4, 5, 5, 6);
        test("(ab|a|c|bcd){4,}(d*)", "", "ababcd", 0, false);
        test("(ab|a|c|bcd){0,10}(d*)", "", "ababcd", 0, true, 0, 6, 4, 5, 5, 6);
        test("(ab|a|c|bcd){1,10}(d*)", "", "ababcd", 0, true, 0, 6, 4, 5, 5, 6);
        test("(ab|a|c|bcd){2,10}(d*)", "", "ababcd", 0, true, 0, 6, 4, 5, 5, 6);
        test("(ab|a|c|bcd){3,10}(d*)", "", "ababcd", 0, true, 0, 6, 4, 5, 5, 6);
        test("(ab|a|c|bcd){4,10}(d*)", "", "ababcd", 0, false);
        test("(ab|a|c|bcd)*(d*)", "", "ababcd", 0, true, 0, 6, 4, 5, 5, 6);
        test("(ab|a|c|bcd)+(d*)", "", "ababcd", 0, true, 0, 6, 4, 5, 5, 6);
        test("(a|ab)(c|bcd)(d*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(a|ab)(bcd|c)(d*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(ab|a)(c|bcd)(d*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("(ab|a)(bcd|c)(d*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("(a*)(b|abc)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a*)(abc|b)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a*)(b|abc)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a*)(abc|b)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a|ab)(c|bcd)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(a|ab)(bcd|c)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(ab|a)(c|bcd)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("(ab|a)(bcd|c)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("(a|ab)(c|bcd)(d*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(a|ab)(bcd|c)(d*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(ab|a)(c|bcd)(d*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("(ab|a)(bcd|c)(d*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("(a*)(b|abc)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a*)(abc|b)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a*)(b|abc)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a*)(abc|b)(c*)", "", "abc", 0, true, 0, 3, 0, 1, 1, 2, 2, 3);
        test("(a|ab)(c|bcd)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(a|ab)(bcd|c)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 1, 1, 4, 4, 4);
        test("(ab|a)(c|bcd)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("(ab|a)(bcd|c)(d|.*)", "", "abcd", 0, true, 0, 4, 0, 2, 2, 3, 3, 4);
        test("\ufb00", "i", "FF", 0, true, 0, 2);
        test("(\ufb00)\\1", "i", "FFFF", 0, true, 0, 4, 0, 2);
        test("(\ufb00)\\1", "i", "FF\ufb00", 0, true, 0, 5, 0, 2);
        test("(\ufb00)\\1", "i", "\ufb00FF", 0, true, 0, 5, 0, 3);
        test("(F\ufb00F)\\1", "i", "\ufb00FF\ufb00FF", 0, true, 0, 10, 0, 5);
        test("(F\ufb00F)\\1", "i", "FFFFFFFF", 0, true, 0, 8, 0, 4);
        test("(F\ufb00F)\\1", "i", "\ufb00\ufb00\ufb00\ufb00", 0, true, 0, 12, 0, 6);
        test("(F\ufb00F)\\1", "i", "\ufb00\ufb00FFFF", 0, true, 0, 10, 0, 6);
        test("(F\ufb00F)\\1", "i", "FFFF\ufb00\ufb00", 0, true, 0, 10, 0, 4);
        test("(\ufb00\ufb00)\\1", "i", "F\ufb00F\ufb00\ufb00", 0, true, 0, 11, 0, 5);
        test("\ufb01", "i", "FI", 0, true, 0, 2);
        test("(\ufb01)\\1", "i", "FIFI", 0, true, 0, 4, 0, 2);
        test("\ufb02", "i", "FL", 0, true, 0, 2);
        test("\ufb03", "i", "FFI", 0, true, 0, 3);
        test("\ufb04", "i", "FFL", 0, true, 0, 3);
        test("\ufb00I", "i", "\ufb03", 0, true, 0, 3);
        test("\ufb03", "i", "\ufb00I", 0, true, 0, 4);
        test("F\ufb01", "i", "\ufb03", 0, true, 0, 3);
        test("\ufb03", "i", "F\ufb01", 0, true, 0, 4);
        test("\ufb00L", "i", "\ufb04", 0, true, 0, 3);
        test("\ufb04", "i", "\ufb00L", 0, true, 0, 4);
        test("F\ufb02", "i", "\ufb04", 0, true, 0, 3);
        test("\ufb04", "i", "F\ufb02", 0, true, 0, 4);
        test("[\ufb04[=a=]o]+", "i", "F\ufb02a\u00c4\u00f6", 0, true, 0, 7);
        test("\u1f50", "i", "\u03c5\u0313", 0, true, 0, 4);
        test("\u1f52", "i", "\u03c5\u0313\u0300", 0, true, 0, 6);
        test("\u1f54", "i", "\u03c5\u0313\u0301", 0, true, 0, 6);
        test("\u1f56", "i", "\u03c5\u0313\u0342", 0, true, 0, 6);
        test("\u1f50\u0300", "i", "\u1f52", 0, true, 0, 3);
        test("\u1f52", "i", "\u1f50\u0300", 0, true, 0, 5);
        test("\u1f50\u0301", "i", "\u1f54", 0, true, 0, 3);
        test("\u1f54", "i", "\u1f50\u0301", 0, true, 0, 5);
        test("\u1f50\u0342", "i", "\u1f56", 0, true, 0, 3);
        test("\u1f56", "i", "\u1f50\u0342", 0, true, 0, 5);
        test("\u1fb6", "i", "\u03b1\u0342", 0, true, 0, 4);
        test("\u1fb7", "i", "\u03b1\u0342\u03b9", 0, true, 0, 6);
        test("\u1fb6\u03b9", "i", "\u1fb7", 0, true, 0, 3);
        test("\u1fb7", "i", "\u1fb6\u03b9", 0, true, 0, 5);
        test("\u1fc6", "i", "\u03b7\u0342", 0, true, 0, 4);
        test("\u1fc7", "i", "\u03b7\u0342\u03b9", 0, true, 0, 6);
        test("\u1fc6\u03b9", "i", "\u1fc7", 0, true, 0, 3);
        test("\u1fc7", "i", "\u1fc6\u03b9", 0, true, 0, 5);
        test("\u1ff6", "i", "\u03c9\u0342", 0, true, 0, 4);
        test("\u1ff7", "i", "\u03c9\u0342\u03b9", 0, true, 0, 6);
        test("\u1ff6\u03b9", "i", "\u1ff7", 0, true, 0, 3);
        test("\u1ff7", "i", "\u1ff6\u03b9", 0, true, 0, 5);
        test("f*", "i", "ff", 0, true, 0, 2);
        test("f*", "i", "\ufb00", 0, true, 0, 0);
        test("f+", "i", "ff", 0, true, 0, 2);
        test("f+", "i", "\ufb00", 0, false);
        test("f{1,}", "i", "ff", 0, true, 0, 2);
        test("f{1,}", "i", "\ufb00", 0, false);
        test("f{1,2}", "i", "ff", 0, true, 0, 2);
        test("f{1,2}", "i", "\ufb00", 0, false);
        test("f{,2}", "i", "ff", 0, false);
        test("f{,2}", "i", "\ufb00", 0, false);
        test("ff?", "i", "ff", 0, true, 0, 2);
        test("ff?", "i", "\ufb00", 0, false);
        test("f{2}", "i", "ff", 0, true, 0, 2);
        test("f{2}", "i", "\ufb00", 0, false);
        test("f{2,2}", "i", "ff", 0, true, 0, 2);
        test("f{2,2}", "i", "\ufb00", 0, false);
        test("K", "i", "\u212a", 0, true, 0, 3);
        test("k", "i", "\u212a", 0, true, 0, 3);
        test("\\w", "i", "\u212a", 0, true, 0, 3);
        test("\\W", "i", "\u212a", 0, false);
        test("[\\w]", "i", "\u212a", 0, false);
        test("[\\w]+", "i", "a\\wWc", 0, true, 1, 4);
        test("[\\W]+", "i", "a\\wWc", 0, true, 1, 4);
        test("[\\d]+", "i", "0\\dD9", 0, true, 1, 4);
        test("[\\D]+", "i", "a\\dDc", 0, true, 1, 4);
        test("[\\s]+", "i", " \\sS\u0009", 0, true, 1, 4);
        test("[\\S]+", "i", " \\sS\u0009", 0, true, 1, 4);
        test("[kx]", "i", "\u212a", 0, true, 0, 3);
        test("ff", "i", "\ufb00", 0, true, 0, 3);
        test("[f]f", "i", "\ufb00", 0, false);
        test("f[f]", "i", "\ufb00", 0, false);
        test("[f][f]", "i", "\ufb00", 0, false);
        test("(?:f)f", "i", "\ufb00", 0, false);
        test("f(?:f)", "i", "\ufb00", 0, false);
        test("(?:f)(?:f)", "i", "\ufb00", 0, false);
        test("\\A[\ufb00]\\z", "i", "\ufb00", 0, true, 0, 3);
        test("\\A[\ufb00]\\z", "i", "ff", 0, true, 0, 2);
        test("\\A[^\ufb00]\\z", "i", "\ufb00", 0, false);
        test("\\A[^\ufb00]\\z", "i", "ff", 0, false);
        test("\\A[^[^\ufb00]]\\z", "i", "\ufb00", 0, false);
        test("\\A[^[^\ufb00]]\\z", "i", "ff", 0, false);
        test("\\A[[^[^\ufb00]]]\\z", "i", "\ufb00", 0, false);
        test("\\A[[^[^\ufb00]]]\\z", "i", "ff", 0, false);
        test("[^a-c]", "i", "A", 0, false);
        test("[[^a-c]]", "i", "A", 0, false);
        test("[^a]", "i", "a", 0, false);
        test("[[^a]]", "i", "a", 0, false);
        test("\\A\\W\\z", "i", "\ufb00", 0, false);
        test("\\A\\W\\z", "i", "ff", 0, false);
        test("\\A[\\p{L}]\\z", "i", "\ufb00", 0, false);
        test("\\A[\\p{L}]\\z", "i", "ff", 0, false);
        test("\\A\\W\\z", "i", "\ufb03", 0, false);
        test("\\A\\W\\z", "i", "ffi", 0, false);
        test("\\A\\W\\z", "i", "\ufb00i", 0, false);
        test("\\A[\\p{L}]\\z", "i", "\ufb03", 0, false);
        test("\\A[\\p{L}]\\z", "i", "ffi", 0, false);
        test("\\A[\\p{L}]\\z", "i", "\ufb00i", 0, false);
        test("([[=a=]])\\1", "i", "aA", 0, true, 0, 2, 0, 1);
        test("([[=a=]])\\1", "i", "Aa", 0, true, 0, 2, 0, 1);
        test("([[=a=]])\\1", "i", "a\u00e4", 0, false);
        test("([[=a=]])\\1", "i", "a\u00c4", 0, false);
        test("([[=a=]])\\1", "i", "\u00e4a", 0, false);
        test("([[=a=]])\\1", "i", "\u00c4a", 0, false);
        test("([[=a=]])\\1", "i", "\u00c4A", 0, false);
        test("[[=a=]o]+", "i", "\u00e4O\u00f6", 0, true, 0, 3);
        test("[[=a=]o]+", "i", "\u00e4O\u00f6", 0, true, 0, 3);
        test("[[=\u00df=]o]+", "i", "s", 0, false);
        test("[[=\u00df=]o]+", "i", "ss", 0, true, 0, 2);
        test("[[=\u00df=]o]+", "", "s", 0, false);
        test("[[=\u00df=]o]+", "", "ss", 0, true, 0, 2);
        test("[\u0132]+", "", "ij", 0, false);
        test("[\u0132]+", "i", "ij", 0, false);
        test("[[=\u0132=]]+", "", "ij", 0, true, 0, 2);
        test("[[=\u0132=]o]+", "", "ij", 0, true, 0, 2);
        test("[[=\u0132=]o]+", "i", "ij", 0, true, 0, 2);
        expectSyntaxError("[\\s-r]+", "", "\\stu", 0, ErrorCode.InvalidCharacterClass);
        test("[\\s-v]+", "", "\\stu", 0, true, 0, 4);
        test("$(\\A|)", "", "x", 0, true, 1, 1, 1, 1);
        test("(^\\w)|()^", "", "empty", 0, true, 0, 1, 0, 1, -1, -1);
        test("a(b|())", "", "ab", 0, true, 0, 2, 1, 2, -1, -1);
        test("(a|())*", "", "aaa", 0, true, 0, 3, 3, 3, 3, 3);
        test("a|", "", "a", 0, true, 0, 1);
        test("a|", "", "b", 0, true, 0, 0);
        test("a|", "", "a|", 0, true, 0, 1);
        test("a|", "", "b|", 0, true, 0, 0);
        test("a|?", "", "a", 0, true, 0, 1);
        test("a|?", "", "b", 0, true, 0, 0);
        test("a|?", "", "?", 0, true, 0, 0);
        test("|", "", "b", 0, true, 0, 0);
        test("a|()", "", "a", 0, true, 0, 1, -1, -1);
        test("a|()", "", "b", 0, true, 0, 0, 0, 0);
        test("a||", "", "a", 0, true, 0, 1);
        test("a||", "", "b", 0, true, 0, 0);
        test("a||b", "", "a", 0, true, 0, 1);
        test("a||b", "", "b", 0, true, 0, 0);
        test("a||b", "", "c", 0, true, 0, 0);
        test("a||b|", "", "a", 0, true, 0, 1);
        test("a||b|", "", "b", 0, true, 0, 0);
        test("a||b|", "", "c", 0, true, 0, 0);
        test("a|()|", "", "a", 0, true, 0, 1, -1, -1);
        test("a|()|", "", "b", 0, true, 0, 0, 0, 0);
        test("(a|)", "", "a", 0, true, 0, 1, 0, 1);
        test("(a|)", "", "b", 0, true, 0, 0, 0, 0);
        test("|a", "", "a", 0, true, 0, 0);
        test("|a", "", "b", 0, true, 0, 0);
        test("|a|", "", "a", 0, true, 0, 0);
        test("|a|", "", "b", 0, true, 0, 0);
        test("||a|", "", "b", 0, true, 0, 0);
        test("|()|", "", "a", 0, true, 0, 0, -1, -1);
        test("a\\Z\\z[[:cntrl:]]", "", "a\n", 0, false);
        test("\\z", "", "a", 0, true, 1, 1);
        test("\\z", "", "\n", 0, true, 1, 1);
        test("\\z", "", "a\n", 0, true, 2, 2);
        test("\\Z", "", "a", 0, true, 1, 1);
        test("\\Z", "", "\n", 0, true, 0, 0);
        test("\\Z", "", "a\n", 0, true, 1, 1);
        test("$", "", "a", 0, true, 1, 1);
        test("$", "", "\n", 0, true, 0, 0);
        test("$", "", "a\n", 0, true, 1, 1);
        test("a\\z", "", "a", 0, true, 0, 1);
        test("a\\z", "", "\n", 0, false);
        test("a\\z", "", "a\n", 0, false);
        test("a\\Z", "", "a", 0, true, 0, 1);
        test("a\\Z", "", "\n", 0, false);
        test("a\\Z", "", "a\n", 0, true, 0, 1);
        test("a$", "", "a", 0, true, 0, 1);
        test("a$", "", "\n", 0, false);
        test("a$", "", "a\n", 0, true, 0, 1);
        test("\n\\z", "", "a", 0, false);
        test("\n\\z", "", "\n", 0, true, 0, 1);
        test("\n\\z", "", "a\n", 0, true, 1, 2);
        test("\n\\Z", "", "a", 0, false);
        test("\n\\Z", "", "\n", 0, true, 0, 1);
        test("\n\\Z", "", "a\n", 0, true, 1, 2);
        test("\n$", "", "a", 0, false);
        test("\n$", "", "\n", 0, true, 0, 1);
        test("\n$", "", "a\n", 0, true, 1, 2);
        test("a\n\\z", "", "a", 0, false);
        test("a\n\\z", "", "\n", 0, false);
        test("a\n\\z", "", "a\n", 0, true, 0, 2);
        test("a\n\\Z", "", "a", 0, false);
        test("a\n\\Z", "", "\n", 0, false);
        test("a\n\\Z", "", "a\n", 0, true, 0, 2);
        test("a\n$", "", "a", 0, false);
        test("a\n$", "", "\n", 0, false);
        test("a\n$", "", "a\n", 0, true, 0, 2);
        test("a\\z\n", "", "a\n", 0, false);
        test("a\\Z\n", "", "a\n", 0, true, 0, 2);
        test("(^|\\Z)a", "", "a", 0, true, 0, 1, 0, 0);
        test("a(\\z|())", "", "a", 0, true, 0, 1, 1, 1, -1, -1);
        test("\\z|\\Z", "", "\n", 0, true, 0, 0);
        test("\\z|\\Z", "", "a\n", 0, true, 1, 1);
        test("\\Z|\\z", "", "\n", 0, true, 0, 0);
        test("\\Z|\\z", "", "a\n", 0, true, 1, 1);
        test("a\\z\\Z", "", "a\n", 0, false);
        test("a\\Z\\z", "", "a\n", 0, false);
        test("a\n\\z\\Z", "", "a\n", 0, true, 0, 2);
        test("a\n\\Z\\z", "", "a\n", 0, true, 0, 2);
        test("a\\z\\Z\n", "", "a\n", 0, false);
        test("a\\Z\\z\n", "", "a\n", 0, false);
        test("a((b?)*)*", "", "ab", 0, true, 0, 2, 2, 2, 2, 2);
        test("(()|()|a)+b", "", "ab", 0, true, 0, 2, 1, 1, 1, 1, -1, -1);
        test("a(|()|())*", "", "a", 0, true, 0, 1, 1, 1, -1, -1, -1, -1);
        test("a(()|()|)*b", "", "ab", 0, true, 0, 2, 1, 1, 1, 1, -1, -1);
        test("a(()|()|())*b", "", "ab", 0, true, 0, 2, 1, 1, 1, 1, -1, -1, -1, -1);
        test("a(()|()|()|())*b", "", "ab", 0, true, 0, 2, 1, 1, 1, 1, -1, -1, -1, -1, -1, -1);
        test("a(b|()|()|()|())*c", "", "abc", 0, true, 0, 3, 2, 2, 2, 2, -1, -1, -1, -1, -1, -1);
        test("a(()|b|()|()|())*c", "", "abc", 0, true, 0, 3, 2, 2, 2, 2, -1, -1, -1, -1, -1, -1);
        test("a(()|()|b|()|())*c", "", "abc", 0, true, 0, 3, 2, 2, 2, 2, -1, -1, -1, -1, -1, -1);
        test("a(()|()|()|b|())*c", "", "abc", 0, true, 0, 3, 2, 2, 2, 2, -1, -1, -1, -1, -1, -1);
        test("a(()|()|()|()|b)*c", "", "abc", 0, true, 0, 3, 2, 2, 2, 2, -1, -1, -1, -1, -1, -1);
        test("a(b|()|()|()|())*c", "", "abbc", 0, true, 0, 4, 3, 3, 3, 3, -1, -1, -1, -1, -1, -1);
        test("a(()|b|()|()|())*c", "", "abbc", 0, true, 0, 4, 3, 3, 3, 3, -1, -1, -1, -1, -1, -1);
        test("a(()|()|b|()|())*c", "", "abbc", 0, true, 0, 4, 3, 3, 3, 3, -1, -1, -1, -1, -1, -1);
        test("a(()|()|()|b|())*c", "", "abbc", 0, true, 0, 4, 3, 3, 3, 3, -1, -1, -1, -1, -1, -1);
        test("a(()|()|()|()|b)*c", "", "abbc", 0, true, 0, 4, 3, 3, 3, 3, -1, -1, -1, -1, -1, -1);
        test("()??()??()??()??()??()??()??()??\\3\\5\\7", "", "a", 0, true, 0, 0, -1, -1, -1, -1, 0, 0, -1, -1, 0, 0, -1, -1, 0, 0, -1, -1);
        test("()*", "", "a", 0, true, 0, 0, 0, 0);
        test("(a|)*", "", "a", 0, true, 0, 1, 1, 1);
        test("(|a)?", "", "a", 0, true, 0, 0, 0, 0);
        test("(a|())*", "", "a", 0, true, 0, 1, 1, 1, 1, 1);
        test("()??\\1", "", "a", 0, true, 0, 0, 0, 0);
        test("(a|())*?\\2", "", "a", 0, true, 0, 1, 1, 1, 1, 1);
        test("(a*)+", "", "a", 0, true, 0, 1, 1, 1);
        test("(\\1a|){2}", "", "aa", 0, true, 0, 0, 0, 0);
        expectSyntaxError("[[.\\].]]", "", "]", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[.\\[.]]", "", "[", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[=\\]=]]", "", "]", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[=\\[=]]", "", "[", 0, ErrorCode.InvalidCharacterClass);
        test("[--I]", "", "-", 0, true, 0, 1);
        expectSyntaxError("[[=^=]--I]", "", "-", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[=\\}=]I]", "", "-", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[I-[=\\}=]]", "", "I", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[y-\\{]", "", "I", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[y-\\{]", "", "I", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[y-\\{][y-\\{]", "", "I", 0, ErrorCode.InvalidCharacterClass);
        test("a?", "", "aaa", 0, true, 0, 1);
        test("a??", "", "aaa", 0, true, 0, 0);
        test("a+?", "", "aaa", 0, true, 0, 1);
        test("((a?)*)??", "", "aaa", 0, true, 0, 0, -1, -1, -1, -1);
        test("((a?)*?)?", "", "aaa", 0, true, 0, 0, 0, 0, -1, -1);
        test("(a+)*?", "", "aaa", 0, true, 0, 0, -1, -1);
        test("((a+)*)?", "", "aaa", 0, true, 0, 3, 0, 3, 0, 3);
        expectSyntaxError("[[.\\a.]]", "", ".", 0, ErrorCode.InvalidCharacterClass);
        test("[[...]]", "", ".", 0, true, 0, 1);
        test("[[...]]", "", "[", 0, false);
        test("[[...]]", "", "]", 0, false);
        expectSyntaxError("[[.\\..]]", "", ".", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[.\\..]]", "", "\\", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[.\\..]]", "", "[", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[.\\..]]", "", "]", 0, ErrorCode.InvalidCharacterClass);
        test("[[...]]", "", ".]", 0, true, 0, 1);
        test("[[...]a]", "", ".a]", 0, true, 0, 1);
        test("[a[...]]", "", "a]", 0, true, 0, 1);
        test("[[...]a]a", "", "a", 0, false);
        test("[[...]a]?a", "", "a", 0, true, 0, 1);
        test("[[...]a]|a", "", "a", 0, true, 0, 1);
        test("[[===]]", "", "=", 0, true, 0, 1);
        expectSyntaxError("[[=\\==]]", "", "=", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[=\\==]]", "", "\\", 0, ErrorCode.InvalidCharacterClass);
        test("[[===]]", "", "=]", 0, true, 0, 1);
        test("[[===]a]", "", "=a]", 0, true, 0, 1);
        test("[a[===]]", "", "a]", 0, true, 0, 1);
        test("[[===]a]a", "", "aa", 0, true, 0, 2);
        test("[[===]a]a", "", "a]a", 0, false);
        test("[[===]a]?a", "", "a", 0, true, 0, 1);
        test("[[===]a]|a", "", "a", 0, true, 0, 1);
        test("\\\\(a+)", "", "(\\a)", 0, true, 1, 3, 2, 3);
        test("\\\\(a+)", "", "\\a)", 0, true, 0, 2, 1, 2);
        test("\\\\(a)", "", "\\a)", 0, true, 0, 2, 1, 2);
        test("\\\\(a)", "", "(\\a)", 0, true, 1, 3, 2, 3);
        test("\\\\(a)", "", "\\(a)", 0, false);
        test("\\\\(a)", "", "\\\\(a)", 0, false);
        test("\\\\(a)+", "", "\\a\\a", 0, true, 0, 2, 1, 2);
        test("\\\\(a)+", "", "\\aa", 0, true, 0, 3, 2, 3);
        expectSyntaxError("\\(a)", "", "\\a)", 0, ErrorCode.UnmatchedParenthesis);
        expectSyntaxError(")", "", "(\\a)", 0, ErrorCode.UnmatchedParenthesis);
        expectSyntaxError("\\()", "", "\\a)", 0, ErrorCode.UnmatchedParenthesis);
        test("\\\\()", "", "\\a)", 0, true, 0, 1, 1, 1);
        test("\\\\(a)b", "", "\\ab", 0, true, 0, 3, 1, 2);
        test("\\\\(a)()b", "", "\\ab", 0, true, 0, 3, 1, 2, 2, 2);
        test("\\\\(a)()()()()()()()()()()b", "", "\\ab", 0, true, 0, 3, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2);
        test("\\[[=a=]c-b]", "", "[ac-b]", 0, true, 0, 6);
        test("\\[[.a.]c-b]", "", "[ac-b]", 0, true, 0, 6);
        test("\\[[=a=]c-b", "", "[ac-b]", 0, true, 0, 5);
        test("\\[[.a.]c-b", "", "[ac-b]", 0, true, 0, 5);
        test("[=a=]c-b]", "", "[ac-b]", 0, true, 1, 6);
        test("\\[[a=]c-b]", "", "[ac-b]", 0, true, 0, 6);
        test("\\[[=a]c-b]", "", "[ac-b]", 0, true, 0, 6);
        test("\\[[=]c-b]", "", "[=c-b]", 0, true, 0, 6);
        test("\\[[a]c-b]", "", "[ac-b]", 0, true, 0, 6);
        test("\\[[=a=]b-b]*", "", "[[aabb", 0, false);
        test("\\[[=a=]b-b]*", "", "[aabb", 0, false);
        test("\\[[=a=]b-b]*", "", "[aabb]]", 0, false);
        test("\\[[=a=]b-b]*", "", "[aabb==]]", 0, false);
        test("\\[[=a=]b-b]*", "", "[aabb==]]", 0, false);
        test("\\[[=a=]b-b]*", "", "aabb", 0, false);
        test("\\[b-b]", "", "[b-b]", 0, true, 0, 5);
        test("\\[[b-b]", "", "[b-b]", 0, true, 0, 2);
        test("\\[c-b]", "", "[c-b]", 0, true, 0, 5);
        expectSyntaxError("\\[[c-b]", "", "[c-b]", 0, ErrorCode.InvalidCharacterClass);
        test("ac??bc?", "", "abc", 0, true, 0, 3);
        test("ac??bc?", "", "acbc", 0, true, 0, 4);
        test("a?", "", "a", 0, true, 0, 1);
        test("a??", "", "a", 0, true, 0, 0);
        test("(a?)??", "", "a", 0, true, 0, 0, -1, -1);
        test("(a??)?", "", "a", 0, true, 0, 0, 0, 0);
        test("(a{0,1})*", "", "aaaaaa", 0, true, 0, 6, 6, 6);
        test("(a{0,2})*", "", "aaaaaa", 0, true, 0, 6, 6, 6);
        test("(a{1,2})*", "", "aaaaaa", 0, true, 0, 6, 4, 6);
        test("(a{2,2})*", "", "aaaaaa", 0, true, 0, 6, 4, 6);
        test("(\\d)", "", "\u1a89", 0, true, 0, 3, 0, 3);
        test("a^^?", "", "a", 0, false);
        test("a||b{2,5}", "", "aaaaa", 0, true, 0, 1);
        test("a||b{1,5}", "", "aaaaa", 0, true, 0, 1);
        test("a|b{1,5}", "", "aaaaa", 0, true, 0, 1);
        test("[^]()|(())||()|u\\A\\6k()\\Z[\\(](|Z^|[[=A=]]|[[=k=]])", "", ":Z", 0, true, 0, 1, 1, 1);
        test("(a*?)*", "", "aaa", 0, true, 0, 0, 0, 0);
        test("?a{1,2}||b", "", "ccc", 0, true, 0, 0);
        test("()?a{1,2}||b", "", "ccc", 0, true, 0, 0, -1, -1);
        test("[a-a]", "", "a", 0, true, 0, 1);
        test("[---]", "", "-", 0, true, 0, 1);
        expectSyntaxError("[a---]", "", "-", 0, ErrorCode.InvalidCharacterClass);
        test("[a-a---]", "", "-", 0, true, 0, 1);
        test("[a-a---[:alpha:]]", "", "-", 0, true, 0, 1);
        test("[---[:alpha:]]", "", "-", 0, true, 0, 1);
        test("[a-c-e]", "", "-", 0, true, 0, 1);
        test("[a-c-e]", "", "a", 0, true, 0, 1);
        test("[a-c-e]", "", "b", 0, true, 0, 1);
        test("[a-c-e]", "", "c", 0, true, 0, 1);
        test("[a-c-e]", "", "d", 0, false);
        test("[a-c-e]", "", "e", 0, true, 0, 1);
        test("a?a+?", "", "aaa", 0, true, 0, 2);
        test("()a\\1\\1", "", "a", 0, true, 0, 1, 0, 0);
        test("()\\1\\1a", "", "a", 0, true, 0, 1, 0, 0);
        test("(a)\\1\\1b", "", "aaab", 0, true, 0, 4, 0, 1);
        test("[[:space:]-y]", "", "empty", 0, true, 4, 5);
        test("[[:space:]-y]", "", "empt-y", 0, true, 4, 5);
        test("[[:space:]--y]", "", "empty", 0, true, 0, 1);
        test("a{1,2}|", "", "ccc", 0, true, 0, 0);
        test("?a{1,2}|", "", "ccc", 0, true, 0, 0);
        test("()?a{1,2}|", "", "ccc", 0, true, 0, 0, -1, -1);
        test("()a{1,2}|", "", "ccc", 0, true, 0, 0, -1, -1);
        test("b?a{1,2}|", "", "ccc", 0, true, 0, 0);
        test("(|[ab]){3,3}?b", "", "aab", 0, true, 0, 3, 2, 2);
        test("(|[ab]){3,3}b", "", "aab", 0, true, 0, 3, 2, 2);
        test("(|[ab]){3}b", "", "aab", 0, true, 0, 3, 2, 2);
        test("(|a){3}b", "", "aab", 0, true, 0, 3, 2, 2);
        test("(|a){2}b", "", "ab", 0, true, 0, 2, 1, 1);
        test("(|a){1}b", "", "b", 0, true, 0, 1, 0, 0);
        test("(|a)b", "", "b", 0, true, 0, 1, 0, 0);
        test("(|a)(|a)(|a)b", "", "aab", 0, true, 0, 3, 0, 0, 0, 1, 1, 2);
        test("(|a)(|a)b", "", "ab", 0, true, 0, 2, 0, 0, 0, 1);
        test("(|a+?){0,4}", "", "aaa", 0, true, 0, 0, 0, 0);
        test("(|a+?){0,4}b", "", "aaab", 0, true, 0, 4, 3, 3);
        expectSyntaxError("[^][.a[.b.]]", "", "ab", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[^[.a[.b.]]", "", "ab", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[.a[.b.]]*", "", "[[[...aabb", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[.a[.b.]]*", "", "[]]]", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[.a[.b.]*]*", "", "[..aa..bb..]][", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[[=abcdefgh=]*]*", "", "[..aa..bb..]][", 0, ErrorCode.InvalidCharacterClass);
        expectSyntaxError("[^]", "", "[]^", 0, ErrorCode.UnmatchedBracket);
        test("[^]]", "", "[]^", 0, true, 0, 1);
        expectSyntaxError("[]", "", "[]", 0, ErrorCode.UnmatchedBracket);
        test("[]]", "", "[]", 0, true, 1, 2);
        expectSyntaxError("[[.a.b.]]", "", "[[.a.b.]]", 0, ErrorCode.InvalidCharacterClass);
        test("\\z|((\\WV|))\\Z", "", "a", 0, true, 1, 1, -1, -1, -1, -1);
        test("\\z|(\\WV|)\\Z", "", "a", 0, true, 1, 1, -1, -1);
        test("\\z|(x|)\\Z", "", "a", 0, true, 1, 1, -1, -1);
        test("a(|())\\2", "", "a", 0, true, 0, 1, 1, 1, 1, 1);
        test("a(|())\\2?", "", "a", 0, true, 0, 1, 1, 1, -1, -1);
        test("a(|())\\2*", "", "a", 0, true, 0, 1, 1, 1, -1, -1);
        test("a(|())\\2+", "", "a", 0, true, 0, 1, 1, 1, 1, 1);
        test("a(|())\\2{1}", "", "a", 0, true, 0, 1, 1, 1, 1, 1);
        test("a(|())\\2{1,2}", "", "a", 0, true, 0, 1, 1, 1, 1, 1);
        test("a(|())\\2{2,3}", "", "a", 0, true, 0, 1, 1, 1, 1, 1);
        test("a(b|()\\1|\\2){4}", "", "abb", 0, true, 0, 3, 3, 3, 2, 2);
        test("\\D([[=O=]S[.f.]]|\\s|(())\\1|\\3(|)){27,222}?", "", "\"O f\u000b  OoOO O\r O", 0, true, 0, 16, 16, 16, 10, 10, 10, 10, 16, 16);
        test("a??|", "", "a", 0, true, 0, 0);
        test("b?a??|", "", "a", 0, true, 0, 0);
        test("b??a??|", "", "a", 0, true, 0, 0);
        test("a??a?|", "", "a", 0, true, 0, 1);
        test("a??a??|", "", "a", 0, true, 0, 0);
        test("(|a)a??|", "", "a", 0, true, 0, 0, 0, 0);
        test("(|a)(|a)|", "", "a", 0, true, 0, 0, 0, 0, 0, 0);
        test("a??a??a??|", "", "a", 0, true, 0, 0);
        test("b??a??", "", "a", 0, true, 0, 0);
        test("(a??)a??|", "", "a", 0, true, 0, 0, 0, 0);
        test("(a??)(a??)|", "", "a", 0, true, 0, 0, 0, 0, 0, 0);
        test("()b??a??|", "", "a", 0, true, 0, 0, 0, 0);
        expectSyntaxError("a{2,71920}", "", "aa", 0, ErrorCode.InvalidQuantifier);
        test("()\\1\\z", "", "a", 0, true, 1, 1, 1, 1);
        test("()\\1\\1\\z", "", "a", 0, true, 1, 1, 1, 1);
        expectSyntaxError("[]\\\\", "", "g", 0, ErrorCode.UnmatchedBracket);
        test("[[:print:][.j.]]\\S;[0-HB]t", "", "$*iAt\u0005;;Et", 0, false);
        test("\\s\\Dz()[[.o.][:blank:][:upper:][=c=]'[.0.][:blank:]<-M}-}[=r=][.h.][=e=]]", "", " &zzh", 0, false);
        test("()(a\\1\\1)*", "", "aaaa", 0, true, 0, 4, 0, 0, 3, 4);
        test("8(?\\A\\D|)[j-{[:punct:][=:=]](\\1[d[./.]]\\1{10,20}?)*\\z|", "", "8://dd/d///d/d", 0, true, 0, 14, 1, 1, 13, 14);
        test("(a*)*", "", "aaaaaaaaaaaaaaa", 0, true, 0, 15, 15, 15);
        test("(a*())*", "", "aaaaaaaaaaaaaaa", 0, true, 0, 15, 15, 15, 15, 15);
        test("([^a])*[^b]+", "", "ccccc", 0, true, 0, 5, 3, 4);
        test("([^a])*[^b]+[^a]", "", "cccccccccccccccc", 0, true, 0, 16, 13, 14);
        test("(\\S)*\\D+\\S", "", "\u0007c\u0004\u0012U\u0019\u001a/to!\u000f\u0012M+~QO", 0, true, 0, 18, 15, 16);
        test("a*?b?|b", "", "b", 0, true, 0, 1);
        test("[][::A]]", "", ":]", 0, true, 0, 2);
        test("[[::AA]]", "", ":]", 0, true, 0, 2);
        test("()^\\s()(\\1\\1|\\\\\\1\\1)+()\\2[_|-~[.>.][:lower:]]|\\w\\s\\1|", "", " \\\\>", 0, true, 0, 4, 0, 0, 1, 1, 3, 3, 3, 3);
        test("\\W\\d\\]+?\\+?'\\D((|\\d\\2)4S(5)O\\W){1,278}\\w", "", "`9]]]]]]]'~4S5O?4S5O`74S5O|4S5O{04S5O}V", 0, true, 0, 39, 32, 38, 32, 33, 35, 36);
        test("H(@\\S|4|\\1\\D|\\z*)+?()H\\Wh", "", "H@\u0012@\u0008@J4%@\u0002\u00054H&h", 0, false);
        test("\\d(A(()()i(|(??\\5)\\W))|\\A)*[T]", "", "3AiAi\u007fAiAi@T", 0, true, 0, 12, 8, 11, 9, 11, 9, 9, 9, 9, 10, 11, 10, 10);
        test("o|([[:blank:][:punct:]I[=_=]])*?$", "", "}~", 0, true, 0, 2, 1, 2);
        test("()*?()a+?", "", "b", 0, false);
        test("()\\1*\\1\\1+\\1?\\1", "", "b", 0, true, 0, 0, 0, 0);
        test("((b\\2|)a)+", "", "aba", 0, true, 0, 3, 1, 3, 1, 2);
        test("(\\A{1,10}a)*?\\z", "", "aa", 0, true, 2, 2, -1, -1);
        test("[0-d]", "i", "L", 0, false);
        test("[W-`]", "i", "hK", 0, false);
        test("[x]|[\\\\t-|]\\+", "i", "X+", 0, true, 0, 1);
        test("[N[:upper:]]|", "i", "empty", 0, true, 0, 0);
        test("\u00df", "i", "ss", 0, true, 0, 2);
        test("\u00dfs", "i", "sss", 0, true, 0, 3);
        test("s\u00dfs", "i", "\u00df\u00df", 0, true, 0, 4);
        test("[s]\u00dfs", "i", "\u00df\u00df", 0, false);
        test("s[\u00df]s", "i", "\u00df\u00df", 0, false);
        test("s\u00df[s]", "i", "\u00df\u00df", 0, false);
        test("\u00df\u00df", "i", "s\u00dfs", 0, true, 0, 4);
        test("[\u00df]\u00df", "i", "s\u00dfs", 0, false);
        test("s\u00df", "i", "sss", 0, true, 0, 3);
        test("s\u00df", "i", "\u00dfs", 0, true, 0, 3);
        test("sss", "i", "\u00dfs", 0, true, 0, 3);
        test("[\u00df]", "i", "ss", 0, true, 0, 2);
        test("[\u00dfs]", "", "ss", 0, true, 0, 1);
        test("[s\u00df]", "", "ss", 0, true, 0, 1);
        test("[[.\u00df.]s]", "", "ss", 0, true, 0, 1);
        test("[[=\u00df=]s]", "", "ss", 0, true, 0, 2);
        test("[s[.\u00df.]]", "", "ss", 0, true, 0, 1);
        test("[s[=\u00df=]]", "", "ss", 0, true, 0, 1);
        test("[\u00dfs]", "i", "ss", 0, true, 0, 2);
        test("[\u00dfs]{2}", "i", "ss", 0, false);
        test("[\u00dfs]{1}", "i", "ss", 0, true, 0, 2);
        test("[\u00dfs]{1}", "i", "s", 0, true, 0, 1);
        test("[s\u00df]", "i", "ss", 0, true, 0, 1);
        test("[\u00dfs-s]", "i", "ss", 0, true, 0, 2);
        test("[s-s\u00df]", "i", "ss", 0, true, 0, 1);
        test("[\u00df-\u00df]", "i", "ss", 0, false);
        test("[^\u00df]", "i", "ss", 0, true, 1, 2);
        test("[^\u00df]", "i", "s", 0, true, 0, 1);
        test("[^\u00df]", "i", "sx", 0, true, 0, 1);
        test("[^\u00df-\u00df]", "i", "ss", 0, true, 0, 1);
        test("[^s]", "i", "ss", 0, false);
        test("[^s]", "i", "SS", 0, false);
        test("[^h-tx]", "i", "SS", 0, false);
        test("[^-f]", "i", "fI", 0, true, 1, 2);
        test("[\ufb03]", "i", "FFI", 0, true, 0, 3);
        test("[\ufb03f]", "i", "FFI", 0, true, 0, 3);
        test("[\ufb03f]", "i", "FF", 0, true, 0, 1);
        test("[\ufb03f]", "i", "FFX", 0, true, 0, 1);
        test("[\ufb03f]", "i", "F", 0, true, 0, 1);
        test("[\ufb03\ufb00]", "i", "FFI", 0, true, 0, 3);
        test("[\ufb03\ufb00]", "i", "FF", 0, true, 0, 2);
        test("[\ufb03\ufb00]i", "i", "FFI", 0, false);
        test("(\ufb03|\ufb00)i", "i", "FFI", 0, true, 0, 3, 0, 2);
        test("[\ufb00]i", "i", "FFI", 0, true, 0, 3);
        test("[\ufb03\ufb00]", "i", "F", 0, false);
        test("[\ufb03\ufb04]", "i", "FFL", 0, true, 0, 3);
        test("\ufb04", "i", "FFL", 0, true, 0, 3);
        test("[[=\u2167=]\u491f-\u9af9[:graph:]\u1490-\u7662[.\u4d67.]\uec18[:print:][.\u45cc.]]", "i", "L", 0, true, 0, 1);
        test("[[=\u1e9a=]]", "i", "a", 0, false);
        test("[\u1f89-\u8ff9]", "i", "empty", 0, true, 0, 1);
        test("[\ud83b\udd2d-\ud83c\udd63]", "i", "w", 0, false);
        test("[[=\u01c6=]]", "i", "Dz", 0, false);
        test("[[=\u00d7=]]", "i", "O", 0, false);
        test("[[.\ud9fa\udc11.][.\ud962\udd68.][:cntrl:][=\ud98e\uddb1=]][[=\ud89d\udfb9=]]+?", "i", "\u0014\u001a\u0008\u007f\u0001\u007f\u007f", 0, false);
        test("\u1f8d", "i", "\u1f85", 0, false);
        test("[\u1f9d]", "i", "\u1f95", 0, false);
        test("[\u1ff3]", "i", "\u1ffc", 0, false);
        test("^", "m", "\n", 0, true, 0, 0);
        test("^", "m", "a\n", 0, true, 0, 0);
        test("^", "m", "a\n", 1, true, 2, 2);
        test("^", "m", "a\na", 1, true, 2, 2);
        test("^", "m", "aa\n", 1, true, 3, 3);
        test("^", "m", "a\na", 0, true, 0, 0);
        test("^", "m", "a\na", 1, true, 2, 2);
        test("^", "m", "aa\na", 1, true, 3, 3);
        test("a^a", "m", "aa\na", 1, false);
        test("a\n^a", "m", "aa\na", 1, false);
        test("a^\na", "m", "aa\na", 1, false);
        test("$", "m", "\n", 0, true, 0, 0);
        test("$", "m", "\na", 0, true, 0, 0);
        test("$", "m", "\na", 1, true, 2, 2);
        test("$", "m", "a\na", 0, true, 1, 1);
        test("$", "m", "a\na", 1, true, 1, 1);
        test("${1}", "m", "a\na", 1, true, 1, 1);
        test("$()", "m", "a\na", 1, true, 1, 1, 1, 1);
        test("$(){1}", "m", "a\na", 1, true, 1, 1, 1, 1);
        test("$", "m", "aa\na", 1, true, 2, 2);
        test("a$a", "m", "aa\na", 1, false);
        test("a\n$a", "m", "aa\na", 1, false);
        test("a$\na", "m", "aa\na", 1, false);
        test("^$", "m", "\n", 0, true, 0, 0);
        test("^$", "m", "a\n", 0, true, 2, 2);
        test("^$", "m", "a\n", 1, true, 2, 2);
        test("^$", "m", "aa\n", 1, true, 3, 3);
        test("^$", "m", "a\na", 0, false);
        test("^$", "m", "a\na", 1, false);
        test("^$", "m", "aa\na", 1, false);
        test("^$", "m", "a\n\na", 0, true, 2, 2);
        test("^$", "m", "a\n\na", 1, true, 2, 2);
        test("^$", "m", "aa\n\na", 1, true, 3, 3);
        test("$^", "m", "\n", 0, true, 0, 0);
        test("$^", "m", "a\n", 0, true, 2, 2);
        test("$^", "m", "a\n", 1, true, 2, 2);
        test("$^", "m", "aa\n", 1, true, 3, 3);
        test("$^", "m", "a\na", 0, false);
        test("$^", "m", "a\na", 1, false);
        test("$^", "m", "aa\na", 1, false);
        test("$^", "m", "a\n\na", 0, true, 2, 2);
        test("$^", "m", "a\n\na", 1, true, 2, 2);
        test("$^", "m", "aa\n\na", 1, true, 3, 3);
        test("$\n", "m", "aa\n\na", 1, false);
        test("$(\n)", "m", "aa\n\na", 1, false);
        test("^", "m", "\na", 0, true, 0, 0);
        test("^", "m", "\na", 1, true, 1, 1);
        test("^a", "m", "aaa\naa", 1, true, 4, 5);
        test("^(^)", "m", "\na", 0, true, 0, 0, 0, 0);
        test("^(^)", "m", "\na", 1, true, 1, 1, 1, 1);
        test("^^a", "m", "\na", 0, true, 1, 2);
        test("^^a", "m", "\n\na", 1, true, 2, 3);
        test("^^", "m", "\n\n\n", 1, true, 1, 1);
        test("^^", "m", "\n\n\n", 0, true, 0, 0);
        test("^^", "m", "\n\na", 1, true, 1, 1);
        test("^^a", "m", "\na", 1, true, 1, 2);
        test("^^", "m", "\na", 1, true, 1, 1);
        test("^(^)a", "m", "\na", 0, true, 1, 2, 1, 1);
        test("$?", "m", "empty", 0, true, 0, 0);
        test("\\A$\\A", "m", "\n", 0, true, 0, 0);
        test("$$", "m", "\n", 0, true, 0, 0);
        test("$($)", "m", "\n", 0, true, 0, 0, 0, 0);
        test("\\Z^", "m", "\n", 0, true, 0, 0);
        test("^\\z|", "m", "\n", 0, true, 0, 0);
        test("$*", "m", "empty", 0, true, 0, 0);
        test("$^()|$?", "m", "empty", 0, true, 0, 0, -1, -1);
        test("\\Z\\W", "m", "\n", 0, true, 0, 1);
        test("\\Z\\D", "m", "\n", 0, true, 0, 1);
        test("\\W\\D()\\Z\\s", "m", "\u17f6\u1855\n", 0, true, 0, 7, 6, 6);
        test("$()\\1", "m", "\n", 0, true, 1, 1, 1, 1);
        test("\\Z()\\1()", "m", "empty", 0, true, 5, 5, 5, 5, 5, 5);
        test("^\\z", "nm", "\n", 0, true, 1, 1);
        test("(()|*?)\\w\\d[[=\ud988\uddf0=][.\udb06\ude6e.][:space:]]*^()k\ud9ae\udc34", "m", "\u05ba\u0a67\u202f \u205f\ud988\uddf0\u205f\u0009\nk\ud9ae\udc34", 0, false);
        test("$\\Z^", "m", "\n", 0, true, 0, 0);
        test("$()\\Z\\1", "m", "\n", 0, true, 1, 1, 1, 1);
        test("${1,2}", "mx", "\n", 0, true, 0, 0);
        test("\\(ADDRESS=(\\([^()]+\\))+\\)", "i", "(address=(x))", 0, true, 0, 13, 9, 12);
        test("[[.\u1ef8.]]", "ic", "\u1ef9", 0, false);
        test("\\D$\\z", "m", "x\n", 0, true, 1, 2);
        test("($)\\z\\1", "m", "\n", 0, true, 1, 1, 1, 1);
        test("$(()+\\2|)\\z", "m", "\n", 0, true, 1, 1, 1, 1, 1, 1);
        test("$()\\z\\1", "m", "\n", 0, true, 1, 1, 1, 1);
        test("$(()\\2|)\\z", "m", "\n", 0, true, 1, 1, 1, 1, 1, 1);
        test("$(())\\z\\2^", "m", "\n", 0, true, 1, 1, 1, 1, 1, 1);
        test("$(())\\z^\\2", "m", "\n\n", 0, true, 2, 2, 2, 2, 2, 2);
        test("\\W$\\z", "m", " \n", 0, true, 1, 2);
        test("($)+?\\1", "m", "\n", 0, true, 1, 1, 1, 1);
        test("\\s$\\z", "m", " \n", 0, true, 1, 2);
        test("[\ua762-\ua78d]", "i", "\ua7ae", 0, false);
        test("[\u1f1d-\ua7ab]", "i", "\u2c63", 0, false);
        test("[\u10b1-\u1cac]", "i", "\u1f04", 0, false);
        test("($)*\\s*", "m", "\n ", 0, true, 0, 0, 0, 0);
        test("$*\\s*", "m", "\n ", 0, true, 0, 0);
        test("(^|(|a))b\\z", "", "b", 0, true, 0, 1, 0, 0, -1, -1);
        test("(a*()*)*", "", "aaa", 0, true, 0, 3, 3, 3, 3, 3);
        test("(a*()+)+", "", "aaa", 0, true, 0, 3, 3, 3, 3, 3);
        test("(a*()+?)+", "", "aaa", 0, true, 0, 3, 3, 3, 3, 3);
        test("a((b|)+)+c", "", "abbbc", 0, true, 0, 5, 4, 4, 4, 4);
        test("((a|)+)+(b)+c", "", "aaabc", 0, true, 0, 5, 3, 3, 3, 3, 3, 4);
        test("(a*()+?b*?)+c", "", "aabaac", 0, true, 0, 6, 5, 5, 5, 5);
        test("((\\w|)()+)+", "i", "empty", 0, true, 0, 5, 5, 5, 5, 5, 5, 5);
        test("(a?()?){3,4}", "", "aa", 0, true, 0, 2, 2, 2, 2, 2);
        test("(a?()+){3,4}", "", "aa", 0, true, 0, 2, 2, 2, 2, 2);
        test("(a*()+?)+", "", "aaa", 0, true, 0, 3, 3, 3, 3, 3);
        test("a(b\\1|)*?()+c", "", "abc", 0, false);
        test("()(a*\\1+)*", "", "aaa", 0, true, 0, 3, 0, 0, 3, 3);
        test("(a(\\2b|)?)+\\1c", "", "aaabaaac", 0, true, 0, 8, 5, 6, 6, 6);
        test("((|ab)+?w\\Z|^c)de()d", "", "ffffff", 0, false);
        test("((a|){2,3}){2,3}", "", "aaaa", 0, true, 0, 4, 4, 4, 4, 4);
        test("((a?){2,3}){2,3}", "", "aaaa", 0, true, 0, 4, 4, 4, 4, 4);
        test("((|a){2,3}){2,3}", "", "aaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((a??){2,3}){2,3}", "", "aaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("(|(a\\zb)|e\\2f)??\\1", "", "xxxxx", 0, true, 0, 0, 0, 0, -1, -1);
        test("(a{2}?|)*", "", "abaaaaaa", 0, true, 0, 0, 0, 0);
        test("(a{2}?|)*", "", "aaaaaaaa", 0, true, 0, 8, 8, 8);
        test("(a$){1,23}?", "", "aaa", 0, true, 2, 3, 2, 3);
        test("(a\\1|){2,5}?", "", "a", 0, true, 0, 0, 0, 0);
        test("(a\\1|){2,5}?$", "", "a", 0, true, 1, 1, 1, 1);
        test("(a\\1\\1|){2,5}?$", "", "a", 0, true, 1, 1, 1, 1);
        test("(a\\1\\1|){2,5}?$||", "", "a", 0, true, 0, 0, -1, -1);
        test("(a\\1\\1|){2,5}?$|()\\2", "", "a", 0, true, 0, 0, -1, -1, 0, 0);
        test("(a\\1\\1|){2,5}?$||()\\2", "", "a", 0, true, 0, 0, -1, -1, -1, -1);
        test("(a{2}|())+", "", "aaaa", 0, true, 0, 4, 4, 4, 4, 4);
        test("(a{0,2})*", "", "aaa", 0, true, 0, 3, 3, 3);
        test("(a{0,2})*", "", "aaaa", 0, true, 0, 4, 4, 4);
        test("(\\1|a){35,74}?", "", "aaaaaaaaaaaaaaaaaaaa", 0, false);
        test("^a(b*)\\1{4,6}?", "", "abbbb", 0, true, 0, 1, 1, 1);
        test("^a(b*)\\1{4,6}?", "", "abbbbb", 0, true, 0, 6, 1, 2);
        test("(a|)\\1{20,20}", "", "aaaa", 0, true, 0, 0, 0, 0);
        test("(a|)a*\\1{20,20}", "", "aaaa", 0, true, 0, 4, 0, 0);
        test("(a)b\\1()?", "", "aba", 0, true, 0, 3, 0, 1, 3, 3);
        test("(a)(|\\1){4,4}bb", "", "aaabb", 0, true, 0, 5, 0, 1, 3, 3);
        test("(\\1|a|)*", "", "aaa", 0, true, 0, 3, 3, 3);
        test("a([bc]|()\\1|\\2){27,222}?", "", "abcbbc", 0, true, 0, 6, 6, 6, 4, 4);
        test("((a)\\2){4,10}", "", "aaaaaaaa", 0, true, 0, 8, 6, 8, 6, 7);
        test("()\\1(|b){2,2}?c", "", "bc", 0, true, 0, 2, 0, 0, 1, 1);
        test("\u0282\\\ud807\udfdd+\u1cf2", "", "\u0282\ud807\udfdd\ud807\udfdd\u1cf2", 0, true, 0, 13);
        test("((A|){7,10}?){10,17}", "", "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA", 0, true, 0, 86, 86, 86, 86, 86);
        test("(a{1,30}){1,4}", "", "a", 0, true, 0, 1, 0, 1);
        test("((a|){4,6}){4,6}", "", "aaaaaaa", 0, true, 0, 7, 7, 7, 7, 7);
        test("((a?){4,6}){4,6}", "", "aaaaaaa", 0, true, 0, 7, 7, 7, 7, 7);
        test("((|a){4,6}){4,6}", "", "aaaaaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((a??){4,6}){4,6}", "", "aaaaaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((a?){4,6}){4,6}", "", "aaaaaa", 0, true, 0, 6, 6, 6, 6, 6);
        test("(a|^){100}", "", "a", 0, true, 0, 0, 0, 0);
        test("(a|^){100}", "", "aa", 0, true, 0, 0, 0, 0);
        test("(a|^){100}", "", "aa", 1, false);
        test("(a|^){100}", "", "ab", 1, false);
        test("(a|){4,6}", "", "a", 0, true, 0, 1, 1, 1);
        test("(a|){4,6}", "", "aa", 0, true, 0, 2, 2, 2);
        test("(a|){4,6}", "", "aaa", 0, true, 0, 3, 3, 3);
        test("(a|){4,6}", "", "aaaa", 0, true, 0, 4, 4, 4);
        test("(a|){4,6}", "", "aaaaa", 0, true, 0, 5, 5, 5);
        test("(a|){4,6}", "", "aaaaaa", 0, true, 0, 6, 5, 6);
        test("(a|){4,6}", "", "aaaaaaa", 0, true, 0, 6, 5, 6);
        test("(a|){4,6}?", "", "a", 0, true, 0, 1, 1, 1);
        test("(a|){4,6}?", "", "aa", 0, true, 0, 2, 2, 2);
        test("(a|){4,6}?", "", "aaa", 0, true, 0, 3, 3, 3);
        test("(a|){4,6}?", "", "aaaa", 0, true, 0, 4, 3, 4);
        test("(a|){4,6}?", "", "aaaaa", 0, true, 0, 4, 3, 4);
        test("(a|){4,6}?", "", "aaaaaa", 0, true, 0, 4, 3, 4);
        test("(a|){4,6}?", "", "aaaaaaa", 0, true, 0, 4, 3, 4);
        test("(a|){4,6}?a", "", "a", 0, true, 0, 1, 0, 0);
        test("(a|){4,6}?a", "", "aa", 0, true, 0, 2, 1, 1);
        test("(a|){4,6}?a", "", "aaa", 0, true, 0, 3, 2, 2);
        test("(a|){4,6}?a", "", "aaaa", 0, true, 0, 4, 3, 3);
        test("(a|){4,6}?a", "", "aaaaa", 0, true, 0, 5, 3, 4);
        test("(a|){4,6}?a", "", "aaaaaa", 0, true, 0, 5, 3, 4);
        test("(a|){4,6}?a", "", "aaaaaaa", 0, true, 0, 5, 3, 4);
        test("(a|){4,6}?a", "", "aaaaaaaa", 0, true, 0, 5, 3, 4);
        test("(|a){4,6}a", "", "a", 0, true, 0, 1, 0, 0);
        test("(|a){4,6}a", "", "aa", 0, true, 0, 1, 0, 0);
        test("(|a){4,6}a", "", "aaa", 0, true, 0, 1, 0, 0);
        test("(|a){4,6}a", "", "aaaa", 0, true, 0, 1, 0, 0);
        test("(|a){4,6}a", "", "aaaaa", 0, true, 0, 1, 0, 0);
        test("(|a){4,6}a", "", "aaaaaa", 0, true, 0, 1, 0, 0);
        test("(|a){4,6}a", "", "aaaaaaa", 0, true, 0, 1, 0, 0);
        test("((a|){4,6}){4,6}", "", "a", 0, true, 0, 1, 1, 1, 1, 1);
        test("((a|){4,6}){4,6}", "", "aa", 0, true, 0, 2, 2, 2, 2, 2);
        test("((a|){4,6}){4,6}", "", "aaa", 0, true, 0, 3, 3, 3, 3, 3);
        test("((a|){4,6}){4,6}", "", "aaaa", 0, true, 0, 4, 4, 4, 4, 4);
        test("((a|){4,6}){4,6}", "", "aaaaa", 0, true, 0, 5, 5, 5, 5, 5);
        test("((a|){4,6}){4,6}", "", "aaaaaa", 0, true, 0, 6, 6, 6, 6, 6);
        test("((a|){4,6}){4,6}", "", "aaaaaaa", 0, true, 0, 7, 7, 7, 7, 7);
        test("((a|){4,6}){4,6}", "", "aaaaaaaa", 0, true, 0, 8, 8, 8, 8, 8);
        test("((a|){4,6}){4,6}", "", "aaaaaaaaa", 0, true, 0, 9, 9, 9, 9, 9);
        test("((a|){4,6}){4,6}", "", "aaaaaaaaaa", 0, true, 0, 10, 10, 10, 10, 10);
        test("((a|){4,6}){4,6}", "", "aaaaaaaaaaa", 0, true, 0, 11, 11, 11, 11, 11);
        test("((a|){4,6}){4,6}", "", "aaaaaaaaaaaa", 0, true, 0, 12, 12, 12, 12, 12);
        test("((a|){4,6}){4,6}", "", "aaaaaaaaaaaaa", 0, true, 0, 13, 13, 13, 13, 13);
        test("((|a){4,6}){4,6}", "", "a", 0, true, 0, 0, 0, 0, 0, 0);
        test("((|a){4,6}){4,6}", "", "aa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((|a){4,6}){4,6}", "", "aaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((|a){4,6}){4,6}", "", "aaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((|a){4,6}){4,6}", "", "aaaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((|a){4,6}){4,6}", "", "aaaaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((|a){4,6}){4,6}", "", "aaaaaaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((|a){4,6}){4,6}", "", "aaaaaaaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((|a){4,6}){4,6}", "", "aaaaaaaaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((|a){4,6}){4,6}", "", "aaaaaaaaaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((|a){4,6}){4,6}", "", "aaaaaaaaaaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((|a){4,6}){4,6}", "", "aaaaaaaaaaaaa", 0, true, 0, 0, 0, 0, 0, 0);
        test("((a|){4,6}?){4,6}", "", "a", 0, true, 0, 1, 1, 1, 1, 1);
        test("((a|){4,6}?){4,6}", "", "aa", 0, true, 0, 2, 2, 2, 2, 2);
        test("((a|){4,6}?){4,6}", "", "aaa", 0, true, 0, 3, 3, 3, 3, 3);
        test("((a|){4,6}?){4,6}", "", "aaaa", 0, true, 0, 4, 4, 4, 4, 4);
        test("((a|){4,6}?){4,6}", "", "aaaaa", 0, true, 0, 5, 5, 5, 5, 5);
        test("((a|){4,6}?){4,6}", "", "aaaaaa", 0, true, 0, 6, 6, 6, 6, 6);
        test("((a|){4,6}?){4,6}", "", "aaaaaaaa", 0, true, 0, 8, 8, 8, 8, 8);
        test("((a|){4,6}?){4,6}", "", "aaaaaaaaa", 0, true, 0, 9, 9, 9, 9, 9);
        test("((a|){4,6}?){4,6}", "", "aaaaaaaaaa", 0, true, 0, 10, 10, 10, 10, 10);
        test("((a|){4,6}?){4,6}", "", "aaaaaaaaaaa", 0, true, 0, 11, 11, 11, 11, 11);
        test("((a|){4,6}?){4,6}", "", "aaaaaaaaaaaa", 0, true, 0, 12, 12, 12, 12, 12);
        test("((a|){4,6}?){4,6}", "", "aaaaaaaaaaaaa", 0, true, 0, 13, 13, 13, 13, 13);
        test("((a|){4,6}?){4,6}", "", "aaaaaaaaaaaaaa", 0, true, 0, 14, 14, 14, 14, 14);
        test("((a|){4,6}?){4,6}", "", "aaaaaaaaaaaaaaa", 0, true, 0, 15, 15, 15, 15, 15);
        test("((a|){4,6}?){4,6}", "", "aaaaaaaaaaaaaaaa", 0, true, 0, 16, 16, 16, 16, 16);
        test("((a|){4,6}?){4,6}", "", "aaaaaaaaaaaaaaaaa", 0, true, 0, 17, 17, 17, 17, 17);
        test("((a|){4,6}?){4,6}", "", "aaaaaaaaaaaaaaaaaa", 0, true, 0, 18, 18, 18, 18, 18);
        test("((a){4,6}?){4,6}", "", "a", 0, false);
        test("((a){4,6}?){4,6}", "", "aa", 0, false);
        test("((a){4,6}?){4,6}", "", "aaa", 0, false);
        test("((a){4,6}?){4,6}", "", "aaaa", 0, false);
        test("((a){4,6}?){4,6}", "", "aaaaa", 0, false);
        test("((a){4,6}?){4,6}", "", "aaaaaa", 0, false);
        test("((a){4,6}?){4,6}", "", "aaaaaaaaaaaaaaaa", 0, true, 0, 16, 12, 16, 15, 16);
        test("((a){4,6}?){4,6}", "", "aaaaaaaaaaaaaaaaa", 0, true, 0, 16, 12, 16, 15, 16);
        test("((a){4,6}?){4,6}", "", "aaaaaaaaaaaaaaaaaaaa", 0, true, 0, 20, 16, 20, 19, 20);
        test("((a){4,6}?){4,6}", "", "aaaaaaaaaaaaaaaaaaaaaaaa", 0, true, 0, 24, 20, 24, 23, 24);
        test("((a){4,6}?){4,6}", "", "aaaaaaaaaaaaaaaaaaaaaaaaa", 0, true, 0, 24, 20, 24, 23, 24);
        test("((a){4,6}){4,6}", "", "a", 0, false);
        test("((a){4,6}){4,6}", "", "aa", 0, false);
        test("((a){4,6}){4,6}", "", "aaa", 0, false);
        test("((a){4,6}){4,6}", "", "aaaa", 0, false);
        test("((a){4,6}){4,6}", "", "aaaaa", 0, false);
        test("((a){4,6}){4,6}", "", "aaaaaa", 0, false);
        test("((a){4,6}){4,6}", "", "aaaaaaaaaaaaaaaa", 0, true, 0, 16, 12, 16, 15, 16);
        test("((a){4,6}){4,6}", "", "aaaaaaaaaaaaaaaaa", 0, true, 0, 17, 13, 17, 16, 17);
        test("((a){4,6}){4,6}", "", "aaaaaaaaaaaaaaaaaaaa", 0, true, 0, 20, 16, 20, 19, 20);
        test("((a){4,6}){4,6}", "", "aaaaaaaaaaaaaaaaaaaaaaaa", 0, true, 0, 24, 18, 24, 23, 24);
        test("((a){4,6}){4,6}", "", "aaaaaaaaaaaaaaaaaaaaaaaaa", 0, true, 0, 24, 18, 24, 23, 24);
        test("((a){4,}){4,6}", "", "a", 0, false);
        test("((a){4,}){4,6}", "", "aa", 0, false);
        test("((a){4,}){4,6}", "", "aaa", 0, false);
        test("((a){4,}){4,6}", "", "aaaa", 0, false);
        test("((a){4,}){4,6}", "", "aaaaa", 0, false);
        test("((a){4,}){4,6}", "", "aaaaaa", 0, false);
        test("((a){4,}){4,6}", "", "aaaaaaaaaaaaaaaa", 0, true, 0, 16, 12, 16, 15, 16);
        test("((a){4,}){4,6}", "", "aaaaaaaaaaaaaaaaa", 0, true, 0, 17, 13, 17, 16, 17);
        test("((a){4,}){4,6}", "", "aaaaaaaaaaaaaaaaaaaa", 0, true, 0, 20, 16, 20, 19, 20);
        test("((a){4,}){4,6}", "", "aaaaaaaaaaaaaaaaaaaaaaaa", 0, true, 0, 24, 20, 24, 23, 24);
        test("((a){4,}){4,6}", "", "aaaaaaaaaaaaaaaaaaaaaaaaa", 0, true, 0, 25, 21, 25, 24, 25);
        test("(.)\\1{2,}", "", "billiam", 0, false);
        test("(^_(a{1,2}[:])*a{1,2}[:]a{1,2}([.]a{1,4})?_)+", "", "_a:a:a.aaa_", 0, true, 0, 11, 0, 11, 1, 3, 6, 10);
        test("(a{2}|())+$", "", "aaaa", 0, true, 0, 4, 4, 4, 4, 4);
        test("^a(b*)\\1{4,6}?", "", "abbbb", 0, true, 0, 1, 1, 1);
        test("^a(b*)\\1{4,6}?", "", "abbbbb", 0, true, 0, 6, 1, 2);
        test("(?<=|$)", "", "a", 0, true, 1, 1, 1, 1);
        test("(?=ab)a", "", "ab", 0, false);
        test("(?=()|^)|x", "", "empty", 0, true, 0, 0, 0, 0, -1, -1);
        test("a(?<=ba)", "", "ba", 0, false);
        expectSyntaxError("(?<=(?<=a)[])", "i", "empty", 0, ErrorCode.UnmatchedBracket);
        test("(?<=(?=|()))", "", "aa", 0, false);
        test("\\d\\W", "i", "4\u017f", 0, false);
        test("[\u08bc-\ucf3a]", "i", "\u03b0", 0, false);
        test("[\u0450-\u6c50]\u7e57\u55ad()\u64e7\\d|", "i", "\u03b0\u7e57\u55ad\u64e79", 0, true, 0, 0, -1, -1);
        test("(?<=(?<=a)b^c)c", "", "abcc", 0, false);
        test("a(?:|()\\1){1,2}", "", "a", 0, false);
        test("|(?<\\d\\1)\ub7e4", "", "error", 0, true, 0, 0, -1, -1);
        test("[a-z][a-z\u2028\u2029].|ab(?<=[a-z]w.)", "", "aac", 0, true, 0, 3, -1, -1);
        test("(animation|animation-name)", "", "animation", 0, true, 0, 9, 0, 9);
        test("(a|){7,7}b", "", "aaab", 0, true, 0, 4, 3, 3);
        test("(a|){7,7}?b", "", "aaab", 0, true, 0, 4, 3, 3);
        test("(|a){7,7}b", "", "aaab", 0, true, 0, 4, 3, 3);
        test("(|a){7,7}?b", "", "aaab", 0, true, 0, 4, 3, 3);
        test("(a||b){7,7}c", "", "aaabc", 0, true, 0, 5, 4, 4);
        test("(a||b){7,7}c", "", "aaac", 0, true, 0, 4, 3, 3);
        test("(a||b){7,7}c", "", "aaabac", 0, true, 0, 6, 5, 5);
        test("($|a){7,7}", "", "aaa", 0, true, 0, 3, 3, 3);
        test("($|a){7,7}?", "", "aaa", 0, true, 0, 3, 3, 3);
        test("(a|$){7,7}", "", "aaa", 0, true, 0, 3, 3, 3);
        test("(a|$){7,7}?", "", "aaa", 0, true, 0, 3, 3, 3);
        test("(a|$|b){7,7}", "", "aaab", 0, true, 0, 4, 4, 4);
        test("(a|$|b){7,7}", "", "aaa", 0, true, 0, 3, 3, 3);
        test("(a|$|b){7,7}", "", "aaaba", 0, true, 0, 5, 5, 5);
        test("((?=a)|a){7,7}b", "", "aaa", 0, false);
        test("((?=[ab])|a){7,7}b", "", "aaab", 0, false);
        test("((?<=a)|a){7,7}b", "", "aaab", 0, false);
        test("a((?<=a)|a){7,7}b", "", "aaab", 0, false);
        test("(a|){0,7}b", "", "aaab", 0, true, 0, 4, 3, 3);
        test("(a|){0,7}?b", "", "aaab", 0, true, 0, 4, 2, 3);
        test("(|a){0,7}b", "", "aaab", 0, true, 0, 4, 3, 3);
        test("(|a){0,7}?b", "", "aaab", 0, true, 0, 4, 2, 3);
        test("(a||b){0,7}c", "", "aaabc", 0, true, 0, 5, 4, 4);
        test("(a||b){0,7}c", "", "aaac", 0, true, 0, 4, 3, 3);
        test("(a||b){0,7}c", "", "aaabac", 0, true, 0, 6, 5, 5);
        test("((?=a)|a){0,7}b", "", "aaab", 0, true, 0, 4, 2, 3, -1, -1);
        test("((?=[ab])|a){0,7}b", "", "aaab", 0, true, 0, 4, 2, 3, -1, -1);
        test("((?<=a)|a){0,7}b", "", "aaab", 0, true, 0, 4, 2, 3, -1, -1);
        test("a((?<=a)|a){0,7}b", "", "aaab", 0, true, 0, 4, 2, 3, -1, -1);
        test("(a*?){11,11}?b", "", "aaaaaaaaaaaaaaaaaaaaaaaaab", 0, true, 0, 26, 10, 25);
        test("\\w(?<=\\W([l-w]{0,19}?){1,2}\\w)\\2\ua2d2\\1\\z", "", "[qowwllu3\u0002\ua2d2qowwlluz", 0, false);
        test("(?:a(b{0,19})c)", "", "abbbbbbbcdebbbbbbbf", 0, false);
        test("(?:a(b{0,19})c)de", "", "abbbbbbbcdebbbbbbbf", 0, false);
        test("(?<=a(b{0,19})c)de", "", "abbbbbbbcdebbbbbbbf", 0, false);
        test("(?<=a(b{0,19}){1,2}c)de", "", "abbbbbbbcdebbbbbbbf", 0, false);
        test("(?<=a(b{0,19}){2,2}c)de", "", "abbbbbbbcdebbbbbbbf", 0, false);
        test("c(?<=a(b{0,19}){1,2}c)de\\1f", "", "abbbbbbbcdebbbbbbbf", 0, false);
        test("[\ud0d9](?<=\\S)", "", "\ud0d9", 0, false);
        test("[\ud0d9](?<=\\W)", "", "\ud0d9", 0, false);
        test("\u0895(?<=\\S)", "", "\u0895", 0, false);
        test("\u0895(?<=\\W)", "", "\u0895", 0, false);
        test("[\u8053](?<=\\S)", "", "\u8053", 0, false);
        test("[\u8053](?<=\\W)", "", "\u8053", 0, false);
        test("\u0895(?<=\\S)", "", "\u0895", 0, false);
        test("\u0895(?<=\\W)", "", "\u0895", 0, false);
        test("\u0895|[\u8053\ud0d9]+(?<=\\S\\W\\S)", "", "\ud0d9\ud0d9\ud0d9\ud0d9", 0, false);
        test("\u0895|[\u8053\ud0d9]+(?<=\\S\\W\\S)", "", "\ud0d9\ud0d9\ud0d9\ud0d9", 0, false);
        test("\u0895|[\u8053\ud0d9]+(?<=\\S\\W\\S)", "", "\ud0d9\ud0d9\ud0d9\ud0d9", 0, false);
        test("a|[bc]+(?<=[abc][abcd][abc])", "", "bbbb", 0, false);
        test("a(b*)*c\\1d", "", "abbbbcbbd", 0, true, 0, 9, 3, 5);
        test("(|a)||b(?<=cde)|", "", "a", 0, true, 0, 0, 0, 0, -1, -1);
        test("^(\\1)?\\D*", "", "empty", 0, true, 0, 5, -1, -1);
        test("abcd(?<=d|c()d)", "", "_abcd", 0, false);
        test("\\Dw\u3aa7\\A\\S(?<=\ue3b3|\\A()\\S)", "", "\udad1\udcfaw\u3aa7A\ue3b3", 0, false);
        test("a(?:c|b(?=()))*", "", "abc", 0, true, 0, 1, -1, -1, -1, -1, -1, -1);
        test("a(?:c|b(?=(c)))*", "", "abc", 0, true, 0, 1, -1, -1, -1, -1, -1, -1);
        test("a(?:c|(?<=(a))b)*", "", "abc", 0, true, 0, 1, -1, -1, -1, -1, -1, -1);
        test("(a||b){15,18}c", "", "ababaabbaaac", 0, true, 0, 12, 11, 11);
        test("(a||b){15,18}?c", "", "ababaabbaaac", 0, true, 0, 12, 11, 11);
        test("(?:ab|c|^){103,104}", "", "abcababccabccabababccabcababcccccabcababababccccabcabcabccabcabcccabababccabababcababababccababccabcababcabcabccabababccccabcab", 0, true, 0, 0, 0, 0);
        test("((?<=a)bec)*d", "", "abecd", 0, true, 4, 5, -1, -1, -1, -1);
        test("(|(^|\\z){2,77}?)?", "", "empty", 0, true, 0, 0, 0, 0, -1, -1);
        test("a(|a{15,36}){10,11}", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", 0, true, 0, 1, 1, 1);
        test("a(|a{15,36}?){10,11}", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", 0, true, 0, 1, 1, 1);
        test("a(|a{15,36}){10,11}$", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", 0, true, 0, 66, 66, 66);
        test("a(|a{15,36}?){10,11}b$", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaab", 0, true, 0, 67, 66, 66);
        test("(?:a()|b??){22,26}c", "", "aabbbaabaaaaaabaaaac", 0, true, 19, 20, 19, 19, -1, -1);
        test("b()(a\\1|){4,4}\\2c", "", "baaaac", 0, false);
        test("a((?=b()|)[a-d])+", "", "abbbcbd", 0, true, 0, 7, 6, 7, 6, 6, -1, -1);
        test("(\ud848\ude53??){7,9}?\\Z", "m", "\ud848\ude53", 0, true, 0, 4, 4, 4);
        test("(|[[.\ud974\udd29.][.\udb03\ude46.]luk-u]){7,7}?$", "", "pr", 0, true, 0, 2, 2, 2);
        test("\\Z?(|l||){4,7}?(x)", "", "lx", 0, true, 0, 2, 1, 1, 1, 2);
        test("\udbe5\udea1\\S\uda61\udf1d(|[\udb7e\ude23]){7,9}\\z", "", "\udbe5\udea1\u18e7\uda61\udf1d\udb7e\ude23\udb7e\ude23\udb7e\ude23\udb7e\ude23", 0, true, 0, 27, 27, 27);
        test("(||j\\Z){5,7}?\\Z+?", "", "jj\n", 0, true, 1, 2, 2, 2);
        test("(|[i-r]){6,7}\\Z", "", "lnijlr", 0, true, 0, 6, 6, 6);
        test("(^|q){5,10}?(\udbb0\udc21)", "", "qqqq\udbb0\udc21", 0, false);
        test("(^|b){6,18}?$", "", "bbbbb", 0, false);
        test("\\A(?|[\ud985\udc7a-\udb1b\ude21]){6,10}?\\z|\\z", "", "\ud9dc\ude93\uda9e\udfd2\udaac\udeb8\uda44\uded1", 0, true, 0, 16, 16, 16);
        test("\uda5b\ude5d(|+?[g-u]){4,8}?\\z", "m", "\uda5b\ude5dsng", 0, true, 0, 7, 7, 7);
        test("(s|^){5,10}", "", "sss", 0, true, 0, 0, 0, 0);
        test("(\\w*?){3,7}?\\Z", "", "\u2148\ud834\udeda\ua887\u0c83\ud806\ude14\u0f35\ud802\udea9\u2102\ud804\udccb\u1234\u0fc6\ud802\udf9f\u3033\ud806\udece\ud802\udd94\ud806\udca4\ud802\udf62\n",
                        0, true, 41, 60, 56, 60);
        test("x$\\S\ud9e9\udddd??\\d|(\\S|^){6,7}|\\d|^?", "", "x\u2019\ud9e9\udddd\ud807\udc53", 0, true, 0, 0, 0, 0);
        test("(\\w*?){3,7}?\\z", "", "\u1994\ud834\udd28\u2414\ud807\udc6e\u0386\ud834\udd8b\u240e\u218d\u321f\u0889\u077f\ud802\ude4f\u1b18\ud807\udc6d\ua6f8\ua69c\ud835\udf58\u168f", 0, true, 48,
                        58, 55, 58);
        test("s()|^(|\\d){6,8}?\\Z", "m", "\n\ua8d1\ua9d6\u1a810\n", 0, true, 1, 11, -1, -1, 11, 11);
        test("(|\\d){7,8}\\z", "", "\u2186\u2188\u0dec", 0, true, 0, 9, 9, 9);
        test("\\A(?|()[r-v]+){6,10}?\\z()??|(+)$", "", "surutrtsuurtvsrssrsvsrvsrvr", 0, true, 0, 27, 27, 27, 0, 0, -1, -1, -1, -1);
        test("\\A(?|()[\udabe\udd1c]){6,10}?\\z|\\z", "", "\udabe\udd1c\udabe\udd1c\udabe\udd1c\udabe\udd1c", 0, true, 0, 16, 16, 16, 12, 12);
        test("(^\udab7\ude3e*){5,10}", "",
                        "\udab7\ude3e\udab7\ude3e\udab7\ude3e\udab7\ude3e\udab7\ude3e\udab7\ude3e\udab7\ude3e\udab7\ude3e\udab7\ude3e\udab7\ude3e\udab7\ude3e\udab7\ude3e\udab7\ude3e", 0, true, 0, 0,
                        0, 0);
        test("\\W(|[\ud9a5\udc2b]\ud982\udf73){6,8}\\z|", "", "\ud805\udc5b\ud9a5\udc2b\ud982\udf73\ud9a5\udc2b\ud982\udf73\ud9a5\udc2b\ud982\udf73", 0, true, 0, 28, 28, 28);
        test("\\W(|\uda5a\udd73){6,7}?\\Z|\\A|", "", "\ua721\uda5a\udd73\uda5a\udd73\uda5a\udd73\uda5a\udd73\n", 0, true, 0, 19, 19, 19);
        test("(|y){3,8}?e", "ci", "Ye", 0, true, 0, 2, 1, 1);
        test("\\W(|\uda5a\udd73){6,7}?\\Z||()", "", "\ud835\udf4f\uda5a\udd73\uda5a\udd73\uda5a\udd73\uda5a\udd73", 0, true, 0, 20, 20, 20, -1, -1);
        test("\\W(|\uda5a\udd73){6,7}?()\\Z|\\A|", "", "\ud804\udde3\uda5a\udd73\uda5a\udd73\n", 0, true, 0, 12, 12, 12, 12, 12);
        test("(\ud848\ude53??){6,7}?\\Z", "", "\ud848\ude53\ud848\ude53\ud848\ude53\n", 0, true, 0, 12, 12, 12);
        test("(\\S\\w|\\A){4,7}||o\udab0\udfb4", "", "empty", 0, true, 0, 0, 0, 0);
        test("[[:cntrl:]\ub4e2-\udaee\udc05]a[b[:xdigit:]]{70,70}\\S", "",
                        "F\uff41aF\u061cae\uff19\uff10fa\uff22BCAfb\uff14\uff11b3baAF\uff214D\uff21\uff25bA2\uff45F\uff415f\uff18C9aDb\uff41\uff11c\uff23f\uff46\uff22\uff45deb\uff26bb0\uff44bb7E4\uff44F\uff4507\uff43\uff42\uff21b\uff15\uff14\u16fc",
                        0, true, 6, 134);
        test("((\\1)a|)\\1\\Z\\s\\1|\\1$\\1a\\1\\1\\2a\\1[[.a.][=b=]]()\\1", "", "a", 0, false);
        test("[[.\uda67\ude18.][=\udaa4\udeb8=]]\udb9d\ude50(\\w\ud815\udf1f(\\1)\\D|)\\1\\Z\\s\\2|\\1$\\1\\w\\1\\1\\2\\D\\1[[.\udae5\ude94.][=\uda47\udf02=]]()\\1", "",
                        "\udae6\udc76\udb98\udfb6\udb8e\udd53\u2284\uda1f\udc8f\ud9f6\ude95\uda46\udf3b\udbe6\udeab\ud95e\udc35\uda0d\udfd9\ud8d5\udc90\ud9c6\udd8a\uda3f\ude7c\udad7\udffe", 19,
                        false);
        test("(e?\\D[xg]){87,87}z", "",
                        "axaxeageagageaxeaxeaxageaxagageaxeaxagageagaxaxeagaxeaxagagaxeagageaxeaxeagageaxeaxagaxaxaxageageagageagaxaxaxageaxageaxeageaxaxaxaxaxagaxagageaxeageageageaxeaxeaxageaxaxeaxeagaxagageaxeageaxeaxaxeaxageaxaxeagaxageageaz",
                        0, false);
        test("((b\\2{1400,1400})+|)*a", "", "a", 0, true, 0, 1, 0, 0, -1, -1);
        test("\\S(\\w?\\W){8,9}\\Z", "", "-a---------  ---------", 0, true, 13, 22, 21, 22);
        test("(a{1100,1100})\\1", "i", "a".repeat(2400), 0, true, 0, 2200, 0, 1100);
        test("[a]\\S{213,213}bcdz", "", "a".repeat(215) + ("bcxd" + "a".repeat(213)).repeat(3), 0, false);

        /* GENERATED CODE END - KEEP THIS MARKER FOR AUTOMATIC UPDATES */
    }

    @Test
    public void gr52933() {
        test("\\z{1,2}a", "", "a", 0, false);
        test("\\z{1,3}a", "", "a", 0, false);
        test("\\z{1,4}a", "", "a", 0, false);
        test("\\z{1,5}a", "", "a", 0, false);
        test("\\z{1,6}a", "", "a", 0, false);
        test("\\z{1,7}a", "", "a", 0, false);
    }

    @Test
    public void testForceLinearExecution() {
        test("(a*)b\\1", "", "_aabaaa_", 0, true, 1, 6, 1, 3);
        expectUnsupported("(a*)b\\1", "", OPT_FORCE_LINEAR_EXECUTION);
        test(".*a{1,65534}.*", "", "_aabaaa_", 0, true, 0, 8);
        expectUnsupported(".*a{1,65534}.*", "", OPT_FORCE_LINEAR_EXECUTION);
    }

    @Test
    public void orcl38190286() {
        test("[[:alpha:]]", "", "\ufffd", 0, true, 0, 3);
        test("[[:alpha:]]", "", "\uD839", 0, false);
        test("[[:alpha:]]", "", "\uDDF2", 0, false);
        test("[[:alpha:]]", "", "\uD839\uDDF2", 0, false);
        test("[[:alpha:]]", "", Collections.emptyMap(), Encodings.UTF_16, "\ufffd", 0, true, 0, 1);
        test("[[:alpha:]]", "", Collections.emptyMap(), Encodings.UTF_16, "\uD839", 0, false);
        test("[[:alpha:]]", "", Collections.emptyMap(), Encodings.UTF_16, "\uDDF2", 0, false);
        test("[[:alpha:]]", "", Collections.emptyMap(), Encodings.UTF_16, "\uD839\uDDF2", 0, false);
    }

    @Test
    public void bqTransitionExplosion() {
        test("(a(b(b(b(b(b(b(b(b(b(b(b(b(b(b(b(b(b(b(b|)|)|)|)|)|)|)|)|)|)|)|)|)|)|)|)|)|)|){2,2}c)de", "", Map.of("regexDummyLang.QuantifierUnrollLimitGroup", "1"),
                        "abbbbbbbcdebbbbbbbf", 0, true, 0, 11, 0, 9, 8, 8, 2, 8, 3, 8, 4, 8, 5, 8, 6, 8, 7, 8, 8, 8);
    }

    @Test
    public void testNestedQuantifierBailout() {
        expectUnsupported("()?*");
        expectUnsupported("()?*|");
        expectUnsupported("()?*||");
        expectUnsupported("()?*||a");
        expectUnsupported("(a)???");
        expectUnsupported("(a?)*??");
        expectUnsupported("(a???)");
        expectUnsupported("a*??");
        expectUnsupported("a+*?");
        expectUnsupported("a+*??");
        expectUnsupported("a++?");
        expectUnsupported("a+??");
        expectUnsupported("a?*?");
        expectUnsupported("a?*??");
        expectUnsupported("a?+");
        expectUnsupported("a?+?");
        expectUnsupported("a?+??");
        expectUnsupported("a??+");
        expectUnsupported("a???");
        expectUnsupported("a??{0,1}");
        expectUnsupported("a{0,1}??");
        expectUnsupported("a{0,1}?{0,1}");
        expectUnsupported("()?*||^a\\Zb");
        expectUnsupported("\\D|++?");
        expectUnsupported("\\D|++?^");
        expectUnsupported("(\\d)|5+*?|[[:lower:]][[=l=]]^%");
        expectUnsupported("\\S|\\D|++?^(3)");
        expectUnsupported("\\S|\\D|++?^((3)|[R-_\\(/])t[[:alnum:]]c");
        expectUnsupported("x???");
        expectUnsupported("x????");
        expectUnsupported("x?????");
        expectUnsupported("x??????");
        expectUnsupported("x{2}*");
        expectUnsupported("x{2}*?");
        expectUnsupported("x{2}*??");
        expectUnsupported("x{2}*???");
        expectUnsupported("x{2}+");
        expectUnsupported("x{2}??");
    }
}
