/*
 * Copyright (c) 2023, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.regex.flavor.oracledb;

import org.graalvm.collections.EconomicMap;

import com.oracle.truffle.regex.charset.CodePointSet;

public final class OracleDBConstants {

    // This map contains the character sets of POSIX character classes like [[:alpha:]] and
    // [[:punct:]].
    public static final EconomicMap<String, CodePointSet> POSIX_CHAR_CLASSES = EconomicMap.create(12);
    // \w
    static final CodePointSet WORD_CHARACTERS;

    static {

        /* GENERATED CODE BEGIN - KEEP THIS MARKER FOR AUTOMATIC UPDATES */

        POSIX_CHAR_CLASSES.put("alpha", CodePointSet.createNoDedup(
                        0x000041, 0x00005a, 0x000061, 0x00007a, 0x0000aa, 0x0000aa, 0x0000b5, 0x0000b5, 0x0000ba, 0x0000ba, 0x0000c0, 0x0000d6, 0x0000d8, 0x0000f6, 0x0000f8, 0x0002b8, 0x0002bb,
                        0x0002c1, 0x0002c7, 0x0002c7, 0x0002c9, 0x0002cb, 0x0002ce, 0x0002ce, 0x0002d1, 0x0002d1, 0x0002d9, 0x0002d9, 0x0002e0, 0x0002e4, 0x0002ec, 0x0002ec, 0x0002ee, 0x0002ee,
                        0x000300, 0x000374, 0x000376, 0x000377, 0x00037a, 0x00037d, 0x00037f, 0x00037f, 0x000386, 0x000386, 0x000388, 0x00038a, 0x00038c, 0x00038c, 0x00038e, 0x0003a1, 0x0003a3,
                        0x0003f5, 0x0003f7, 0x000481, 0x000483, 0x00052f, 0x000531, 0x000556, 0x000559, 0x000559, 0x000560, 0x000588, 0x000591, 0x0005af, 0x0005ba, 0x0005ba, 0x0005c5, 0x0005c5,
                        0x0005c7, 0x0005c7, 0x0005d0, 0x0005ea, 0x0005ef, 0x0005f2, 0x000610, 0x00061a, 0x000620, 0x00065f, 0x00066e, 0x0006d3, 0x0006d5, 0x0006dc, 0x0006df, 0x0006e8, 0x0006ea,
                        0x0006ef, 0x0006fa, 0x0006fc, 0x0006ff, 0x0006ff, 0x000710, 0x00074a, 0x00074d, 0x0007b1, 0x0007ca, 0x0007f5, 0x0007fa, 0x0007fa, 0x0007fd, 0x0007fd, 0x000800, 0x00082d,
                        0x000840, 0x00085b, 0x000860, 0x00086a, 0x000870, 0x000887, 0x000889, 0x00088e, 0x000898, 0x0008e1, 0x0008e3, 0x000900, 0x000904, 0x00093b, 0x00094e, 0x000963, 0x000971,
                        0x000983, 0x000985, 0x00098c, 0x00098f, 0x000990, 0x000993, 0x0009a8, 0x0009aa, 0x0009b0, 0x0009b2, 0x0009b2, 0x0009b6, 0x0009b9, 0x0009bc, 0x0009c4, 0x0009c7, 0x0009c8,
                        0x0009cb, 0x0009ce, 0x0009d7, 0x0009d7, 0x0009dc, 0x0009dd, 0x0009df, 0x0009e3, 0x0009f0, 0x0009f1, 0x0009fc, 0x0009fc, 0x0009fe, 0x0009fe, 0x000a01, 0x000a03, 0x000a05,
                        0x000a0a, 0x000a0f, 0x000a10, 0x000a13, 0x000a28, 0x000a2a, 0x000a30, 0x000a32, 0x000a33, 0x000a35, 0x000a36, 0x000a38, 0x000a39, 0x000a3c, 0x000a3c, 0x000a3e, 0x000a42,
                        0x000a47, 0x000a48, 0x000a4b, 0x000a4d, 0x000a51, 0x000a51, 0x000a59, 0x000a5c, 0x000a5e, 0x000a5e, 0x000a70, 0x000a75, 0x000a81, 0x000a83, 0x000a85, 0x000a8d, 0x000a8f,
                        0x000a91, 0x000a93, 0x000aa8, 0x000aaa, 0x000ab0, 0x000ab2, 0x000ab3, 0x000ab5, 0x000ab9, 0x000abc, 0x000ac5, 0x000ac7, 0x000ac9, 0x000acb, 0x000acd, 0x000ad0, 0x000ad0,
                        0x000ae0, 0x000ae3, 0x000af9, 0x000aff, 0x000b01, 0x000b03, 0x000b05, 0x000b0c, 0x000b0f, 0x000b10, 0x000b13, 0x000b28, 0x000b2a, 0x000b30, 0x000b32, 0x000b33, 0x000b35,
                        0x000b39, 0x000b3c, 0x000b44, 0x000b47, 0x000b48, 0x000b4b, 0x000b4d, 0x000b55, 0x000b57, 0x000b5c, 0x000b5d, 0x000b5f, 0x000b63, 0x000b71, 0x000b71, 0x000b82, 0x000b83,
                        0x000b85, 0x000b8a, 0x000b8e, 0x000b90, 0x000b92, 0x000b95, 0x000b99, 0x000b9a, 0x000b9c, 0x000b9c, 0x000b9e, 0x000b9f, 0x000ba3, 0x000ba4, 0x000ba8, 0x000baa, 0x000bae,
                        0x000bb9, 0x000bbe, 0x000bc2, 0x000bc6, 0x000bc8, 0x000bca, 0x000bcd, 0x000bd0, 0x000bd0, 0x000bd7, 0x000bd7, 0x000c00, 0x000c0c, 0x000c0e, 0x000c10, 0x000c12, 0x000c28,
                        0x000c2a, 0x000c39, 0x000c3c, 0x000c44, 0x000c46, 0x000c48, 0x000c4a, 0x000c4d, 0x000c55, 0x000c56, 0x000c58, 0x000c5a, 0x000c5d, 0x000c5d, 0x000c60, 0x000c63, 0x000c80,
                        0x000c83, 0x000c85, 0x000c8c, 0x000c8e, 0x000c90, 0x000c92, 0x000ca8, 0x000caa, 0x000cb3, 0x000cb5, 0x000cb9, 0x000cbc, 0x000cc4, 0x000cc6, 0x000cc8, 0x000cca, 0x000ccd,
                        0x000cd5, 0x000cd6, 0x000cdd, 0x000cde, 0x000ce0, 0x000ce3, 0x000cf1, 0x000cf3, 0x000d00, 0x000d0c, 0x000d0e, 0x000d10, 0x000d12, 0x000d44, 0x000d46, 0x000d48, 0x000d4a,
                        0x000d4e, 0x000d54, 0x000d57, 0x000d5f, 0x000d63, 0x000d7a, 0x000d7f, 0x000d81, 0x000d83, 0x000d85, 0x000d96, 0x000d9a, 0x000db1, 0x000db3, 0x000dbb, 0x000dbd, 0x000dbd,
                        0x000dc0, 0x000dc6, 0x000dca, 0x000dca, 0x000dcf, 0x000dd4, 0x000dd6, 0x000dd6, 0x000dd8, 0x000ddf, 0x000df2, 0x000df3, 0x000e01, 0x000e2e, 0x000e30, 0x000e3a, 0x000e40,
                        0x000e45, 0x000e47, 0x000e4d, 0x000e81, 0x000e82, 0x000e84, 0x000e84, 0x000e86, 0x000e8a, 0x000e8c, 0x000ea3, 0x000ea5, 0x000ea5, 0x000ea7, 0x000ebd, 0x000ec0, 0x000ec4,
                        0x000ec6, 0x000ec6, 0x000ec8, 0x000ece, 0x000edc, 0x000edf, 0x000f00, 0x000f00, 0x000f18, 0x000f19, 0x000f35, 0x000f35, 0x000f37, 0x000f37, 0x000f39, 0x000f39, 0x000f3e,
                        0x000f47, 0x000f49, 0x000f6c, 0x000f71, 0x000f84, 0x000f86, 0x000f97, 0x000f99, 0x000fbc, 0x000fc6, 0x000fc6, 0x001000, 0x00103f, 0x001050, 0x00108f, 0x00109a, 0x00109d,
                        0x0010a0, 0x0010c5, 0x0010c7, 0x0010c7, 0x0010cd, 0x0010cd, 0x0010d0, 0x0010fa, 0x0010fc, 0x001248, 0x00124a, 0x00124d, 0x001250, 0x001256, 0x001258, 0x001258, 0x00125a,
                        0x00125d, 0x001260, 0x001288, 0x00128a, 0x00128d, 0x001290, 0x0012b0, 0x0012b2, 0x0012b5, 0x0012b8, 0x0012be, 0x0012c0, 0x0012c0, 0x0012c2, 0x0012c5, 0x0012c8, 0x0012d6,
                        0x0012d8, 0x001310, 0x001312, 0x001315, 0x001318, 0x00135a, 0x00135d, 0x00135f, 0x001380, 0x00138f, 0x0013a0, 0x0013f5, 0x0013f8, 0x0013fd, 0x001401, 0x00166c, 0x00166f,
                        0x00167f, 0x001681, 0x00169a, 0x0016a0, 0x0016ea, 0x0016ee, 0x0016f8, 0x001700, 0x001715, 0x00171f, 0x001734, 0x001740, 0x001753, 0x001760, 0x00176c, 0x00176e, 0x001770,
                        0x001772, 0x001773, 0x001780, 0x0017d3, 0x0017d7, 0x0017d7, 0x0017dc, 0x0017dd, 0x00180b, 0x00180d, 0x00180f, 0x00180f, 0x001820, 0x001878, 0x001880, 0x0018aa, 0x0018b0,
                        0x0018f5, 0x001900, 0x00191e, 0x001920, 0x00192b, 0x001930, 0x00193b, 0x001950, 0x00196d, 0x001970, 0x001974, 0x001980, 0x0019ab, 0x0019b0, 0x0019c9, 0x001a00, 0x001a1b,
                        0x001a20, 0x001a5e, 0x001a60, 0x001a7c, 0x001a7f, 0x001a7f, 0x001aa7, 0x001aa7, 0x001ab0, 0x001ace, 0x001b00, 0x001b4c, 0x001b6b, 0x001b73, 0x001b80, 0x001baf, 0x001bba,
                        0x001bf3, 0x001c00, 0x001c37, 0x001c4d, 0x001c4f, 0x001c5a, 0x001c7d, 0x001c80, 0x001c88, 0x001c90, 0x001cba, 0x001cbd, 0x001cbf, 0x001cd0, 0x001cd2, 0x001cd4, 0x001cfa,
                        0x001d00, 0x001f15, 0x001f18, 0x001f1d, 0x001f20, 0x001f45, 0x001f48, 0x001f4d, 0x001f50, 0x001f57, 0x001f59, 0x001f59, 0x001f5b, 0x001f5b, 0x001f5d, 0x001f5d, 0x001f5f,
                        0x001f7d, 0x001f80, 0x001fb4, 0x001fb6, 0x001fbc, 0x001fbe, 0x001fbe, 0x001fc2, 0x001fc4, 0x001fc6, 0x001fcc, 0x001fd0, 0x001fd3, 0x001fd6, 0x001fdb, 0x001fe0, 0x001fec,
                        0x001ff2, 0x001ff4, 0x001ff6, 0x001ffc, 0x002015, 0x002015, 0x002071, 0x002071, 0x002090, 0x00209c, 0x0020d0, 0x0020f0, 0x002102, 0x002102, 0x002107, 0x002107, 0x00210a,
                        0x002113, 0x002115, 0x002115, 0x002119, 0x00211d, 0x002124, 0x002124, 0x002128, 0x002128, 0x00212a, 0x00212a, 0x00212c, 0x00212d, 0x00212f, 0x002139, 0x00213c, 0x00213f,
                        0x002145, 0x002149, 0x00214e, 0x00214e, 0x002160, 0x002188, 0x002400, 0x002421, 0x002460, 0x002487, 0x002c00, 0x002ce4, 0x002ceb, 0x002cf3, 0x002d00, 0x002d25, 0x002d27,
                        0x002d27, 0x002d2d, 0x002d2d, 0x002d30, 0x002d67, 0x002d6f, 0x002d6f, 0x002d7f, 0x002d96, 0x002da0, 0x002da6, 0x002da8, 0x002dae, 0x002db0, 0x002db6, 0x002db8, 0x002dbe,
                        0x002dc0, 0x002dc6, 0x002dc8, 0x002dce, 0x002dd0, 0x002dd6, 0x002dd8, 0x002dde, 0x002de0, 0x002dff, 0x002e2f, 0x002e2f, 0x002e80, 0x002e99, 0x002e9b, 0x002ef3, 0x002f00,
                        0x002fd5, 0x003003, 0x003003, 0x003005, 0x003007, 0x00301c, 0x00301c, 0x003021, 0x00302f, 0x003031, 0x003035, 0x003038, 0x00303c, 0x003041, 0x003096, 0x003099, 0x00309a,
                        0x00309d, 0x00309f, 0x0030a1, 0x0030fa, 0x0030fc, 0x0030ff, 0x003105, 0x00312f, 0x003131, 0x00318e, 0x003192, 0x0031bf, 0x0031f0, 0x0031ff, 0x003400, 0x004dbf, 0x004e00,
                        0x00a48c, 0x00a4d0, 0x00a4fd, 0x00a500, 0x00a60c, 0x00a610, 0x00a61f, 0x00a62a, 0x00a62b, 0x00a640, 0x00a672, 0x00a674, 0x00a67d, 0x00a67f, 0x00a6f1, 0x00a717, 0x00a71f,
                        0x00a722, 0x00a788, 0x00a78b, 0x00a7ca, 0x00a7d0, 0x00a7d1, 0x00a7d3, 0x00a7d3, 0x00a7d5, 0x00a7d9, 0x00a7f2, 0x00a827, 0x00a82c, 0x00a82c, 0x00a840, 0x00a873, 0x00a880,
                        0x00a8c5, 0x00a8e0, 0x00a8f7, 0x00a8fb, 0x00a8fb, 0x00a8fd, 0x00a8ff, 0x00a90a, 0x00a92d, 0x00a930, 0x00a953, 0x00a960, 0x00a97c, 0x00a980, 0x00a9c0, 0x00a9cf, 0x00a9cf,
                        0x00a9e0, 0x00a9ef, 0x00a9fa, 0x00a9fe, 0x00aa00, 0x00aa36, 0x00aa40, 0x00aa4d, 0x00aa60, 0x00aa76, 0x00aa7a, 0x00aac2, 0x00aadb, 0x00aadd, 0x00aae0, 0x00aaef, 0x00aaf2,
                        0x00aaf6, 0x00ab01, 0x00ab06, 0x00ab09, 0x00ab0e, 0x00ab11, 0x00ab16, 0x00ab20, 0x00ab26, 0x00ab28, 0x00ab2e, 0x00ab30, 0x00ab5a, 0x00ab5c, 0x00ab69, 0x00ab70, 0x00abea,
                        0x00abec, 0x00abed, 0x00ac00, 0x00d7a3, 0x00d7b0, 0x00d7c6, 0x00d7cb, 0x00d7fb, 0x00e000, 0x00fa6d, 0x00fa70, 0x00fad9, 0x00fb00, 0x00fb06, 0x00fb13, 0x00fb17, 0x00fb1d,
                        0x00fb1e, 0x00fb20, 0x00fb28, 0x00fb2a, 0x00fb36, 0x00fb38, 0x00fb3c, 0x00fb3e, 0x00fb3e, 0x00fb40, 0x00fb41, 0x00fb43, 0x00fb44, 0x00fb46, 0x00fbb1, 0x00fbd3, 0x00fd3d,
                        0x00fd50, 0x00fd8f, 0x00fd92, 0x00fdc7, 0x00fdf0, 0x00fdfb, 0x00fe00, 0x00fe0f, 0x00fe20, 0x00fe2f, 0x00fe70, 0x00fe74, 0x00fe76, 0x00fefc, 0x00ff21, 0x00ff3a, 0x00ff3f,
                        0x00ff3f, 0x00ff41, 0x00ff5a, 0x00ff66, 0x00ffbe, 0x00ffc2, 0x00ffc7, 0x00ffca, 0x00ffcf, 0x00ffd2, 0x00ffd7, 0x00ffda, 0x00ffdc, 0x00fffd, 0x00fffd, 0x010000, 0x01000b,
                        0x01000d, 0x010026, 0x010028, 0x01003a, 0x01003c, 0x01003d, 0x01003f, 0x01004d, 0x010050, 0x01005d, 0x010080, 0x0100fa, 0x010140, 0x010174, 0x0101fd, 0x0101fd, 0x010280,
                        0x01029c, 0x0102a0, 0x0102d0, 0x0102e0, 0x0102e0, 0x010300, 0x01031f, 0x01032d, 0x01034a, 0x010350, 0x01037a, 0x010380, 0x01039d, 0x0103a0, 0x0103c3, 0x0103c8, 0x0103cf,
                        0x0103d1, 0x0103d5, 0x010400, 0x01049d, 0x0104b0, 0x0104d3, 0x0104d8, 0x0104fb, 0x010500, 0x010527, 0x010530, 0x010563, 0x010570, 0x01057a, 0x01057c, 0x01058a, 0x01058c,
                        0x010592, 0x010594, 0x010595, 0x010597, 0x0105a1, 0x0105a3, 0x0105b1, 0x0105b3, 0x0105b9, 0x0105bb, 0x0105bc, 0x010600, 0x010736, 0x010740, 0x010755, 0x010760, 0x010767,
                        0x010780, 0x010785, 0x010787, 0x0107b0, 0x0107b2, 0x0107ba, 0x010800, 0x010805, 0x010808, 0x010808, 0x01080a, 0x010835, 0x010837, 0x010838, 0x01083c, 0x01083c, 0x01083f,
                        0x010855, 0x010860, 0x010876, 0x010880, 0x01089e, 0x0108e0, 0x0108f2, 0x0108f4, 0x0108f5, 0x010900, 0x010915, 0x010920, 0x010939, 0x010980, 0x0109b7, 0x0109be, 0x0109bf,
                        0x010a00, 0x010a03, 0x010a05, 0x010a06, 0x010a0c, 0x010a13, 0x010a15, 0x010a17, 0x010a19, 0x010a35, 0x010a38, 0x010a3a, 0x010a3f, 0x010a3f, 0x010a60, 0x010a7c, 0x010a80,
                        0x010a9c, 0x010ac0, 0x010ac7, 0x010ac9, 0x010ae6, 0x010b00, 0x010b35, 0x010b40, 0x010b55, 0x010b60, 0x010b72, 0x010b80, 0x010b91, 0x010c00, 0x010c48, 0x010c80, 0x010cb2,
                        0x010cc0, 0x010cf2, 0x010d00, 0x010d27, 0x010e80, 0x010ea9, 0x010eab, 0x010eac, 0x010eb0, 0x010eb1, 0x010efd, 0x010f1c, 0x010f27, 0x010f27, 0x010f30, 0x010f50, 0x010f70,
                        0x010f85, 0x010fb0, 0x010fc4, 0x010fe0, 0x010ff6, 0x011000, 0x011046, 0x011070, 0x011075, 0x01107f, 0x0110ba, 0x0110c2, 0x0110c2, 0x0110d0, 0x0110e8, 0x011100, 0x011134,
                        0x011144, 0x011147, 0x011150, 0x011173, 0x011176, 0x011176, 0x011180, 0x0111c4, 0x0111c9, 0x0111cc, 0x0111ce, 0x0111cf, 0x0111da, 0x0111da, 0x0111dc, 0x0111dc, 0x011200,
                        0x011211, 0x011213, 0x011237, 0x01123e, 0x011241, 0x011280, 0x011286, 0x011288, 0x011288, 0x01128a, 0x01128d, 0x01128f, 0x01129d, 0x01129f, 0x0112a8, 0x0112b0, 0x0112ea,
                        0x011300, 0x011303, 0x011305, 0x01130c, 0x01130f, 0x011310, 0x011313, 0x011328, 0x01132a, 0x011330, 0x011332, 0x011333, 0x011335, 0x011339, 0x01133b, 0x011344, 0x011347,
                        0x011348, 0x01134b, 0x01134d, 0x011350, 0x011350, 0x011357, 0x011357, 0x01135d, 0x011363, 0x011366, 0x01136c, 0x011370, 0x011374, 0x011400, 0x01144a, 0x01145e, 0x011461,
                        0x011480, 0x0114c5, 0x0114c7, 0x0114c7, 0x011580, 0x0115b5, 0x0115b8, 0x0115c0, 0x0115d8, 0x0115dd, 0x011600, 0x011640, 0x011644, 0x011644, 0x011680, 0x0116b8, 0x011700,
                        0x01171a, 0x01171d, 0x01172b, 0x011740, 0x011746, 0x011800, 0x01183a, 0x0118a0, 0x0118df, 0x0118ff, 0x011906, 0x011909, 0x011909, 0x01190c, 0x011913, 0x011915, 0x011916,
                        0x011918, 0x011935, 0x011937, 0x011938, 0x01193b, 0x011943, 0x0119a0, 0x0119a7, 0x0119aa, 0x0119d7, 0x0119da, 0x0119e1, 0x0119e3, 0x0119e4, 0x011a00, 0x011a3e, 0x011a47,
                        0x011a47, 0x011a50, 0x011a99, 0x011a9d, 0x011a9d, 0x011ab0, 0x011af8, 0x011c00, 0x011c08, 0x011c0a, 0x011c36, 0x011c38, 0x011c40, 0x011c72, 0x011c8f, 0x011c92, 0x011ca7,
                        0x011ca9, 0x011cb6, 0x011d00, 0x011d06, 0x011d08, 0x011d09, 0x011d0b, 0x011d36, 0x011d3a, 0x011d3a, 0x011d3c, 0x011d3d, 0x011d3f, 0x011d47, 0x011d60, 0x011d65, 0x011d67,
                        0x011d68, 0x011d6a, 0x011d8e, 0x011d90, 0x011d91, 0x011d93, 0x011d98, 0x011ee0, 0x011ef6, 0x011f00, 0x011f10, 0x011f12, 0x011f3a, 0x011f3e, 0x011f42, 0x011fb0, 0x011fb0,
                        0x012000, 0x012399, 0x012400, 0x01246e, 0x012480, 0x012543, 0x012f90, 0x012ff0, 0x013000, 0x01342f, 0x013440, 0x013455, 0x014400, 0x014646, 0x016800, 0x016a38, 0x016a40,
                        0x016a5e, 0x016a70, 0x016abe, 0x016ad0, 0x016aed, 0x016af0, 0x016af4, 0x016b00, 0x016b36, 0x016b40, 0x016b43, 0x016b63, 0x016b77, 0x016b7d, 0x016b8f, 0x016e40, 0x016e7f,
                        0x016f00, 0x016f4a, 0x016f4f, 0x016f87, 0x016f8f, 0x016f9f, 0x016fe0, 0x016fe1, 0x016fe3, 0x016fe4, 0x016ff0, 0x016ff1, 0x017000, 0x0187f7, 0x018800, 0x018cd5, 0x018d00,
                        0x018d08, 0x01aff0, 0x01aff3, 0x01aff5, 0x01affb, 0x01affd, 0x01affe, 0x01b000, 0x01b122, 0x01b132, 0x01b132, 0x01b150, 0x01b152, 0x01b155, 0x01b155, 0x01b164, 0x01b167,
                        0x01b170, 0x01b2fb, 0x01bc00, 0x01bc6a, 0x01bc70, 0x01bc7c, 0x01bc80, 0x01bc88, 0x01bc90, 0x01bc99, 0x01bc9d, 0x01bc9e, 0x01cf00, 0x01cf2d, 0x01cf30, 0x01cf46, 0x01d165,
                        0x01d169, 0x01d16d, 0x01d172, 0x01d17b, 0x01d182, 0x01d185, 0x01d18b, 0x01d1aa, 0x01d1ad, 0x01d242, 0x01d244, 0x01d400, 0x01d454, 0x01d456, 0x01d49c, 0x01d49e, 0x01d49f,
                        0x01d4a2, 0x01d4a2, 0x01d4a5, 0x01d4a6, 0x01d4a9, 0x01d4ac, 0x01d4ae, 0x01d4b9, 0x01d4bb, 0x01d4bb, 0x01d4bd, 0x01d4c3, 0x01d4c5, 0x01d505, 0x01d507, 0x01d50a, 0x01d50d,
                        0x01d514, 0x01d516, 0x01d51c, 0x01d51e, 0x01d539, 0x01d53b, 0x01d53e, 0x01d540, 0x01d544, 0x01d546, 0x01d546, 0x01d54a, 0x01d550, 0x01d552, 0x01d6a5, 0x01d6a8, 0x01d6c0,
                        0x01d6c2, 0x01d6da, 0x01d6dc, 0x01d6fa, 0x01d6fc, 0x01d714, 0x01d716, 0x01d734, 0x01d736, 0x01d74e, 0x01d750, 0x01d76e, 0x01d770, 0x01d788, 0x01d78a, 0x01d7a8, 0x01d7aa,
                        0x01d7c2, 0x01d7c4, 0x01d7cb, 0x01da00, 0x01da36, 0x01da3b, 0x01da6c, 0x01da75, 0x01da75, 0x01da84, 0x01da84, 0x01da9b, 0x01da9f, 0x01daa1, 0x01daaf, 0x01df00, 0x01df1e,
                        0x01df25, 0x01df2a, 0x01e000, 0x01e006, 0x01e008, 0x01e018, 0x01e01b, 0x01e021, 0x01e023, 0x01e024, 0x01e026, 0x01e02a, 0x01e030, 0x01e06d, 0x01e08f, 0x01e08f, 0x01e100,
                        0x01e12c, 0x01e130, 0x01e13d, 0x01e14e, 0x01e14e, 0x01e290, 0x01e2ae, 0x01e2c0, 0x01e2ef, 0x01e4d0, 0x01e4ef, 0x01e7e0, 0x01e7e6, 0x01e7e8, 0x01e7eb, 0x01e7ed, 0x01e7ee,
                        0x01e7f0, 0x01e7fe, 0x01e800, 0x01e8c4, 0x01e8d0, 0x01e8d6, 0x01e900, 0x01e94b, 0x01ee00, 0x01ee03, 0x01ee05, 0x01ee1f, 0x01ee21, 0x01ee22, 0x01ee24, 0x01ee24, 0x01ee27,
                        0x01ee27, 0x01ee29, 0x01ee32, 0x01ee34, 0x01ee37, 0x01ee39, 0x01ee39, 0x01ee3b, 0x01ee3b, 0x01ee42, 0x01ee42, 0x01ee47, 0x01ee47, 0x01ee49, 0x01ee49, 0x01ee4b, 0x01ee4b,
                        0x01ee4d, 0x01ee4f, 0x01ee51, 0x01ee52, 0x01ee54, 0x01ee54, 0x01ee57, 0x01ee57, 0x01ee59, 0x01ee59, 0x01ee5b, 0x01ee5b, 0x01ee5d, 0x01ee5d, 0x01ee5f, 0x01ee5f, 0x01ee61,
                        0x01ee62, 0x01ee64, 0x01ee64, 0x01ee67, 0x01ee6a, 0x01ee6c, 0x01ee72, 0x01ee74, 0x01ee77, 0x01ee79, 0x01ee7c, 0x01ee7e, 0x01ee7e, 0x01ee80, 0x01ee89, 0x01ee8b, 0x01ee9b,
                        0x01eea1, 0x01eea3, 0x01eea5, 0x01eea9, 0x01eeab, 0x01eebb, 0x020000, 0x02a6df, 0x02a700, 0x02b739, 0x02b740, 0x02b81d, 0x02b820, 0x02cea1, 0x02ceb0, 0x02ebe0, 0x02f800,
                        0x02fa1d, 0x030000, 0x03134a, 0x031350, 0x0323af, 0x0e0100, 0x0e01ef, 0x0f0000, 0x0ffffd, 0x100000, 0x10fffd));

        POSIX_CHAR_CLASSES.put("blank", CodePointSet.createNoDedup(
                        0x000020, 0x000020, 0x001680, 0x001680, 0x002000, 0x00200a, 0x002028, 0x002029, 0x00202f, 0x00202f, 0x00205f, 0x00205f, 0x003000, 0x003000));

        POSIX_CHAR_CLASSES.put("cntrl", CodePointSet.createNoDedup(
                        0x000001, 0x00001f, 0x00007f, 0x00009f, 0x0000ad, 0x0000ad, 0x000600, 0x000605, 0x00061c, 0x00061c, 0x0006dd, 0x0006dd, 0x00070f, 0x00070f, 0x000890, 0x000891, 0x0008e2,
                        0x0008e2, 0x00180e, 0x00180e, 0x00200b, 0x00200f, 0x00202a, 0x00202e, 0x002060, 0x002064, 0x002066, 0x00206f, 0x00feff, 0x00feff, 0x00fff9, 0x00fffb, 0x0110bd, 0x0110bd,
                        0x0110cd, 0x0110cd, 0x013430, 0x01343f, 0x01bca0, 0x01bca3, 0x01d173, 0x01d17a, 0x0e0001, 0x0e0001, 0x0e0020, 0x0e007f));

        POSIX_CHAR_CLASSES.put("digit", CodePointSet.createNoDedup(
                        0x000030, 0x000039, 0x000660, 0x000669, 0x0006f0, 0x0006f9, 0x0007c0, 0x0007c9, 0x000966, 0x00096f, 0x0009e6, 0x0009ef, 0x000a66, 0x000a6f, 0x000ae6, 0x000aef, 0x000b66,
                        0x000b6f, 0x000be6, 0x000bef, 0x000c66, 0x000c6f, 0x000ce6, 0x000cef, 0x000d66, 0x000d6f, 0x000de6, 0x000def, 0x000e50, 0x000e59, 0x000ed0, 0x000ed9, 0x000f20, 0x000f29,
                        0x001040, 0x001049, 0x001090, 0x001099, 0x0016ee, 0x0016f0, 0x0017e0, 0x0017e9, 0x001810, 0x001819, 0x001946, 0x00194f, 0x0019d0, 0x0019d9, 0x001a80, 0x001a89, 0x001a90,
                        0x001a99, 0x001b50, 0x001b59, 0x001bb0, 0x001bb9, 0x001c40, 0x001c49, 0x001c50, 0x001c59, 0x002160, 0x002182, 0x002185, 0x002188, 0x003007, 0x003007, 0x003021, 0x003029,
                        0x003038, 0x00303a, 0x00a620, 0x00a629, 0x00a6e6, 0x00a6ef, 0x00a8d0, 0x00a8d9, 0x00a900, 0x00a909, 0x00a9d0, 0x00a9d9, 0x00a9f0, 0x00a9f9, 0x00aa50, 0x00aa59, 0x00abf0,
                        0x00abf9, 0x00ff10, 0x00ff19, 0x010140, 0x010174, 0x010341, 0x010341, 0x01034a, 0x01034a, 0x0103d1, 0x0103d5, 0x0104a0, 0x0104a9, 0x010d30, 0x010d39, 0x011066, 0x01106f,
                        0x0110f0, 0x0110f9, 0x011136, 0x01113f, 0x0111d0, 0x0111d9, 0x0112f0, 0x0112f9, 0x011450, 0x011459, 0x0114d0, 0x0114d9, 0x011650, 0x011659, 0x0116c0, 0x0116c9, 0x011730,
                        0x011739, 0x0118e0, 0x0118e9, 0x011950, 0x011959, 0x011c50, 0x011c59, 0x011d50, 0x011d59, 0x011da0, 0x011da9, 0x011f50, 0x011f59, 0x012400, 0x01246e, 0x016a60, 0x016a69,
                        0x016ac0, 0x016ac9, 0x016b50, 0x016b59, 0x01d7ce, 0x01d7ff, 0x01e140, 0x01e149, 0x01e2f0, 0x01e2f9, 0x01e4f0, 0x01e4f9, 0x01e950, 0x01e959, 0x01fbf0, 0x01fbf9));

        POSIX_CHAR_CLASSES.put("graph", CodePointSet.createNoDedup(
                        0x000021, 0x00007e, 0x0000a0, 0x0000ac, 0x0000ae, 0x0000b1, 0x0000b4, 0x0000b8, 0x0000ba, 0x0000bb, 0x0000bf, 0x000377, 0x00037a, 0x00037f, 0x000384, 0x00038a, 0x00038c,
                        0x00038c, 0x00038e, 0x0003a1, 0x0003a3, 0x00052f, 0x000531, 0x000556, 0x000559, 0x00058a, 0x00058d, 0x00058f, 0x000591, 0x0005c7, 0x0005d0, 0x0005ea, 0x0005ef, 0x0005f4,
                        0x000606, 0x00061b, 0x00061d, 0x0006dc, 0x0006de, 0x00070d, 0x000710, 0x00074a, 0x00074d, 0x0007b1, 0x0007c0, 0x0007fa, 0x0007fd, 0x00082d, 0x000830, 0x00083e, 0x000840,
                        0x00085b, 0x00085e, 0x00085e, 0x000860, 0x00086a, 0x000870, 0x00088e, 0x000898, 0x0008e1, 0x0008e3, 0x000983, 0x000985, 0x00098c, 0x00098f, 0x000990, 0x000993, 0x0009a8,
                        0x0009aa, 0x0009b0, 0x0009b2, 0x0009b2, 0x0009b6, 0x0009b9, 0x0009bc, 0x0009c4, 0x0009c7, 0x0009c8, 0x0009cb, 0x0009ce, 0x0009d7, 0x0009d7, 0x0009dc, 0x0009dd, 0x0009df,
                        0x0009e3, 0x0009e6, 0x0009f3, 0x0009fa, 0x0009fe, 0x000a01, 0x000a03, 0x000a05, 0x000a0a, 0x000a0f, 0x000a10, 0x000a13, 0x000a28, 0x000a2a, 0x000a30, 0x000a32, 0x000a33,
                        0x000a35, 0x000a36, 0x000a38, 0x000a39, 0x000a3c, 0x000a3c, 0x000a3e, 0x000a42, 0x000a47, 0x000a48, 0x000a4b, 0x000a4d, 0x000a51, 0x000a51, 0x000a59, 0x000a5c, 0x000a5e,
                        0x000a5e, 0x000a66, 0x000a76, 0x000a81, 0x000a83, 0x000a85, 0x000a8d, 0x000a8f, 0x000a91, 0x000a93, 0x000aa8, 0x000aaa, 0x000ab0, 0x000ab2, 0x000ab3, 0x000ab5, 0x000ab9,
                        0x000abc, 0x000ac5, 0x000ac7, 0x000ac9, 0x000acb, 0x000acd, 0x000ad0, 0x000ad0, 0x000ae0, 0x000ae3, 0x000ae6, 0x000af1, 0x000af9, 0x000aff, 0x000b01, 0x000b03, 0x000b05,
                        0x000b0c, 0x000b0f, 0x000b10, 0x000b13, 0x000b28, 0x000b2a, 0x000b30, 0x000b32, 0x000b33, 0x000b35, 0x000b39, 0x000b3c, 0x000b44, 0x000b47, 0x000b48, 0x000b4b, 0x000b4d,
                        0x000b55, 0x000b57, 0x000b5c, 0x000b5d, 0x000b5f, 0x000b63, 0x000b66, 0x000b71, 0x000b82, 0x000b83, 0x000b85, 0x000b8a, 0x000b8e, 0x000b90, 0x000b92, 0x000b95, 0x000b99,
                        0x000b9a, 0x000b9c, 0x000b9c, 0x000b9e, 0x000b9f, 0x000ba3, 0x000ba4, 0x000ba8, 0x000baa, 0x000bae, 0x000bb9, 0x000bbe, 0x000bc2, 0x000bc6, 0x000bc8, 0x000bca, 0x000bcd,
                        0x000bd0, 0x000bd0, 0x000bd7, 0x000bd7, 0x000be6, 0x000bef, 0x000bf3, 0x000bfa, 0x000c00, 0x000c0c, 0x000c0e, 0x000c10, 0x000c12, 0x000c28, 0x000c2a, 0x000c39, 0x000c3c,
                        0x000c44, 0x000c46, 0x000c48, 0x000c4a, 0x000c4d, 0x000c55, 0x000c56, 0x000c58, 0x000c5a, 0x000c5d, 0x000c5d, 0x000c60, 0x000c63, 0x000c66, 0x000c6f, 0x000c77, 0x000c77,
                        0x000c7f, 0x000c8c, 0x000c8e, 0x000c90, 0x000c92, 0x000ca8, 0x000caa, 0x000cb3, 0x000cb5, 0x000cb9, 0x000cbc, 0x000cc4, 0x000cc6, 0x000cc8, 0x000cca, 0x000ccd, 0x000cd5,
                        0x000cd6, 0x000cdd, 0x000cde, 0x000ce0, 0x000ce3, 0x000ce6, 0x000cef, 0x000cf1, 0x000cf3, 0x000d00, 0x000d0c, 0x000d0e, 0x000d10, 0x000d12, 0x000d44, 0x000d46, 0x000d48,
                        0x000d4a, 0x000d4f, 0x000d54, 0x000d57, 0x000d5f, 0x000d63, 0x000d66, 0x000d6f, 0x000d79, 0x000d7f, 0x000d81, 0x000d83, 0x000d85, 0x000d96, 0x000d9a, 0x000db1, 0x000db3,
                        0x000dbb, 0x000dbd, 0x000dbd, 0x000dc0, 0x000dc6, 0x000dca, 0x000dca, 0x000dcf, 0x000dd4, 0x000dd6, 0x000dd6, 0x000dd8, 0x000ddf, 0x000de6, 0x000def, 0x000df2, 0x000df4,
                        0x000e01, 0x000e3a, 0x000e3f, 0x000e5b, 0x000e81, 0x000e82, 0x000e84, 0x000e84, 0x000e86, 0x000e8a, 0x000e8c, 0x000ea3, 0x000ea5, 0x000ea5, 0x000ea7, 0x000ebd, 0x000ec0,
                        0x000ec4, 0x000ec6, 0x000ec6, 0x000ec8, 0x000ece, 0x000ed0, 0x000ed9, 0x000edc, 0x000edf, 0x000f00, 0x000f29, 0x000f34, 0x000f47, 0x000f49, 0x000f6c, 0x000f71, 0x000f97,
                        0x000f99, 0x000fbc, 0x000fbe, 0x000fcc, 0x000fce, 0x000fda, 0x001000, 0x0010c5, 0x0010c7, 0x0010c7, 0x0010cd, 0x0010cd, 0x0010d0, 0x001248, 0x00124a, 0x00124d, 0x001250,
                        0x001256, 0x001258, 0x001258, 0x00125a, 0x00125d, 0x001260, 0x001288, 0x00128a, 0x00128d, 0x001290, 0x0012b0, 0x0012b2, 0x0012b5, 0x0012b8, 0x0012be, 0x0012c0, 0x0012c0,
                        0x0012c2, 0x0012c5, 0x0012c8, 0x0012d6, 0x0012d8, 0x001310, 0x001312, 0x001315, 0x001318, 0x00135a, 0x00135d, 0x001368, 0x001380, 0x001399, 0x0013a0, 0x0013f5, 0x0013f8,
                        0x0013fd, 0x001400, 0x00167f, 0x001681, 0x00169c, 0x0016a0, 0x0016f8, 0x001700, 0x001715, 0x00171f, 0x001736, 0x001740, 0x001753, 0x001760, 0x00176c, 0x00176e, 0x001770,
                        0x001772, 0x001773, 0x001780, 0x0017dd, 0x0017e0, 0x0017e9, 0x001800, 0x00180d, 0x00180f, 0x001819, 0x001820, 0x001878, 0x001880, 0x0018aa, 0x0018b0, 0x0018f5, 0x001900,
                        0x00191e, 0x001920, 0x00192b, 0x001930, 0x00193b, 0x001940, 0x001940, 0x001944, 0x00196d, 0x001970, 0x001974, 0x001980, 0x0019ab, 0x0019b0, 0x0019c9, 0x0019d0, 0x0019d9,
                        0x0019de, 0x001a1b, 0x001a1e, 0x001a5e, 0x001a60, 0x001a7c, 0x001a7f, 0x001a89, 0x001a90, 0x001a99, 0x001aa0, 0x001aad, 0x001ab0, 0x001ace, 0x001b00, 0x001b4c, 0x001b50,
                        0x001b7e, 0x001b80, 0x001bf3, 0x001bfc, 0x001c37, 0x001c3b, 0x001c49, 0x001c4d, 0x001c88, 0x001c90, 0x001cba, 0x001cbd, 0x001cc7, 0x001cd0, 0x001cfa, 0x001d00, 0x001f15,
                        0x001f18, 0x001f1d, 0x001f20, 0x001f45, 0x001f48, 0x001f4d, 0x001f50, 0x001f57, 0x001f59, 0x001f59, 0x001f5b, 0x001f5b, 0x001f5d, 0x001f5d, 0x001f5f, 0x001f7d, 0x001f80,
                        0x001fb4, 0x001fb6, 0x001fc4, 0x001fc6, 0x001fd3, 0x001fd6, 0x001fdb, 0x001fdd, 0x001fef, 0x001ff2, 0x001ff4, 0x001ff6, 0x001ffe, 0x002010, 0x002027, 0x002030, 0x00205e,
                        0x002071, 0x002071, 0x00207a, 0x00207f, 0x00208a, 0x00208e, 0x002090, 0x00209c, 0x0020a0, 0x0020c0, 0x0020d0, 0x0020f0, 0x002100, 0x00212a, 0x00212c, 0x00214f, 0x002160,
                        0x002188, 0x00218a, 0x00218b, 0x002190, 0x002426, 0x002440, 0x00244a, 0x002460, 0x002487, 0x00249c, 0x0024e9, 0x002500, 0x002775, 0x002794, 0x002b73, 0x002b76, 0x002b95,
                        0x002b97, 0x002cf3, 0x002cf9, 0x002cfc, 0x002cfe, 0x002d25, 0x002d27, 0x002d27, 0x002d2d, 0x002d2d, 0x002d30, 0x002d67, 0x002d6f, 0x002d70, 0x002d7f, 0x002d96, 0x002da0,
                        0x002da6, 0x002da8, 0x002dae, 0x002db0, 0x002db6, 0x002db8, 0x002dbe, 0x002dc0, 0x002dc6, 0x002dc8, 0x002dce, 0x002dd0, 0x002dd6, 0x002dd8, 0x002dde, 0x002de0, 0x002e5d,
                        0x002e80, 0x002e99, 0x002e9b, 0x002ef3, 0x002f00, 0x002fd5, 0x002ff0, 0x002ffb, 0x003001, 0x00303f, 0x003041, 0x003096, 0x003099, 0x0030ff, 0x003105, 0x00312f, 0x003131,
                        0x00318e, 0x003190, 0x0031e3, 0x0031f0, 0x00321e, 0x00322a, 0x003247, 0x003250, 0x003250, 0x003260, 0x00327f, 0x00328a, 0x0032b0, 0x0032c0, 0x00a48c, 0x00a490, 0x00a4c6,
                        0x00a4d0, 0x00a62b, 0x00a640, 0x00a6f7, 0x00a700, 0x00a7ca, 0x00a7d0, 0x00a7d1, 0x00a7d3, 0x00a7d3, 0x00a7d5, 0x00a7d9, 0x00a7f2, 0x00a82c, 0x00a836, 0x00a839, 0x00a840,
                        0x00a877, 0x00a880, 0x00a8c5, 0x00a8ce, 0x00a8d9, 0x00a8e0, 0x00a953, 0x00a95f, 0x00a97c, 0x00a980, 0x00a9cd, 0x00a9cf, 0x00a9d9, 0x00a9de, 0x00a9fe, 0x00aa00, 0x00aa36,
                        0x00aa40, 0x00aa4d, 0x00aa50, 0x00aa59, 0x00aa5c, 0x00aac2, 0x00aadb, 0x00aaf6, 0x00ab01, 0x00ab06, 0x00ab09, 0x00ab0e, 0x00ab11, 0x00ab16, 0x00ab20, 0x00ab26, 0x00ab28,
                        0x00ab2e, 0x00ab30, 0x00ab6b, 0x00ab70, 0x00abed, 0x00abf0, 0x00abf9, 0x00ac00, 0x00d7a3, 0x00d7b0, 0x00d7c6, 0x00d7cb, 0x00d7fb, 0x00e000, 0x00fa6d, 0x00fa70, 0x00fad9,
                        0x00fb00, 0x00fb06, 0x00fb13, 0x00fb17, 0x00fb1d, 0x00fb36, 0x00fb38, 0x00fb3c, 0x00fb3e, 0x00fb3e, 0x00fb40, 0x00fb41, 0x00fb43, 0x00fb44, 0x00fb46, 0x00fbc2, 0x00fbd3,
                        0x00fd8f, 0x00fd92, 0x00fdc7, 0x00fdcf, 0x00fdcf, 0x00fdf0, 0x00fe19, 0x00fe20, 0x00fe52, 0x00fe54, 0x00fe66, 0x00fe68, 0x00fe6b, 0x00fe70, 0x00fe74, 0x00fe76, 0x00fefc,
                        0x00ff01, 0x00ffbe, 0x00ffc2, 0x00ffc7, 0x00ffca, 0x00ffcf, 0x00ffd2, 0x00ffd7, 0x00ffda, 0x00ffdc, 0x00ffe0, 0x00ffe6, 0x00ffe8, 0x00ffee, 0x00fffc, 0x00fffd, 0x010000,
                        0x01000b, 0x01000d, 0x010026, 0x010028, 0x01003a, 0x01003c, 0x01003d, 0x01003f, 0x01004d, 0x010050, 0x01005d, 0x010080, 0x0100fa, 0x010100, 0x010102, 0x010137, 0x010174,
                        0x010179, 0x010189, 0x01018c, 0x01018e, 0x010190, 0x01019c, 0x0101a0, 0x0101a0, 0x0101d0, 0x0101fd, 0x010280, 0x01029c, 0x0102a0, 0x0102d0, 0x0102e0, 0x0102e0, 0x010300,
                        0x01031f, 0x01032d, 0x01034a, 0x010350, 0x01037a, 0x010380, 0x01039d, 0x01039f, 0x0103c3, 0x0103c8, 0x0103d5, 0x010400, 0x01049d, 0x0104a0, 0x0104a9, 0x0104b0, 0x0104d3,
                        0x0104d8, 0x0104fb, 0x010500, 0x010527, 0x010530, 0x010563, 0x01056f, 0x01057a, 0x01057c, 0x01058a, 0x01058c, 0x010592, 0x010594, 0x010595, 0x010597, 0x0105a1, 0x0105a3,
                        0x0105b1, 0x0105b3, 0x0105b9, 0x0105bb, 0x0105bc, 0x010600, 0x010736, 0x010740, 0x010755, 0x010760, 0x010767, 0x010780, 0x010785, 0x010787, 0x0107b0, 0x0107b2, 0x0107ba,
                        0x010800, 0x010805, 0x010808, 0x010808, 0x01080a, 0x010835, 0x010837, 0x010838, 0x01083c, 0x01083c, 0x01083f, 0x010855, 0x010857, 0x010857, 0x010860, 0x010878, 0x010880,
                        0x01089e, 0x0108e0, 0x0108f2, 0x0108f4, 0x0108f5, 0x010900, 0x010915, 0x01091f, 0x010939, 0x01093f, 0x01093f, 0x010980, 0x0109b7, 0x0109be, 0x0109bf, 0x010a00, 0x010a03,
                        0x010a05, 0x010a06, 0x010a0c, 0x010a13, 0x010a15, 0x010a17, 0x010a19, 0x010a35, 0x010a38, 0x010a3a, 0x010a3f, 0x010a3f, 0x010a50, 0x010a58, 0x010a60, 0x010a7c, 0x010a7f,
                        0x010a9c, 0x010ac0, 0x010ae6, 0x010af0, 0x010af6, 0x010b00, 0x010b35, 0x010b39, 0x010b55, 0x010b60, 0x010b72, 0x010b80, 0x010b91, 0x010b99, 0x010b9c, 0x010c00, 0x010c48,
                        0x010c80, 0x010cb2, 0x010cc0, 0x010cf2, 0x010d00, 0x010d27, 0x010d30, 0x010d39, 0x010e80, 0x010ea9, 0x010eab, 0x010ead, 0x010eb0, 0x010eb1, 0x010efd, 0x010f1c, 0x010f27,
                        0x010f27, 0x010f30, 0x010f50, 0x010f55, 0x010f59, 0x010f70, 0x010f89, 0x010fb0, 0x010fc4, 0x010fe0, 0x010ff6, 0x011000, 0x01104d, 0x011066, 0x011075, 0x01107f, 0x0110bc,
                        0x0110be, 0x0110c2, 0x0110d0, 0x0110e8, 0x0110f0, 0x0110f9, 0x011100, 0x011134, 0x011136, 0x011147, 0x011150, 0x011176, 0x011180, 0x0111df, 0x011200, 0x011211, 0x011213,
                        0x011241, 0x011280, 0x011286, 0x011288, 0x011288, 0x01128a, 0x01128d, 0x01128f, 0x01129d, 0x01129f, 0x0112a9, 0x0112b0, 0x0112ea, 0x0112f0, 0x0112f9, 0x011300, 0x011303,
                        0x011305, 0x01130c, 0x01130f, 0x011310, 0x011313, 0x011328, 0x01132a, 0x011330, 0x011332, 0x011333, 0x011335, 0x011339, 0x01133b, 0x011344, 0x011347, 0x011348, 0x01134b,
                        0x01134d, 0x011350, 0x011350, 0x011357, 0x011357, 0x01135d, 0x011363, 0x011366, 0x01136c, 0x011370, 0x011374, 0x011400, 0x01145b, 0x01145d, 0x011461, 0x011480, 0x0114c7,
                        0x0114d0, 0x0114d9, 0x011580, 0x0115b5, 0x0115b8, 0x0115dd, 0x011600, 0x011644, 0x011650, 0x011659, 0x011660, 0x01166c, 0x011680, 0x0116b9, 0x0116c0, 0x0116c9, 0x011700,
                        0x01171a, 0x01171d, 0x01172b, 0x011730, 0x011739, 0x01173c, 0x011746, 0x011800, 0x01183b, 0x0118a0, 0x0118e9, 0x0118ff, 0x011906, 0x011909, 0x011909, 0x01190c, 0x011913,
                        0x011915, 0x011916, 0x011918, 0x011935, 0x011937, 0x011938, 0x01193b, 0x011946, 0x011950, 0x011959, 0x0119a0, 0x0119a7, 0x0119aa, 0x0119d7, 0x0119da, 0x0119e4, 0x011a00,
                        0x011a47, 0x011a50, 0x011aa2, 0x011ab0, 0x011af8, 0x011b00, 0x011b09, 0x011c00, 0x011c08, 0x011c0a, 0x011c36, 0x011c38, 0x011c45, 0x011c50, 0x011c59, 0x011c70, 0x011c8f,
                        0x011c92, 0x011ca7, 0x011ca9, 0x011cb6, 0x011d00, 0x011d06, 0x011d08, 0x011d09, 0x011d0b, 0x011d36, 0x011d3a, 0x011d3a, 0x011d3c, 0x011d3d, 0x011d3f, 0x011d47, 0x011d50,
                        0x011d59, 0x011d60, 0x011d65, 0x011d67, 0x011d68, 0x011d6a, 0x011d8e, 0x011d90, 0x011d91, 0x011d93, 0x011d98, 0x011da0, 0x011da9, 0x011ee0, 0x011ef8, 0x011f00, 0x011f10,
                        0x011f12, 0x011f3a, 0x011f3e, 0x011f59, 0x011fb0, 0x011fb0, 0x011fd5, 0x011ff1, 0x011fff, 0x012399, 0x012400, 0x01246e, 0x012470, 0x012474, 0x012480, 0x012543, 0x012f90,
                        0x012ff2, 0x013000, 0x01342f, 0x013440, 0x013455, 0x014400, 0x014646, 0x016800, 0x016a38, 0x016a40, 0x016a5e, 0x016a60, 0x016a69, 0x016a6e, 0x016abe, 0x016ac0, 0x016ac9,
                        0x016ad0, 0x016aed, 0x016af0, 0x016af5, 0x016b00, 0x016b45, 0x016b50, 0x016b59, 0x016b63, 0x016b77, 0x016b7d, 0x016b8f, 0x016e40, 0x016e7f, 0x016e97, 0x016e9a, 0x016f00,
                        0x016f4a, 0x016f4f, 0x016f87, 0x016f8f, 0x016f9f, 0x016fe0, 0x016fe4, 0x016ff0, 0x016ff1, 0x017000, 0x0187f7, 0x018800, 0x018cd5, 0x018d00, 0x018d08, 0x01aff0, 0x01aff3,
                        0x01aff5, 0x01affb, 0x01affd, 0x01affe, 0x01b000, 0x01b122, 0x01b132, 0x01b132, 0x01b150, 0x01b152, 0x01b155, 0x01b155, 0x01b164, 0x01b167, 0x01b170, 0x01b2fb, 0x01bc00,
                        0x01bc6a, 0x01bc70, 0x01bc7c, 0x01bc80, 0x01bc88, 0x01bc90, 0x01bc99, 0x01bc9c, 0x01bc9f, 0x01cf00, 0x01cf2d, 0x01cf30, 0x01cf46, 0x01cf50, 0x01cfc3, 0x01d000, 0x01d0f5,
                        0x01d100, 0x01d126, 0x01d129, 0x01d172, 0x01d17b, 0x01d1ea, 0x01d200, 0x01d245, 0x01d300, 0x01d356, 0x01d400, 0x01d454, 0x01d456, 0x01d49c, 0x01d49e, 0x01d49f, 0x01d4a2,
                        0x01d4a2, 0x01d4a5, 0x01d4a6, 0x01d4a9, 0x01d4ac, 0x01d4ae, 0x01d4b9, 0x01d4bb, 0x01d4bb, 0x01d4bd, 0x01d4c3, 0x01d4c5, 0x01d505, 0x01d507, 0x01d50a, 0x01d50d, 0x01d514,
                        0x01d516, 0x01d51c, 0x01d51e, 0x01d539, 0x01d53b, 0x01d53e, 0x01d540, 0x01d544, 0x01d546, 0x01d546, 0x01d54a, 0x01d550, 0x01d552, 0x01d6a5, 0x01d6a8, 0x01d7cb, 0x01d7ce,
                        0x01da8b, 0x01da9b, 0x01da9f, 0x01daa1, 0x01daaf, 0x01df00, 0x01df1e, 0x01df25, 0x01df2a, 0x01e000, 0x01e006, 0x01e008, 0x01e018, 0x01e01b, 0x01e021, 0x01e023, 0x01e024,
                        0x01e026, 0x01e02a, 0x01e030, 0x01e06d, 0x01e08f, 0x01e08f, 0x01e100, 0x01e12c, 0x01e130, 0x01e13d, 0x01e140, 0x01e149, 0x01e14e, 0x01e14f, 0x01e290, 0x01e2ae, 0x01e2c0,
                        0x01e2f9, 0x01e2ff, 0x01e2ff, 0x01e4d0, 0x01e4f9, 0x01e7e0, 0x01e7e6, 0x01e7e8, 0x01e7eb, 0x01e7ed, 0x01e7ee, 0x01e7f0, 0x01e7fe, 0x01e800, 0x01e8c4, 0x01e8d0, 0x01e8d6,
                        0x01e900, 0x01e94b, 0x01e950, 0x01e959, 0x01e95e, 0x01e95f, 0x01ecac, 0x01ecac, 0x01ecb0, 0x01ecb0, 0x01ed2e, 0x01ed2e, 0x01ee00, 0x01ee03, 0x01ee05, 0x01ee1f, 0x01ee21,
                        0x01ee22, 0x01ee24, 0x01ee24, 0x01ee27, 0x01ee27, 0x01ee29, 0x01ee32, 0x01ee34, 0x01ee37, 0x01ee39, 0x01ee39, 0x01ee3b, 0x01ee3b, 0x01ee42, 0x01ee42, 0x01ee47, 0x01ee47,
                        0x01ee49, 0x01ee49, 0x01ee4b, 0x01ee4b, 0x01ee4d, 0x01ee4f, 0x01ee51, 0x01ee52, 0x01ee54, 0x01ee54, 0x01ee57, 0x01ee57, 0x01ee59, 0x01ee59, 0x01ee5b, 0x01ee5b, 0x01ee5d,
                        0x01ee5d, 0x01ee5f, 0x01ee5f, 0x01ee61, 0x01ee62, 0x01ee64, 0x01ee64, 0x01ee67, 0x01ee6a, 0x01ee6c, 0x01ee72, 0x01ee74, 0x01ee77, 0x01ee79, 0x01ee7c, 0x01ee7e, 0x01ee7e,
                        0x01ee80, 0x01ee89, 0x01ee8b, 0x01ee9b, 0x01eea1, 0x01eea3, 0x01eea5, 0x01eea9, 0x01eeab, 0x01eebb, 0x01eef0, 0x01eef1, 0x01f000, 0x01f02b, 0x01f030, 0x01f093, 0x01f0a0,
                        0x01f0ae, 0x01f0b1, 0x01f0bf, 0x01f0c1, 0x01f0cf, 0x01f0d1, 0x01f0f5, 0x01f10d, 0x01f1ad, 0x01f1e6, 0x01f202, 0x01f210, 0x01f23b, 0x01f240, 0x01f248, 0x01f250, 0x01f251,
                        0x01f260, 0x01f265, 0x01f300, 0x01f6d7, 0x01f6dc, 0x01f6ec, 0x01f6f0, 0x01f6fc, 0x01f700, 0x01f776, 0x01f77b, 0x01f7d9, 0x01f7e0, 0x01f7eb, 0x01f7f0, 0x01f7f0, 0x01f800,
                        0x01f80b, 0x01f810, 0x01f847, 0x01f850, 0x01f859, 0x01f860, 0x01f887, 0x01f890, 0x01f8ad, 0x01f8b0, 0x01f8b1, 0x01f900, 0x01fa53, 0x01fa60, 0x01fa6d, 0x01fa70, 0x01fa7c,
                        0x01fa80, 0x01fa88, 0x01fa90, 0x01fabd, 0x01fabf, 0x01fac5, 0x01face, 0x01fadb, 0x01fae0, 0x01fae8, 0x01faf0, 0x01faf8, 0x01fb00, 0x01fb92, 0x01fb94, 0x01fbca, 0x01fbf0,
                        0x01fbf9, 0x020000, 0x02a6df, 0x02a700, 0x02b739, 0x02b740, 0x02b81d, 0x02b820, 0x02cea1, 0x02ceb0, 0x02ebe0, 0x02f800, 0x02fa1d, 0x030000, 0x03134a, 0x031350, 0x0323af,
                        0x0e0100, 0x0e01ef, 0x0f0000, 0x0ffffd, 0x100000, 0x10fffd));

        POSIX_CHAR_CLASSES.put("lower", CodePointSet.createNoDedup(
                        0x000061, 0x00007a, 0x0000b5, 0x0000b5, 0x0000df, 0x0000f6, 0x0000f8, 0x0000ff, 0x000101, 0x000101, 0x000103, 0x000103, 0x000105, 0x000105, 0x000107, 0x000107, 0x000109,
                        0x000109, 0x00010b, 0x00010b, 0x00010d, 0x00010d, 0x00010f, 0x00010f, 0x000111, 0x000111, 0x000113, 0x000113, 0x000115, 0x000115, 0x000117, 0x000117, 0x000119, 0x000119,
                        0x00011b, 0x00011b, 0x00011d, 0x00011d, 0x00011f, 0x00011f, 0x000121, 0x000121, 0x000123, 0x000123, 0x000125, 0x000125, 0x000127, 0x000127, 0x000129, 0x000129, 0x00012b,
                        0x00012b, 0x00012d, 0x00012d, 0x00012f, 0x00012f, 0x000131, 0x000131, 0x000133, 0x000133, 0x000135, 0x000135, 0x000137, 0x000138, 0x00013a, 0x00013a, 0x00013c, 0x00013c,
                        0x00013e, 0x00013e, 0x000140, 0x000140, 0x000142, 0x000142, 0x000144, 0x000144, 0x000146, 0x000146, 0x000148, 0x000149, 0x00014b, 0x00014b, 0x00014d, 0x00014d, 0x00014f,
                        0x00014f, 0x000151, 0x000151, 0x000153, 0x000153, 0x000155, 0x000155, 0x000157, 0x000157, 0x000159, 0x000159, 0x00015b, 0x00015b, 0x00015d, 0x00015d, 0x00015f, 0x00015f,
                        0x000161, 0x000161, 0x000163, 0x000163, 0x000165, 0x000165, 0x000167, 0x000167, 0x000169, 0x000169, 0x00016b, 0x00016b, 0x00016d, 0x00016d, 0x00016f, 0x00016f, 0x000171,
                        0x000171, 0x000173, 0x000173, 0x000175, 0x000175, 0x000177, 0x000177, 0x00017a, 0x00017a, 0x00017c, 0x00017c, 0x00017e, 0x000180, 0x000183, 0x000183, 0x000185, 0x000185,
                        0x000188, 0x000188, 0x00018c, 0x00018d, 0x000192, 0x000192, 0x000195, 0x000195, 0x000199, 0x00019b, 0x00019e, 0x00019e, 0x0001a1, 0x0001a1, 0x0001a3, 0x0001a3, 0x0001a5,
                        0x0001a5, 0x0001a8, 0x0001a8, 0x0001aa, 0x0001ab, 0x0001ad, 0x0001ad, 0x0001b0, 0x0001b0, 0x0001b4, 0x0001b4, 0x0001b6, 0x0001b6, 0x0001b9, 0x0001ba, 0x0001bd, 0x0001bf,
                        0x0001c5, 0x0001c6, 0x0001c8, 0x0001c9, 0x0001cb, 0x0001cc, 0x0001ce, 0x0001ce, 0x0001d0, 0x0001d0, 0x0001d2, 0x0001d2, 0x0001d4, 0x0001d4, 0x0001d6, 0x0001d6, 0x0001d8,
                        0x0001d8, 0x0001da, 0x0001da, 0x0001dc, 0x0001dd, 0x0001df, 0x0001df, 0x0001e1, 0x0001e1, 0x0001e3, 0x0001e3, 0x0001e5, 0x0001e5, 0x0001e7, 0x0001e7, 0x0001e9, 0x0001e9,
                        0x0001eb, 0x0001eb, 0x0001ed, 0x0001ed, 0x0001ef, 0x0001f0, 0x0001f2, 0x0001f3, 0x0001f5, 0x0001f5, 0x0001f9, 0x0001f9, 0x0001fb, 0x0001fb, 0x0001fd, 0x0001fd, 0x0001ff,
                        0x0001ff, 0x000201, 0x000201, 0x000203, 0x000203, 0x000205, 0x000205, 0x000207, 0x000207, 0x000209, 0x000209, 0x00020b, 0x00020b, 0x00020d, 0x00020d, 0x00020f, 0x00020f,
                        0x000211, 0x000211, 0x000213, 0x000213, 0x000215, 0x000215, 0x000217, 0x000217, 0x000219, 0x000219, 0x00021b, 0x00021b, 0x00021d, 0x00021d, 0x00021f, 0x00021f, 0x000221,
                        0x000221, 0x000223, 0x000223, 0x000225, 0x000225, 0x000227, 0x000227, 0x000229, 0x000229, 0x00022b, 0x00022b, 0x00022d, 0x00022d, 0x00022f, 0x00022f, 0x000231, 0x000231,
                        0x000233, 0x000239, 0x00023c, 0x00023c, 0x00023f, 0x000240, 0x000242, 0x000242, 0x000247, 0x000247, 0x000249, 0x000249, 0x00024b, 0x00024b, 0x00024d, 0x00024d, 0x00024f,
                        0x000293, 0x000295, 0x0002af, 0x000345, 0x000345, 0x000371, 0x000371, 0x000373, 0x000373, 0x000377, 0x000377, 0x00037b, 0x00037d, 0x000390, 0x000390, 0x0003ac, 0x0003ce,
                        0x0003d0, 0x0003d1, 0x0003d5, 0x0003d7, 0x0003d9, 0x0003d9, 0x0003db, 0x0003db, 0x0003dd, 0x0003dd, 0x0003df, 0x0003df, 0x0003e1, 0x0003e1, 0x0003e3, 0x0003e3, 0x0003e5,
                        0x0003e5, 0x0003e7, 0x0003e7, 0x0003e9, 0x0003e9, 0x0003eb, 0x0003eb, 0x0003ed, 0x0003ed, 0x0003ef, 0x0003f3, 0x0003f5, 0x0003f5, 0x0003f8, 0x0003f8, 0x0003fb, 0x0003fc,
                        0x000430, 0x00045f, 0x000461, 0x000461, 0x000463, 0x000463, 0x000465, 0x000465, 0x000467, 0x000467, 0x000469, 0x000469, 0x00046b, 0x00046b, 0x00046d, 0x00046d, 0x00046f,
                        0x00046f, 0x000471, 0x000471, 0x000473, 0x000473, 0x000475, 0x000475, 0x000477, 0x000477, 0x000479, 0x000479, 0x00047b, 0x00047b, 0x00047d, 0x00047d, 0x00047f, 0x00047f,
                        0x000481, 0x000481, 0x00048b, 0x00048b, 0x00048d, 0x00048d, 0x00048f, 0x00048f, 0x000491, 0x000491, 0x000493, 0x000493, 0x000495, 0x000495, 0x000497, 0x000497, 0x000499,
                        0x000499, 0x00049b, 0x00049b, 0x00049d, 0x00049d, 0x00049f, 0x00049f, 0x0004a1, 0x0004a1, 0x0004a3, 0x0004a3, 0x0004a5, 0x0004a5, 0x0004a7, 0x0004a7, 0x0004a9, 0x0004a9,
                        0x0004ab, 0x0004ab, 0x0004ad, 0x0004ad, 0x0004af, 0x0004af, 0x0004b1, 0x0004b1, 0x0004b3, 0x0004b3, 0x0004b5, 0x0004b5, 0x0004b7, 0x0004b7, 0x0004b9, 0x0004b9, 0x0004bb,
                        0x0004bb, 0x0004bd, 0x0004bd, 0x0004bf, 0x0004bf, 0x0004c2, 0x0004c2, 0x0004c4, 0x0004c4, 0x0004c6, 0x0004c6, 0x0004c8, 0x0004c8, 0x0004ca, 0x0004ca, 0x0004cc, 0x0004cc,
                        0x0004ce, 0x0004cf, 0x0004d1, 0x0004d1, 0x0004d3, 0x0004d3, 0x0004d5, 0x0004d5, 0x0004d7, 0x0004d7, 0x0004d9, 0x0004d9, 0x0004db, 0x0004db, 0x0004dd, 0x0004dd, 0x0004df,
                        0x0004df, 0x0004e1, 0x0004e1, 0x0004e3, 0x0004e3, 0x0004e5, 0x0004e5, 0x0004e7, 0x0004e7, 0x0004e9, 0x0004e9, 0x0004eb, 0x0004eb, 0x0004ed, 0x0004ed, 0x0004ef, 0x0004ef,
                        0x0004f1, 0x0004f1, 0x0004f3, 0x0004f3, 0x0004f5, 0x0004f5, 0x0004f7, 0x0004f7, 0x0004f9, 0x0004f9, 0x0004fb, 0x0004fb, 0x0004fd, 0x0004fd, 0x0004ff, 0x0004ff, 0x000501,
                        0x000501, 0x000503, 0x000503, 0x000505, 0x000505, 0x000507, 0x000507, 0x000509, 0x000509, 0x00050b, 0x00050b, 0x00050d, 0x00050d, 0x00050f, 0x00050f, 0x000511, 0x000511,
                        0x000513, 0x000513, 0x000515, 0x000515, 0x000517, 0x000517, 0x000519, 0x000519, 0x00051b, 0x00051b, 0x00051d, 0x00051d, 0x00051f, 0x00051f, 0x000521, 0x000521, 0x000523,
                        0x000523, 0x000525, 0x000525, 0x000527, 0x000527, 0x000529, 0x000529, 0x00052b, 0x00052b, 0x00052d, 0x00052d, 0x00052f, 0x00052f, 0x000560, 0x000588, 0x0010d0, 0x0010fa,
                        0x0010fd, 0x0010ff, 0x0013f8, 0x0013fd, 0x001c80, 0x001c88, 0x001d00, 0x001d2b, 0x001d6b, 0x001d77, 0x001d79, 0x001d9a, 0x001e01, 0x001e01, 0x001e03, 0x001e03, 0x001e05,
                        0x001e05, 0x001e07, 0x001e07, 0x001e09, 0x001e09, 0x001e0b, 0x001e0b, 0x001e0d, 0x001e0d, 0x001e0f, 0x001e0f, 0x001e11, 0x001e11, 0x001e13, 0x001e13, 0x001e15, 0x001e15,
                        0x001e17, 0x001e17, 0x001e19, 0x001e19, 0x001e1b, 0x001e1b, 0x001e1d, 0x001e1d, 0x001e1f, 0x001e1f, 0x001e21, 0x001e21, 0x001e23, 0x001e23, 0x001e25, 0x001e25, 0x001e27,
                        0x001e27, 0x001e29, 0x001e29, 0x001e2b, 0x001e2b, 0x001e2d, 0x001e2d, 0x001e2f, 0x001e2f, 0x001e31, 0x001e31, 0x001e33, 0x001e33, 0x001e35, 0x001e35, 0x001e37, 0x001e37,
                        0x001e39, 0x001e39, 0x001e3b, 0x001e3b, 0x001e3d, 0x001e3d, 0x001e3f, 0x001e3f, 0x001e41, 0x001e41, 0x001e43, 0x001e43, 0x001e45, 0x001e45, 0x001e47, 0x001e47, 0x001e49,
                        0x001e49, 0x001e4b, 0x001e4b, 0x001e4d, 0x001e4d, 0x001e4f, 0x001e4f, 0x001e51, 0x001e51, 0x001e53, 0x001e53, 0x001e55, 0x001e55, 0x001e57, 0x001e57, 0x001e59, 0x001e59,
                        0x001e5b, 0x001e5b, 0x001e5d, 0x001e5d, 0x001e5f, 0x001e5f, 0x001e61, 0x001e61, 0x001e63, 0x001e63, 0x001e65, 0x001e65, 0x001e67, 0x001e67, 0x001e69, 0x001e69, 0x001e6b,
                        0x001e6b, 0x001e6d, 0x001e6d, 0x001e6f, 0x001e6f, 0x001e71, 0x001e71, 0x001e73, 0x001e73, 0x001e75, 0x001e75, 0x001e77, 0x001e77, 0x001e79, 0x001e79, 0x001e7b, 0x001e7b,
                        0x001e7d, 0x001e7d, 0x001e7f, 0x001e7f, 0x001e81, 0x001e81, 0x001e83, 0x001e83, 0x001e85, 0x001e85, 0x001e87, 0x001e87, 0x001e89, 0x001e89, 0x001e8b, 0x001e8b, 0x001e8d,
                        0x001e8d, 0x001e8f, 0x001e8f, 0x001e91, 0x001e91, 0x001e93, 0x001e93, 0x001e95, 0x001e9d, 0x001e9f, 0x001e9f, 0x001ea1, 0x001ea1, 0x001ea3, 0x001ea3, 0x001ea5, 0x001ea5,
                        0x001ea7, 0x001ea7, 0x001ea9, 0x001ea9, 0x001eab, 0x001eab, 0x001ead, 0x001ead, 0x001eaf, 0x001eaf, 0x001eb1, 0x001eb1, 0x001eb3, 0x001eb3, 0x001eb5, 0x001eb5, 0x001eb7,
                        0x001eb7, 0x001eb9, 0x001eb9, 0x001ebb, 0x001ebb, 0x001ebd, 0x001ebd, 0x001ebf, 0x001ebf, 0x001ec1, 0x001ec1, 0x001ec3, 0x001ec3, 0x001ec5, 0x001ec5, 0x001ec7, 0x001ec7,
                        0x001ec9, 0x001ec9, 0x001ecb, 0x001ecb, 0x001ecd, 0x001ecd, 0x001ecf, 0x001ecf, 0x001ed1, 0x001ed1, 0x001ed3, 0x001ed3, 0x001ed5, 0x001ed5, 0x001ed7, 0x001ed7, 0x001ed9,
                        0x001ed9, 0x001edb, 0x001edb, 0x001edd, 0x001edd, 0x001edf, 0x001edf, 0x001ee1, 0x001ee1, 0x001ee3, 0x001ee3, 0x001ee5, 0x001ee5, 0x001ee7, 0x001ee7, 0x001ee9, 0x001ee9,
                        0x001eeb, 0x001eeb, 0x001eed, 0x001eed, 0x001eef, 0x001eef, 0x001ef1, 0x001ef1, 0x001ef3, 0x001ef3, 0x001ef5, 0x001ef5, 0x001ef7, 0x001ef7, 0x001ef9, 0x001ef9, 0x001efb,
                        0x001efb, 0x001efd, 0x001efd, 0x001eff, 0x001f07, 0x001f10, 0x001f15, 0x001f20, 0x001f27, 0x001f30, 0x001f37, 0x001f40, 0x001f45, 0x001f50, 0x001f57, 0x001f60, 0x001f67,
                        0x001f70, 0x001f7d, 0x001f80, 0x001f87, 0x001f90, 0x001f97, 0x001fa0, 0x001fa7, 0x001fb0, 0x001fb4, 0x001fb6, 0x001fb7, 0x001fbe, 0x001fbe, 0x001fc2, 0x001fc4, 0x001fc6,
                        0x001fc7, 0x001fd0, 0x001fd3, 0x001fd6, 0x001fd7, 0x001fe0, 0x001fe7, 0x001ff2, 0x001ff4, 0x001ff6, 0x001ff7, 0x00207f, 0x00207f, 0x00210a, 0x00210a, 0x00210e, 0x00210f,
                        0x002113, 0x002113, 0x00212f, 0x00212f, 0x002134, 0x002134, 0x002139, 0x002139, 0x00213c, 0x00213d, 0x002146, 0x002149, 0x00214e, 0x00214e, 0x002170, 0x00217f, 0x002184,
                        0x002184, 0x0024d0, 0x0024e9, 0x002c30, 0x002c5f, 0x002c61, 0x002c61, 0x002c65, 0x002c66, 0x002c68, 0x002c68, 0x002c6a, 0x002c6a, 0x002c6c, 0x002c6c, 0x002c71, 0x002c71,
                        0x002c73, 0x002c74, 0x002c76, 0x002c7b, 0x002c81, 0x002c81, 0x002c83, 0x002c83, 0x002c85, 0x002c85, 0x002c87, 0x002c87, 0x002c89, 0x002c89, 0x002c8b, 0x002c8b, 0x002c8d,
                        0x002c8d, 0x002c8f, 0x002c8f, 0x002c91, 0x002c91, 0x002c93, 0x002c93, 0x002c95, 0x002c95, 0x002c97, 0x002c97, 0x002c99, 0x002c99, 0x002c9b, 0x002c9b, 0x002c9d, 0x002c9d,
                        0x002c9f, 0x002c9f, 0x002ca1, 0x002ca1, 0x002ca3, 0x002ca3, 0x002ca5, 0x002ca5, 0x002ca7, 0x002ca7, 0x002ca9, 0x002ca9, 0x002cab, 0x002cab, 0x002cad, 0x002cad, 0x002caf,
                        0x002caf, 0x002cb1, 0x002cb1, 0x002cb3, 0x002cb3, 0x002cb5, 0x002cb5, 0x002cb7, 0x002cb7, 0x002cb9, 0x002cb9, 0x002cbb, 0x002cbb, 0x002cbd, 0x002cbd, 0x002cbf, 0x002cbf,
                        0x002cc1, 0x002cc1, 0x002cc3, 0x002cc3, 0x002cc5, 0x002cc5, 0x002cc7, 0x002cc7, 0x002cc9, 0x002cc9, 0x002ccb, 0x002ccb, 0x002ccd, 0x002ccd, 0x002ccf, 0x002ccf, 0x002cd1,
                        0x002cd1, 0x002cd3, 0x002cd3, 0x002cd5, 0x002cd5, 0x002cd7, 0x002cd7, 0x002cd9, 0x002cd9, 0x002cdb, 0x002cdb, 0x002cdd, 0x002cdd, 0x002cdf, 0x002cdf, 0x002ce1, 0x002ce1,
                        0x002ce3, 0x002ce4, 0x002cec, 0x002cec, 0x002cee, 0x002cee, 0x002cf3, 0x002cf3, 0x002d00, 0x002d25, 0x002d27, 0x002d27, 0x002d2d, 0x002d2d, 0x00a641, 0x00a641, 0x00a643,
                        0x00a643, 0x00a645, 0x00a645, 0x00a647, 0x00a647, 0x00a649, 0x00a649, 0x00a64b, 0x00a64b, 0x00a64d, 0x00a64d, 0x00a64f, 0x00a64f, 0x00a651, 0x00a651, 0x00a653, 0x00a653,
                        0x00a655, 0x00a655, 0x00a657, 0x00a657, 0x00a659, 0x00a659, 0x00a65b, 0x00a65b, 0x00a65d, 0x00a65d, 0x00a65f, 0x00a65f, 0x00a661, 0x00a661, 0x00a663, 0x00a663, 0x00a665,
                        0x00a665, 0x00a667, 0x00a667, 0x00a669, 0x00a669, 0x00a66b, 0x00a66b, 0x00a66d, 0x00a66d, 0x00a681, 0x00a681, 0x00a683, 0x00a683, 0x00a685, 0x00a685, 0x00a687, 0x00a687,
                        0x00a689, 0x00a689, 0x00a68b, 0x00a68b, 0x00a68d, 0x00a68d, 0x00a68f, 0x00a68f, 0x00a691, 0x00a691, 0x00a693, 0x00a693, 0x00a695, 0x00a695, 0x00a697, 0x00a697, 0x00a699,
                        0x00a699, 0x00a69b, 0x00a69b, 0x00a723, 0x00a723, 0x00a725, 0x00a725, 0x00a727, 0x00a727, 0x00a729, 0x00a729, 0x00a72b, 0x00a72b, 0x00a72d, 0x00a72d, 0x00a72f, 0x00a731,
                        0x00a733, 0x00a733, 0x00a735, 0x00a735, 0x00a737, 0x00a737, 0x00a739, 0x00a739, 0x00a73b, 0x00a73b, 0x00a73d, 0x00a73d, 0x00a73f, 0x00a73f, 0x00a741, 0x00a741, 0x00a743,
                        0x00a743, 0x00a745, 0x00a745, 0x00a747, 0x00a747, 0x00a749, 0x00a749, 0x00a74b, 0x00a74b, 0x00a74d, 0x00a74d, 0x00a74f, 0x00a74f, 0x00a751, 0x00a751, 0x00a753, 0x00a753,
                        0x00a755, 0x00a755, 0x00a757, 0x00a757, 0x00a759, 0x00a759, 0x00a75b, 0x00a75b, 0x00a75d, 0x00a75d, 0x00a75f, 0x00a75f, 0x00a761, 0x00a761, 0x00a763, 0x00a763, 0x00a765,
                        0x00a765, 0x00a767, 0x00a767, 0x00a769, 0x00a769, 0x00a76b, 0x00a76b, 0x00a76d, 0x00a76d, 0x00a76f, 0x00a76f, 0x00a771, 0x00a778, 0x00a77a, 0x00a77a, 0x00a77c, 0x00a77c,
                        0x00a77f, 0x00a77f, 0x00a781, 0x00a781, 0x00a783, 0x00a783, 0x00a785, 0x00a785, 0x00a787, 0x00a787, 0x00a78c, 0x00a78c, 0x00a78e, 0x00a78e, 0x00a791, 0x00a791, 0x00a793,
                        0x00a795, 0x00a797, 0x00a797, 0x00a799, 0x00a799, 0x00a79b, 0x00a79b, 0x00a79d, 0x00a79d, 0x00a79f, 0x00a79f, 0x00a7a1, 0x00a7a1, 0x00a7a3, 0x00a7a3, 0x00a7a5, 0x00a7a5,
                        0x00a7a7, 0x00a7a7, 0x00a7a9, 0x00a7a9, 0x00a7af, 0x00a7af, 0x00a7b5, 0x00a7b5, 0x00a7b7, 0x00a7b7, 0x00a7b9, 0x00a7b9, 0x00a7bb, 0x00a7bb, 0x00a7bd, 0x00a7bd, 0x00a7bf,
                        0x00a7bf, 0x00a7c1, 0x00a7c1, 0x00a7c3, 0x00a7c3, 0x00a7c8, 0x00a7c8, 0x00a7ca, 0x00a7ca, 0x00a7d1, 0x00a7d1, 0x00a7d3, 0x00a7d3, 0x00a7d5, 0x00a7d5, 0x00a7d7, 0x00a7d7,
                        0x00a7d9, 0x00a7d9, 0x00a7f6, 0x00a7f6, 0x00a7fa, 0x00a7fa, 0x00ab30, 0x00ab5a, 0x00ab60, 0x00ab68, 0x00ab70, 0x00abbf, 0x00fb00, 0x00fb06, 0x00fb13, 0x00fb17, 0x00ff41,
                        0x00ff5a, 0x010428, 0x01044f, 0x0104d8, 0x0104fb, 0x010597, 0x0105a1, 0x0105a3, 0x0105b1, 0x0105b3, 0x0105b9, 0x0105bb, 0x0105bc, 0x010cc0, 0x010cf2, 0x0118c0, 0x0118df,
                        0x016e60, 0x016e7f, 0x01d41a, 0x01d433, 0x01d44e, 0x01d454, 0x01d456, 0x01d467, 0x01d482, 0x01d49b, 0x01d4b6, 0x01d4b9, 0x01d4bb, 0x01d4bb, 0x01d4bd, 0x01d4c3, 0x01d4c5,
                        0x01d4cf, 0x01d4ea, 0x01d503, 0x01d51e, 0x01d537, 0x01d552, 0x01d56b, 0x01d586, 0x01d59f, 0x01d5ba, 0x01d5d3, 0x01d5ee, 0x01d607, 0x01d622, 0x01d63b, 0x01d656, 0x01d66f,
                        0x01d68a, 0x01d6a5, 0x01d6c2, 0x01d6da, 0x01d6dc, 0x01d6e1, 0x01d6fc, 0x01d714, 0x01d716, 0x01d71b, 0x01d736, 0x01d74e, 0x01d750, 0x01d755, 0x01d770, 0x01d788, 0x01d78a,
                        0x01d78f, 0x01d7aa, 0x01d7c2, 0x01d7c4, 0x01d7c9, 0x01d7cb, 0x01d7cb, 0x01df00, 0x01df09, 0x01df0b, 0x01df1e, 0x01df25, 0x01df2a, 0x01e922, 0x01e943));

        POSIX_CHAR_CLASSES.put("print", CodePointSet.createNoDedup(
                        0x000020, 0x00007e, 0x0000a0, 0x0000ac, 0x0000ae, 0x000377, 0x00037a, 0x00037f, 0x000384, 0x00038a, 0x00038c, 0x00038c, 0x00038e, 0x0003a1, 0x0003a3, 0x00052f, 0x000531,
                        0x000556, 0x000559, 0x00058a, 0x00058d, 0x00058f, 0x000591, 0x0005c7, 0x0005d0, 0x0005ea, 0x0005ef, 0x0005f4, 0x000606, 0x00061b, 0x00061d, 0x0006dc, 0x0006de, 0x00070d,
                        0x000710, 0x00074a, 0x00074d, 0x0007b1, 0x0007c0, 0x0007fa, 0x0007fd, 0x00082d, 0x000830, 0x00083e, 0x000840, 0x00085b, 0x00085e, 0x00085e, 0x000860, 0x00086a, 0x000870,
                        0x00088e, 0x000898, 0x0008e1, 0x0008e3, 0x000983, 0x000985, 0x00098c, 0x00098f, 0x000990, 0x000993, 0x0009a8, 0x0009aa, 0x0009b0, 0x0009b2, 0x0009b2, 0x0009b6, 0x0009b9,
                        0x0009bc, 0x0009c4, 0x0009c7, 0x0009c8, 0x0009cb, 0x0009ce, 0x0009d7, 0x0009d7, 0x0009dc, 0x0009dd, 0x0009df, 0x0009e3, 0x0009e6, 0x0009fe, 0x000a01, 0x000a03, 0x000a05,
                        0x000a0a, 0x000a0f, 0x000a10, 0x000a13, 0x000a28, 0x000a2a, 0x000a30, 0x000a32, 0x000a33, 0x000a35, 0x000a36, 0x000a38, 0x000a39, 0x000a3c, 0x000a3c, 0x000a3e, 0x000a42,
                        0x000a47, 0x000a48, 0x000a4b, 0x000a4d, 0x000a51, 0x000a51, 0x000a59, 0x000a5c, 0x000a5e, 0x000a5e, 0x000a66, 0x000a76, 0x000a81, 0x000a83, 0x000a85, 0x000a8d, 0x000a8f,
                        0x000a91, 0x000a93, 0x000aa8, 0x000aaa, 0x000ab0, 0x000ab2, 0x000ab3, 0x000ab5, 0x000ab9, 0x000abc, 0x000ac5, 0x000ac7, 0x000ac9, 0x000acb, 0x000acd, 0x000ad0, 0x000ad0,
                        0x000ae0, 0x000ae3, 0x000ae6, 0x000af1, 0x000af9, 0x000aff, 0x000b01, 0x000b03, 0x000b05, 0x000b0c, 0x000b0f, 0x000b10, 0x000b13, 0x000b28, 0x000b2a, 0x000b30, 0x000b32,
                        0x000b33, 0x000b35, 0x000b39, 0x000b3c, 0x000b44, 0x000b47, 0x000b48, 0x000b4b, 0x000b4d, 0x000b55, 0x000b57, 0x000b5c, 0x000b5d, 0x000b5f, 0x000b63, 0x000b66, 0x000b77,
                        0x000b82, 0x000b83, 0x000b85, 0x000b8a, 0x000b8e, 0x000b90, 0x000b92, 0x000b95, 0x000b99, 0x000b9a, 0x000b9c, 0x000b9c, 0x000b9e, 0x000b9f, 0x000ba3, 0x000ba4, 0x000ba8,
                        0x000baa, 0x000bae, 0x000bb9, 0x000bbe, 0x000bc2, 0x000bc6, 0x000bc8, 0x000bca, 0x000bcd, 0x000bd0, 0x000bd0, 0x000bd7, 0x000bd7, 0x000be6, 0x000bfa, 0x000c00, 0x000c0c,
                        0x000c0e, 0x000c10, 0x000c12, 0x000c28, 0x000c2a, 0x000c39, 0x000c3c, 0x000c44, 0x000c46, 0x000c48, 0x000c4a, 0x000c4d, 0x000c55, 0x000c56, 0x000c58, 0x000c5a, 0x000c5d,
                        0x000c5d, 0x000c60, 0x000c63, 0x000c66, 0x000c6f, 0x000c77, 0x000c8c, 0x000c8e, 0x000c90, 0x000c92, 0x000ca8, 0x000caa, 0x000cb3, 0x000cb5, 0x000cb9, 0x000cbc, 0x000cc4,
                        0x000cc6, 0x000cc8, 0x000cca, 0x000ccd, 0x000cd5, 0x000cd6, 0x000cdd, 0x000cde, 0x000ce0, 0x000ce3, 0x000ce6, 0x000cef, 0x000cf1, 0x000cf3, 0x000d00, 0x000d0c, 0x000d0e,
                        0x000d10, 0x000d12, 0x000d44, 0x000d46, 0x000d48, 0x000d4a, 0x000d4f, 0x000d54, 0x000d63, 0x000d66, 0x000d7f, 0x000d81, 0x000d83, 0x000d85, 0x000d96, 0x000d9a, 0x000db1,
                        0x000db3, 0x000dbb, 0x000dbd, 0x000dbd, 0x000dc0, 0x000dc6, 0x000dca, 0x000dca, 0x000dcf, 0x000dd4, 0x000dd6, 0x000dd6, 0x000dd8, 0x000ddf, 0x000de6, 0x000def, 0x000df2,
                        0x000df4, 0x000e01, 0x000e3a, 0x000e3f, 0x000e5b, 0x000e81, 0x000e82, 0x000e84, 0x000e84, 0x000e86, 0x000e8a, 0x000e8c, 0x000ea3, 0x000ea5, 0x000ea5, 0x000ea7, 0x000ebd,
                        0x000ec0, 0x000ec4, 0x000ec6, 0x000ec6, 0x000ec8, 0x000ece, 0x000ed0, 0x000ed9, 0x000edc, 0x000edf, 0x000f00, 0x000f47, 0x000f49, 0x000f6c, 0x000f71, 0x000f97, 0x000f99,
                        0x000fbc, 0x000fbe, 0x000fcc, 0x000fce, 0x000fda, 0x001000, 0x0010c5, 0x0010c7, 0x0010c7, 0x0010cd, 0x0010cd, 0x0010d0, 0x001248, 0x00124a, 0x00124d, 0x001250, 0x001256,
                        0x001258, 0x001258, 0x00125a, 0x00125d, 0x001260, 0x001288, 0x00128a, 0x00128d, 0x001290, 0x0012b0, 0x0012b2, 0x0012b5, 0x0012b8, 0x0012be, 0x0012c0, 0x0012c0, 0x0012c2,
                        0x0012c5, 0x0012c8, 0x0012d6, 0x0012d8, 0x001310, 0x001312, 0x001315, 0x001318, 0x00135a, 0x00135d, 0x00137c, 0x001380, 0x001399, 0x0013a0, 0x0013f5, 0x0013f8, 0x0013fd,
                        0x001400, 0x00169c, 0x0016a0, 0x0016f8, 0x001700, 0x001715, 0x00171f, 0x001736, 0x001740, 0x001753, 0x001760, 0x00176c, 0x00176e, 0x001770, 0x001772, 0x001773, 0x001780,
                        0x0017dd, 0x0017e0, 0x0017e9, 0x0017f0, 0x0017f9, 0x001800, 0x00180d, 0x00180f, 0x001819, 0x001820, 0x001878, 0x001880, 0x0018aa, 0x0018b0, 0x0018f5, 0x001900, 0x00191e,
                        0x001920, 0x00192b, 0x001930, 0x00193b, 0x001940, 0x001940, 0x001944, 0x00196d, 0x001970, 0x001974, 0x001980, 0x0019ab, 0x0019b0, 0x0019c9, 0x0019d0, 0x0019da, 0x0019de,
                        0x001a1b, 0x001a1e, 0x001a5e, 0x001a60, 0x001a7c, 0x001a7f, 0x001a89, 0x001a90, 0x001a99, 0x001aa0, 0x001aad, 0x001ab0, 0x001ace, 0x001b00, 0x001b4c, 0x001b50, 0x001b7e,
                        0x001b80, 0x001bf3, 0x001bfc, 0x001c37, 0x001c3b, 0x001c49, 0x001c4d, 0x001c88, 0x001c90, 0x001cba, 0x001cbd, 0x001cc7, 0x001cd0, 0x001cfa, 0x001d00, 0x001f15, 0x001f18,
                        0x001f1d, 0x001f20, 0x001f45, 0x001f48, 0x001f4d, 0x001f50, 0x001f57, 0x001f59, 0x001f59, 0x001f5b, 0x001f5b, 0x001f5d, 0x001f5d, 0x001f5f, 0x001f7d, 0x001f80, 0x001fb4,
                        0x001fb6, 0x001fc4, 0x001fc6, 0x001fd3, 0x001fd6, 0x001fdb, 0x001fdd, 0x001fef, 0x001ff2, 0x001ff4, 0x001ff6, 0x001ffe, 0x002000, 0x00200a, 0x002010, 0x002029, 0x00202f,
                        0x00205f, 0x002070, 0x002071, 0x002074, 0x00208e, 0x002090, 0x00209c, 0x0020a0, 0x0020c0, 0x0020d0, 0x0020f0, 0x002100, 0x00218b, 0x002190, 0x002426, 0x002440, 0x00244a,
                        0x002460, 0x002b73, 0x002b76, 0x002b95, 0x002b97, 0x002cf3, 0x002cf9, 0x002d25, 0x002d27, 0x002d27, 0x002d2d, 0x002d2d, 0x002d30, 0x002d67, 0x002d6f, 0x002d70, 0x002d7f,
                        0x002d96, 0x002da0, 0x002da6, 0x002da8, 0x002dae, 0x002db0, 0x002db6, 0x002db8, 0x002dbe, 0x002dc0, 0x002dc6, 0x002dc8, 0x002dce, 0x002dd0, 0x002dd6, 0x002dd8, 0x002dde,
                        0x002de0, 0x002e5d, 0x002e80, 0x002e99, 0x002e9b, 0x002ef3, 0x002f00, 0x002fd5, 0x002ff0, 0x002ffb, 0x003000, 0x00303f, 0x003041, 0x003096, 0x003099, 0x0030ff, 0x003105,
                        0x00312f, 0x003131, 0x00318e, 0x003190, 0x0031e3, 0x0031f0, 0x00321e, 0x003220, 0x00a48c, 0x00a490, 0x00a4c6, 0x00a4d0, 0x00a62b, 0x00a640, 0x00a6f7, 0x00a700, 0x00a7ca,
                        0x00a7d0, 0x00a7d1, 0x00a7d3, 0x00a7d3, 0x00a7d5, 0x00a7d9, 0x00a7f2, 0x00a82c, 0x00a830, 0x00a839, 0x00a840, 0x00a877, 0x00a880, 0x00a8c5, 0x00a8ce, 0x00a8d9, 0x00a8e0,
                        0x00a953, 0x00a95f, 0x00a97c, 0x00a980, 0x00a9cd, 0x00a9cf, 0x00a9d9, 0x00a9de, 0x00a9fe, 0x00aa00, 0x00aa36, 0x00aa40, 0x00aa4d, 0x00aa50, 0x00aa59, 0x00aa5c, 0x00aac2,
                        0x00aadb, 0x00aaf6, 0x00ab01, 0x00ab06, 0x00ab09, 0x00ab0e, 0x00ab11, 0x00ab16, 0x00ab20, 0x00ab26, 0x00ab28, 0x00ab2e, 0x00ab30, 0x00ab6b, 0x00ab70, 0x00abed, 0x00abf0,
                        0x00abf9, 0x00ac00, 0x00d7a3, 0x00d7b0, 0x00d7c6, 0x00d7cb, 0x00d7fb, 0x00e000, 0x00fa6d, 0x00fa70, 0x00fad9, 0x00fb00, 0x00fb06, 0x00fb13, 0x00fb17, 0x00fb1d, 0x00fb36,
                        0x00fb38, 0x00fb3c, 0x00fb3e, 0x00fb3e, 0x00fb40, 0x00fb41, 0x00fb43, 0x00fb44, 0x00fb46, 0x00fbc2, 0x00fbd3, 0x00fd8f, 0x00fd92, 0x00fdc7, 0x00fdcf, 0x00fdcf, 0x00fdf0,
                        0x00fe19, 0x00fe20, 0x00fe52, 0x00fe54, 0x00fe66, 0x00fe68, 0x00fe6b, 0x00fe70, 0x00fe74, 0x00fe76, 0x00fefc, 0x00ff01, 0x00ffbe, 0x00ffc2, 0x00ffc7, 0x00ffca, 0x00ffcf,
                        0x00ffd2, 0x00ffd7, 0x00ffda, 0x00ffdc, 0x00ffe0, 0x00ffe6, 0x00ffe8, 0x00ffee, 0x00fffc, 0x00fffd, 0x010000, 0x01000b, 0x01000d, 0x010026, 0x010028, 0x01003a, 0x01003c,
                        0x01003d, 0x01003f, 0x01004d, 0x010050, 0x01005d, 0x010080, 0x0100fa, 0x010100, 0x010102, 0x010107, 0x010133, 0x010137, 0x01018e, 0x010190, 0x01019c, 0x0101a0, 0x0101a0,
                        0x0101d0, 0x0101fd, 0x010280, 0x01029c, 0x0102a0, 0x0102d0, 0x0102e0, 0x0102fb, 0x010300, 0x010323, 0x01032d, 0x01034a, 0x010350, 0x01037a, 0x010380, 0x01039d, 0x01039f,
                        0x0103c3, 0x0103c8, 0x0103d5, 0x010400, 0x01049d, 0x0104a0, 0x0104a9, 0x0104b0, 0x0104d3, 0x0104d8, 0x0104fb, 0x010500, 0x010527, 0x010530, 0x010563, 0x01056f, 0x01057a,
                        0x01057c, 0x01058a, 0x01058c, 0x010592, 0x010594, 0x010595, 0x010597, 0x0105a1, 0x0105a3, 0x0105b1, 0x0105b3, 0x0105b9, 0x0105bb, 0x0105bc, 0x010600, 0x010736, 0x010740,
                        0x010755, 0x010760, 0x010767, 0x010780, 0x010785, 0x010787, 0x0107b0, 0x0107b2, 0x0107ba, 0x010800, 0x010805, 0x010808, 0x010808, 0x01080a, 0x010835, 0x010837, 0x010838,
                        0x01083c, 0x01083c, 0x01083f, 0x010855, 0x010857, 0x01089e, 0x0108a7, 0x0108af, 0x0108e0, 0x0108f2, 0x0108f4, 0x0108f5, 0x0108fb, 0x01091b, 0x01091f, 0x010939, 0x01093f,
                        0x01093f, 0x010980, 0x0109b7, 0x0109bc, 0x0109cf, 0x0109d2, 0x010a03, 0x010a05, 0x010a06, 0x010a0c, 0x010a13, 0x010a15, 0x010a17, 0x010a19, 0x010a35, 0x010a38, 0x010a3a,
                        0x010a3f, 0x010a48, 0x010a50, 0x010a58, 0x010a60, 0x010a9f, 0x010ac0, 0x010ae6, 0x010aeb, 0x010af6, 0x010b00, 0x010b35, 0x010b39, 0x010b55, 0x010b58, 0x010b72, 0x010b78,
                        0x010b91, 0x010b99, 0x010b9c, 0x010ba9, 0x010baf, 0x010c00, 0x010c48, 0x010c80, 0x010cb2, 0x010cc0, 0x010cf2, 0x010cfa, 0x010d27, 0x010d30, 0x010d39, 0x010e60, 0x010e7e,
                        0x010e80, 0x010ea9, 0x010eab, 0x010ead, 0x010eb0, 0x010eb1, 0x010efd, 0x010f27, 0x010f30, 0x010f59, 0x010f70, 0x010f89, 0x010fb0, 0x010fcb, 0x010fe0, 0x010ff6, 0x011000,
                        0x01104d, 0x011052, 0x011075, 0x01107f, 0x0110bc, 0x0110be, 0x0110c2, 0x0110d0, 0x0110e8, 0x0110f0, 0x0110f9, 0x011100, 0x011134, 0x011136, 0x011147, 0x011150, 0x011176,
                        0x011180, 0x0111df, 0x0111e1, 0x0111f4, 0x011200, 0x011211, 0x011213, 0x011241, 0x011280, 0x011286, 0x011288, 0x011288, 0x01128a, 0x01128d, 0x01128f, 0x01129d, 0x01129f,
                        0x0112a9, 0x0112b0, 0x0112ea, 0x0112f0, 0x0112f9, 0x011300, 0x011303, 0x011305, 0x01130c, 0x01130f, 0x011310, 0x011313, 0x011328, 0x01132a, 0x011330, 0x011332, 0x011333,
                        0x011335, 0x011339, 0x01133b, 0x011344, 0x011347, 0x011348, 0x01134b, 0x01134d, 0x011350, 0x011350, 0x011357, 0x011357, 0x01135d, 0x011363, 0x011366, 0x01136c, 0x011370,
                        0x011374, 0x011400, 0x01145b, 0x01145d, 0x011461, 0x011480, 0x0114c7, 0x0114d0, 0x0114d9, 0x011580, 0x0115b5, 0x0115b8, 0x0115dd, 0x011600, 0x011644, 0x011650, 0x011659,
                        0x011660, 0x01166c, 0x011680, 0x0116b9, 0x0116c0, 0x0116c9, 0x011700, 0x01171a, 0x01171d, 0x01172b, 0x011730, 0x011746, 0x011800, 0x01183b, 0x0118a0, 0x0118f2, 0x0118ff,
                        0x011906, 0x011909, 0x011909, 0x01190c, 0x011913, 0x011915, 0x011916, 0x011918, 0x011935, 0x011937, 0x011938, 0x01193b, 0x011946, 0x011950, 0x011959, 0x0119a0, 0x0119a7,
                        0x0119aa, 0x0119d7, 0x0119da, 0x0119e4, 0x011a00, 0x011a47, 0x011a50, 0x011aa2, 0x011ab0, 0x011af8, 0x011b00, 0x011b09, 0x011c00, 0x011c08, 0x011c0a, 0x011c36, 0x011c38,
                        0x011c45, 0x011c50, 0x011c6c, 0x011c70, 0x011c8f, 0x011c92, 0x011ca7, 0x011ca9, 0x011cb6, 0x011d00, 0x011d06, 0x011d08, 0x011d09, 0x011d0b, 0x011d36, 0x011d3a, 0x011d3a,
                        0x011d3c, 0x011d3d, 0x011d3f, 0x011d47, 0x011d50, 0x011d59, 0x011d60, 0x011d65, 0x011d67, 0x011d68, 0x011d6a, 0x011d8e, 0x011d90, 0x011d91, 0x011d93, 0x011d98, 0x011da0,
                        0x011da9, 0x011ee0, 0x011ef8, 0x011f00, 0x011f10, 0x011f12, 0x011f3a, 0x011f3e, 0x011f59, 0x011fb0, 0x011fb0, 0x011fc0, 0x011ff1, 0x011fff, 0x012399, 0x012400, 0x01246e,
                        0x012470, 0x012474, 0x012480, 0x012543, 0x012f90, 0x012ff2, 0x013000, 0x01342f, 0x013440, 0x013455, 0x014400, 0x014646, 0x016800, 0x016a38, 0x016a40, 0x016a5e, 0x016a60,
                        0x016a69, 0x016a6e, 0x016abe, 0x016ac0, 0x016ac9, 0x016ad0, 0x016aed, 0x016af0, 0x016af5, 0x016b00, 0x016b45, 0x016b50, 0x016b59, 0x016b5b, 0x016b61, 0x016b63, 0x016b77,
                        0x016b7d, 0x016b8f, 0x016e40, 0x016e9a, 0x016f00, 0x016f4a, 0x016f4f, 0x016f87, 0x016f8f, 0x016f9f, 0x016fe0, 0x016fe4, 0x016ff0, 0x016ff1, 0x017000, 0x0187f7, 0x018800,
                        0x018cd5, 0x018d00, 0x018d08, 0x01aff0, 0x01aff3, 0x01aff5, 0x01affb, 0x01affd, 0x01affe, 0x01b000, 0x01b122, 0x01b132, 0x01b132, 0x01b150, 0x01b152, 0x01b155, 0x01b155,
                        0x01b164, 0x01b167, 0x01b170, 0x01b2fb, 0x01bc00, 0x01bc6a, 0x01bc70, 0x01bc7c, 0x01bc80, 0x01bc88, 0x01bc90, 0x01bc99, 0x01bc9c, 0x01bc9f, 0x01cf00, 0x01cf2d, 0x01cf30,
                        0x01cf46, 0x01cf50, 0x01cfc3, 0x01d000, 0x01d0f5, 0x01d100, 0x01d126, 0x01d129, 0x01d172, 0x01d17b, 0x01d1ea, 0x01d200, 0x01d245, 0x01d2c0, 0x01d2d3, 0x01d2e0, 0x01d2f3,
                        0x01d300, 0x01d356, 0x01d360, 0x01d378, 0x01d400, 0x01d454, 0x01d456, 0x01d49c, 0x01d49e, 0x01d49f, 0x01d4a2, 0x01d4a2, 0x01d4a5, 0x01d4a6, 0x01d4a9, 0x01d4ac, 0x01d4ae,
                        0x01d4b9, 0x01d4bb, 0x01d4bb, 0x01d4bd, 0x01d4c3, 0x01d4c5, 0x01d505, 0x01d507, 0x01d50a, 0x01d50d, 0x01d514, 0x01d516, 0x01d51c, 0x01d51e, 0x01d539, 0x01d53b, 0x01d53e,
                        0x01d540, 0x01d544, 0x01d546, 0x01d546, 0x01d54a, 0x01d550, 0x01d552, 0x01d6a5, 0x01d6a8, 0x01d7cb, 0x01d7ce, 0x01da8b, 0x01da9b, 0x01da9f, 0x01daa1, 0x01daaf, 0x01df00,
                        0x01df1e, 0x01df25, 0x01df2a, 0x01e000, 0x01e006, 0x01e008, 0x01e018, 0x01e01b, 0x01e021, 0x01e023, 0x01e024, 0x01e026, 0x01e02a, 0x01e030, 0x01e06d, 0x01e08f, 0x01e08f,
                        0x01e100, 0x01e12c, 0x01e130, 0x01e13d, 0x01e140, 0x01e149, 0x01e14e, 0x01e14f, 0x01e290, 0x01e2ae, 0x01e2c0, 0x01e2f9, 0x01e2ff, 0x01e2ff, 0x01e4d0, 0x01e4f9, 0x01e7e0,
                        0x01e7e6, 0x01e7e8, 0x01e7eb, 0x01e7ed, 0x01e7ee, 0x01e7f0, 0x01e7fe, 0x01e800, 0x01e8c4, 0x01e8c7, 0x01e8d6, 0x01e900, 0x01e94b, 0x01e950, 0x01e959, 0x01e95e, 0x01e95f,
                        0x01ec71, 0x01ecb4, 0x01ed01, 0x01ed3d, 0x01ee00, 0x01ee03, 0x01ee05, 0x01ee1f, 0x01ee21, 0x01ee22, 0x01ee24, 0x01ee24, 0x01ee27, 0x01ee27, 0x01ee29, 0x01ee32, 0x01ee34,
                        0x01ee37, 0x01ee39, 0x01ee39, 0x01ee3b, 0x01ee3b, 0x01ee42, 0x01ee42, 0x01ee47, 0x01ee47, 0x01ee49, 0x01ee49, 0x01ee4b, 0x01ee4b, 0x01ee4d, 0x01ee4f, 0x01ee51, 0x01ee52,
                        0x01ee54, 0x01ee54, 0x01ee57, 0x01ee57, 0x01ee59, 0x01ee59, 0x01ee5b, 0x01ee5b, 0x01ee5d, 0x01ee5d, 0x01ee5f, 0x01ee5f, 0x01ee61, 0x01ee62, 0x01ee64, 0x01ee64, 0x01ee67,
                        0x01ee6a, 0x01ee6c, 0x01ee72, 0x01ee74, 0x01ee77, 0x01ee79, 0x01ee7c, 0x01ee7e, 0x01ee7e, 0x01ee80, 0x01ee89, 0x01ee8b, 0x01ee9b, 0x01eea1, 0x01eea3, 0x01eea5, 0x01eea9,
                        0x01eeab, 0x01eebb, 0x01eef0, 0x01eef1, 0x01f000, 0x01f02b, 0x01f030, 0x01f093, 0x01f0a0, 0x01f0ae, 0x01f0b1, 0x01f0bf, 0x01f0c1, 0x01f0cf, 0x01f0d1, 0x01f0f5, 0x01f100,
                        0x01f1ad, 0x01f1e6, 0x01f202, 0x01f210, 0x01f23b, 0x01f240, 0x01f248, 0x01f250, 0x01f251, 0x01f260, 0x01f265, 0x01f300, 0x01f6d7, 0x01f6dc, 0x01f6ec, 0x01f6f0, 0x01f6fc,
                        0x01f700, 0x01f776, 0x01f77b, 0x01f7d9, 0x01f7e0, 0x01f7eb, 0x01f7f0, 0x01f7f0, 0x01f800, 0x01f80b, 0x01f810, 0x01f847, 0x01f850, 0x01f859, 0x01f860, 0x01f887, 0x01f890,
                        0x01f8ad, 0x01f8b0, 0x01f8b1, 0x01f900, 0x01fa53, 0x01fa60, 0x01fa6d, 0x01fa70, 0x01fa7c, 0x01fa80, 0x01fa88, 0x01fa90, 0x01fabd, 0x01fabf, 0x01fac5, 0x01face, 0x01fadb,
                        0x01fae0, 0x01fae8, 0x01faf0, 0x01faf8, 0x01fb00, 0x01fb92, 0x01fb94, 0x01fbca, 0x01fbf0, 0x01fbf9, 0x020000, 0x02a6df, 0x02a700, 0x02b739, 0x02b740, 0x02b81d, 0x02b820,
                        0x02cea1, 0x02ceb0, 0x02ebe0, 0x02f800, 0x02fa1d, 0x030000, 0x03134a, 0x031350, 0x0323af, 0x0e0100, 0x0e01ef, 0x0f0000, 0x0ffffd, 0x100000, 0x10fffd));

        POSIX_CHAR_CLASSES.put("punct", CodePointSet.createNoDedup(
                        0x000021, 0x00002f, 0x00003a, 0x000040, 0x00005b, 0x000060, 0x00007b, 0x00007e, 0x0000a0, 0x0000a9, 0x0000ab, 0x0000ac, 0x0000ae, 0x0000b1, 0x0000b4, 0x0000b4, 0x0000b6,
                        0x0000b8, 0x0000bb, 0x0000bb, 0x0000bf, 0x0000bf, 0x0000d7, 0x0000d7, 0x0000f7, 0x0000f7, 0x0002b9, 0x0002ba, 0x0002c2, 0x0002c6, 0x0002c8, 0x0002c8, 0x0002cc, 0x0002cd,
                        0x0002cf, 0x0002d0, 0x0002d2, 0x0002d8, 0x0002da, 0x0002df, 0x0002e5, 0x0002eb, 0x0002ed, 0x0002ed, 0x0002ef, 0x0002ff, 0x000375, 0x000375, 0x00037e, 0x00037e, 0x000384,
                        0x000385, 0x000387, 0x000387, 0x0003f6, 0x0003f6, 0x000482, 0x000482, 0x00055a, 0x00055f, 0x000589, 0x00058a, 0x00058d, 0x00058f, 0x0005b0, 0x0005b9, 0x0005bb, 0x0005c4,
                        0x0005c6, 0x0005c6, 0x0005f3, 0x0005f4, 0x000606, 0x00060f, 0x00061b, 0x00061b, 0x00061d, 0x00061f, 0x00066a, 0x00066d, 0x0006d4, 0x0006d4, 0x0006de, 0x0006de, 0x0006e9,
                        0x0006e9, 0x0006fd, 0x0006fe, 0x000700, 0x00070d, 0x0007f6, 0x0007f9, 0x0007fe, 0x0007ff, 0x000830, 0x00083e, 0x00085e, 0x00085e, 0x000888, 0x000888, 0x000901, 0x000903,
                        0x00093c, 0x00094d, 0x000964, 0x000965, 0x000970, 0x000970, 0x0009f2, 0x0009f3, 0x0009fa, 0x0009fb, 0x0009fd, 0x0009fd, 0x000a76, 0x000a76, 0x000af0, 0x000af1, 0x000b70,
                        0x000b70, 0x000bf3, 0x000bfa, 0x000c77, 0x000c77, 0x000c7f, 0x000c7f, 0x000c84, 0x000c84, 0x000d4f, 0x000d4f, 0x000d79, 0x000d79, 0x000df4, 0x000df4, 0x000e2f, 0x000e2f,
                        0x000e3f, 0x000e3f, 0x000e46, 0x000e46, 0x000e4e, 0x000e4f, 0x000e5a, 0x000e5b, 0x000f01, 0x000f17, 0x000f1a, 0x000f1f, 0x000f34, 0x000f34, 0x000f36, 0x000f36, 0x000f38,
                        0x000f38, 0x000f3a, 0x000f3d, 0x000f85, 0x000f85, 0x000fbe, 0x000fc5, 0x000fc7, 0x000fcc, 0x000fce, 0x000fda, 0x00104a, 0x00104f, 0x00109e, 0x00109f, 0x0010fb, 0x0010fb,
                        0x001360, 0x001368, 0x001390, 0x001399, 0x001400, 0x001400, 0x00166d, 0x00166e, 0x00169b, 0x00169c, 0x0016eb, 0x0016ed, 0x001735, 0x001736, 0x0017d4, 0x0017d6, 0x0017d8,
                        0x0017db, 0x001800, 0x00180a, 0x001940, 0x001940, 0x001944, 0x001945, 0x0019de, 0x0019ff, 0x001a1e, 0x001a1f, 0x001aa0, 0x001aa6, 0x001aa8, 0x001aad, 0x001b5a, 0x001b6a,
                        0x001b74, 0x001b7e, 0x001bfc, 0x001bff, 0x001c3b, 0x001c3f, 0x001c7e, 0x001c7f, 0x001cc0, 0x001cc7, 0x001cd3, 0x001cd3, 0x001fbd, 0x001fbd, 0x001fbf, 0x001fc1, 0x001fcd,
                        0x001fcf, 0x001fdd, 0x001fdf, 0x001fed, 0x001fef, 0x001ffd, 0x001ffe, 0x002010, 0x002014, 0x002016, 0x002027, 0x002030, 0x00205e, 0x00207a, 0x00207f, 0x00208a, 0x00208e,
                        0x0020a0, 0x0020c0, 0x002100, 0x002101, 0x002103, 0x002106, 0x002108, 0x002109, 0x002114, 0x002114, 0x002116, 0x002118, 0x00211e, 0x002123, 0x002125, 0x002127, 0x002129,
                        0x002129, 0x00212e, 0x00212e, 0x00213a, 0x00213b, 0x002140, 0x002144, 0x00214a, 0x00214d, 0x00214f, 0x00214f, 0x00218a, 0x00218b, 0x002190, 0x0023ff, 0x002422, 0x002426,
                        0x002440, 0x00244a, 0x00249c, 0x0024e9, 0x002500, 0x002775, 0x002794, 0x002b73, 0x002b76, 0x002b95, 0x002b97, 0x002bff, 0x002ce5, 0x002cea, 0x002cf9, 0x002cfc, 0x002cfe,
                        0x002cff, 0x002d70, 0x002d70, 0x002e00, 0x002e2e, 0x002e30, 0x002e5d, 0x002ff0, 0x002ffb, 0x003001, 0x003002, 0x003004, 0x003004, 0x003008, 0x00301b, 0x00301d, 0x003020,
                        0x003030, 0x003030, 0x003036, 0x003037, 0x00303d, 0x00303f, 0x00309b, 0x00309c, 0x0030a0, 0x0030a0, 0x0030fb, 0x0030fb, 0x003190, 0x003191, 0x0031c0, 0x0031e3, 0x003200,
                        0x00321e, 0x00322a, 0x003247, 0x003250, 0x003250, 0x003260, 0x00327f, 0x00328a, 0x0032b0, 0x0032c0, 0x0033ff, 0x004dc0, 0x004dff, 0x00a490, 0x00a4c6, 0x00a4fe, 0x00a4ff,
                        0x00a60d, 0x00a60f, 0x00a673, 0x00a673, 0x00a67e, 0x00a67e, 0x00a6f2, 0x00a6f7, 0x00a700, 0x00a716, 0x00a720, 0x00a721, 0x00a789, 0x00a78a, 0x00a828, 0x00a82b, 0x00a836,
                        0x00a839, 0x00a874, 0x00a877, 0x00a8ce, 0x00a8cf, 0x00a8f8, 0x00a8fa, 0x00a8fc, 0x00a8fc, 0x00a92e, 0x00a92f, 0x00a95f, 0x00a95f, 0x00a9c1, 0x00a9cd, 0x00a9de, 0x00a9df,
                        0x00aa5c, 0x00aa5f, 0x00aa77, 0x00aa79, 0x00aade, 0x00aadf, 0x00aaf0, 0x00aaf1, 0x00ab5b, 0x00ab5b, 0x00ab6a, 0x00ab6b, 0x00abeb, 0x00abeb, 0x00fb1f, 0x00fb1f, 0x00fb29,
                        0x00fb29, 0x00fbb2, 0x00fbc2, 0x00fd3e, 0x00fd4f, 0x00fdcf, 0x00fdcf, 0x00fdfc, 0x00fdff, 0x00fe10, 0x00fe19, 0x00fe30, 0x00fe52, 0x00fe54, 0x00fe66, 0x00fe68, 0x00fe6b,
                        0x00ff01, 0x00ff0f, 0x00ff1a, 0x00ff20, 0x00ff3b, 0x00ff3e, 0x00ff40, 0x00ff40, 0x00ff5b, 0x00ff65, 0x00ffe0, 0x00ffe6, 0x00ffe8, 0x00ffee, 0x00fffc, 0x00fffc, 0x010100,
                        0x010102, 0x010137, 0x01013f, 0x010179, 0x010189, 0x01018c, 0x01018e, 0x010190, 0x01019c, 0x0101a0, 0x0101a0, 0x0101d0, 0x0101fc, 0x01039f, 0x01039f, 0x0103d0, 0x0103d0,
                        0x01056f, 0x01056f, 0x010857, 0x010857, 0x010877, 0x010878, 0x01091f, 0x01091f, 0x01093f, 0x01093f, 0x010a50, 0x010a58, 0x010a7f, 0x010a7f, 0x010ac8, 0x010ac8, 0x010af0,
                        0x010af6, 0x010b39, 0x010b3f, 0x010b99, 0x010b9c, 0x010ead, 0x010ead, 0x010f55, 0x010f59, 0x010f86, 0x010f89, 0x011047, 0x01104d, 0x0110bb, 0x0110bc, 0x0110be, 0x0110c1,
                        0x011140, 0x011143, 0x011174, 0x011175, 0x0111c5, 0x0111c8, 0x0111cd, 0x0111cd, 0x0111db, 0x0111db, 0x0111dd, 0x0111df, 0x011238, 0x01123d, 0x0112a9, 0x0112a9, 0x01144b,
                        0x01144f, 0x01145a, 0x01145b, 0x01145d, 0x01145d, 0x0114c6, 0x0114c6, 0x0115c1, 0x0115d7, 0x011641, 0x011643, 0x011660, 0x01166c, 0x0116b9, 0x0116b9, 0x01173c, 0x01173f,
                        0x01183b, 0x01183b, 0x011944, 0x011946, 0x0119e2, 0x0119e2, 0x011a3f, 0x011a46, 0x011a9a, 0x011a9c, 0x011a9e, 0x011aa2, 0x011b00, 0x011b09, 0x011c41, 0x011c45, 0x011c70,
                        0x011c71, 0x011ef7, 0x011ef8, 0x011f43, 0x011f4f, 0x011fd5, 0x011ff1, 0x011fff, 0x011fff, 0x012470, 0x012474, 0x012ff1, 0x012ff2, 0x016a6e, 0x016a6f, 0x016af5, 0x016af5,
                        0x016b37, 0x016b3f, 0x016b44, 0x016b45, 0x016e97, 0x016e9a, 0x016fe2, 0x016fe2, 0x01bc9c, 0x01bc9c, 0x01bc9f, 0x01bc9f, 0x01cf50, 0x01cfc3, 0x01d000, 0x01d0f5, 0x01d100,
                        0x01d126, 0x01d129, 0x01d164, 0x01d16a, 0x01d16c, 0x01d183, 0x01d184, 0x01d18c, 0x01d1a9, 0x01d1ae, 0x01d1ea, 0x01d200, 0x01d241, 0x01d245, 0x01d245, 0x01d300, 0x01d356,
                        0x01d6c1, 0x01d6c1, 0x01d6db, 0x01d6db, 0x01d6fb, 0x01d6fb, 0x01d715, 0x01d715, 0x01d735, 0x01d735, 0x01d74f, 0x01d74f, 0x01d76f, 0x01d76f, 0x01d789, 0x01d789, 0x01d7a9,
                        0x01d7a9, 0x01d7c3, 0x01d7c3, 0x01d800, 0x01d9ff, 0x01da37, 0x01da3a, 0x01da6d, 0x01da74, 0x01da76, 0x01da83, 0x01da85, 0x01da8b, 0x01e14f, 0x01e14f, 0x01e2ff, 0x01e2ff,
                        0x01e95e, 0x01e95f, 0x01ecac, 0x01ecac, 0x01ecb0, 0x01ecb0, 0x01ed2e, 0x01ed2e, 0x01eef0, 0x01eef1, 0x01f000, 0x01f02b, 0x01f030, 0x01f093, 0x01f0a0, 0x01f0ae, 0x01f0b1,
                        0x01f0bf, 0x01f0c1, 0x01f0cf, 0x01f0d1, 0x01f0f5, 0x01f10d, 0x01f1ad, 0x01f1e6, 0x01f202, 0x01f210, 0x01f23b, 0x01f240, 0x01f248, 0x01f250, 0x01f251, 0x01f260, 0x01f265,
                        0x01f300, 0x01f6d7, 0x01f6dc, 0x01f6ec, 0x01f6f0, 0x01f6fc, 0x01f700, 0x01f776, 0x01f77b, 0x01f7d9, 0x01f7e0, 0x01f7eb, 0x01f7f0, 0x01f7f0, 0x01f800, 0x01f80b, 0x01f810,
                        0x01f847, 0x01f850, 0x01f859, 0x01f860, 0x01f887, 0x01f890, 0x01f8ad, 0x01f8b0, 0x01f8b1, 0x01f900, 0x01fa53, 0x01fa60, 0x01fa6d, 0x01fa70, 0x01fa7c, 0x01fa80, 0x01fa88,
                        0x01fa90, 0x01fabd, 0x01fabf, 0x01fac5, 0x01face, 0x01fadb, 0x01fae0, 0x01fae8, 0x01faf0, 0x01faf8, 0x01fb00, 0x01fb92, 0x01fb94, 0x01fbca));

        POSIX_CHAR_CLASSES.put("space", CodePointSet.createNoDedup(
                        0x000009, 0x00000d, 0x000020, 0x000020, 0x001680, 0x001680, 0x002000, 0x00200a, 0x002028, 0x002029, 0x00202f, 0x00202f, 0x00205f, 0x00205f, 0x003000, 0x003000));

        POSIX_CHAR_CLASSES.put("upper", CodePointSet.createNoDedup(
                        0x000041, 0x00005a, 0x0000c0, 0x0000d6, 0x0000d8, 0x0000de, 0x000100, 0x000100, 0x000102, 0x000102, 0x000104, 0x000104, 0x000106, 0x000106, 0x000108, 0x000108, 0x00010a,
                        0x00010a, 0x00010c, 0x00010c, 0x00010e, 0x00010e, 0x000110, 0x000110, 0x000112, 0x000112, 0x000114, 0x000114, 0x000116, 0x000116, 0x000118, 0x000118, 0x00011a, 0x00011a,
                        0x00011c, 0x00011c, 0x00011e, 0x00011e, 0x000120, 0x000120, 0x000122, 0x000122, 0x000124, 0x000124, 0x000126, 0x000126, 0x000128, 0x000128, 0x00012a, 0x00012a, 0x00012c,
                        0x00012c, 0x00012e, 0x00012e, 0x000130, 0x000130, 0x000132, 0x000132, 0x000134, 0x000134, 0x000136, 0x000136, 0x000139, 0x000139, 0x00013b, 0x00013b, 0x00013d, 0x00013d,
                        0x00013f, 0x00013f, 0x000141, 0x000141, 0x000143, 0x000143, 0x000145, 0x000145, 0x000147, 0x000147, 0x00014a, 0x00014a, 0x00014c, 0x00014c, 0x00014e, 0x00014e, 0x000150,
                        0x000150, 0x000152, 0x000152, 0x000154, 0x000154, 0x000156, 0x000156, 0x000158, 0x000158, 0x00015a, 0x00015a, 0x00015c, 0x00015c, 0x00015e, 0x00015e, 0x000160, 0x000160,
                        0x000162, 0x000162, 0x000164, 0x000164, 0x000166, 0x000166, 0x000168, 0x000168, 0x00016a, 0x00016a, 0x00016c, 0x00016c, 0x00016e, 0x00016e, 0x000170, 0x000170, 0x000172,
                        0x000172, 0x000174, 0x000174, 0x000176, 0x000176, 0x000178, 0x000179, 0x00017b, 0x00017b, 0x00017d, 0x00017d, 0x000181, 0x000182, 0x000184, 0x000184, 0x000186, 0x000187,
                        0x000189, 0x00018b, 0x00018e, 0x000191, 0x000193, 0x000194, 0x000196, 0x000198, 0x00019c, 0x00019d, 0x00019f, 0x0001a0, 0x0001a2, 0x0001a2, 0x0001a4, 0x0001a4, 0x0001a6,
                        0x0001a7, 0x0001a9, 0x0001a9, 0x0001ac, 0x0001ac, 0x0001ae, 0x0001af, 0x0001b1, 0x0001b3, 0x0001b5, 0x0001b5, 0x0001b7, 0x0001b8, 0x0001bc, 0x0001bc, 0x0001c4, 0x0001c5,
                        0x0001c7, 0x0001c8, 0x0001ca, 0x0001cb, 0x0001cd, 0x0001cd, 0x0001cf, 0x0001cf, 0x0001d1, 0x0001d1, 0x0001d3, 0x0001d3, 0x0001d5, 0x0001d5, 0x0001d7, 0x0001d7, 0x0001d9,
                        0x0001d9, 0x0001db, 0x0001db, 0x0001de, 0x0001de, 0x0001e0, 0x0001e0, 0x0001e2, 0x0001e2, 0x0001e4, 0x0001e4, 0x0001e6, 0x0001e6, 0x0001e8, 0x0001e8, 0x0001ea, 0x0001ea,
                        0x0001ec, 0x0001ec, 0x0001ee, 0x0001ee, 0x0001f1, 0x0001f2, 0x0001f4, 0x0001f4, 0x0001f6, 0x0001f8, 0x0001fa, 0x0001fa, 0x0001fc, 0x0001fc, 0x0001fe, 0x0001fe, 0x000200,
                        0x000200, 0x000202, 0x000202, 0x000204, 0x000204, 0x000206, 0x000206, 0x000208, 0x000208, 0x00020a, 0x00020a, 0x00020c, 0x00020c, 0x00020e, 0x00020e, 0x000210, 0x000210,
                        0x000212, 0x000212, 0x000214, 0x000214, 0x000216, 0x000216, 0x000218, 0x000218, 0x00021a, 0x00021a, 0x00021c, 0x00021c, 0x00021e, 0x00021e, 0x000220, 0x000220, 0x000222,
                        0x000222, 0x000224, 0x000224, 0x000226, 0x000226, 0x000228, 0x000228, 0x00022a, 0x00022a, 0x00022c, 0x00022c, 0x00022e, 0x00022e, 0x000230, 0x000230, 0x000232, 0x000232,
                        0x00023a, 0x00023b, 0x00023d, 0x00023e, 0x000241, 0x000241, 0x000243, 0x000246, 0x000248, 0x000248, 0x00024a, 0x00024a, 0x00024c, 0x00024c, 0x00024e, 0x00024e, 0x000370,
                        0x000370, 0x000372, 0x000372, 0x000376, 0x000376, 0x00037f, 0x00037f, 0x000386, 0x000386, 0x000388, 0x00038a, 0x00038c, 0x00038c, 0x00038e, 0x00038f, 0x000391, 0x0003a1,
                        0x0003a3, 0x0003ab, 0x0003cf, 0x0003cf, 0x0003d2, 0x0003d4, 0x0003d8, 0x0003d8, 0x0003da, 0x0003da, 0x0003dc, 0x0003dc, 0x0003de, 0x0003de, 0x0003e0, 0x0003e0, 0x0003e2,
                        0x0003e2, 0x0003e4, 0x0003e4, 0x0003e6, 0x0003e6, 0x0003e8, 0x0003e8, 0x0003ea, 0x0003ea, 0x0003ec, 0x0003ec, 0x0003ee, 0x0003ee, 0x0003f4, 0x0003f4, 0x0003f7, 0x0003f7,
                        0x0003f9, 0x0003fa, 0x0003fd, 0x00042f, 0x000460, 0x000460, 0x000462, 0x000462, 0x000464, 0x000464, 0x000466, 0x000466, 0x000468, 0x000468, 0x00046a, 0x00046a, 0x00046c,
                        0x00046c, 0x00046e, 0x00046e, 0x000470, 0x000470, 0x000472, 0x000472, 0x000474, 0x000474, 0x000476, 0x000476, 0x000478, 0x000478, 0x00047a, 0x00047a, 0x00047c, 0x00047c,
                        0x00047e, 0x00047e, 0x000480, 0x000480, 0x00048a, 0x00048a, 0x00048c, 0x00048c, 0x00048e, 0x00048e, 0x000490, 0x000490, 0x000492, 0x000492, 0x000494, 0x000494, 0x000496,
                        0x000496, 0x000498, 0x000498, 0x00049a, 0x00049a, 0x00049c, 0x00049c, 0x00049e, 0x00049e, 0x0004a0, 0x0004a0, 0x0004a2, 0x0004a2, 0x0004a4, 0x0004a4, 0x0004a6, 0x0004a6,
                        0x0004a8, 0x0004a8, 0x0004aa, 0x0004aa, 0x0004ac, 0x0004ac, 0x0004ae, 0x0004ae, 0x0004b0, 0x0004b0, 0x0004b2, 0x0004b2, 0x0004b4, 0x0004b4, 0x0004b6, 0x0004b6, 0x0004b8,
                        0x0004b8, 0x0004ba, 0x0004ba, 0x0004bc, 0x0004bc, 0x0004be, 0x0004be, 0x0004c0, 0x0004c1, 0x0004c3, 0x0004c3, 0x0004c5, 0x0004c5, 0x0004c7, 0x0004c7, 0x0004c9, 0x0004c9,
                        0x0004cb, 0x0004cb, 0x0004cd, 0x0004cd, 0x0004d0, 0x0004d0, 0x0004d2, 0x0004d2, 0x0004d4, 0x0004d4, 0x0004d6, 0x0004d6, 0x0004d8, 0x0004d8, 0x0004da, 0x0004da, 0x0004dc,
                        0x0004dc, 0x0004de, 0x0004de, 0x0004e0, 0x0004e0, 0x0004e2, 0x0004e2, 0x0004e4, 0x0004e4, 0x0004e6, 0x0004e6, 0x0004e8, 0x0004e8, 0x0004ea, 0x0004ea, 0x0004ec, 0x0004ec,
                        0x0004ee, 0x0004ee, 0x0004f0, 0x0004f0, 0x0004f2, 0x0004f2, 0x0004f4, 0x0004f4, 0x0004f6, 0x0004f6, 0x0004f8, 0x0004f8, 0x0004fa, 0x0004fa, 0x0004fc, 0x0004fc, 0x0004fe,
                        0x0004fe, 0x000500, 0x000500, 0x000502, 0x000502, 0x000504, 0x000504, 0x000506, 0x000506, 0x000508, 0x000508, 0x00050a, 0x00050a, 0x00050c, 0x00050c, 0x00050e, 0x00050e,
                        0x000510, 0x000510, 0x000512, 0x000512, 0x000514, 0x000514, 0x000516, 0x000516, 0x000518, 0x000518, 0x00051a, 0x00051a, 0x00051c, 0x00051c, 0x00051e, 0x00051e, 0x000520,
                        0x000520, 0x000522, 0x000522, 0x000524, 0x000524, 0x000526, 0x000526, 0x000528, 0x000528, 0x00052a, 0x00052a, 0x00052c, 0x00052c, 0x00052e, 0x00052e, 0x000531, 0x000556,
                        0x0010a0, 0x0010c5, 0x0010c7, 0x0010c7, 0x0010cd, 0x0010cd, 0x0013a0, 0x0013f5, 0x001c90, 0x001cba, 0x001cbd, 0x001cbf, 0x001e00, 0x001e00, 0x001e02, 0x001e02, 0x001e04,
                        0x001e04, 0x001e06, 0x001e06, 0x001e08, 0x001e08, 0x001e0a, 0x001e0a, 0x001e0c, 0x001e0c, 0x001e0e, 0x001e0e, 0x001e10, 0x001e10, 0x001e12, 0x001e12, 0x001e14, 0x001e14,
                        0x001e16, 0x001e16, 0x001e18, 0x001e18, 0x001e1a, 0x001e1a, 0x001e1c, 0x001e1c, 0x001e1e, 0x001e1e, 0x001e20, 0x001e20, 0x001e22, 0x001e22, 0x001e24, 0x001e24, 0x001e26,
                        0x001e26, 0x001e28, 0x001e28, 0x001e2a, 0x001e2a, 0x001e2c, 0x001e2c, 0x001e2e, 0x001e2e, 0x001e30, 0x001e30, 0x001e32, 0x001e32, 0x001e34, 0x001e34, 0x001e36, 0x001e36,
                        0x001e38, 0x001e38, 0x001e3a, 0x001e3a, 0x001e3c, 0x001e3c, 0x001e3e, 0x001e3e, 0x001e40, 0x001e40, 0x001e42, 0x001e42, 0x001e44, 0x001e44, 0x001e46, 0x001e46, 0x001e48,
                        0x001e48, 0x001e4a, 0x001e4a, 0x001e4c, 0x001e4c, 0x001e4e, 0x001e4e, 0x001e50, 0x001e50, 0x001e52, 0x001e52, 0x001e54, 0x001e54, 0x001e56, 0x001e56, 0x001e58, 0x001e58,
                        0x001e5a, 0x001e5a, 0x001e5c, 0x001e5c, 0x001e5e, 0x001e5e, 0x001e60, 0x001e60, 0x001e62, 0x001e62, 0x001e64, 0x001e64, 0x001e66, 0x001e66, 0x001e68, 0x001e68, 0x001e6a,
                        0x001e6a, 0x001e6c, 0x001e6c, 0x001e6e, 0x001e6e, 0x001e70, 0x001e70, 0x001e72, 0x001e72, 0x001e74, 0x001e74, 0x001e76, 0x001e76, 0x001e78, 0x001e78, 0x001e7a, 0x001e7a,
                        0x001e7c, 0x001e7c, 0x001e7e, 0x001e7e, 0x001e80, 0x001e80, 0x001e82, 0x001e82, 0x001e84, 0x001e84, 0x001e86, 0x001e86, 0x001e88, 0x001e88, 0x001e8a, 0x001e8a, 0x001e8c,
                        0x001e8c, 0x001e8e, 0x001e8e, 0x001e90, 0x001e90, 0x001e92, 0x001e92, 0x001e94, 0x001e94, 0x001e9e, 0x001e9e, 0x001ea0, 0x001ea0, 0x001ea2, 0x001ea2, 0x001ea4, 0x001ea4,
                        0x001ea6, 0x001ea6, 0x001ea8, 0x001ea8, 0x001eaa, 0x001eaa, 0x001eac, 0x001eac, 0x001eae, 0x001eae, 0x001eb0, 0x001eb0, 0x001eb2, 0x001eb2, 0x001eb4, 0x001eb4, 0x001eb6,
                        0x001eb6, 0x001eb8, 0x001eb8, 0x001eba, 0x001eba, 0x001ebc, 0x001ebc, 0x001ebe, 0x001ebe, 0x001ec0, 0x001ec0, 0x001ec2, 0x001ec2, 0x001ec4, 0x001ec4, 0x001ec6, 0x001ec6,
                        0x001ec8, 0x001ec8, 0x001eca, 0x001eca, 0x001ecc, 0x001ecc, 0x001ece, 0x001ece, 0x001ed0, 0x001ed0, 0x001ed2, 0x001ed2, 0x001ed4, 0x001ed4, 0x001ed6, 0x001ed6, 0x001ed8,
                        0x001ed8, 0x001eda, 0x001eda, 0x001edc, 0x001edc, 0x001ede, 0x001ede, 0x001ee0, 0x001ee0, 0x001ee2, 0x001ee2, 0x001ee4, 0x001ee4, 0x001ee6, 0x001ee6, 0x001ee8, 0x001ee8,
                        0x001eea, 0x001eea, 0x001eec, 0x001eec, 0x001eee, 0x001eee, 0x001ef0, 0x001ef0, 0x001ef2, 0x001ef2, 0x001ef4, 0x001ef4, 0x001ef6, 0x001ef6, 0x001ef8, 0x001ef8, 0x001efa,
                        0x001efa, 0x001efc, 0x001efc, 0x001efe, 0x001efe, 0x001f08, 0x001f0f, 0x001f18, 0x001f1d, 0x001f28, 0x001f2f, 0x001f38, 0x001f3f, 0x001f48, 0x001f4d, 0x001f59, 0x001f59,
                        0x001f5b, 0x001f5b, 0x001f5d, 0x001f5d, 0x001f5f, 0x001f5f, 0x001f68, 0x001f6f, 0x001f88, 0x001f8f, 0x001f98, 0x001f9f, 0x001fa8, 0x001faf, 0x001fb8, 0x001fbc, 0x001fc8,
                        0x001fcc, 0x001fd8, 0x001fdb, 0x001fe8, 0x001fec, 0x001ff8, 0x001ffc, 0x002102, 0x002102, 0x002107, 0x002107, 0x00210b, 0x00210d, 0x002110, 0x002112, 0x002115, 0x002115,
                        0x002119, 0x00211d, 0x002124, 0x002124, 0x002126, 0x002126, 0x002128, 0x002128, 0x00212a, 0x00212d, 0x002130, 0x002133, 0x00213e, 0x00213f, 0x002145, 0x002145, 0x002160,
                        0x00216f, 0x002183, 0x002183, 0x0024b6, 0x0024cf, 0x002c00, 0x002c2f, 0x002c60, 0x002c60, 0x002c62, 0x002c64, 0x002c67, 0x002c67, 0x002c69, 0x002c69, 0x002c6b, 0x002c6b,
                        0x002c6d, 0x002c70, 0x002c72, 0x002c72, 0x002c75, 0x002c75, 0x002c7e, 0x002c80, 0x002c82, 0x002c82, 0x002c84, 0x002c84, 0x002c86, 0x002c86, 0x002c88, 0x002c88, 0x002c8a,
                        0x002c8a, 0x002c8c, 0x002c8c, 0x002c8e, 0x002c8e, 0x002c90, 0x002c90, 0x002c92, 0x002c92, 0x002c94, 0x002c94, 0x002c96, 0x002c96, 0x002c98, 0x002c98, 0x002c9a, 0x002c9a,
                        0x002c9c, 0x002c9c, 0x002c9e, 0x002c9e, 0x002ca0, 0x002ca0, 0x002ca2, 0x002ca2, 0x002ca4, 0x002ca4, 0x002ca6, 0x002ca6, 0x002ca8, 0x002ca8, 0x002caa, 0x002caa, 0x002cac,
                        0x002cac, 0x002cae, 0x002cae, 0x002cb0, 0x002cb0, 0x002cb2, 0x002cb2, 0x002cb4, 0x002cb4, 0x002cb6, 0x002cb6, 0x002cb8, 0x002cb8, 0x002cba, 0x002cba, 0x002cbc, 0x002cbc,
                        0x002cbe, 0x002cbe, 0x002cc0, 0x002cc0, 0x002cc2, 0x002cc2, 0x002cc4, 0x002cc4, 0x002cc6, 0x002cc6, 0x002cc8, 0x002cc8, 0x002cca, 0x002cca, 0x002ccc, 0x002ccc, 0x002cce,
                        0x002cce, 0x002cd0, 0x002cd0, 0x002cd2, 0x002cd2, 0x002cd4, 0x002cd4, 0x002cd6, 0x002cd6, 0x002cd8, 0x002cd8, 0x002cda, 0x002cda, 0x002cdc, 0x002cdc, 0x002cde, 0x002cde,
                        0x002ce0, 0x002ce0, 0x002ce2, 0x002ce2, 0x002ceb, 0x002ceb, 0x002ced, 0x002ced, 0x002cf2, 0x002cf2, 0x00a640, 0x00a640, 0x00a642, 0x00a642, 0x00a644, 0x00a644, 0x00a646,
                        0x00a646, 0x00a648, 0x00a648, 0x00a64a, 0x00a64a, 0x00a64c, 0x00a64c, 0x00a64e, 0x00a64e, 0x00a650, 0x00a650, 0x00a652, 0x00a652, 0x00a654, 0x00a654, 0x00a656, 0x00a656,
                        0x00a658, 0x00a658, 0x00a65a, 0x00a65a, 0x00a65c, 0x00a65c, 0x00a65e, 0x00a65e, 0x00a660, 0x00a660, 0x00a662, 0x00a662, 0x00a664, 0x00a664, 0x00a666, 0x00a666, 0x00a668,
                        0x00a668, 0x00a66a, 0x00a66a, 0x00a66c, 0x00a66c, 0x00a680, 0x00a680, 0x00a682, 0x00a682, 0x00a684, 0x00a684, 0x00a686, 0x00a686, 0x00a688, 0x00a688, 0x00a68a, 0x00a68a,
                        0x00a68c, 0x00a68c, 0x00a68e, 0x00a68e, 0x00a690, 0x00a690, 0x00a692, 0x00a692, 0x00a694, 0x00a694, 0x00a696, 0x00a696, 0x00a698, 0x00a698, 0x00a69a, 0x00a69a, 0x00a722,
                        0x00a722, 0x00a724, 0x00a724, 0x00a726, 0x00a726, 0x00a728, 0x00a728, 0x00a72a, 0x00a72a, 0x00a72c, 0x00a72c, 0x00a72e, 0x00a72e, 0x00a732, 0x00a732, 0x00a734, 0x00a734,
                        0x00a736, 0x00a736, 0x00a738, 0x00a738, 0x00a73a, 0x00a73a, 0x00a73c, 0x00a73c, 0x00a73e, 0x00a73e, 0x00a740, 0x00a740, 0x00a742, 0x00a742, 0x00a744, 0x00a744, 0x00a746,
                        0x00a746, 0x00a748, 0x00a748, 0x00a74a, 0x00a74a, 0x00a74c, 0x00a74c, 0x00a74e, 0x00a74e, 0x00a750, 0x00a750, 0x00a752, 0x00a752, 0x00a754, 0x00a754, 0x00a756, 0x00a756,
                        0x00a758, 0x00a758, 0x00a75a, 0x00a75a, 0x00a75c, 0x00a75c, 0x00a75e, 0x00a75e, 0x00a760, 0x00a760, 0x00a762, 0x00a762, 0x00a764, 0x00a764, 0x00a766, 0x00a766, 0x00a768,
                        0x00a768, 0x00a76a, 0x00a76a, 0x00a76c, 0x00a76c, 0x00a76e, 0x00a76e, 0x00a779, 0x00a779, 0x00a77b, 0x00a77b, 0x00a77d, 0x00a77e, 0x00a780, 0x00a780, 0x00a782, 0x00a782,
                        0x00a784, 0x00a784, 0x00a786, 0x00a786, 0x00a78b, 0x00a78b, 0x00a78d, 0x00a78d, 0x00a790, 0x00a790, 0x00a792, 0x00a792, 0x00a796, 0x00a796, 0x00a798, 0x00a798, 0x00a79a,
                        0x00a79a, 0x00a79c, 0x00a79c, 0x00a79e, 0x00a79e, 0x00a7a0, 0x00a7a0, 0x00a7a2, 0x00a7a2, 0x00a7a4, 0x00a7a4, 0x00a7a6, 0x00a7a6, 0x00a7a8, 0x00a7a8, 0x00a7aa, 0x00a7ae,
                        0x00a7b0, 0x00a7b4, 0x00a7b6, 0x00a7b6, 0x00a7b8, 0x00a7b8, 0x00a7ba, 0x00a7ba, 0x00a7bc, 0x00a7bc, 0x00a7be, 0x00a7be, 0x00a7c0, 0x00a7c0, 0x00a7c2, 0x00a7c2, 0x00a7c4,
                        0x00a7c7, 0x00a7c9, 0x00a7c9, 0x00a7d0, 0x00a7d0, 0x00a7d6, 0x00a7d6, 0x00a7d8, 0x00a7d8, 0x00a7f5, 0x00a7f5, 0x00ff21, 0x00ff3a, 0x010400, 0x010427, 0x0104b0, 0x0104d3,
                        0x010570, 0x01057a, 0x01057c, 0x01058a, 0x01058c, 0x010592, 0x010594, 0x010595, 0x010c80, 0x010cb2, 0x0118a0, 0x0118bf, 0x016e40, 0x016e5f, 0x01d400, 0x01d419, 0x01d434,
                        0x01d44d, 0x01d468, 0x01d481, 0x01d49c, 0x01d49c, 0x01d49e, 0x01d49f, 0x01d4a2, 0x01d4a2, 0x01d4a5, 0x01d4a6, 0x01d4a9, 0x01d4ac, 0x01d4ae, 0x01d4b5, 0x01d4d0, 0x01d4e9,
                        0x01d504, 0x01d505, 0x01d507, 0x01d50a, 0x01d50d, 0x01d514, 0x01d516, 0x01d51c, 0x01d538, 0x01d539, 0x01d53b, 0x01d53e, 0x01d540, 0x01d544, 0x01d546, 0x01d546, 0x01d54a,
                        0x01d550, 0x01d56c, 0x01d585, 0x01d5a0, 0x01d5b9, 0x01d5d4, 0x01d5ed, 0x01d608, 0x01d621, 0x01d63c, 0x01d655, 0x01d670, 0x01d689, 0x01d6a8, 0x01d6c0, 0x01d6e2, 0x01d6fa,
                        0x01d71c, 0x01d734, 0x01d756, 0x01d76e, 0x01d790, 0x01d7a8, 0x01d7ca, 0x01d7ca, 0x01e900, 0x01e921));

        POSIX_CHAR_CLASSES.put("xdigit", CodePointSet.createNoDedup(
                        0x000030, 0x000039, 0x000041, 0x000046, 0x000061, 0x000066, 0x00ff10, 0x00ff19, 0x00ff21, 0x00ff26, 0x00ff41, 0x00ff46));

        /* GENERATED CODE END - KEEP THIS MARKER FOR AUTOMATIC UPDATES */

        POSIX_CHAR_CLASSES.put("alnum", POSIX_CHAR_CLASSES.get("alpha").union(POSIX_CHAR_CLASSES.get("digit")));
        WORD_CHARACTERS = POSIX_CHAR_CLASSES.get("alnum").union(CodePointSet.create('_'));
    }
}
