/*
 * Copyright (c) 2025, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.regex.tregex.nodes.dfa;

import java.util.Objects;

import com.oracle.truffle.api.CompilerAsserts;

public final class CGTrackingTransitionNode extends CGTrackingAbstractTransitionNode {

    private final DFACaptureGroupLazyTransition transition;
    private final short lastTransitionIndex;

    public CGTrackingTransitionNode(short id, short successor, DFACaptureGroupLazyTransition transition, short lastTransitionIndex) {
        super(id, successor);
        this.transition = transition;
        this.lastTransitionIndex = lastTransitionIndex;
    }

    public static CGTrackingTransitionNode create(short id, short successor, DFACaptureGroupLazyTransition transition, short lastTransitionIndex) {
        if (transition.isEmpty() && lastTransitionIndex < 0) {
            return null;
        }
        return new CGTrackingTransitionNode(id, successor, transition, lastTransitionIndex);
    }

    @Override
    public int getCGTrackingCost() {
        return transition.getCost();
    }

    @Override
    public void apply(TRegexDFAExecutorLocals locals, TRegexDFAExecutorNode executor) {
        CompilerAsserts.partialEvaluationConstant(this);
        transition.apply(locals, executor, false);
        if (lastTransitionIndex >= 0) {
            locals.setLastTransition(lastTransitionIndex);
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof CGTrackingTransitionNode o)) {
            return false;
        }
        return getSuccessor() == o.getSuccessor() && lastTransitionIndex == o.lastTransitionIndex && Objects.equals(transition, o.transition);
    }

    @Override
    public int hashCode() {
        return Objects.hash(getSuccessor(), transition, lastTransitionIndex);
    }
}
