/*
 * Copyright (c) 2023, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.oracle.svm.configure.command;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Iterator;
import java.util.List;

import com.oracle.svm.configure.ConfigurationUsageException;

public final class ConfigurationCommandFileCommand extends ConfigurationCommand {
    @Override
    public String getName() {
        return "command-file";
    }

    @Override
    public void apply(Iterator<String> argumentsIterator) throws IOException {
        throw new UnsupportedOperationException("Operation is not supported.");
    }

    public static Iterator<String> handleCommandFile(Iterator<String> argumentsIterator) {
        if (!argumentsIterator.hasNext()) {
            throw new ConfigurationUsageException("Path to a command file must be provided.");
        }
        Path filePath = Paths.get(argumentsIterator.next());

        if (argumentsIterator.hasNext()) {
            throw new ConfigurationUsageException("Too many arguments to command-file passed. Expected a single argument: <path to a command file>.");
        }

        try {
            List<String> lines = Files.readAllLines(filePath);
            return lines.iterator();
        } catch (IOException e) {
            throw new ConfigurationUsageException(
                            "Failed to read the command file at " + filePath + ". Check if the file exists, you have the required permissions and that the file is actually a text file.");
        }
    }

    @Override
    public String getUsage() {
        return "native-image-configure command-file <command-file-path>";
    }

    @Override
    protected String getDescription0() {
        return """
                                      reads the command-line arguments from the given file.
                                                  The command-file contains one argument per line, with
                                                  no quoting of whitespace necessary (newline
                                                  characters in an argument cannot be represented).
                                                  An example command file:
                                                  generate
                                                  --input-dir=input-dir-1-path
                                                  --input-dir=input-dir-2-path
                                                  --output-dir=output-dir-path
                                                  This option is meant for executing commands with
                                                  an argument list that is too large to be passed by
                                                  regular means, which might be generated by other
                                                  tools.
                            <command-file-path>
                                                  specifies the path to a command file
                        """.replaceAll("\n", System.lineSeparator());
    }
}
