/*
 * Copyright (c) 2025, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.svm.test;

import com.oracle.svm.hosted.jdk.jniRegistration.BaseClassMemberRegistration;
import com.oracle.svm.hosted.jdk.jniRegistration.ClassJavaAwtWindowRegistration;
import com.oracle.svm.hosted.jdk.jniRegistration.ClassSunFont2DRegistration;
import com.oracle.svm.hosted.jdk.jniRegistration.ClassSunJava2dWindowsWindowsFlags;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Test;

import java.io.File;
import java.io.IOException;

import java.lang.classfile.ClassFile;
import java.lang.classfile.ClassModel;
import java.lang.classfile.FieldModel;
import java.lang.classfile.MethodModel;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * {@see mx.substratevm/suite.py:290 } This shaded ASM project is just a quickfix. Eventually, we
 * should migrate to the Classfile API [JDK-8294982] (GR-61102). {@href <a href=
 * "https://bugs.openjdk.org/browse/JDK-8294982?page=com.atlassian.jira.plugin.system.issuetabpanels%3Aall-tabpanel">JDK-8294982</a>}
 * ... "shadedDependencies" : [ "compiler:ASM_9.7.1", "compiler:ASM_TREE_9.7.1", ], ...
 */
public class JNIClassMemberRegistrationTest {
    private static final String CLASSES_SUB_DIRECTORY = "classes/";
    private static final String CLASS_EXT = ".class";
    private static final String JAVA_HOME = System.getProperty("java.home");

    private static final Set<String> VERIFIED_CLASS_NAMES = new HashSet<>();
    private static final Map<String, byte[]> VERIFIED_CLASS_TO_CONTENT;

    static {
        Assert.assertNotNull(JAVA_HOME);
        VERIFIED_CLASS_NAMES.add("java.awt.Window");
        VERIFIED_CLASS_NAMES.add("sun.font.Font2D");
        if (isWindows()) {
            VERIFIED_CLASS_NAMES.add("sun.java2d.windows.WindowsFlags");
        }
        VERIFIED_CLASS_TO_CONTENT = readClassesFromJMODArchives(JAVA_HOME);
        Assert.assertEquals(VERIFIED_CLASS_NAMES.size(), VERIFIED_CLASS_TO_CONTENT.size());
    }

    @Test
    public void testAwtWindowMemberAccessMethod() {
        testWorker(new ClassJavaAwtWindowRegistration());
    }

    @Test
    public void testSunJava2dWindowsWindowsFlagsMemberAccess() {
        Assume.assumeTrue("Skip test for Windows OS", isWindows());
        testWorker(new ClassSunJava2dWindowsWindowsFlags());
    }

    @Test
    public void testSunJava2dFont2DMemberAccess() {
        testWorker(new ClassSunFont2DRegistration());
    }

    private static void testWorker(BaseClassMemberRegistration expectedMembers) {
        String testedClassName = expectedMembers.getClassName();
        Assert.assertNotNull(expectedMembers);

        byte[] classData = VERIFIED_CLASS_TO_CONTENT.get(testedClassName);
        Assert.assertNotNull(expectedMembers);

        ClassFileData classFileData = ClassFileData.read(classData);
        Assert.assertNotNull(classFileData);

        for (String fieldName : expectedMembers.getFields()) {
            Assert.assertTrue(classFileData.fieldNames.contains(fieldName));
        }

        for (BaseClassMemberRegistration.MethodSignature signature : expectedMembers.getMethods()) {
            Assert.assertTrue(classFileData.methodDescriptors.contains(signature));
        }
    }

    private static String zippedClassPathToClassName(String relativeClassPath) {
        String classPath = relativeClassPath.replace(CLASSES_SUB_DIRECTORY, "");
        String className = classPath.replace(CLASS_EXT, "");
        return replaceSlashWithDot(className);
    }

    private static String replaceSlashWithDot(String className) {
        return className.replace('/', '.');
    }

    private static Map<String, byte[]> readClassesFromJMODArchives(String javaHome) {
        File jmodDirectory = new File(javaHome, "jmods");
        Assert.assertTrue(jmodDirectory.exists());

        File[] jmodFiles = jmodDirectory.listFiles();
        if (jmodFiles == null) {
            Assert.fail("Cannot find .jmods in JAVA_HOME");
        }

        Map<String, byte[]> classToContent = new HashMap<>();
        for (File jmod : jmodFiles) {
            String fileName = jmod.getName();
            if (!fileName.endsWith(".jmod")) {
                continue;
            }
            Path absolutePath = jmod.toPath().toAbsolutePath();
            File jmodFile = new File(absolutePath.toString());

            try (var zipFile = new ZipFile(jmodFile)) {
                for (var entries = zipFile.entries(); entries.hasMoreElements();) {
                    ZipEntry entry = entries.nextElement();
                    String zipBasedClassPath = entry.getName();
                    String classFQNName = zippedClassPathToClassName(zipBasedClassPath);
                    boolean isInClassesDirectory = zipBasedClassPath.startsWith(CLASSES_SUB_DIRECTORY);
                    boolean expectedToBeChecked = VERIFIED_CLASS_NAMES.contains(classFQNName);
                    if (isInClassesDirectory && expectedToBeChecked) {
                        try (var is = zipFile.getInputStream(entry)) {
                            byte[] classBytes = is.readAllBytes();
                            Assert.assertTrue(classBytes.length > 0);
                            classToContent.putIfAbsent(classFQNName, classBytes);
                        } catch (IOException e) {
                            Assert.fail("Could not read class file " + classFQNName + ".\n" + e.getMessage());
                        }
                    }
                }
            } catch (IOException e) {
                Assert.fail("Could not read jmod file " + jmod.getAbsolutePath() + ".\n" + e.getMessage());
            }
        }
        return classToContent;
    }

    private static boolean isWindows() {
        String osName = System.getProperty("os.name");
        if (osName == null) {
            throw new RuntimeException("Cannot determine OS");
        }
        return osName.contains("Windows");
    }

    private record ClassFileData(
                    String className,
                    Set<String> fieldNames,
                    Set<BaseClassMemberRegistration.MethodSignature> methodDescriptors) {

        public static ClassFileData read(byte[] classData) {
            try {
                ClassFile cf = ClassFile.of();
                ClassModel classModel = cf.parse(classData);

                Set<String> fieldNames = new HashSet<>();
                Set<BaseClassMemberRegistration.MethodSignature> methodDescriptors = new HashSet<>();
                List<FieldModel> fields = classModel.fields();
                for (FieldModel field : fields) {
                    fieldNames.add(field.fieldName().stringValue());
                }

                List<MethodModel> methods = classModel.methods();
                for (MethodModel method : methods) {
                    var methodData = new BaseClassMemberRegistration.MethodSignature(
                            method.methodName().stringValue(), method.methodTypeSymbol());
                    methodDescriptors.add(methodData);

                }
                String className = replaceSlashWithDot(classModel.thisClass().name().stringValue());
                return new ClassFileData(className, Set.copyOf(fieldNames), Set.copyOf(methodDescriptors));
            } catch (Exception e) {
                System.err.println(e.getLocalizedMessage());
                return null;
            }
        }
    }
}
