/*
 * Copyright (c) 2021, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.oracle.truffle.api.strings.test;

import static com.oracle.truffle.api.strings.TruffleString.Encoding.UTF_16;
import static com.oracle.truffle.api.strings.TruffleString.Encoding.UTF_32;

import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.stream.Collectors;

import org.graalvm.collections.EconomicMap;
import org.graalvm.shadowed.org.jcodings.Encoding;
import org.graalvm.shadowed.org.jcodings.EncodingDB;
import org.graalvm.shadowed.org.jcodings.util.CaseInsensitiveBytesHash;

import com.oracle.truffle.api.strings.TruffleString;

public final class Encodings {

    public static final TruffleString.Encoding[] PRIMARY_ENCODINGS = {
                    TruffleString.Encoding.US_ASCII,
                    TruffleString.Encoding.ISO_8859_1,
                    TruffleString.Encoding.BYTES,
                    TruffleString.Encoding.UTF_8,
                    UTF_16,
                    UTF_32};
    private static final int MAX_J_CODINGS_INDEX_VALUE = 0x7f;
    static final EconomicMap<String, Encoding> J_CODINGS_MAP = createJCodingsMap();

    static Encoding getJCoding(TruffleString.Encoding encoding) {
        if (encoding == UTF_16) {
            return J_CODINGS_MAP.get(ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN ? "UTF_16LE" : "UTF_16BE");
        }
        if (encoding == UTF_32) {
            return J_CODINGS_MAP.get(ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN ? "UTF_32LE" : "UTF_32BE");
        }
        return J_CODINGS_MAP.get(encoding.name());
    }

    private static EconomicMap<String, Encoding> createJCodingsMap() {
        CaseInsensitiveBytesHash<EncodingDB.Entry> encodings = EncodingDB.getEncodings();
        if (encodings.size() > MAX_J_CODINGS_INDEX_VALUE) {
            throw new RuntimeException(String.format("Assumption broken: org.graalvm.shadowed.org.jcodings has more than %d encodings (actual: %d)!", MAX_J_CODINGS_INDEX_VALUE, encodings.size()));
        }
        EconomicMap<String, Encoding> allEncodings = EconomicMap.create(encodings.size());
        for (EncodingDB.Entry entry : encodings) {
            Encoding enc = entry.getEncoding();
            int i = enc.getIndex();
            if (i < 0 || i >= encodings.size()) {
                throw new RuntimeException(String.format(
                                "Assumption broken: index of org.graalvm.shadowed.org.jcodings encoding \"%s\" is greater than number of encodings (index: %d, number of encodings: %d)!", enc, i,
                                encodings.size()));
            }
            allEncodings.put(toEnumName(enc.toString()), enc);
        }
        return allEncodings;
    }

    static String toEnumName(String encodingName) {
        if ("ASCII-8BIT".equals(encodingName)) {
            return "BYTES";
        }
        String capitalized = encodingName;
        if (Character.isLowerCase(encodingName.charAt(0))) {
            capitalized = Character.toUpperCase(encodingName.charAt(0)) + encodingName.substring(1);
        }
        return capitalized.replace('-', '_');
    }

    public static final class TestData {

        public final int[] codepoints;
        public final int[] byteIndices;
        final byte[] encodedAscii;
        final byte[] encodedLatin;
        final byte[] encodedBMP;
        final byte[] encodedValid;
        final byte[] encodedBroken;
        public final int[] codepointsBroken;

        private TestData(int[] codepoints, int[] byteIndices, byte[] encodedAscii, byte[] encodedLatin, byte[] encodedBMP, byte[] encodedValid, byte[] encodedBroken, int[] codepointsBroken) {
            this.codepoints = codepoints;
            this.byteIndices = byteIndices;
            this.encodedAscii = encodedAscii;
            this.encodedLatin = encodedLatin;
            this.encodedBMP = encodedBMP;
            this.encodedValid = encodedValid;
            this.encodedBroken = encodedBroken;
            this.codepointsBroken = codepointsBroken;
        }

        private String serialize() {
            StringBuilder sb = new StringBuilder("new TestData(").append(serialize(codepoints)).append(", ").append(serialize(byteIndices));
            for (byte[] array : new byte[][]{encodedAscii, encodedLatin, encodedBMP, encodedValid, encodedBroken}) {
                sb.append(", ").append(serialize(array));
            }
            return sb.append(", ").append(serialize(codepointsBroken)).append(")").toString();
        }

        private static String serialize(int[] array) {
            if (array == null) {
                return "null";
            }
            String formatString = "0x%0" + Integer.toHexString(array[array.length - 1]).length() + "x";
            return Arrays.stream(array).mapToObj(i -> String.format(formatString, i)).collect(Collectors.joining(", ", "new int[]{", "}"));
        }

        private static String serialize(byte[] array) {
            if (array == null) {
                return "null";
            }
            StringBuilder sb = new StringBuilder("new byte[]{");
            String formatString = "0x%02x";
            for (int i = 0; i < array.length; i++) {
                if (i > 0) {
                    sb.append(", ");
                }
                sb.append("(byte) ").append(String.format(formatString, array[i]));
            }
            return sb.append("}").toString();
        }
    }

    /**
     * Generate unit test character tables for all encodings. Invoke with the following mx command:
     *
     * <pre>
     * mx java -cp `mx paths truffle:TRUFFLE_API`:`mx paths truffle:TRUFFLE_TEST`:`mx paths sdk:GRAAL_SDK` com.oracle.truffle.api.strings.test.Encodings
     * </pre>
     */
    public static void main(String[] args) {
        byte[] bytesAscii = {0x00, 0x7f};
        byte[] bytesLatin = {0x00, (byte) 0xff};
        int[] indices01 = {0, 1};
        int[] codepointsUTF8 = {0x000000, 0x00007f, 0x000080, 0x00d7ff, 0x00e000, 0x10ffff};
        int[] codepointsUTF16 = {0x000000, 0x00007f, 0x000080, 0x0000ff, 0x000100, 0x00d7ff, 0x00e000, 0x00ffff, 0x010000, 0x10ffff};
        TestData[] testData = new TestData[TruffleString.Encoding.values().length];
        testData[TruffleString.Encoding.US_ASCII.ordinal()] = new TestData(
                        new int[]{0x00, 0x7f},
                        indices01, bytesAscii, null, null, bytesAscii, new byte[]{(byte) 0xff}, new int[]{0xff});
        testData[TruffleString.Encoding.ISO_8859_1.ordinal()] = new TestData(
                        new int[]{0x00, 0xff},
                        new int[]{0, 1}, bytesAscii, bytesLatin, null, bytesLatin, null, null);
        testData[TruffleString.Encoding.UTF_8.ordinal()] = new TestData(
                        codepointsUTF8,
                        codePointByteIndices(codepointsUTF8, getJCoding(TruffleString.Encoding.UTF_8)),
                        bytesAscii,
                        null,
                        null,
                        encodeCodePoints(codepointsUTF8, getJCoding(TruffleString.Encoding.UTF_8)),
                        new byte[]{(byte) 0xc0}, new int[]{0xfffd});
        testData[TruffleString.Encoding.UTF_16.ordinal()] = new TestData(
                        codepointsUTF16,
                        codePointByteIndices(codepointsUTF16, getJCoding(TruffleString.Encoding.UTF_16)),
                        asBytes(new int[]{0x00, 0x7f}, 1),
                        asBytes(new int[]{0x00, 0xff}, 1),
                        asBytes(new int[]{0x0000, 0xffff}, 1),
                        encodeCodePoints(codepointsUTF16, getJCoding(TruffleString.Encoding.UTF_16)),
                        asBytes(new int[]{Character.MIN_LOW_SURROGATE}, 1),
                        new int[]{Character.MIN_LOW_SURROGATE});
        testData[TruffleString.Encoding.UTF_32.ordinal()] = new TestData(
                        codepointsUTF16,
                        codePointByteIndices(codepointsUTF16, getJCoding(TruffleString.Encoding.UTF_32)),
                        asBytes(new int[]{0x00, 0x7f}, 2),
                        asBytes(new int[]{0x00, 0xff}, 2),
                        asBytes(new int[]{0x0000, 0xffff}, 2),
                        asBytes(codepointsUTF16, 2),
                        asBytes(new int[]{Character.MIN_LOW_SURROGATE}, 2),
                        new int[]{Character.MIN_LOW_SURROGATE});
        testData[TruffleString.Encoding.BYTES.ordinal()] = new TestData(
                        new int[]{0x00, 0x7f, 0x80, 0xff},
                        new int[]{0, 1, 2, 3}, null, null, null, new byte[]{0x00, 0x7f, (byte) 0x80, (byte) 0xff}, null, null);
        for (TruffleString.Encoding e : TruffleString.Encoding.values()) {
            Encoding jCoding = getJCoding(e);
            if (testData[e.ordinal()] == null) {
                int iBuf = 0;
                int[] buf = new int[16];
                int lo = -1;
                for (int i = 0; i < 0x10ffff; i++) {
                    if (isValidCodePoint(i, jCoding)) {
                        if (lo < 0) {
                            lo = i;
                        }
                    } else {
                        if (lo >= 0) {
                            if (iBuf + 2 >= buf.length) {
                                buf = Arrays.copyOf(buf, buf.length * 2);
                            }
                            buf[iBuf++] = lo;
                            buf[iBuf++] = i - 1;
                            lo = -1;
                        }
                    }
                }
                if (lo >= 0) {
                    if (iBuf + 2 >= buf.length) {
                        buf = Arrays.copyOf(buf, buf.length * 2);
                    }
                    buf[iBuf++] = lo;
                    buf[iBuf++] = 0x10ffff;
                }
                int[] codepoints = Arrays.copyOf(buf, iBuf);

                byte[] encodedBroken = null;
                int[] codepointsBroken = null;

                if (e == TruffleString.Encoding.UTF_16BE) {
                    encodedBroken = asBytes(new int[]{Character.MIN_LOW_SURROGATE}, 1, ByteOrder.BIG_ENDIAN);
                    codepointsBroken = new int[]{Character.MIN_LOW_SURROGATE};
                } else if (e == TruffleString.Encoding.UTF_32BE) {
                    encodedBroken = asBytes(new int[]{Character.MIN_LOW_SURROGATE}, 2, ByteOrder.BIG_ENDIAN);
                    codepointsBroken = new int[]{Character.MIN_LOW_SURROGATE};
                }

                testData[e.ordinal()] = new TestData(
                                codepoints,
                                codePointByteIndices(codepoints, jCoding),
                                jCoding.isAsciiCompatible() ? bytesAscii : null,
                                null,
                                null,
                                encodeCodePoints(codepoints, jCoding),
                                encodedBroken,
                                codepointsBroken);
            }
        }
        // Checkstyle: stop
        System.out.println("static final TestData[] TEST_DATA = {");
        for (TruffleString.Encoding e : TruffleString.Encoding.values()) {
            System.out.println("data" + e.name().replace("_", "") + "(),");
        }
        System.out.println("};");
        for (TruffleString.Encoding e : TruffleString.Encoding.values()) {
            System.out.println("static TestData data" + e.name().replace("_", "") + "() {");
            System.out.println("assert TruffleString.Encoding." + e.name() + ".ordinal() == " + e.ordinal() + ";");
            System.out.println("return " + testData[e.ordinal()].serialize() + ";");
            System.out.println("}");
        }
        // Checkstyle: resume
    }

    private static byte[] encodeCodePoints(int[] codepoints, Encoding jCoding) {
        int length = 0;
        for (int c : codepoints) {
            assert isValidCodePoint(c, jCoding);
            length += jCoding.codeToMbcLength(c);
        }
        byte[] ret = new byte[length];
        int i = 0;
        for (int c : codepoints) {
            i += jCoding.codeToMbc(c, ret, i);
        }
        return ret;
    }

    public static int[] codePointByteIndices(int[] codepoints, Encoding jCoding) {
        int[] ret = new int[codepoints.length];
        int length = 0;
        for (int i = 0; i < codepoints.length; i++) {
            ret[i] = length;
            length += jCoding.codeToMbcLength(codepoints[i]);
        }
        return ret;
    }

    private static byte[] asBytes(int[] values, int stride) {
        return asBytes(values, stride, ByteOrder.nativeOrder());
    }

    private static byte[] asBytes(int[] values, int stride, ByteOrder byteOrder) {
        byte[] ret = new byte[values.length << stride];
        for (int i = 0; i < values.length; i++) {
            TStringTestUtil.writeValue(ret, stride, i, values[i], byteOrder);
        }
        return ret;
    }

    static boolean isValidCodePoint(int codePoint, Encoding jCoding) {
        int length = jCoding.codeToMbcLength(codePoint);
        if (length < 1) {
            return false;
        }
        byte[] tmp = new byte[length];
        int lengthEncoded = jCoding.codeToMbc(codePoint, tmp, 0);
        return lengthEncoded == length && jCoding.length(tmp, 0, length) == length && jCoding.mbcToCode(tmp, 0, length) == codePoint;
    }

    static int[] codepoints(TruffleString.Encoding encoding) {
        return TEST_DATA[encoding.ordinal()].codepoints;
    }

    /**
     * GENERATED CODE.
     */
    static final TestData[] TEST_DATA = {
                    dataUTF32LE(),
                    dataUTF32BE(),
                    dataUTF16LE(),
                    dataUTF16BE(),
                    dataISO88591(),
                    dataUTF8(),
                    dataUSASCII(),
                    dataBYTES(),
                    dataBig5(),
                    dataBig5HKSCS(),
                    dataBig5UAO(),
                    dataCESU8(),
                    dataCP51932(),
                    dataCP850(),
                    dataCP852(),
                    dataCP855(),
                    dataCP949(),
                    dataCP950(),
                    dataCP951(),
                    dataEUCJIS2004(),
                    dataEUCJP(),
                    dataEUCKR(),
                    dataEUCTW(),
                    dataEmacsMule(),
                    dataEucJPms(),
                    dataGB12345(),
                    dataGB18030(),
                    dataGB1988(),
                    dataGB2312(),
                    dataGBK(),
                    dataIBM437(),
                    dataIBM720(),
                    dataIBM737(),
                    dataIBM775(),
                    dataIBM852(),
                    dataIBM855(),
                    dataIBM857(),
                    dataIBM860(),
                    dataIBM861(),
                    dataIBM862(),
                    dataIBM863(),
                    dataIBM864(),
                    dataIBM865(),
                    dataIBM866(),
                    dataIBM869(),
                    dataISO885910(),
                    dataISO885911(),
                    dataISO885913(),
                    dataISO885914(),
                    dataISO885915(),
                    dataISO885916(),
                    dataISO88592(),
                    dataISO88593(),
                    dataISO88594(),
                    dataISO88595(),
                    dataISO88596(),
                    dataISO88597(),
                    dataISO88598(),
                    dataISO88599(),
                    dataKOI8R(),
                    dataKOI8U(),
                    dataMacCentEuro(),
                    dataMacCroatian(),
                    dataMacCyrillic(),
                    dataMacGreek(),
                    dataMacIceland(),
                    dataMacJapanese(),
                    dataMacRoman(),
                    dataMacRomania(),
                    dataMacThai(),
                    dataMacTurkish(),
                    dataMacUkraine(),
                    dataSJISDoCoMo(),
                    dataSJISKDDI(),
                    dataSJISSoftBank(),
                    dataShiftJIS(),
                    dataStatelessISO2022JP(),
                    dataStatelessISO2022JPKDDI(),
                    dataTIS620(),
                    dataUTF8DoCoMo(),
                    dataUTF8KDDI(),
                    dataUTF8MAC(),
                    dataUTF8SoftBank(),
                    dataWindows1250(),
                    dataWindows1251(),
                    dataWindows1252(),
                    dataWindows1253(),
                    dataWindows1254(),
                    dataWindows1255(),
                    dataWindows1256(),
                    dataWindows1257(),
                    dataWindows1258(),
                    dataWindows31J(),
                    dataWindows874(),
                    dataCP50220(),
                    dataCP50221(),
                    dataIBM037(),
                    dataISO2022JP(),
                    dataISO2022JP2(),
                    dataISO2022JPKDDI(),
                    dataUTF7(),
    };

    static TestData dataUTF32LE() {
        assert TruffleString.Encoding.UTF_32LE.ordinal() == 0;
        return new TestData(new int[]{0x000000, 0x00007f, 0x000080, 0x0000ff, 0x000100, 0x00d7ff, 0x00e000, 0x00ffff, 0x010000, 0x10ffff},
                        new int[]{0x00, 0x04, 0x08, 0x0c, 0x10, 0x14, 0x18, 0x1c, 0x20, 0x24},
                        new byte[]{(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x7f, (byte) 0x00, (byte) 0x00, (byte) 0x00},
                        new byte[]{(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0xff, (byte) 0x00, (byte) 0x00, (byte) 0x00},
                        new byte[]{(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0xff, (byte) 0xff, (byte) 0x00, (byte) 0x00},
                        new byte[]{(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x7f, (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x80, (byte) 0x00, (byte) 0x00, (byte) 0x00,
                                        (byte) 0xff, (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x01, (byte) 0x00, (byte) 0x00, (byte) 0xff, (byte) 0xd7, (byte) 0x00, (byte) 0x00,
                                        (byte) 0x00, (byte) 0xe0, (byte) 0x00, (byte) 0x00, (byte) 0xff, (byte) 0xff, (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x01, (byte) 0x00,
                                        (byte) 0xff, (byte) 0xff, (byte) 0x10, (byte) 0x00},
                        new byte[]{(byte) 0x00, (byte) 0xdc, (byte) 0x00, (byte) 0x00}, new int[]{0xdc00});
    }

    static TestData dataUTF32BE() {
        assert TruffleString.Encoding.UTF_32BE.ordinal() == 1;
        return new TestData(
                        new int[]{0x000000, 0x00d7ff, 0x00e000, 0x10ffff}, new int[]{0x0, 0x4, 0x8, 0xc}, null, null, null, new byte[]{(byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00,
                                        (byte) 0x00, (byte) 0xd7, (byte) 0xff, (byte) 0x00, (byte) 0x00, (byte) 0xe0, (byte) 0x00, (byte) 0x00, (byte) 0x10, (byte) 0xff, (byte) 0xff},
                        new byte[]{(byte) 0x00, (byte) 0x00, (byte) 0xdc, (byte) 0x00}, new int[]{0xdc00});
    }

    static TestData dataUTF16LE() {
        assert TruffleString.Encoding.UTF_16LE.ordinal() == 2;
        return new TestData(new int[]{0x000000, 0x00007f, 0x000080, 0x0000ff, 0x000100, 0x00d7ff, 0x00e000, 0x00ffff, 0x010000, 0x10ffff},
                        new int[]{0x00, 0x02, 0x04, 0x06, 0x08, 0x0a, 0x0c, 0x0e, 0x10, 0x14}, new byte[]{(byte) 0x00, (byte) 0x00, (byte) 0x7f, (byte) 0x00},
                        new byte[]{(byte) 0x00, (byte) 0x00, (byte) 0xff, (byte) 0x00}, new byte[]{(byte) 0x00, (byte) 0x00, (byte) 0xff, (byte) 0xff},
                        new byte[]{(byte) 0x00, (byte) 0x00, (byte) 0x7f, (byte) 0x00, (byte) 0x80, (byte) 0x00, (byte) 0xff, (byte) 0x00, (byte) 0x00, (byte) 0x01, (byte) 0xff, (byte) 0xd7,
                                        (byte) 0x00, (byte) 0xe0, (byte) 0xff, (byte) 0xff, (byte) 0x00, (byte) 0xd8, (byte) 0x00, (byte) 0xdc, (byte) 0xff, (byte) 0xdb, (byte) 0xff, (byte) 0xdf},
                        new byte[]{(byte) 0x00, (byte) 0xdc}, new int[]{0xdc00});
    }

    static TestData dataUTF16BE() {
        assert TruffleString.Encoding.UTF_16BE.ordinal() == 3;
        return new TestData(new int[]{0x000000, 0x00d7ff, 0x00e000, 0x10ffff}, new int[]{0x0, 0x2, 0x4, 0x6}, null, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x00, (byte) 0xd7, (byte) 0xff, (byte) 0xe0, (byte) 0x00, (byte) 0xdb, (byte) 0xff, (byte) 0xdf, (byte) 0xff},
                        new byte[]{(byte) 0xdc, (byte) 0x00}, new int[]{0xdc00});
    }

    static TestData dataISO88591() {
        assert TruffleString.Encoding.ISO_8859_1.ordinal() == 4;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, new byte[]{(byte) 0x00, (byte) 0xff}, null, new byte[]{(byte) 0x00, (byte) 0xff}, null,
                        null);
    }

    static TestData dataUTF8() {
        assert TruffleString.Encoding.UTF_8.ordinal() == 5;
        return new TestData(new int[]{0x000000, 0x00007f, 0x000080, 0x00d7ff, 0x00e000, 0x10ffff}, new int[]{0x0, 0x1, 0x2, 0x4, 0x7, 0xa}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xc2, (byte) 0x80, (byte) 0xed, (byte) 0x9f, (byte) 0xbf, (byte) 0xee, (byte) 0x80, (byte) 0x80, (byte) 0xf4, (byte) 0x8f,
                                        (byte) 0xbf, (byte) 0xbf},
                        new byte[]{(byte) 0xc0}, new int[]{0xfffd});
    }

    static TestData dataUSASCII() {
        assert TruffleString.Encoding.US_ASCII.ordinal() == 6;
        return new TestData(new int[]{0x00, 0x7f}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0x7f}, new byte[]{(byte) 0xff},
                        new int[]{0xff});
    }

    static TestData dataBYTES() {
        assert TruffleString.Encoding.BYTES.ordinal() == 7;
        return new TestData(new int[]{0x00, 0x7f, 0x80, 0xff}, new int[]{0x0, 0x1, 0x2, 0x3}, null, null, null, new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0x80, (byte) 0xff}, null, null);
    }

    static TestData dataBig5() {
        assert TruffleString.Encoding.Big5.ordinal() == 8;
        return new TestData(new int[]{0x0000, 0x007f, 0xa140, 0xa17e, 0xa1a1, 0xa1fe, 0xa240, 0xa27e, 0xa2a1, 0xa2fe, 0xa340, 0xa37e, 0xa3a1, 0xa3fe, 0xa440, 0xa47e, 0xa4a1, 0xa4fe, 0xa540, 0xa57e,
                        0xa5a1, 0xa5fe, 0xa640, 0xa67e, 0xa6a1, 0xa6fe, 0xa740, 0xa77e, 0xa7a1, 0xa7fe, 0xa840, 0xa87e, 0xa8a1, 0xa8fe, 0xa940, 0xa97e, 0xa9a1, 0xa9fe, 0xaa40, 0xaa7e, 0xaaa1, 0xaafe,
                        0xab40, 0xab7e, 0xaba1, 0xabfe, 0xac40, 0xac7e, 0xaca1, 0xacfe, 0xad40, 0xad7e, 0xada1, 0xadfe, 0xae40, 0xae7e, 0xaea1, 0xaefe, 0xaf40, 0xaf7e, 0xafa1, 0xaffe, 0xb040, 0xb07e,
                        0xb0a1, 0xb0fe, 0xb140, 0xb17e, 0xb1a1, 0xb1fe, 0xb240, 0xb27e, 0xb2a1, 0xb2fe, 0xb340, 0xb37e, 0xb3a1, 0xb3fe, 0xb440, 0xb47e, 0xb4a1, 0xb4fe, 0xb540, 0xb57e, 0xb5a1, 0xb5fe,
                        0xb640, 0xb67e, 0xb6a1, 0xb6fe, 0xb740, 0xb77e, 0xb7a1, 0xb7fe, 0xb840, 0xb87e, 0xb8a1, 0xb8fe, 0xb940, 0xb97e, 0xb9a1, 0xb9fe, 0xba40, 0xba7e, 0xbaa1, 0xbafe, 0xbb40, 0xbb7e,
                        0xbba1, 0xbbfe, 0xbc40, 0xbc7e, 0xbca1, 0xbcfe, 0xbd40, 0xbd7e, 0xbda1, 0xbdfe, 0xbe40, 0xbe7e, 0xbea1, 0xbefe, 0xbf40, 0xbf7e, 0xbfa1, 0xbffe, 0xc040, 0xc07e, 0xc0a1, 0xc0fe,
                        0xc140, 0xc17e, 0xc1a1, 0xc1fe, 0xc240, 0xc27e, 0xc2a1, 0xc2fe, 0xc340, 0xc37e, 0xc3a1, 0xc3fe, 0xc440, 0xc47e, 0xc4a1, 0xc4fe, 0xc540, 0xc57e, 0xc5a1, 0xc5fe, 0xc640, 0xc67e,
                        0xc6a1, 0xc6fe, 0xc740, 0xc77e, 0xc7a1, 0xc7fe, 0xc840, 0xc87e, 0xc8a1, 0xc8fe, 0xc940, 0xc97e, 0xc9a1, 0xc9fe, 0xca40, 0xca7e, 0xcaa1, 0xcafe, 0xcb40, 0xcb7e, 0xcba1, 0xcbfe,
                        0xcc40, 0xcc7e, 0xcca1, 0xccfe, 0xcd40, 0xcd7e, 0xcda1, 0xcdfe, 0xce40, 0xce7e, 0xcea1, 0xcefe, 0xcf40, 0xcf7e, 0xcfa1, 0xcffe, 0xd040, 0xd07e, 0xd0a1, 0xd0fe, 0xd140, 0xd17e,
                        0xd1a1, 0xd1fe, 0xd240, 0xd27e, 0xd2a1, 0xd2fe, 0xd340, 0xd37e, 0xd3a1, 0xd3fe, 0xd440, 0xd47e, 0xd4a1, 0xd4fe, 0xd540, 0xd57e, 0xd5a1, 0xd5fe, 0xd640, 0xd67e, 0xd6a1, 0xd6fe,
                        0xd740, 0xd77e, 0xd7a1, 0xd7fe, 0xd840, 0xd87e, 0xd8a1, 0xd8fe, 0xd940, 0xd97e, 0xd9a1, 0xd9fe, 0xda40, 0xda7e, 0xdaa1, 0xdafe, 0xdb40, 0xdb7e, 0xdba1, 0xdbfe, 0xdc40, 0xdc7e,
                        0xdca1, 0xdcfe, 0xdd40, 0xdd7e, 0xdda1, 0xddfe, 0xde40, 0xde7e, 0xdea1, 0xdefe, 0xdf40, 0xdf7e, 0xdfa1, 0xdffe, 0xe040, 0xe07e, 0xe0a1, 0xe0fe, 0xe140, 0xe17e, 0xe1a1, 0xe1fe,
                        0xe240, 0xe27e, 0xe2a1, 0xe2fe, 0xe340, 0xe37e, 0xe3a1, 0xe3fe, 0xe440, 0xe47e, 0xe4a1, 0xe4fe, 0xe540, 0xe57e, 0xe5a1, 0xe5fe, 0xe640, 0xe67e, 0xe6a1, 0xe6fe, 0xe740, 0xe77e,
                        0xe7a1, 0xe7fe, 0xe840, 0xe87e, 0xe8a1, 0xe8fe, 0xe940, 0xe97e, 0xe9a1, 0xe9fe, 0xea40, 0xea7e, 0xeaa1, 0xeafe, 0xeb40, 0xeb7e, 0xeba1, 0xebfe, 0xec40, 0xec7e, 0xeca1, 0xecfe,
                        0xed40, 0xed7e, 0xeda1, 0xedfe, 0xee40, 0xee7e, 0xeea1, 0xeefe, 0xef40, 0xef7e, 0xefa1, 0xeffe, 0xf040, 0xf07e, 0xf0a1, 0xf0fe, 0xf140, 0xf17e, 0xf1a1, 0xf1fe, 0xf240, 0xf27e,
                        0xf2a1, 0xf2fe, 0xf340, 0xf37e, 0xf3a1, 0xf3fe, 0xf440, 0xf47e, 0xf4a1, 0xf4fe, 0xf540, 0xf57e, 0xf5a1, 0xf5fe, 0xf640, 0xf67e, 0xf6a1, 0xf6fe, 0xf740, 0xf77e, 0xf7a1, 0xf7fe,
                        0xf840, 0xf87e, 0xf8a1, 0xf8fe, 0xf940, 0xf97e, 0xf9a1, 0xf9fe, 0xfa40, 0xfa7e, 0xfaa1, 0xfafe, 0xfb40, 0xfb7e, 0xfba1, 0xfbfe, 0xfc40, 0xfc7e, 0xfca1, 0xfcfe, 0xfd40, 0xfd7e,
                        0xfda1, 0xfdfe, 0xfe40, 0xfe7e, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198, 0x19a,
                                        0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6, 0x1c8,
                                        0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2, 0x1e4, 0x1e6, 0x1e8, 0x1ea, 0x1ec, 0x1ee, 0x1f0, 0x1f2, 0x1f4, 0x1f6,
                                        0x1f8, 0x1fa, 0x1fc, 0x1fe, 0x200, 0x202, 0x204, 0x206, 0x208, 0x20a, 0x20c, 0x20e, 0x210, 0x212, 0x214, 0x216, 0x218, 0x21a, 0x21c, 0x21e, 0x220, 0x222, 0x224,
                                        0x226, 0x228, 0x22a, 0x22c, 0x22e, 0x230, 0x232, 0x234, 0x236, 0x238, 0x23a, 0x23c, 0x23e, 0x240, 0x242, 0x244, 0x246, 0x248, 0x24a, 0x24c, 0x24e, 0x250, 0x252,
                                        0x254, 0x256, 0x258, 0x25a, 0x25c, 0x25e, 0x260, 0x262, 0x264, 0x266, 0x268, 0x26a, 0x26c, 0x26e, 0x270, 0x272, 0x274, 0x276, 0x278, 0x27a, 0x27c, 0x27e, 0x280,
                                        0x282, 0x284, 0x286, 0x288, 0x28a, 0x28c, 0x28e, 0x290, 0x292, 0x294, 0x296, 0x298, 0x29a, 0x29c, 0x29e, 0x2a0, 0x2a2, 0x2a4, 0x2a6, 0x2a8, 0x2aa, 0x2ac, 0x2ae,
                                        0x2b0, 0x2b2, 0x2b4, 0x2b6, 0x2b8, 0x2ba, 0x2bc, 0x2be, 0x2c0, 0x2c2, 0x2c4, 0x2c6, 0x2c8, 0x2ca, 0x2cc, 0x2ce, 0x2d0, 0x2d2, 0x2d4, 0x2d6, 0x2d8, 0x2da, 0x2dc,
                                        0x2de, 0x2e0, 0x2e2, 0x2e4, 0x2e6, 0x2e8, 0x2ea, 0x2ec, 0x2ee, 0x2f0},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0x40, (byte) 0xa1, (byte) 0x7e, (byte) 0xa1, (byte) 0xa1, (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0x40,
                                        (byte) 0xa2, (byte) 0x7e, (byte) 0xa2, (byte) 0xa1, (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0x40, (byte) 0xa3, (byte) 0x7e, (byte) 0xa3, (byte) 0xa1,
                                        (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0x40, (byte) 0xa4, (byte) 0x7e, (byte) 0xa4, (byte) 0xa1, (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0x40,
                                        (byte) 0xa5, (byte) 0x7e, (byte) 0xa5, (byte) 0xa1, (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0x40, (byte) 0xa6, (byte) 0x7e, (byte) 0xa6, (byte) 0xa1,
                                        (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0x40, (byte) 0xa7, (byte) 0x7e, (byte) 0xa7, (byte) 0xa1, (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0x40,
                                        (byte) 0xa8, (byte) 0x7e, (byte) 0xa8, (byte) 0xa1, (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0x40, (byte) 0xa9, (byte) 0x7e, (byte) 0xa9, (byte) 0xa1,
                                        (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0x40, (byte) 0xaa, (byte) 0x7e, (byte) 0xaa, (byte) 0xa1, (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0x40,
                                        (byte) 0xab, (byte) 0x7e, (byte) 0xab, (byte) 0xa1, (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0x40, (byte) 0xac, (byte) 0x7e, (byte) 0xac, (byte) 0xa1,
                                        (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0x40, (byte) 0xad, (byte) 0x7e, (byte) 0xad, (byte) 0xa1, (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0x40,
                                        (byte) 0xae, (byte) 0x7e, (byte) 0xae, (byte) 0xa1, (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0x40, (byte) 0xaf, (byte) 0x7e, (byte) 0xaf, (byte) 0xa1,
                                        (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0x40, (byte) 0xb0, (byte) 0x7e, (byte) 0xb0, (byte) 0xa1, (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0x40,
                                        (byte) 0xb1, (byte) 0x7e, (byte) 0xb1, (byte) 0xa1, (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0x40, (byte) 0xb2, (byte) 0x7e, (byte) 0xb2, (byte) 0xa1,
                                        (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0x40, (byte) 0xb3, (byte) 0x7e, (byte) 0xb3, (byte) 0xa1, (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0x40,
                                        (byte) 0xb4, (byte) 0x7e, (byte) 0xb4, (byte) 0xa1, (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0x40, (byte) 0xb5, (byte) 0x7e, (byte) 0xb5, (byte) 0xa1,
                                        (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0x40, (byte) 0xb6, (byte) 0x7e, (byte) 0xb6, (byte) 0xa1, (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0x40,
                                        (byte) 0xb7, (byte) 0x7e, (byte) 0xb7, (byte) 0xa1, (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0x40, (byte) 0xb8, (byte) 0x7e, (byte) 0xb8, (byte) 0xa1,
                                        (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0x40, (byte) 0xb9, (byte) 0x7e, (byte) 0xb9, (byte) 0xa1, (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0x40,
                                        (byte) 0xba, (byte) 0x7e, (byte) 0xba, (byte) 0xa1, (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0x40, (byte) 0xbb, (byte) 0x7e, (byte) 0xbb, (byte) 0xa1,
                                        (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0x40, (byte) 0xbc, (byte) 0x7e, (byte) 0xbc, (byte) 0xa1, (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0x40,
                                        (byte) 0xbd, (byte) 0x7e, (byte) 0xbd, (byte) 0xa1, (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0x40, (byte) 0xbe, (byte) 0x7e, (byte) 0xbe, (byte) 0xa1,
                                        (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0x40, (byte) 0xbf, (byte) 0x7e, (byte) 0xbf, (byte) 0xa1, (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0x40,
                                        (byte) 0xc0, (byte) 0x7e, (byte) 0xc0, (byte) 0xa1, (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0x40, (byte) 0xc1, (byte) 0x7e, (byte) 0xc1, (byte) 0xa1,
                                        (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0x40, (byte) 0xc2, (byte) 0x7e, (byte) 0xc2, (byte) 0xa1, (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0x40,
                                        (byte) 0xc3, (byte) 0x7e, (byte) 0xc3, (byte) 0xa1, (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0x40, (byte) 0xc4, (byte) 0x7e, (byte) 0xc4, (byte) 0xa1,
                                        (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0x40, (byte) 0xc5, (byte) 0x7e, (byte) 0xc5, (byte) 0xa1, (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0x40,
                                        (byte) 0xc6, (byte) 0x7e, (byte) 0xc6, (byte) 0xa1, (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0x40, (byte) 0xc7, (byte) 0x7e, (byte) 0xc7, (byte) 0xa1,
                                        (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0x40, (byte) 0xc8, (byte) 0x7e, (byte) 0xc8, (byte) 0xa1, (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0x40,
                                        (byte) 0xc9, (byte) 0x7e, (byte) 0xc9, (byte) 0xa1, (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0x40, (byte) 0xca, (byte) 0x7e, (byte) 0xca, (byte) 0xa1,
                                        (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0x40, (byte) 0xcb, (byte) 0x7e, (byte) 0xcb, (byte) 0xa1, (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0x40,
                                        (byte) 0xcc, (byte) 0x7e, (byte) 0xcc, (byte) 0xa1, (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0x40, (byte) 0xcd, (byte) 0x7e, (byte) 0xcd, (byte) 0xa1,
                                        (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0x40, (byte) 0xce, (byte) 0x7e, (byte) 0xce, (byte) 0xa1, (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0x40,
                                        (byte) 0xcf, (byte) 0x7e, (byte) 0xcf, (byte) 0xa1, (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0x40, (byte) 0xd0, (byte) 0x7e, (byte) 0xd0, (byte) 0xa1,
                                        (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0x40, (byte) 0xd1, (byte) 0x7e, (byte) 0xd1, (byte) 0xa1, (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0x40,
                                        (byte) 0xd2, (byte) 0x7e, (byte) 0xd2, (byte) 0xa1, (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0x40, (byte) 0xd3, (byte) 0x7e, (byte) 0xd3, (byte) 0xa1,
                                        (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0x40, (byte) 0xd4, (byte) 0x7e, (byte) 0xd4, (byte) 0xa1, (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0x40,
                                        (byte) 0xd5, (byte) 0x7e, (byte) 0xd5, (byte) 0xa1, (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0x40, (byte) 0xd6, (byte) 0x7e, (byte) 0xd6, (byte) 0xa1,
                                        (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0x40, (byte) 0xd7, (byte) 0x7e, (byte) 0xd7, (byte) 0xa1, (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0x40,
                                        (byte) 0xd8, (byte) 0x7e, (byte) 0xd8, (byte) 0xa1, (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0x40, (byte) 0xd9, (byte) 0x7e, (byte) 0xd9, (byte) 0xa1,
                                        (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0x40, (byte) 0xda, (byte) 0x7e, (byte) 0xda, (byte) 0xa1, (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0x40,
                                        (byte) 0xdb, (byte) 0x7e, (byte) 0xdb, (byte) 0xa1, (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0x40, (byte) 0xdc, (byte) 0x7e, (byte) 0xdc, (byte) 0xa1,
                                        (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0x40, (byte) 0xdd, (byte) 0x7e, (byte) 0xdd, (byte) 0xa1, (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0x40,
                                        (byte) 0xde, (byte) 0x7e, (byte) 0xde, (byte) 0xa1, (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0x40, (byte) 0xdf, (byte) 0x7e, (byte) 0xdf, (byte) 0xa1,
                                        (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0xa1, (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0x40,
                                        (byte) 0xe1, (byte) 0x7e, (byte) 0xe1, (byte) 0xa1, (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0x40, (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0xa1,
                                        (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0xa1, (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0x40,
                                        (byte) 0xe4, (byte) 0x7e, (byte) 0xe4, (byte) 0xa1, (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0x40, (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0xa1,
                                        (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0xa1, (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0x40,
                                        (byte) 0xe7, (byte) 0x7e, (byte) 0xe7, (byte) 0xa1, (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0x40, (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0xa1,
                                        (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0xa1, (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0x40,
                                        (byte) 0xea, (byte) 0x7e, (byte) 0xea, (byte) 0xa1, (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0x40, (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0xa1,
                                        (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0xa1, (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0x40,
                                        (byte) 0xed, (byte) 0x7e, (byte) 0xed, (byte) 0xa1, (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0x40, (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0xa1,
                                        (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0xa1, (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0x40,
                                        (byte) 0xf0, (byte) 0x7e, (byte) 0xf0, (byte) 0xa1, (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0x40, (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0xa1,
                                        (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0xa1, (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0x40,
                                        (byte) 0xf3, (byte) 0x7e, (byte) 0xf3, (byte) 0xa1, (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0x40, (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0xa1,
                                        (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0xa1, (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0x40,
                                        (byte) 0xf6, (byte) 0x7e, (byte) 0xf6, (byte) 0xa1, (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0x40, (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0xa1,
                                        (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0xa1, (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0x40,
                                        (byte) 0xf9, (byte) 0x7e, (byte) 0xf9, (byte) 0xa1, (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0x40, (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0xa1,
                                        (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0xa1, (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0x40,
                                        (byte) 0xfc, (byte) 0x7e, (byte) 0xfc, (byte) 0xa1, (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0x40, (byte) 0xfd, (byte) 0x7e, (byte) 0xfd, (byte) 0xa1,
                                        (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0x40, (byte) 0xfe, (byte) 0x7e, (byte) 0xfe, (byte) 0xa1, (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataBig5HKSCS() {
        assert TruffleString.Encoding.Big5_HKSCS.ordinal() == 9;
        return new TestData(new int[]{0x0000, 0x007f, 0x8740, 0x877e, 0x87a1, 0x87fe, 0x8840, 0x887e, 0x88a1, 0x88fe, 0x8940, 0x897e, 0x89a1, 0x89fe, 0x8a40, 0x8a7e, 0x8aa1, 0x8afe, 0x8b40, 0x8b7e,
                        0x8ba1, 0x8bfe, 0x8c40, 0x8c7e, 0x8ca1, 0x8cfe, 0x8d40, 0x8d7e, 0x8da1, 0x8dfe, 0x8e40, 0x8e7e, 0x8ea1, 0x8efe, 0x8f40, 0x8f7e, 0x8fa1, 0x8ffe, 0x9040, 0x907e, 0x90a1, 0x90fe,
                        0x9140, 0x917e, 0x91a1, 0x91fe, 0x9240, 0x927e, 0x92a1, 0x92fe, 0x9340, 0x937e, 0x93a1, 0x93fe, 0x9440, 0x947e, 0x94a1, 0x94fe, 0x9540, 0x957e, 0x95a1, 0x95fe, 0x9640, 0x967e,
                        0x96a1, 0x96fe, 0x9740, 0x977e, 0x97a1, 0x97fe, 0x9840, 0x987e, 0x98a1, 0x98fe, 0x9940, 0x997e, 0x99a1, 0x99fe, 0x9a40, 0x9a7e, 0x9aa1, 0x9afe, 0x9b40, 0x9b7e, 0x9ba1, 0x9bfe,
                        0x9c40, 0x9c7e, 0x9ca1, 0x9cfe, 0x9d40, 0x9d7e, 0x9da1, 0x9dfe, 0x9e40, 0x9e7e, 0x9ea1, 0x9efe, 0x9f40, 0x9f7e, 0x9fa1, 0x9ffe, 0xa040, 0xa07e, 0xa0a1, 0xa0fe, 0xa140, 0xa17e,
                        0xa1a1, 0xa1fe, 0xa240, 0xa27e, 0xa2a1, 0xa2fe, 0xa340, 0xa37e, 0xa3a1, 0xa3fe, 0xa440, 0xa47e, 0xa4a1, 0xa4fe, 0xa540, 0xa57e, 0xa5a1, 0xa5fe, 0xa640, 0xa67e, 0xa6a1, 0xa6fe,
                        0xa740, 0xa77e, 0xa7a1, 0xa7fe, 0xa840, 0xa87e, 0xa8a1, 0xa8fe, 0xa940, 0xa97e, 0xa9a1, 0xa9fe, 0xaa40, 0xaa7e, 0xaaa1, 0xaafe, 0xab40, 0xab7e, 0xaba1, 0xabfe, 0xac40, 0xac7e,
                        0xaca1, 0xacfe, 0xad40, 0xad7e, 0xada1, 0xadfe, 0xae40, 0xae7e, 0xaea1, 0xaefe, 0xaf40, 0xaf7e, 0xafa1, 0xaffe, 0xb040, 0xb07e, 0xb0a1, 0xb0fe, 0xb140, 0xb17e, 0xb1a1, 0xb1fe,
                        0xb240, 0xb27e, 0xb2a1, 0xb2fe, 0xb340, 0xb37e, 0xb3a1, 0xb3fe, 0xb440, 0xb47e, 0xb4a1, 0xb4fe, 0xb540, 0xb57e, 0xb5a1, 0xb5fe, 0xb640, 0xb67e, 0xb6a1, 0xb6fe, 0xb740, 0xb77e,
                        0xb7a1, 0xb7fe, 0xb840, 0xb87e, 0xb8a1, 0xb8fe, 0xb940, 0xb97e, 0xb9a1, 0xb9fe, 0xba40, 0xba7e, 0xbaa1, 0xbafe, 0xbb40, 0xbb7e, 0xbba1, 0xbbfe, 0xbc40, 0xbc7e, 0xbca1, 0xbcfe,
                        0xbd40, 0xbd7e, 0xbda1, 0xbdfe, 0xbe40, 0xbe7e, 0xbea1, 0xbefe, 0xbf40, 0xbf7e, 0xbfa1, 0xbffe, 0xc040, 0xc07e, 0xc0a1, 0xc0fe, 0xc140, 0xc17e, 0xc1a1, 0xc1fe, 0xc240, 0xc27e,
                        0xc2a1, 0xc2fe, 0xc340, 0xc37e, 0xc3a1, 0xc3fe, 0xc440, 0xc47e, 0xc4a1, 0xc4fe, 0xc540, 0xc57e, 0xc5a1, 0xc5fe, 0xc640, 0xc67e, 0xc6a1, 0xc6fe, 0xc740, 0xc77e, 0xc7a1, 0xc7fe,
                        0xc840, 0xc87e, 0xc8a1, 0xc8fe, 0xc940, 0xc97e, 0xc9a1, 0xc9fe, 0xca40, 0xca7e, 0xcaa1, 0xcafe, 0xcb40, 0xcb7e, 0xcba1, 0xcbfe, 0xcc40, 0xcc7e, 0xcca1, 0xccfe, 0xcd40, 0xcd7e,
                        0xcda1, 0xcdfe, 0xce40, 0xce7e, 0xcea1, 0xcefe, 0xcf40, 0xcf7e, 0xcfa1, 0xcffe, 0xd040, 0xd07e, 0xd0a1, 0xd0fe, 0xd140, 0xd17e, 0xd1a1, 0xd1fe, 0xd240, 0xd27e, 0xd2a1, 0xd2fe,
                        0xd340, 0xd37e, 0xd3a1, 0xd3fe, 0xd440, 0xd47e, 0xd4a1, 0xd4fe, 0xd540, 0xd57e, 0xd5a1, 0xd5fe, 0xd640, 0xd67e, 0xd6a1, 0xd6fe, 0xd740, 0xd77e, 0xd7a1, 0xd7fe, 0xd840, 0xd87e,
                        0xd8a1, 0xd8fe, 0xd940, 0xd97e, 0xd9a1, 0xd9fe, 0xda40, 0xda7e, 0xdaa1, 0xdafe, 0xdb40, 0xdb7e, 0xdba1, 0xdbfe, 0xdc40, 0xdc7e, 0xdca1, 0xdcfe, 0xdd40, 0xdd7e, 0xdda1, 0xddfe,
                        0xde40, 0xde7e, 0xdea1, 0xdefe, 0xdf40, 0xdf7e, 0xdfa1, 0xdffe, 0xe040, 0xe07e, 0xe0a1, 0xe0fe, 0xe140, 0xe17e, 0xe1a1, 0xe1fe, 0xe240, 0xe27e, 0xe2a1, 0xe2fe, 0xe340, 0xe37e,
                        0xe3a1, 0xe3fe, 0xe440, 0xe47e, 0xe4a1, 0xe4fe, 0xe540, 0xe57e, 0xe5a1, 0xe5fe, 0xe640, 0xe67e, 0xe6a1, 0xe6fe, 0xe740, 0xe77e, 0xe7a1, 0xe7fe, 0xe840, 0xe87e, 0xe8a1, 0xe8fe,
                        0xe940, 0xe97e, 0xe9a1, 0xe9fe, 0xea40, 0xea7e, 0xeaa1, 0xeafe, 0xeb40, 0xeb7e, 0xeba1, 0xebfe, 0xec40, 0xec7e, 0xeca1, 0xecfe, 0xed40, 0xed7e, 0xeda1, 0xedfe, 0xee40, 0xee7e,
                        0xeea1, 0xeefe, 0xef40, 0xef7e, 0xefa1, 0xeffe, 0xf040, 0xf07e, 0xf0a1, 0xf0fe, 0xf140, 0xf17e, 0xf1a1, 0xf1fe, 0xf240, 0xf27e, 0xf2a1, 0xf2fe, 0xf340, 0xf37e, 0xf3a1, 0xf3fe,
                        0xf440, 0xf47e, 0xf4a1, 0xf4fe, 0xf540, 0xf57e, 0xf5a1, 0xf5fe, 0xf640, 0xf67e, 0xf6a1, 0xf6fe, 0xf740, 0xf77e, 0xf7a1, 0xf7fe, 0xf840, 0xf87e, 0xf8a1, 0xf8fe, 0xf940, 0xf97e,
                        0xf9a1, 0xf9fe, 0xfa40, 0xfa7e, 0xfaa1, 0xfafe, 0xfb40, 0xfb7e, 0xfba1, 0xfbfe, 0xfc40, 0xfc7e, 0xfca1, 0xfcfe, 0xfd40, 0xfd7e, 0xfda1, 0xfdfe, 0xfe40, 0xfe7e, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198, 0x19a,
                                        0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6, 0x1c8,
                                        0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2, 0x1e4, 0x1e6, 0x1e8, 0x1ea, 0x1ec, 0x1ee, 0x1f0, 0x1f2, 0x1f4, 0x1f6,
                                        0x1f8, 0x1fa, 0x1fc, 0x1fe, 0x200, 0x202, 0x204, 0x206, 0x208, 0x20a, 0x20c, 0x20e, 0x210, 0x212, 0x214, 0x216, 0x218, 0x21a, 0x21c, 0x21e, 0x220, 0x222, 0x224,
                                        0x226, 0x228, 0x22a, 0x22c, 0x22e, 0x230, 0x232, 0x234, 0x236, 0x238, 0x23a, 0x23c, 0x23e, 0x240, 0x242, 0x244, 0x246, 0x248, 0x24a, 0x24c, 0x24e, 0x250, 0x252,
                                        0x254, 0x256, 0x258, 0x25a, 0x25c, 0x25e, 0x260, 0x262, 0x264, 0x266, 0x268, 0x26a, 0x26c, 0x26e, 0x270, 0x272, 0x274, 0x276, 0x278, 0x27a, 0x27c, 0x27e, 0x280,
                                        0x282, 0x284, 0x286, 0x288, 0x28a, 0x28c, 0x28e, 0x290, 0x292, 0x294, 0x296, 0x298, 0x29a, 0x29c, 0x29e, 0x2a0, 0x2a2, 0x2a4, 0x2a6, 0x2a8, 0x2aa, 0x2ac, 0x2ae,
                                        0x2b0, 0x2b2, 0x2b4, 0x2b6, 0x2b8, 0x2ba, 0x2bc, 0x2be, 0x2c0, 0x2c2, 0x2c4, 0x2c6, 0x2c8, 0x2ca, 0x2cc, 0x2ce, 0x2d0, 0x2d2, 0x2d4, 0x2d6, 0x2d8, 0x2da, 0x2dc,
                                        0x2de, 0x2e0, 0x2e2, 0x2e4, 0x2e6, 0x2e8, 0x2ea, 0x2ec, 0x2ee, 0x2f0, 0x2f2, 0x2f4, 0x2f6, 0x2f8, 0x2fa, 0x2fc, 0x2fe, 0x300, 0x302, 0x304, 0x306, 0x308, 0x30a,
                                        0x30c, 0x30e, 0x310, 0x312, 0x314, 0x316, 0x318, 0x31a, 0x31c, 0x31e, 0x320, 0x322, 0x324, 0x326, 0x328, 0x32a, 0x32c, 0x32e, 0x330, 0x332, 0x334, 0x336, 0x338,
                                        0x33a, 0x33c, 0x33e, 0x340, 0x342, 0x344, 0x346, 0x348, 0x34a, 0x34c, 0x34e, 0x350, 0x352, 0x354, 0x356, 0x358, 0x35a, 0x35c, 0x35e, 0x360, 0x362, 0x364, 0x366,
                                        0x368, 0x36a, 0x36c, 0x36e, 0x370, 0x372, 0x374, 0x376, 0x378, 0x37a, 0x37c, 0x37e, 0x380, 0x382, 0x384, 0x386, 0x388, 0x38a, 0x38c, 0x38e, 0x390, 0x392, 0x394,
                                        0x396, 0x398, 0x39a, 0x39c, 0x39e, 0x3a0, 0x3a2, 0x3a4, 0x3a6, 0x3a8, 0x3aa, 0x3ac, 0x3ae, 0x3b0, 0x3b2, 0x3b4, 0x3b6, 0x3b8, 0x3ba, 0x3bc, 0x3be, 0x3c0},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0x87, (byte) 0x40, (byte) 0x87, (byte) 0x7e, (byte) 0x87, (byte) 0xa1, (byte) 0x87, (byte) 0xfe, (byte) 0x88, (byte) 0x40,
                                        (byte) 0x88, (byte) 0x7e, (byte) 0x88, (byte) 0xa1, (byte) 0x88, (byte) 0xfe, (byte) 0x89, (byte) 0x40, (byte) 0x89, (byte) 0x7e, (byte) 0x89, (byte) 0xa1,
                                        (byte) 0x89, (byte) 0xfe, (byte) 0x8a, (byte) 0x40, (byte) 0x8a, (byte) 0x7e, (byte) 0x8a, (byte) 0xa1, (byte) 0x8a, (byte) 0xfe, (byte) 0x8b, (byte) 0x40,
                                        (byte) 0x8b, (byte) 0x7e, (byte) 0x8b, (byte) 0xa1, (byte) 0x8b, (byte) 0xfe, (byte) 0x8c, (byte) 0x40, (byte) 0x8c, (byte) 0x7e, (byte) 0x8c, (byte) 0xa1,
                                        (byte) 0x8c, (byte) 0xfe, (byte) 0x8d, (byte) 0x40, (byte) 0x8d, (byte) 0x7e, (byte) 0x8d, (byte) 0xa1, (byte) 0x8d, (byte) 0xfe, (byte) 0x8e, (byte) 0x40,
                                        (byte) 0x8e, (byte) 0x7e, (byte) 0x8e, (byte) 0xa1, (byte) 0x8e, (byte) 0xfe, (byte) 0x8f, (byte) 0x40, (byte) 0x8f, (byte) 0x7e, (byte) 0x8f, (byte) 0xa1,
                                        (byte) 0x8f, (byte) 0xfe, (byte) 0x90, (byte) 0x40, (byte) 0x90, (byte) 0x7e, (byte) 0x90, (byte) 0xa1, (byte) 0x90, (byte) 0xfe, (byte) 0x91, (byte) 0x40,
                                        (byte) 0x91, (byte) 0x7e, (byte) 0x91, (byte) 0xa1, (byte) 0x91, (byte) 0xfe, (byte) 0x92, (byte) 0x40, (byte) 0x92, (byte) 0x7e, (byte) 0x92, (byte) 0xa1,
                                        (byte) 0x92, (byte) 0xfe, (byte) 0x93, (byte) 0x40, (byte) 0x93, (byte) 0x7e, (byte) 0x93, (byte) 0xa1, (byte) 0x93, (byte) 0xfe, (byte) 0x94, (byte) 0x40,
                                        (byte) 0x94, (byte) 0x7e, (byte) 0x94, (byte) 0xa1, (byte) 0x94, (byte) 0xfe, (byte) 0x95, (byte) 0x40, (byte) 0x95, (byte) 0x7e, (byte) 0x95, (byte) 0xa1,
                                        (byte) 0x95, (byte) 0xfe, (byte) 0x96, (byte) 0x40, (byte) 0x96, (byte) 0x7e, (byte) 0x96, (byte) 0xa1, (byte) 0x96, (byte) 0xfe, (byte) 0x97, (byte) 0x40,
                                        (byte) 0x97, (byte) 0x7e, (byte) 0x97, (byte) 0xa1, (byte) 0x97, (byte) 0xfe, (byte) 0x98, (byte) 0x40, (byte) 0x98, (byte) 0x7e, (byte) 0x98, (byte) 0xa1,
                                        (byte) 0x98, (byte) 0xfe, (byte) 0x99, (byte) 0x40, (byte) 0x99, (byte) 0x7e, (byte) 0x99, (byte) 0xa1, (byte) 0x99, (byte) 0xfe, (byte) 0x9a, (byte) 0x40,
                                        (byte) 0x9a, (byte) 0x7e, (byte) 0x9a, (byte) 0xa1, (byte) 0x9a, (byte) 0xfe, (byte) 0x9b, (byte) 0x40, (byte) 0x9b, (byte) 0x7e, (byte) 0x9b, (byte) 0xa1,
                                        (byte) 0x9b, (byte) 0xfe, (byte) 0x9c, (byte) 0x40, (byte) 0x9c, (byte) 0x7e, (byte) 0x9c, (byte) 0xa1, (byte) 0x9c, (byte) 0xfe, (byte) 0x9d, (byte) 0x40,
                                        (byte) 0x9d, (byte) 0x7e, (byte) 0x9d, (byte) 0xa1, (byte) 0x9d, (byte) 0xfe, (byte) 0x9e, (byte) 0x40, (byte) 0x9e, (byte) 0x7e, (byte) 0x9e, (byte) 0xa1,
                                        (byte) 0x9e, (byte) 0xfe, (byte) 0x9f, (byte) 0x40, (byte) 0x9f, (byte) 0x7e, (byte) 0x9f, (byte) 0xa1, (byte) 0x9f, (byte) 0xfe, (byte) 0xa0, (byte) 0x40,
                                        (byte) 0xa0, (byte) 0x7e, (byte) 0xa0, (byte) 0xa1, (byte) 0xa0, (byte) 0xfe, (byte) 0xa1, (byte) 0x40, (byte) 0xa1, (byte) 0x7e, (byte) 0xa1, (byte) 0xa1,
                                        (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0x40, (byte) 0xa2, (byte) 0x7e, (byte) 0xa2, (byte) 0xa1, (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0x40,
                                        (byte) 0xa3, (byte) 0x7e, (byte) 0xa3, (byte) 0xa1, (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0x40, (byte) 0xa4, (byte) 0x7e, (byte) 0xa4, (byte) 0xa1,
                                        (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0x40, (byte) 0xa5, (byte) 0x7e, (byte) 0xa5, (byte) 0xa1, (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0x40,
                                        (byte) 0xa6, (byte) 0x7e, (byte) 0xa6, (byte) 0xa1, (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0x40, (byte) 0xa7, (byte) 0x7e, (byte) 0xa7, (byte) 0xa1,
                                        (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0x40, (byte) 0xa8, (byte) 0x7e, (byte) 0xa8, (byte) 0xa1, (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0x40,
                                        (byte) 0xa9, (byte) 0x7e, (byte) 0xa9, (byte) 0xa1, (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0x40, (byte) 0xaa, (byte) 0x7e, (byte) 0xaa, (byte) 0xa1,
                                        (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0x40, (byte) 0xab, (byte) 0x7e, (byte) 0xab, (byte) 0xa1, (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0x40,
                                        (byte) 0xac, (byte) 0x7e, (byte) 0xac, (byte) 0xa1, (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0x40, (byte) 0xad, (byte) 0x7e, (byte) 0xad, (byte) 0xa1,
                                        (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0x40, (byte) 0xae, (byte) 0x7e, (byte) 0xae, (byte) 0xa1, (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0x40,
                                        (byte) 0xaf, (byte) 0x7e, (byte) 0xaf, (byte) 0xa1, (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0x40, (byte) 0xb0, (byte) 0x7e, (byte) 0xb0, (byte) 0xa1,
                                        (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0x40, (byte) 0xb1, (byte) 0x7e, (byte) 0xb1, (byte) 0xa1, (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0x40,
                                        (byte) 0xb2, (byte) 0x7e, (byte) 0xb2, (byte) 0xa1, (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0x40, (byte) 0xb3, (byte) 0x7e, (byte) 0xb3, (byte) 0xa1,
                                        (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0x40, (byte) 0xb4, (byte) 0x7e, (byte) 0xb4, (byte) 0xa1, (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0x40,
                                        (byte) 0xb5, (byte) 0x7e, (byte) 0xb5, (byte) 0xa1, (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0x40, (byte) 0xb6, (byte) 0x7e, (byte) 0xb6, (byte) 0xa1,
                                        (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0x40, (byte) 0xb7, (byte) 0x7e, (byte) 0xb7, (byte) 0xa1, (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0x40,
                                        (byte) 0xb8, (byte) 0x7e, (byte) 0xb8, (byte) 0xa1, (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0x40, (byte) 0xb9, (byte) 0x7e, (byte) 0xb9, (byte) 0xa1,
                                        (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0x40, (byte) 0xba, (byte) 0x7e, (byte) 0xba, (byte) 0xa1, (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0x40,
                                        (byte) 0xbb, (byte) 0x7e, (byte) 0xbb, (byte) 0xa1, (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0x40, (byte) 0xbc, (byte) 0x7e, (byte) 0xbc, (byte) 0xa1,
                                        (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0x40, (byte) 0xbd, (byte) 0x7e, (byte) 0xbd, (byte) 0xa1, (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0x40,
                                        (byte) 0xbe, (byte) 0x7e, (byte) 0xbe, (byte) 0xa1, (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0x40, (byte) 0xbf, (byte) 0x7e, (byte) 0xbf, (byte) 0xa1,
                                        (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0x40, (byte) 0xc0, (byte) 0x7e, (byte) 0xc0, (byte) 0xa1, (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0x40,
                                        (byte) 0xc1, (byte) 0x7e, (byte) 0xc1, (byte) 0xa1, (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0x40, (byte) 0xc2, (byte) 0x7e, (byte) 0xc2, (byte) 0xa1,
                                        (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0x40, (byte) 0xc3, (byte) 0x7e, (byte) 0xc3, (byte) 0xa1, (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0x40,
                                        (byte) 0xc4, (byte) 0x7e, (byte) 0xc4, (byte) 0xa1, (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0x40, (byte) 0xc5, (byte) 0x7e, (byte) 0xc5, (byte) 0xa1,
                                        (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0x40, (byte) 0xc6, (byte) 0x7e, (byte) 0xc6, (byte) 0xa1, (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0x40,
                                        (byte) 0xc7, (byte) 0x7e, (byte) 0xc7, (byte) 0xa1, (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0x40, (byte) 0xc8, (byte) 0x7e, (byte) 0xc8, (byte) 0xa1,
                                        (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0x40, (byte) 0xc9, (byte) 0x7e, (byte) 0xc9, (byte) 0xa1, (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0x40,
                                        (byte) 0xca, (byte) 0x7e, (byte) 0xca, (byte) 0xa1, (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0x40, (byte) 0xcb, (byte) 0x7e, (byte) 0xcb, (byte) 0xa1,
                                        (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0x40, (byte) 0xcc, (byte) 0x7e, (byte) 0xcc, (byte) 0xa1, (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0x40,
                                        (byte) 0xcd, (byte) 0x7e, (byte) 0xcd, (byte) 0xa1, (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0x40, (byte) 0xce, (byte) 0x7e, (byte) 0xce, (byte) 0xa1,
                                        (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0x40, (byte) 0xcf, (byte) 0x7e, (byte) 0xcf, (byte) 0xa1, (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0x40,
                                        (byte) 0xd0, (byte) 0x7e, (byte) 0xd0, (byte) 0xa1, (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0x40, (byte) 0xd1, (byte) 0x7e, (byte) 0xd1, (byte) 0xa1,
                                        (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0x40, (byte) 0xd2, (byte) 0x7e, (byte) 0xd2, (byte) 0xa1, (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0x40,
                                        (byte) 0xd3, (byte) 0x7e, (byte) 0xd3, (byte) 0xa1, (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0x40, (byte) 0xd4, (byte) 0x7e, (byte) 0xd4, (byte) 0xa1,
                                        (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0x40, (byte) 0xd5, (byte) 0x7e, (byte) 0xd5, (byte) 0xa1, (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0x40,
                                        (byte) 0xd6, (byte) 0x7e, (byte) 0xd6, (byte) 0xa1, (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0x40, (byte) 0xd7, (byte) 0x7e, (byte) 0xd7, (byte) 0xa1,
                                        (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0x40, (byte) 0xd8, (byte) 0x7e, (byte) 0xd8, (byte) 0xa1, (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0x40,
                                        (byte) 0xd9, (byte) 0x7e, (byte) 0xd9, (byte) 0xa1, (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0x40, (byte) 0xda, (byte) 0x7e, (byte) 0xda, (byte) 0xa1,
                                        (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0x40, (byte) 0xdb, (byte) 0x7e, (byte) 0xdb, (byte) 0xa1, (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0x40,
                                        (byte) 0xdc, (byte) 0x7e, (byte) 0xdc, (byte) 0xa1, (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0x40, (byte) 0xdd, (byte) 0x7e, (byte) 0xdd, (byte) 0xa1,
                                        (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0x40, (byte) 0xde, (byte) 0x7e, (byte) 0xde, (byte) 0xa1, (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0x40,
                                        (byte) 0xdf, (byte) 0x7e, (byte) 0xdf, (byte) 0xa1, (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0xa1,
                                        (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0x40, (byte) 0xe1, (byte) 0x7e, (byte) 0xe1, (byte) 0xa1, (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0x40,
                                        (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0xa1, (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0xa1,
                                        (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0x40, (byte) 0xe4, (byte) 0x7e, (byte) 0xe4, (byte) 0xa1, (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0x40,
                                        (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0xa1, (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0xa1,
                                        (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0x40, (byte) 0xe7, (byte) 0x7e, (byte) 0xe7, (byte) 0xa1, (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0x40,
                                        (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0xa1, (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0xa1,
                                        (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0x40, (byte) 0xea, (byte) 0x7e, (byte) 0xea, (byte) 0xa1, (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0x40,
                                        (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0xa1, (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0xa1,
                                        (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0x40, (byte) 0xed, (byte) 0x7e, (byte) 0xed, (byte) 0xa1, (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0x40,
                                        (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0xa1, (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0xa1,
                                        (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0x40, (byte) 0xf0, (byte) 0x7e, (byte) 0xf0, (byte) 0xa1, (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0x40,
                                        (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0xa1, (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0xa1,
                                        (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0x40, (byte) 0xf3, (byte) 0x7e, (byte) 0xf3, (byte) 0xa1, (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0x40,
                                        (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0xa1, (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0xa1,
                                        (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0x40, (byte) 0xf6, (byte) 0x7e, (byte) 0xf6, (byte) 0xa1, (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0x40,
                                        (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0xa1, (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0xa1,
                                        (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0x40, (byte) 0xf9, (byte) 0x7e, (byte) 0xf9, (byte) 0xa1, (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0x40,
                                        (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0xa1, (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0xa1,
                                        (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0x40, (byte) 0xfc, (byte) 0x7e, (byte) 0xfc, (byte) 0xa1, (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0x40,
                                        (byte) 0xfd, (byte) 0x7e, (byte) 0xfd, (byte) 0xa1, (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0x40, (byte) 0xfe, (byte) 0x7e, (byte) 0xfe, (byte) 0xa1,
                                        (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataBig5UAO() {
        assert TruffleString.Encoding.Big5_UAO.ordinal() == 10;
        return new TestData(new int[]{0x0000, 0x007f, 0x8740, 0x877e, 0x87a1, 0x87fe, 0x8840, 0x887e, 0x88a1, 0x88fe, 0x8940, 0x897e, 0x89a1, 0x89fe, 0x8a40, 0x8a7e, 0x8aa1, 0x8afe, 0x8b40, 0x8b7e,
                        0x8ba1, 0x8bfe, 0x8c40, 0x8c7e, 0x8ca1, 0x8cfe, 0x8d40, 0x8d7e, 0x8da1, 0x8dfe, 0x8e40, 0x8e7e, 0x8ea1, 0x8efe, 0x8f40, 0x8f7e, 0x8fa1, 0x8ffe, 0x9040, 0x907e, 0x90a1, 0x90fe,
                        0x9140, 0x917e, 0x91a1, 0x91fe, 0x9240, 0x927e, 0x92a1, 0x92fe, 0x9340, 0x937e, 0x93a1, 0x93fe, 0x9440, 0x947e, 0x94a1, 0x94fe, 0x9540, 0x957e, 0x95a1, 0x95fe, 0x9640, 0x967e,
                        0x96a1, 0x96fe, 0x9740, 0x977e, 0x97a1, 0x97fe, 0x9840, 0x987e, 0x98a1, 0x98fe, 0x9940, 0x997e, 0x99a1, 0x99fe, 0x9a40, 0x9a7e, 0x9aa1, 0x9afe, 0x9b40, 0x9b7e, 0x9ba1, 0x9bfe,
                        0x9c40, 0x9c7e, 0x9ca1, 0x9cfe, 0x9d40, 0x9d7e, 0x9da1, 0x9dfe, 0x9e40, 0x9e7e, 0x9ea1, 0x9efe, 0x9f40, 0x9f7e, 0x9fa1, 0x9ffe, 0xa040, 0xa07e, 0xa0a1, 0xa0fe, 0xa140, 0xa17e,
                        0xa1a1, 0xa1fe, 0xa240, 0xa27e, 0xa2a1, 0xa2fe, 0xa340, 0xa37e, 0xa3a1, 0xa3fe, 0xa440, 0xa47e, 0xa4a1, 0xa4fe, 0xa540, 0xa57e, 0xa5a1, 0xa5fe, 0xa640, 0xa67e, 0xa6a1, 0xa6fe,
                        0xa740, 0xa77e, 0xa7a1, 0xa7fe, 0xa840, 0xa87e, 0xa8a1, 0xa8fe, 0xa940, 0xa97e, 0xa9a1, 0xa9fe, 0xaa40, 0xaa7e, 0xaaa1, 0xaafe, 0xab40, 0xab7e, 0xaba1, 0xabfe, 0xac40, 0xac7e,
                        0xaca1, 0xacfe, 0xad40, 0xad7e, 0xada1, 0xadfe, 0xae40, 0xae7e, 0xaea1, 0xaefe, 0xaf40, 0xaf7e, 0xafa1, 0xaffe, 0xb040, 0xb07e, 0xb0a1, 0xb0fe, 0xb140, 0xb17e, 0xb1a1, 0xb1fe,
                        0xb240, 0xb27e, 0xb2a1, 0xb2fe, 0xb340, 0xb37e, 0xb3a1, 0xb3fe, 0xb440, 0xb47e, 0xb4a1, 0xb4fe, 0xb540, 0xb57e, 0xb5a1, 0xb5fe, 0xb640, 0xb67e, 0xb6a1, 0xb6fe, 0xb740, 0xb77e,
                        0xb7a1, 0xb7fe, 0xb840, 0xb87e, 0xb8a1, 0xb8fe, 0xb940, 0xb97e, 0xb9a1, 0xb9fe, 0xba40, 0xba7e, 0xbaa1, 0xbafe, 0xbb40, 0xbb7e, 0xbba1, 0xbbfe, 0xbc40, 0xbc7e, 0xbca1, 0xbcfe,
                        0xbd40, 0xbd7e, 0xbda1, 0xbdfe, 0xbe40, 0xbe7e, 0xbea1, 0xbefe, 0xbf40, 0xbf7e, 0xbfa1, 0xbffe, 0xc040, 0xc07e, 0xc0a1, 0xc0fe, 0xc140, 0xc17e, 0xc1a1, 0xc1fe, 0xc240, 0xc27e,
                        0xc2a1, 0xc2fe, 0xc340, 0xc37e, 0xc3a1, 0xc3fe, 0xc440, 0xc47e, 0xc4a1, 0xc4fe, 0xc540, 0xc57e, 0xc5a1, 0xc5fe, 0xc640, 0xc67e, 0xc6a1, 0xc6fe, 0xc740, 0xc77e, 0xc7a1, 0xc7fe,
                        0xc840, 0xc87e, 0xc8a1, 0xc8fe, 0xc940, 0xc97e, 0xc9a1, 0xc9fe, 0xca40, 0xca7e, 0xcaa1, 0xcafe, 0xcb40, 0xcb7e, 0xcba1, 0xcbfe, 0xcc40, 0xcc7e, 0xcca1, 0xccfe, 0xcd40, 0xcd7e,
                        0xcda1, 0xcdfe, 0xce40, 0xce7e, 0xcea1, 0xcefe, 0xcf40, 0xcf7e, 0xcfa1, 0xcffe, 0xd040, 0xd07e, 0xd0a1, 0xd0fe, 0xd140, 0xd17e, 0xd1a1, 0xd1fe, 0xd240, 0xd27e, 0xd2a1, 0xd2fe,
                        0xd340, 0xd37e, 0xd3a1, 0xd3fe, 0xd440, 0xd47e, 0xd4a1, 0xd4fe, 0xd540, 0xd57e, 0xd5a1, 0xd5fe, 0xd640, 0xd67e, 0xd6a1, 0xd6fe, 0xd740, 0xd77e, 0xd7a1, 0xd7fe, 0xd840, 0xd87e,
                        0xd8a1, 0xd8fe, 0xd940, 0xd97e, 0xd9a1, 0xd9fe, 0xda40, 0xda7e, 0xdaa1, 0xdafe, 0xdb40, 0xdb7e, 0xdba1, 0xdbfe, 0xdc40, 0xdc7e, 0xdca1, 0xdcfe, 0xdd40, 0xdd7e, 0xdda1, 0xddfe,
                        0xde40, 0xde7e, 0xdea1, 0xdefe, 0xdf40, 0xdf7e, 0xdfa1, 0xdffe, 0xe040, 0xe07e, 0xe0a1, 0xe0fe, 0xe140, 0xe17e, 0xe1a1, 0xe1fe, 0xe240, 0xe27e, 0xe2a1, 0xe2fe, 0xe340, 0xe37e,
                        0xe3a1, 0xe3fe, 0xe440, 0xe47e, 0xe4a1, 0xe4fe, 0xe540, 0xe57e, 0xe5a1, 0xe5fe, 0xe640, 0xe67e, 0xe6a1, 0xe6fe, 0xe740, 0xe77e, 0xe7a1, 0xe7fe, 0xe840, 0xe87e, 0xe8a1, 0xe8fe,
                        0xe940, 0xe97e, 0xe9a1, 0xe9fe, 0xea40, 0xea7e, 0xeaa1, 0xeafe, 0xeb40, 0xeb7e, 0xeba1, 0xebfe, 0xec40, 0xec7e, 0xeca1, 0xecfe, 0xed40, 0xed7e, 0xeda1, 0xedfe, 0xee40, 0xee7e,
                        0xeea1, 0xeefe, 0xef40, 0xef7e, 0xefa1, 0xeffe, 0xf040, 0xf07e, 0xf0a1, 0xf0fe, 0xf140, 0xf17e, 0xf1a1, 0xf1fe, 0xf240, 0xf27e, 0xf2a1, 0xf2fe, 0xf340, 0xf37e, 0xf3a1, 0xf3fe,
                        0xf440, 0xf47e, 0xf4a1, 0xf4fe, 0xf540, 0xf57e, 0xf5a1, 0xf5fe, 0xf640, 0xf67e, 0xf6a1, 0xf6fe, 0xf740, 0xf77e, 0xf7a1, 0xf7fe, 0xf840, 0xf87e, 0xf8a1, 0xf8fe, 0xf940, 0xf97e,
                        0xf9a1, 0xf9fe, 0xfa40, 0xfa7e, 0xfaa1, 0xfafe, 0xfb40, 0xfb7e, 0xfba1, 0xfbfe, 0xfc40, 0xfc7e, 0xfca1, 0xfcfe, 0xfd40, 0xfd7e, 0xfda1, 0xfdfe, 0xfe40, 0xfe7e, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198, 0x19a,
                                        0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6, 0x1c8,
                                        0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2, 0x1e4, 0x1e6, 0x1e8, 0x1ea, 0x1ec, 0x1ee, 0x1f0, 0x1f2, 0x1f4, 0x1f6,
                                        0x1f8, 0x1fa, 0x1fc, 0x1fe, 0x200, 0x202, 0x204, 0x206, 0x208, 0x20a, 0x20c, 0x20e, 0x210, 0x212, 0x214, 0x216, 0x218, 0x21a, 0x21c, 0x21e, 0x220, 0x222, 0x224,
                                        0x226, 0x228, 0x22a, 0x22c, 0x22e, 0x230, 0x232, 0x234, 0x236, 0x238, 0x23a, 0x23c, 0x23e, 0x240, 0x242, 0x244, 0x246, 0x248, 0x24a, 0x24c, 0x24e, 0x250, 0x252,
                                        0x254, 0x256, 0x258, 0x25a, 0x25c, 0x25e, 0x260, 0x262, 0x264, 0x266, 0x268, 0x26a, 0x26c, 0x26e, 0x270, 0x272, 0x274, 0x276, 0x278, 0x27a, 0x27c, 0x27e, 0x280,
                                        0x282, 0x284, 0x286, 0x288, 0x28a, 0x28c, 0x28e, 0x290, 0x292, 0x294, 0x296, 0x298, 0x29a, 0x29c, 0x29e, 0x2a0, 0x2a2, 0x2a4, 0x2a6, 0x2a8, 0x2aa, 0x2ac, 0x2ae,
                                        0x2b0, 0x2b2, 0x2b4, 0x2b6, 0x2b8, 0x2ba, 0x2bc, 0x2be, 0x2c0, 0x2c2, 0x2c4, 0x2c6, 0x2c8, 0x2ca, 0x2cc, 0x2ce, 0x2d0, 0x2d2, 0x2d4, 0x2d6, 0x2d8, 0x2da, 0x2dc,
                                        0x2de, 0x2e0, 0x2e2, 0x2e4, 0x2e6, 0x2e8, 0x2ea, 0x2ec, 0x2ee, 0x2f0, 0x2f2, 0x2f4, 0x2f6, 0x2f8, 0x2fa, 0x2fc, 0x2fe, 0x300, 0x302, 0x304, 0x306, 0x308, 0x30a,
                                        0x30c, 0x30e, 0x310, 0x312, 0x314, 0x316, 0x318, 0x31a, 0x31c, 0x31e, 0x320, 0x322, 0x324, 0x326, 0x328, 0x32a, 0x32c, 0x32e, 0x330, 0x332, 0x334, 0x336, 0x338,
                                        0x33a, 0x33c, 0x33e, 0x340, 0x342, 0x344, 0x346, 0x348, 0x34a, 0x34c, 0x34e, 0x350, 0x352, 0x354, 0x356, 0x358, 0x35a, 0x35c, 0x35e, 0x360, 0x362, 0x364, 0x366,
                                        0x368, 0x36a, 0x36c, 0x36e, 0x370, 0x372, 0x374, 0x376, 0x378, 0x37a, 0x37c, 0x37e, 0x380, 0x382, 0x384, 0x386, 0x388, 0x38a, 0x38c, 0x38e, 0x390, 0x392, 0x394,
                                        0x396, 0x398, 0x39a, 0x39c, 0x39e, 0x3a0, 0x3a2, 0x3a4, 0x3a6, 0x3a8, 0x3aa, 0x3ac, 0x3ae, 0x3b0, 0x3b2, 0x3b4, 0x3b6, 0x3b8, 0x3ba, 0x3bc, 0x3be, 0x3c0},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0x87, (byte) 0x40, (byte) 0x87, (byte) 0x7e, (byte) 0x87, (byte) 0xa1, (byte) 0x87, (byte) 0xfe, (byte) 0x88, (byte) 0x40,
                                        (byte) 0x88, (byte) 0x7e, (byte) 0x88, (byte) 0xa1, (byte) 0x88, (byte) 0xfe, (byte) 0x89, (byte) 0x40, (byte) 0x89, (byte) 0x7e, (byte) 0x89, (byte) 0xa1,
                                        (byte) 0x89, (byte) 0xfe, (byte) 0x8a, (byte) 0x40, (byte) 0x8a, (byte) 0x7e, (byte) 0x8a, (byte) 0xa1, (byte) 0x8a, (byte) 0xfe, (byte) 0x8b, (byte) 0x40,
                                        (byte) 0x8b, (byte) 0x7e, (byte) 0x8b, (byte) 0xa1, (byte) 0x8b, (byte) 0xfe, (byte) 0x8c, (byte) 0x40, (byte) 0x8c, (byte) 0x7e, (byte) 0x8c, (byte) 0xa1,
                                        (byte) 0x8c, (byte) 0xfe, (byte) 0x8d, (byte) 0x40, (byte) 0x8d, (byte) 0x7e, (byte) 0x8d, (byte) 0xa1, (byte) 0x8d, (byte) 0xfe, (byte) 0x8e, (byte) 0x40,
                                        (byte) 0x8e, (byte) 0x7e, (byte) 0x8e, (byte) 0xa1, (byte) 0x8e, (byte) 0xfe, (byte) 0x8f, (byte) 0x40, (byte) 0x8f, (byte) 0x7e, (byte) 0x8f, (byte) 0xa1,
                                        (byte) 0x8f, (byte) 0xfe, (byte) 0x90, (byte) 0x40, (byte) 0x90, (byte) 0x7e, (byte) 0x90, (byte) 0xa1, (byte) 0x90, (byte) 0xfe, (byte) 0x91, (byte) 0x40,
                                        (byte) 0x91, (byte) 0x7e, (byte) 0x91, (byte) 0xa1, (byte) 0x91, (byte) 0xfe, (byte) 0x92, (byte) 0x40, (byte) 0x92, (byte) 0x7e, (byte) 0x92, (byte) 0xa1,
                                        (byte) 0x92, (byte) 0xfe, (byte) 0x93, (byte) 0x40, (byte) 0x93, (byte) 0x7e, (byte) 0x93, (byte) 0xa1, (byte) 0x93, (byte) 0xfe, (byte) 0x94, (byte) 0x40,
                                        (byte) 0x94, (byte) 0x7e, (byte) 0x94, (byte) 0xa1, (byte) 0x94, (byte) 0xfe, (byte) 0x95, (byte) 0x40, (byte) 0x95, (byte) 0x7e, (byte) 0x95, (byte) 0xa1,
                                        (byte) 0x95, (byte) 0xfe, (byte) 0x96, (byte) 0x40, (byte) 0x96, (byte) 0x7e, (byte) 0x96, (byte) 0xa1, (byte) 0x96, (byte) 0xfe, (byte) 0x97, (byte) 0x40,
                                        (byte) 0x97, (byte) 0x7e, (byte) 0x97, (byte) 0xa1, (byte) 0x97, (byte) 0xfe, (byte) 0x98, (byte) 0x40, (byte) 0x98, (byte) 0x7e, (byte) 0x98, (byte) 0xa1,
                                        (byte) 0x98, (byte) 0xfe, (byte) 0x99, (byte) 0x40, (byte) 0x99, (byte) 0x7e, (byte) 0x99, (byte) 0xa1, (byte) 0x99, (byte) 0xfe, (byte) 0x9a, (byte) 0x40,
                                        (byte) 0x9a, (byte) 0x7e, (byte) 0x9a, (byte) 0xa1, (byte) 0x9a, (byte) 0xfe, (byte) 0x9b, (byte) 0x40, (byte) 0x9b, (byte) 0x7e, (byte) 0x9b, (byte) 0xa1,
                                        (byte) 0x9b, (byte) 0xfe, (byte) 0x9c, (byte) 0x40, (byte) 0x9c, (byte) 0x7e, (byte) 0x9c, (byte) 0xa1, (byte) 0x9c, (byte) 0xfe, (byte) 0x9d, (byte) 0x40,
                                        (byte) 0x9d, (byte) 0x7e, (byte) 0x9d, (byte) 0xa1, (byte) 0x9d, (byte) 0xfe, (byte) 0x9e, (byte) 0x40, (byte) 0x9e, (byte) 0x7e, (byte) 0x9e, (byte) 0xa1,
                                        (byte) 0x9e, (byte) 0xfe, (byte) 0x9f, (byte) 0x40, (byte) 0x9f, (byte) 0x7e, (byte) 0x9f, (byte) 0xa1, (byte) 0x9f, (byte) 0xfe, (byte) 0xa0, (byte) 0x40,
                                        (byte) 0xa0, (byte) 0x7e, (byte) 0xa0, (byte) 0xa1, (byte) 0xa0, (byte) 0xfe, (byte) 0xa1, (byte) 0x40, (byte) 0xa1, (byte) 0x7e, (byte) 0xa1, (byte) 0xa1,
                                        (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0x40, (byte) 0xa2, (byte) 0x7e, (byte) 0xa2, (byte) 0xa1, (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0x40,
                                        (byte) 0xa3, (byte) 0x7e, (byte) 0xa3, (byte) 0xa1, (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0x40, (byte) 0xa4, (byte) 0x7e, (byte) 0xa4, (byte) 0xa1,
                                        (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0x40, (byte) 0xa5, (byte) 0x7e, (byte) 0xa5, (byte) 0xa1, (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0x40,
                                        (byte) 0xa6, (byte) 0x7e, (byte) 0xa6, (byte) 0xa1, (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0x40, (byte) 0xa7, (byte) 0x7e, (byte) 0xa7, (byte) 0xa1,
                                        (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0x40, (byte) 0xa8, (byte) 0x7e, (byte) 0xa8, (byte) 0xa1, (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0x40,
                                        (byte) 0xa9, (byte) 0x7e, (byte) 0xa9, (byte) 0xa1, (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0x40, (byte) 0xaa, (byte) 0x7e, (byte) 0xaa, (byte) 0xa1,
                                        (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0x40, (byte) 0xab, (byte) 0x7e, (byte) 0xab, (byte) 0xa1, (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0x40,
                                        (byte) 0xac, (byte) 0x7e, (byte) 0xac, (byte) 0xa1, (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0x40, (byte) 0xad, (byte) 0x7e, (byte) 0xad, (byte) 0xa1,
                                        (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0x40, (byte) 0xae, (byte) 0x7e, (byte) 0xae, (byte) 0xa1, (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0x40,
                                        (byte) 0xaf, (byte) 0x7e, (byte) 0xaf, (byte) 0xa1, (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0x40, (byte) 0xb0, (byte) 0x7e, (byte) 0xb0, (byte) 0xa1,
                                        (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0x40, (byte) 0xb1, (byte) 0x7e, (byte) 0xb1, (byte) 0xa1, (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0x40,
                                        (byte) 0xb2, (byte) 0x7e, (byte) 0xb2, (byte) 0xa1, (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0x40, (byte) 0xb3, (byte) 0x7e, (byte) 0xb3, (byte) 0xa1,
                                        (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0x40, (byte) 0xb4, (byte) 0x7e, (byte) 0xb4, (byte) 0xa1, (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0x40,
                                        (byte) 0xb5, (byte) 0x7e, (byte) 0xb5, (byte) 0xa1, (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0x40, (byte) 0xb6, (byte) 0x7e, (byte) 0xb6, (byte) 0xa1,
                                        (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0x40, (byte) 0xb7, (byte) 0x7e, (byte) 0xb7, (byte) 0xa1, (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0x40,
                                        (byte) 0xb8, (byte) 0x7e, (byte) 0xb8, (byte) 0xa1, (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0x40, (byte) 0xb9, (byte) 0x7e, (byte) 0xb9, (byte) 0xa1,
                                        (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0x40, (byte) 0xba, (byte) 0x7e, (byte) 0xba, (byte) 0xa1, (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0x40,
                                        (byte) 0xbb, (byte) 0x7e, (byte) 0xbb, (byte) 0xa1, (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0x40, (byte) 0xbc, (byte) 0x7e, (byte) 0xbc, (byte) 0xa1,
                                        (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0x40, (byte) 0xbd, (byte) 0x7e, (byte) 0xbd, (byte) 0xa1, (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0x40,
                                        (byte) 0xbe, (byte) 0x7e, (byte) 0xbe, (byte) 0xa1, (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0x40, (byte) 0xbf, (byte) 0x7e, (byte) 0xbf, (byte) 0xa1,
                                        (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0x40, (byte) 0xc0, (byte) 0x7e, (byte) 0xc0, (byte) 0xa1, (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0x40,
                                        (byte) 0xc1, (byte) 0x7e, (byte) 0xc1, (byte) 0xa1, (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0x40, (byte) 0xc2, (byte) 0x7e, (byte) 0xc2, (byte) 0xa1,
                                        (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0x40, (byte) 0xc3, (byte) 0x7e, (byte) 0xc3, (byte) 0xa1, (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0x40,
                                        (byte) 0xc4, (byte) 0x7e, (byte) 0xc4, (byte) 0xa1, (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0x40, (byte) 0xc5, (byte) 0x7e, (byte) 0xc5, (byte) 0xa1,
                                        (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0x40, (byte) 0xc6, (byte) 0x7e, (byte) 0xc6, (byte) 0xa1, (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0x40,
                                        (byte) 0xc7, (byte) 0x7e, (byte) 0xc7, (byte) 0xa1, (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0x40, (byte) 0xc8, (byte) 0x7e, (byte) 0xc8, (byte) 0xa1,
                                        (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0x40, (byte) 0xc9, (byte) 0x7e, (byte) 0xc9, (byte) 0xa1, (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0x40,
                                        (byte) 0xca, (byte) 0x7e, (byte) 0xca, (byte) 0xa1, (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0x40, (byte) 0xcb, (byte) 0x7e, (byte) 0xcb, (byte) 0xa1,
                                        (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0x40, (byte) 0xcc, (byte) 0x7e, (byte) 0xcc, (byte) 0xa1, (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0x40,
                                        (byte) 0xcd, (byte) 0x7e, (byte) 0xcd, (byte) 0xa1, (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0x40, (byte) 0xce, (byte) 0x7e, (byte) 0xce, (byte) 0xa1,
                                        (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0x40, (byte) 0xcf, (byte) 0x7e, (byte) 0xcf, (byte) 0xa1, (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0x40,
                                        (byte) 0xd0, (byte) 0x7e, (byte) 0xd0, (byte) 0xa1, (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0x40, (byte) 0xd1, (byte) 0x7e, (byte) 0xd1, (byte) 0xa1,
                                        (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0x40, (byte) 0xd2, (byte) 0x7e, (byte) 0xd2, (byte) 0xa1, (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0x40,
                                        (byte) 0xd3, (byte) 0x7e, (byte) 0xd3, (byte) 0xa1, (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0x40, (byte) 0xd4, (byte) 0x7e, (byte) 0xd4, (byte) 0xa1,
                                        (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0x40, (byte) 0xd5, (byte) 0x7e, (byte) 0xd5, (byte) 0xa1, (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0x40,
                                        (byte) 0xd6, (byte) 0x7e, (byte) 0xd6, (byte) 0xa1, (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0x40, (byte) 0xd7, (byte) 0x7e, (byte) 0xd7, (byte) 0xa1,
                                        (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0x40, (byte) 0xd8, (byte) 0x7e, (byte) 0xd8, (byte) 0xa1, (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0x40,
                                        (byte) 0xd9, (byte) 0x7e, (byte) 0xd9, (byte) 0xa1, (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0x40, (byte) 0xda, (byte) 0x7e, (byte) 0xda, (byte) 0xa1,
                                        (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0x40, (byte) 0xdb, (byte) 0x7e, (byte) 0xdb, (byte) 0xa1, (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0x40,
                                        (byte) 0xdc, (byte) 0x7e, (byte) 0xdc, (byte) 0xa1, (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0x40, (byte) 0xdd, (byte) 0x7e, (byte) 0xdd, (byte) 0xa1,
                                        (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0x40, (byte) 0xde, (byte) 0x7e, (byte) 0xde, (byte) 0xa1, (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0x40,
                                        (byte) 0xdf, (byte) 0x7e, (byte) 0xdf, (byte) 0xa1, (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0xa1,
                                        (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0x40, (byte) 0xe1, (byte) 0x7e, (byte) 0xe1, (byte) 0xa1, (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0x40,
                                        (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0xa1, (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0xa1,
                                        (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0x40, (byte) 0xe4, (byte) 0x7e, (byte) 0xe4, (byte) 0xa1, (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0x40,
                                        (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0xa1, (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0xa1,
                                        (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0x40, (byte) 0xe7, (byte) 0x7e, (byte) 0xe7, (byte) 0xa1, (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0x40,
                                        (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0xa1, (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0xa1,
                                        (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0x40, (byte) 0xea, (byte) 0x7e, (byte) 0xea, (byte) 0xa1, (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0x40,
                                        (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0xa1, (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0xa1,
                                        (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0x40, (byte) 0xed, (byte) 0x7e, (byte) 0xed, (byte) 0xa1, (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0x40,
                                        (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0xa1, (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0xa1,
                                        (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0x40, (byte) 0xf0, (byte) 0x7e, (byte) 0xf0, (byte) 0xa1, (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0x40,
                                        (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0xa1, (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0xa1,
                                        (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0x40, (byte) 0xf3, (byte) 0x7e, (byte) 0xf3, (byte) 0xa1, (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0x40,
                                        (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0xa1, (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0xa1,
                                        (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0x40, (byte) 0xf6, (byte) 0x7e, (byte) 0xf6, (byte) 0xa1, (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0x40,
                                        (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0xa1, (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0xa1,
                                        (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0x40, (byte) 0xf9, (byte) 0x7e, (byte) 0xf9, (byte) 0xa1, (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0x40,
                                        (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0xa1, (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0xa1,
                                        (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0x40, (byte) 0xfc, (byte) 0x7e, (byte) 0xfc, (byte) 0xa1, (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0x40,
                                        (byte) 0xfd, (byte) 0x7e, (byte) 0xfd, (byte) 0xa1, (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0x40, (byte) 0xfe, (byte) 0x7e, (byte) 0xfe, (byte) 0xa1,
                                        (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataCESU8() {
        assert TruffleString.Encoding.CESU_8.ordinal() == 11;
        return new TestData(new int[]{0x000000, 0x00d7ff, 0x00e000, 0x10ffff}, new int[]{0x0, 0x1, 0x4, 0x7}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xed,
                        (byte) 0x9f, (byte) 0xbf, (byte) 0xee, (byte) 0x80, (byte) 0x80, (byte) 0xed, (byte) 0xaf, (byte) 0xbf, (byte) 0xed, (byte) 0xbf, (byte) 0xbf}, null, null);
    }

    static TestData dataCP51932() {
        assert TruffleString.Encoding.CP51932.ordinal() == 12;
        return new TestData(new int[]{0x0000, 0x007f, 0x8ea1, 0x8efe, 0xa1a1, 0xa1fe, 0xa2a1, 0xa2fe, 0xa3a1, 0xa3fe, 0xa4a1, 0xa4fe, 0xa5a1, 0xa5fe, 0xa6a1, 0xa6fe, 0xa7a1, 0xa7fe, 0xa8a1, 0xa8fe,
                        0xa9a1, 0xa9fe, 0xaaa1, 0xaafe, 0xaba1, 0xabfe, 0xaca1, 0xacfe, 0xada1, 0xadfe, 0xaea1, 0xaefe, 0xafa1, 0xaffe, 0xb0a1, 0xb0fe, 0xb1a1, 0xb1fe, 0xb2a1, 0xb2fe, 0xb3a1, 0xb3fe,
                        0xb4a1, 0xb4fe, 0xb5a1, 0xb5fe, 0xb6a1, 0xb6fe, 0xb7a1, 0xb7fe, 0xb8a1, 0xb8fe, 0xb9a1, 0xb9fe, 0xbaa1, 0xbafe, 0xbba1, 0xbbfe, 0xbca1, 0xbcfe, 0xbda1, 0xbdfe, 0xbea1, 0xbefe,
                        0xbfa1, 0xbffe, 0xc0a1, 0xc0fe, 0xc1a1, 0xc1fe, 0xc2a1, 0xc2fe, 0xc3a1, 0xc3fe, 0xc4a1, 0xc4fe, 0xc5a1, 0xc5fe, 0xc6a1, 0xc6fe, 0xc7a1, 0xc7fe, 0xc8a1, 0xc8fe, 0xc9a1, 0xc9fe,
                        0xcaa1, 0xcafe, 0xcba1, 0xcbfe, 0xcca1, 0xccfe, 0xcda1, 0xcdfe, 0xcea1, 0xcefe, 0xcfa1, 0xcffe, 0xd0a1, 0xd0fe, 0xd1a1, 0xd1fe, 0xd2a1, 0xd2fe, 0xd3a1, 0xd3fe, 0xd4a1, 0xd4fe,
                        0xd5a1, 0xd5fe, 0xd6a1, 0xd6fe, 0xd7a1, 0xd7fe, 0xd8a1, 0xd8fe, 0xd9a1, 0xd9fe, 0xdaa1, 0xdafe, 0xdba1, 0xdbfe, 0xdca1, 0xdcfe, 0xdda1, 0xddfe, 0xdea1, 0xdefe, 0xdfa1, 0xdffe,
                        0xe0a1, 0xe0fe, 0xe1a1, 0xe1fe, 0xe2a1, 0xe2fe, 0xe3a1, 0xe3fe, 0xe4a1, 0xe4fe, 0xe5a1, 0xe5fe, 0xe6a1, 0xe6fe, 0xe7a1, 0xe7fe, 0xe8a1, 0xe8fe, 0xe9a1, 0xe9fe, 0xeaa1, 0xeafe,
                        0xeba1, 0xebfe, 0xeca1, 0xecfe, 0xeda1, 0xedfe, 0xeea1, 0xeefe, 0xefa1, 0xeffe, 0xf0a1, 0xf0fe, 0xf1a1, 0xf1fe, 0xf2a1, 0xf2fe, 0xf3a1, 0xf3fe, 0xf4a1, 0xf4fe, 0xf5a1, 0xf5fe,
                        0xf6a1, 0xf6fe, 0xf7a1, 0xf7fe, 0xf8a1, 0xf8fe, 0xf9a1, 0xf9fe, 0xfaa1, 0xfafe, 0xfba1, 0xfbfe, 0xfca1, 0xfcfe, 0xfda1, 0xfdfe, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0x8e, (byte) 0xa1, (byte) 0x8e, (byte) 0xfe, (byte) 0xa1, (byte) 0xa1, (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0xa1,
                                        (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0xa1, (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0xa1, (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0xa1,
                                        (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0xa1, (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0xa1, (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0xa1,
                                        (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0xa1, (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0xa1, (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0xa1,
                                        (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0xa1, (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0xa1, (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0xa1,
                                        (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0xa1, (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0xa1, (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0xa1,
                                        (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0xa1, (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0xa1, (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0xa1,
                                        (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0xa1, (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0xa1, (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0xa1,
                                        (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0xa1, (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0xa1, (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0xa1,
                                        (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0xa1, (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0xa1, (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0xa1,
                                        (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0xa1, (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0xa1, (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0xa1,
                                        (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0xa1, (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0xa1, (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0xa1,
                                        (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0xa1, (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0xa1, (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0xa1,
                                        (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0xa1, (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0xa1, (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0xa1,
                                        (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0xa1, (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0xa1, (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0xa1,
                                        (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0xa1, (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0xa1, (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0xa1,
                                        (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0xa1, (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0xa1, (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0xa1,
                                        (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0xa1, (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0xa1, (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0xa1,
                                        (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0xa1, (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0xa1, (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0xa1,
                                        (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0xa1, (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0xa1, (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0xa1,
                                        (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0xa1, (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0xa1, (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0xa1,
                                        (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0xa1, (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0xa1, (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0xa1,
                                        (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0xa1, (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0xa1, (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0xa1,
                                        (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0xa1, (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0xa1, (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0xa1,
                                        (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0xa1, (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0xa1, (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0xa1,
                                        (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0xa1, (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0xa1, (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0xa1,
                                        (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0xa1, (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0xa1, (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0xa1,
                                        (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0xa1, (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0xa1, (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0xa1,
                                        (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0xa1, (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0xa1, (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0xa1,
                                        (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0xa1, (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0xa1, (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0xa1,
                                        (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0xa1, (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0xa1, (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0xa1,
                                        (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0xa1, (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0xa1, (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataCP850() {
        assert TruffleString.Encoding.CP850.ordinal() == 13;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataCP852() {
        assert TruffleString.Encoding.CP852.ordinal() == 14;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataCP855() {
        assert TruffleString.Encoding.CP855.ordinal() == 15;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataCP949() {
        assert TruffleString.Encoding.CP949.ordinal() == 16;
        return new TestData(new int[]{0x0000, 0x0080, 0x8141, 0x815a, 0x8161, 0x817a, 0x8181, 0x81fe, 0x8241, 0x825a, 0x8261, 0x827a, 0x8281, 0x82fe, 0x8341, 0x835a, 0x8361, 0x837a, 0x8381, 0x83fe,
                        0x8441, 0x845a, 0x8461, 0x847a, 0x8481, 0x84fe, 0x8541, 0x855a, 0x8561, 0x857a, 0x8581, 0x85fe, 0x8641, 0x865a, 0x8661, 0x867a, 0x8681, 0x86fe, 0x8741, 0x875a, 0x8761, 0x877a,
                        0x8781, 0x87fe, 0x8841, 0x885a, 0x8861, 0x887a, 0x8881, 0x88fe, 0x8941, 0x895a, 0x8961, 0x897a, 0x8981, 0x89fe, 0x8a41, 0x8a5a, 0x8a61, 0x8a7a, 0x8a81, 0x8afe, 0x8b41, 0x8b5a,
                        0x8b61, 0x8b7a, 0x8b81, 0x8bfe, 0x8c41, 0x8c5a, 0x8c61, 0x8c7a, 0x8c81, 0x8cfe, 0x8d41, 0x8d5a, 0x8d61, 0x8d7a, 0x8d81, 0x8dfe, 0x8e41, 0x8e5a, 0x8e61, 0x8e7a, 0x8e81, 0x8efe,
                        0x8f41, 0x8f5a, 0x8f61, 0x8f7a, 0x8f81, 0x8ffe, 0x9041, 0x905a, 0x9061, 0x907a, 0x9081, 0x90fe, 0x9141, 0x915a, 0x9161, 0x917a, 0x9181, 0x91fe, 0x9241, 0x925a, 0x9261, 0x927a,
                        0x9281, 0x92fe, 0x9341, 0x935a, 0x9361, 0x937a, 0x9381, 0x93fe, 0x9441, 0x945a, 0x9461, 0x947a, 0x9481, 0x94fe, 0x9541, 0x955a, 0x9561, 0x957a, 0x9581, 0x95fe, 0x9641, 0x965a,
                        0x9661, 0x967a, 0x9681, 0x96fe, 0x9741, 0x975a, 0x9761, 0x977a, 0x9781, 0x97fe, 0x9841, 0x985a, 0x9861, 0x987a, 0x9881, 0x98fe, 0x9941, 0x995a, 0x9961, 0x997a, 0x9981, 0x99fe,
                        0x9a41, 0x9a5a, 0x9a61, 0x9a7a, 0x9a81, 0x9afe, 0x9b41, 0x9b5a, 0x9b61, 0x9b7a, 0x9b81, 0x9bfe, 0x9c41, 0x9c5a, 0x9c61, 0x9c7a, 0x9c81, 0x9cfe, 0x9d41, 0x9d5a, 0x9d61, 0x9d7a,
                        0x9d81, 0x9dfe, 0x9e41, 0x9e5a, 0x9e61, 0x9e7a, 0x9e81, 0x9efe, 0x9f41, 0x9f5a, 0x9f61, 0x9f7a, 0x9f81, 0x9ffe, 0xa041, 0xa05a, 0xa061, 0xa07a, 0xa081, 0xa0fe, 0xa141, 0xa15a,
                        0xa161, 0xa17a, 0xa181, 0xa1fe, 0xa241, 0xa25a, 0xa261, 0xa27a, 0xa281, 0xa2fe, 0xa341, 0xa35a, 0xa361, 0xa37a, 0xa381, 0xa3fe, 0xa441, 0xa45a, 0xa461, 0xa47a, 0xa481, 0xa4fe,
                        0xa541, 0xa55a, 0xa561, 0xa57a, 0xa581, 0xa5fe, 0xa641, 0xa65a, 0xa661, 0xa67a, 0xa681, 0xa6fe, 0xa741, 0xa75a, 0xa761, 0xa77a, 0xa781, 0xa7fe, 0xa841, 0xa85a, 0xa861, 0xa87a,
                        0xa881, 0xa8fe, 0xa941, 0xa95a, 0xa961, 0xa97a, 0xa981, 0xa9fe, 0xaa41, 0xaa5a, 0xaa61, 0xaa7a, 0xaa81, 0xaafe, 0xab41, 0xab5a, 0xab61, 0xab7a, 0xab81, 0xabfe, 0xac41, 0xac5a,
                        0xac61, 0xac7a, 0xac81, 0xacfe, 0xad41, 0xad5a, 0xad61, 0xad7a, 0xad81, 0xadfe, 0xae41, 0xae5a, 0xae61, 0xae7a, 0xae81, 0xaefe, 0xaf41, 0xaf5a, 0xaf61, 0xaf7a, 0xaf81, 0xaffe,
                        0xb041, 0xb05a, 0xb061, 0xb07a, 0xb081, 0xb0fe, 0xb141, 0xb15a, 0xb161, 0xb17a, 0xb181, 0xb1fe, 0xb241, 0xb25a, 0xb261, 0xb27a, 0xb281, 0xb2fe, 0xb341, 0xb35a, 0xb361, 0xb37a,
                        0xb381, 0xb3fe, 0xb441, 0xb45a, 0xb461, 0xb47a, 0xb481, 0xb4fe, 0xb541, 0xb55a, 0xb561, 0xb57a, 0xb581, 0xb5fe, 0xb641, 0xb65a, 0xb661, 0xb67a, 0xb681, 0xb6fe, 0xb741, 0xb75a,
                        0xb761, 0xb77a, 0xb781, 0xb7fe, 0xb841, 0xb85a, 0xb861, 0xb87a, 0xb881, 0xb8fe, 0xb941, 0xb95a, 0xb961, 0xb97a, 0xb981, 0xb9fe, 0xba41, 0xba5a, 0xba61, 0xba7a, 0xba81, 0xbafe,
                        0xbb41, 0xbb5a, 0xbb61, 0xbb7a, 0xbb81, 0xbbfe, 0xbc41, 0xbc5a, 0xbc61, 0xbc7a, 0xbc81, 0xbcfe, 0xbd41, 0xbd5a, 0xbd61, 0xbd7a, 0xbd81, 0xbdfe, 0xbe41, 0xbe5a, 0xbe61, 0xbe7a,
                        0xbe81, 0xbefe, 0xbf41, 0xbf5a, 0xbf61, 0xbf7a, 0xbf81, 0xbffe, 0xc041, 0xc05a, 0xc061, 0xc07a, 0xc081, 0xc0fe, 0xc141, 0xc15a, 0xc161, 0xc17a, 0xc181, 0xc1fe, 0xc241, 0xc25a,
                        0xc261, 0xc27a, 0xc281, 0xc2fe, 0xc341, 0xc35a, 0xc361, 0xc37a, 0xc381, 0xc3fe, 0xc441, 0xc45a, 0xc461, 0xc47a, 0xc481, 0xc4fe, 0xc541, 0xc55a, 0xc561, 0xc57a, 0xc581, 0xc5fe,
                        0xc641, 0xc65a, 0xc661, 0xc67a, 0xc681, 0xc6fe, 0xc741, 0xc75a, 0xc761, 0xc77a, 0xc781, 0xc7fe, 0xc841, 0xc85a, 0xc861, 0xc87a, 0xc881, 0xc8fe, 0xc941, 0xc95a, 0xc961, 0xc97a,
                        0xc981, 0xc9fe, 0xca41, 0xca5a, 0xca61, 0xca7a, 0xca81, 0xcafe, 0xcb41, 0xcb5a, 0xcb61, 0xcb7a, 0xcb81, 0xcbfe, 0xcc41, 0xcc5a, 0xcc61, 0xcc7a, 0xcc81, 0xccfe, 0xcd41, 0xcd5a,
                        0xcd61, 0xcd7a, 0xcd81, 0xcdfe, 0xce41, 0xce5a, 0xce61, 0xce7a, 0xce81, 0xcefe, 0xcf41, 0xcf5a, 0xcf61, 0xcf7a, 0xcf81, 0xcffe, 0xd041, 0xd05a, 0xd061, 0xd07a, 0xd081, 0xd0fe,
                        0xd141, 0xd15a, 0xd161, 0xd17a, 0xd181, 0xd1fe, 0xd241, 0xd25a, 0xd261, 0xd27a, 0xd281, 0xd2fe, 0xd341, 0xd35a, 0xd361, 0xd37a, 0xd381, 0xd3fe, 0xd441, 0xd45a, 0xd461, 0xd47a,
                        0xd481, 0xd4fe, 0xd541, 0xd55a, 0xd561, 0xd57a, 0xd581, 0xd5fe, 0xd641, 0xd65a, 0xd661, 0xd67a, 0xd681, 0xd6fe, 0xd741, 0xd75a, 0xd761, 0xd77a, 0xd781, 0xd7fe, 0xd841, 0xd85a,
                        0xd861, 0xd87a, 0xd881, 0xd8fe, 0xd941, 0xd95a, 0xd961, 0xd97a, 0xd981, 0xd9fe, 0xda41, 0xda5a, 0xda61, 0xda7a, 0xda81, 0xdafe, 0xdb41, 0xdb5a, 0xdb61, 0xdb7a, 0xdb81, 0xdbfe,
                        0xdc41, 0xdc5a, 0xdc61, 0xdc7a, 0xdc81, 0xdcfe, 0xdd41, 0xdd5a, 0xdd61, 0xdd7a, 0xdd81, 0xddfe, 0xde41, 0xde5a, 0xde61, 0xde7a, 0xde81, 0xdefe, 0xdf41, 0xdf5a, 0xdf61, 0xdf7a,
                        0xdf81, 0xdffe, 0xe041, 0xe05a, 0xe061, 0xe07a, 0xe081, 0xe0fe, 0xe141, 0xe15a, 0xe161, 0xe17a, 0xe181, 0xe1fe, 0xe241, 0xe25a, 0xe261, 0xe27a, 0xe281, 0xe2fe, 0xe341, 0xe35a,
                        0xe361, 0xe37a, 0xe381, 0xe3fe, 0xe441, 0xe45a, 0xe461, 0xe47a, 0xe481, 0xe4fe, 0xe541, 0xe55a, 0xe561, 0xe57a, 0xe581, 0xe5fe, 0xe641, 0xe65a, 0xe661, 0xe67a, 0xe681, 0xe6fe,
                        0xe741, 0xe75a, 0xe761, 0xe77a, 0xe781, 0xe7fe, 0xe841, 0xe85a, 0xe861, 0xe87a, 0xe881, 0xe8fe, 0xe941, 0xe95a, 0xe961, 0xe97a, 0xe981, 0xe9fe, 0xea41, 0xea5a, 0xea61, 0xea7a,
                        0xea81, 0xeafe, 0xeb41, 0xeb5a, 0xeb61, 0xeb7a, 0xeb81, 0xebfe, 0xec41, 0xec5a, 0xec61, 0xec7a, 0xec81, 0xecfe, 0xed41, 0xed5a, 0xed61, 0xed7a, 0xed81, 0xedfe, 0xee41, 0xee5a,
                        0xee61, 0xee7a, 0xee81, 0xeefe, 0xef41, 0xef5a, 0xef61, 0xef7a, 0xef81, 0xeffe, 0xf041, 0xf05a, 0xf061, 0xf07a, 0xf081, 0xf0fe, 0xf141, 0xf15a, 0xf161, 0xf17a, 0xf181, 0xf1fe,
                        0xf241, 0xf25a, 0xf261, 0xf27a, 0xf281, 0xf2fe, 0xf341, 0xf35a, 0xf361, 0xf37a, 0xf381, 0xf3fe, 0xf441, 0xf45a, 0xf461, 0xf47a, 0xf481, 0xf4fe, 0xf541, 0xf55a, 0xf561, 0xf57a,
                        0xf581, 0xf5fe, 0xf641, 0xf65a, 0xf661, 0xf67a, 0xf681, 0xf6fe, 0xf741, 0xf75a, 0xf761, 0xf77a, 0xf781, 0xf7fe, 0xf841, 0xf85a, 0xf861, 0xf87a, 0xf881, 0xf8fe, 0xf941, 0xf95a,
                        0xf961, 0xf97a, 0xf981, 0xf9fe, 0xfa41, 0xfa5a, 0xfa61, 0xfa7a, 0xfa81, 0xfafe, 0xfb41, 0xfb5a, 0xfb61, 0xfb7a, 0xfb81, 0xfbfe, 0xfc41, 0xfc5a, 0xfc61, 0xfc7a, 0xfc81, 0xfcfe,
                        0xfd41, 0xfd5a, 0xfd61, 0xfd7a, 0xfd81, 0xfdfe, 0xfe41, 0xfe5a, 0xfe61, 0xfe7a, 0xfe81, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198, 0x19a,
                                        0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6, 0x1c8,
                                        0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2, 0x1e4, 0x1e6, 0x1e8, 0x1ea, 0x1ec, 0x1ee, 0x1f0, 0x1f2, 0x1f4, 0x1f6,
                                        0x1f8, 0x1fa, 0x1fc, 0x1fe, 0x200, 0x202, 0x204, 0x206, 0x208, 0x20a, 0x20c, 0x20e, 0x210, 0x212, 0x214, 0x216, 0x218, 0x21a, 0x21c, 0x21e, 0x220, 0x222, 0x224,
                                        0x226, 0x228, 0x22a, 0x22c, 0x22e, 0x230, 0x232, 0x234, 0x236, 0x238, 0x23a, 0x23c, 0x23e, 0x240, 0x242, 0x244, 0x246, 0x248, 0x24a, 0x24c, 0x24e, 0x250, 0x252,
                                        0x254, 0x256, 0x258, 0x25a, 0x25c, 0x25e, 0x260, 0x262, 0x264, 0x266, 0x268, 0x26a, 0x26c, 0x26e, 0x270, 0x272, 0x274, 0x276, 0x278, 0x27a, 0x27c, 0x27e, 0x280,
                                        0x282, 0x284, 0x286, 0x288, 0x28a, 0x28c, 0x28e, 0x290, 0x292, 0x294, 0x296, 0x298, 0x29a, 0x29c, 0x29e, 0x2a0, 0x2a2, 0x2a4, 0x2a6, 0x2a8, 0x2aa, 0x2ac, 0x2ae,
                                        0x2b0, 0x2b2, 0x2b4, 0x2b6, 0x2b8, 0x2ba, 0x2bc, 0x2be, 0x2c0, 0x2c2, 0x2c4, 0x2c6, 0x2c8, 0x2ca, 0x2cc, 0x2ce, 0x2d0, 0x2d2, 0x2d4, 0x2d6, 0x2d8, 0x2da, 0x2dc,
                                        0x2de, 0x2e0, 0x2e2, 0x2e4, 0x2e6, 0x2e8, 0x2ea, 0x2ec, 0x2ee, 0x2f0, 0x2f2, 0x2f4, 0x2f6, 0x2f8, 0x2fa, 0x2fc, 0x2fe, 0x300, 0x302, 0x304, 0x306, 0x308, 0x30a,
                                        0x30c, 0x30e, 0x310, 0x312, 0x314, 0x316, 0x318, 0x31a, 0x31c, 0x31e, 0x320, 0x322, 0x324, 0x326, 0x328, 0x32a, 0x32c, 0x32e, 0x330, 0x332, 0x334, 0x336, 0x338,
                                        0x33a, 0x33c, 0x33e, 0x340, 0x342, 0x344, 0x346, 0x348, 0x34a, 0x34c, 0x34e, 0x350, 0x352, 0x354, 0x356, 0x358, 0x35a, 0x35c, 0x35e, 0x360, 0x362, 0x364, 0x366,
                                        0x368, 0x36a, 0x36c, 0x36e, 0x370, 0x372, 0x374, 0x376, 0x378, 0x37a, 0x37c, 0x37e, 0x380, 0x382, 0x384, 0x386, 0x388, 0x38a, 0x38c, 0x38e, 0x390, 0x392, 0x394,
                                        0x396, 0x398, 0x39a, 0x39c, 0x39e, 0x3a0, 0x3a2, 0x3a4, 0x3a6, 0x3a8, 0x3aa, 0x3ac, 0x3ae, 0x3b0, 0x3b2, 0x3b4, 0x3b6, 0x3b8, 0x3ba, 0x3bc, 0x3be, 0x3c0, 0x3c2,
                                        0x3c4, 0x3c6, 0x3c8, 0x3ca, 0x3cc, 0x3ce, 0x3d0, 0x3d2, 0x3d4, 0x3d6, 0x3d8, 0x3da, 0x3dc, 0x3de, 0x3e0, 0x3e2, 0x3e4, 0x3e6, 0x3e8, 0x3ea, 0x3ec, 0x3ee, 0x3f0,
                                        0x3f2, 0x3f4, 0x3f6, 0x3f8, 0x3fa, 0x3fc, 0x3fe, 0x400, 0x402, 0x404, 0x406, 0x408, 0x40a, 0x40c, 0x40e, 0x410, 0x412, 0x414, 0x416, 0x418, 0x41a, 0x41c, 0x41e,
                                        0x420, 0x422, 0x424, 0x426, 0x428, 0x42a, 0x42c, 0x42e, 0x430, 0x432, 0x434, 0x436, 0x438, 0x43a, 0x43c, 0x43e, 0x440, 0x442, 0x444, 0x446, 0x448, 0x44a, 0x44c,
                                        0x44e, 0x450, 0x452, 0x454, 0x456, 0x458, 0x45a, 0x45c, 0x45e, 0x460, 0x462, 0x464, 0x466, 0x468, 0x46a, 0x46c, 0x46e, 0x470, 0x472, 0x474, 0x476, 0x478, 0x47a,
                                        0x47c, 0x47e, 0x480, 0x482, 0x484, 0x486, 0x488, 0x48a, 0x48c, 0x48e, 0x490, 0x492, 0x494, 0x496, 0x498, 0x49a, 0x49c, 0x49e, 0x4a0, 0x4a2, 0x4a4, 0x4a6, 0x4a8,
                                        0x4aa, 0x4ac, 0x4ae, 0x4b0, 0x4b2, 0x4b4, 0x4b6, 0x4b8, 0x4ba, 0x4bc, 0x4be, 0x4c0, 0x4c2, 0x4c4, 0x4c6, 0x4c8, 0x4ca, 0x4cc, 0x4ce, 0x4d0, 0x4d2, 0x4d4, 0x4d6,
                                        0x4d8, 0x4da, 0x4dc, 0x4de, 0x4e0, 0x4e2, 0x4e4, 0x4e6, 0x4e8, 0x4ea, 0x4ec, 0x4ee, 0x4f0, 0x4f2, 0x4f4, 0x4f6, 0x4f8, 0x4fa, 0x4fc, 0x4fe, 0x500, 0x502, 0x504,
                                        0x506, 0x508, 0x50a, 0x50c, 0x50e, 0x510, 0x512, 0x514, 0x516, 0x518, 0x51a, 0x51c, 0x51e, 0x520, 0x522, 0x524, 0x526, 0x528, 0x52a, 0x52c, 0x52e, 0x530, 0x532,
                                        0x534, 0x536, 0x538, 0x53a, 0x53c, 0x53e, 0x540, 0x542, 0x544, 0x546, 0x548, 0x54a, 0x54c, 0x54e, 0x550, 0x552, 0x554, 0x556, 0x558, 0x55a, 0x55c, 0x55e, 0x560,
                                        0x562, 0x564, 0x566, 0x568, 0x56a, 0x56c, 0x56e, 0x570, 0x572, 0x574, 0x576, 0x578, 0x57a, 0x57c, 0x57e, 0x580, 0x582, 0x584, 0x586, 0x588, 0x58a, 0x58c, 0x58e,
                                        0x590, 0x592, 0x594, 0x596, 0x598, 0x59a, 0x59c, 0x59e, 0x5a0, 0x5a2, 0x5a4, 0x5a6, 0x5a8, 0x5aa, 0x5ac, 0x5ae, 0x5b0, 0x5b2, 0x5b4, 0x5b6, 0x5b8, 0x5ba, 0x5bc,
                                        0x5be, 0x5c0, 0x5c2, 0x5c4, 0x5c6, 0x5c8, 0x5ca, 0x5cc, 0x5ce, 0x5d0, 0x5d2, 0x5d4, 0x5d6, 0x5d8, 0x5da, 0x5dc, 0x5de, 0x5e0, 0x5e2, 0x5e4, 0x5e6, 0x5e8},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x80, (byte) 0x81, (byte) 0x41, (byte) 0x81, (byte) 0x5a, (byte) 0x81, (byte) 0x61, (byte) 0x81, (byte) 0x7a, (byte) 0x81, (byte) 0x81,
                                        (byte) 0x81, (byte) 0xfe, (byte) 0x82, (byte) 0x41, (byte) 0x82, (byte) 0x5a, (byte) 0x82, (byte) 0x61, (byte) 0x82, (byte) 0x7a, (byte) 0x82, (byte) 0x81,
                                        (byte) 0x82, (byte) 0xfe, (byte) 0x83, (byte) 0x41, (byte) 0x83, (byte) 0x5a, (byte) 0x83, (byte) 0x61, (byte) 0x83, (byte) 0x7a, (byte) 0x83, (byte) 0x81,
                                        (byte) 0x83, (byte) 0xfe, (byte) 0x84, (byte) 0x41, (byte) 0x84, (byte) 0x5a, (byte) 0x84, (byte) 0x61, (byte) 0x84, (byte) 0x7a, (byte) 0x84, (byte) 0x81,
                                        (byte) 0x84, (byte) 0xfe, (byte) 0x85, (byte) 0x41, (byte) 0x85, (byte) 0x5a, (byte) 0x85, (byte) 0x61, (byte) 0x85, (byte) 0x7a, (byte) 0x85, (byte) 0x81,
                                        (byte) 0x85, (byte) 0xfe, (byte) 0x86, (byte) 0x41, (byte) 0x86, (byte) 0x5a, (byte) 0x86, (byte) 0x61, (byte) 0x86, (byte) 0x7a, (byte) 0x86, (byte) 0x81,
                                        (byte) 0x86, (byte) 0xfe, (byte) 0x87, (byte) 0x41, (byte) 0x87, (byte) 0x5a, (byte) 0x87, (byte) 0x61, (byte) 0x87, (byte) 0x7a, (byte) 0x87, (byte) 0x81,
                                        (byte) 0x87, (byte) 0xfe, (byte) 0x88, (byte) 0x41, (byte) 0x88, (byte) 0x5a, (byte) 0x88, (byte) 0x61, (byte) 0x88, (byte) 0x7a, (byte) 0x88, (byte) 0x81,
                                        (byte) 0x88, (byte) 0xfe, (byte) 0x89, (byte) 0x41, (byte) 0x89, (byte) 0x5a, (byte) 0x89, (byte) 0x61, (byte) 0x89, (byte) 0x7a, (byte) 0x89, (byte) 0x81,
                                        (byte) 0x89, (byte) 0xfe, (byte) 0x8a, (byte) 0x41, (byte) 0x8a, (byte) 0x5a, (byte) 0x8a, (byte) 0x61, (byte) 0x8a, (byte) 0x7a, (byte) 0x8a, (byte) 0x81,
                                        (byte) 0x8a, (byte) 0xfe, (byte) 0x8b, (byte) 0x41, (byte) 0x8b, (byte) 0x5a, (byte) 0x8b, (byte) 0x61, (byte) 0x8b, (byte) 0x7a, (byte) 0x8b, (byte) 0x81,
                                        (byte) 0x8b, (byte) 0xfe, (byte) 0x8c, (byte) 0x41, (byte) 0x8c, (byte) 0x5a, (byte) 0x8c, (byte) 0x61, (byte) 0x8c, (byte) 0x7a, (byte) 0x8c, (byte) 0x81,
                                        (byte) 0x8c, (byte) 0xfe, (byte) 0x8d, (byte) 0x41, (byte) 0x8d, (byte) 0x5a, (byte) 0x8d, (byte) 0x61, (byte) 0x8d, (byte) 0x7a, (byte) 0x8d, (byte) 0x81,
                                        (byte) 0x8d, (byte) 0xfe, (byte) 0x8e, (byte) 0x41, (byte) 0x8e, (byte) 0x5a, (byte) 0x8e, (byte) 0x61, (byte) 0x8e, (byte) 0x7a, (byte) 0x8e, (byte) 0x81,
                                        (byte) 0x8e, (byte) 0xfe, (byte) 0x8f, (byte) 0x41, (byte) 0x8f, (byte) 0x5a, (byte) 0x8f, (byte) 0x61, (byte) 0x8f, (byte) 0x7a, (byte) 0x8f, (byte) 0x81,
                                        (byte) 0x8f, (byte) 0xfe, (byte) 0x90, (byte) 0x41, (byte) 0x90, (byte) 0x5a, (byte) 0x90, (byte) 0x61, (byte) 0x90, (byte) 0x7a, (byte) 0x90, (byte) 0x81,
                                        (byte) 0x90, (byte) 0xfe, (byte) 0x91, (byte) 0x41, (byte) 0x91, (byte) 0x5a, (byte) 0x91, (byte) 0x61, (byte) 0x91, (byte) 0x7a, (byte) 0x91, (byte) 0x81,
                                        (byte) 0x91, (byte) 0xfe, (byte) 0x92, (byte) 0x41, (byte) 0x92, (byte) 0x5a, (byte) 0x92, (byte) 0x61, (byte) 0x92, (byte) 0x7a, (byte) 0x92, (byte) 0x81,
                                        (byte) 0x92, (byte) 0xfe, (byte) 0x93, (byte) 0x41, (byte) 0x93, (byte) 0x5a, (byte) 0x93, (byte) 0x61, (byte) 0x93, (byte) 0x7a, (byte) 0x93, (byte) 0x81,
                                        (byte) 0x93, (byte) 0xfe, (byte) 0x94, (byte) 0x41, (byte) 0x94, (byte) 0x5a, (byte) 0x94, (byte) 0x61, (byte) 0x94, (byte) 0x7a, (byte) 0x94, (byte) 0x81,
                                        (byte) 0x94, (byte) 0xfe, (byte) 0x95, (byte) 0x41, (byte) 0x95, (byte) 0x5a, (byte) 0x95, (byte) 0x61, (byte) 0x95, (byte) 0x7a, (byte) 0x95, (byte) 0x81,
                                        (byte) 0x95, (byte) 0xfe, (byte) 0x96, (byte) 0x41, (byte) 0x96, (byte) 0x5a, (byte) 0x96, (byte) 0x61, (byte) 0x96, (byte) 0x7a, (byte) 0x96, (byte) 0x81,
                                        (byte) 0x96, (byte) 0xfe, (byte) 0x97, (byte) 0x41, (byte) 0x97, (byte) 0x5a, (byte) 0x97, (byte) 0x61, (byte) 0x97, (byte) 0x7a, (byte) 0x97, (byte) 0x81,
                                        (byte) 0x97, (byte) 0xfe, (byte) 0x98, (byte) 0x41, (byte) 0x98, (byte) 0x5a, (byte) 0x98, (byte) 0x61, (byte) 0x98, (byte) 0x7a, (byte) 0x98, (byte) 0x81,
                                        (byte) 0x98, (byte) 0xfe, (byte) 0x99, (byte) 0x41, (byte) 0x99, (byte) 0x5a, (byte) 0x99, (byte) 0x61, (byte) 0x99, (byte) 0x7a, (byte) 0x99, (byte) 0x81,
                                        (byte) 0x99, (byte) 0xfe, (byte) 0x9a, (byte) 0x41, (byte) 0x9a, (byte) 0x5a, (byte) 0x9a, (byte) 0x61, (byte) 0x9a, (byte) 0x7a, (byte) 0x9a, (byte) 0x81,
                                        (byte) 0x9a, (byte) 0xfe, (byte) 0x9b, (byte) 0x41, (byte) 0x9b, (byte) 0x5a, (byte) 0x9b, (byte) 0x61, (byte) 0x9b, (byte) 0x7a, (byte) 0x9b, (byte) 0x81,
                                        (byte) 0x9b, (byte) 0xfe, (byte) 0x9c, (byte) 0x41, (byte) 0x9c, (byte) 0x5a, (byte) 0x9c, (byte) 0x61, (byte) 0x9c, (byte) 0x7a, (byte) 0x9c, (byte) 0x81,
                                        (byte) 0x9c, (byte) 0xfe, (byte) 0x9d, (byte) 0x41, (byte) 0x9d, (byte) 0x5a, (byte) 0x9d, (byte) 0x61, (byte) 0x9d, (byte) 0x7a, (byte) 0x9d, (byte) 0x81,
                                        (byte) 0x9d, (byte) 0xfe, (byte) 0x9e, (byte) 0x41, (byte) 0x9e, (byte) 0x5a, (byte) 0x9e, (byte) 0x61, (byte) 0x9e, (byte) 0x7a, (byte) 0x9e, (byte) 0x81,
                                        (byte) 0x9e, (byte) 0xfe, (byte) 0x9f, (byte) 0x41, (byte) 0x9f, (byte) 0x5a, (byte) 0x9f, (byte) 0x61, (byte) 0x9f, (byte) 0x7a, (byte) 0x9f, (byte) 0x81,
                                        (byte) 0x9f, (byte) 0xfe, (byte) 0xa0, (byte) 0x41, (byte) 0xa0, (byte) 0x5a, (byte) 0xa0, (byte) 0x61, (byte) 0xa0, (byte) 0x7a, (byte) 0xa0, (byte) 0x81,
                                        (byte) 0xa0, (byte) 0xfe, (byte) 0xa1, (byte) 0x41, (byte) 0xa1, (byte) 0x5a, (byte) 0xa1, (byte) 0x61, (byte) 0xa1, (byte) 0x7a, (byte) 0xa1, (byte) 0x81,
                                        (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0x41, (byte) 0xa2, (byte) 0x5a, (byte) 0xa2, (byte) 0x61, (byte) 0xa2, (byte) 0x7a, (byte) 0xa2, (byte) 0x81,
                                        (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0x41, (byte) 0xa3, (byte) 0x5a, (byte) 0xa3, (byte) 0x61, (byte) 0xa3, (byte) 0x7a, (byte) 0xa3, (byte) 0x81,
                                        (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0x41, (byte) 0xa4, (byte) 0x5a, (byte) 0xa4, (byte) 0x61, (byte) 0xa4, (byte) 0x7a, (byte) 0xa4, (byte) 0x81,
                                        (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0x41, (byte) 0xa5, (byte) 0x5a, (byte) 0xa5, (byte) 0x61, (byte) 0xa5, (byte) 0x7a, (byte) 0xa5, (byte) 0x81,
                                        (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0x41, (byte) 0xa6, (byte) 0x5a, (byte) 0xa6, (byte) 0x61, (byte) 0xa6, (byte) 0x7a, (byte) 0xa6, (byte) 0x81,
                                        (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0x41, (byte) 0xa7, (byte) 0x5a, (byte) 0xa7, (byte) 0x61, (byte) 0xa7, (byte) 0x7a, (byte) 0xa7, (byte) 0x81,
                                        (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0x41, (byte) 0xa8, (byte) 0x5a, (byte) 0xa8, (byte) 0x61, (byte) 0xa8, (byte) 0x7a, (byte) 0xa8, (byte) 0x81,
                                        (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0x41, (byte) 0xa9, (byte) 0x5a, (byte) 0xa9, (byte) 0x61, (byte) 0xa9, (byte) 0x7a, (byte) 0xa9, (byte) 0x81,
                                        (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0x41, (byte) 0xaa, (byte) 0x5a, (byte) 0xaa, (byte) 0x61, (byte) 0xaa, (byte) 0x7a, (byte) 0xaa, (byte) 0x81,
                                        (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0x41, (byte) 0xab, (byte) 0x5a, (byte) 0xab, (byte) 0x61, (byte) 0xab, (byte) 0x7a, (byte) 0xab, (byte) 0x81,
                                        (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0x41, (byte) 0xac, (byte) 0x5a, (byte) 0xac, (byte) 0x61, (byte) 0xac, (byte) 0x7a, (byte) 0xac, (byte) 0x81,
                                        (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0x41, (byte) 0xad, (byte) 0x5a, (byte) 0xad, (byte) 0x61, (byte) 0xad, (byte) 0x7a, (byte) 0xad, (byte) 0x81,
                                        (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0x41, (byte) 0xae, (byte) 0x5a, (byte) 0xae, (byte) 0x61, (byte) 0xae, (byte) 0x7a, (byte) 0xae, (byte) 0x81,
                                        (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0x41, (byte) 0xaf, (byte) 0x5a, (byte) 0xaf, (byte) 0x61, (byte) 0xaf, (byte) 0x7a, (byte) 0xaf, (byte) 0x81,
                                        (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0x41, (byte) 0xb0, (byte) 0x5a, (byte) 0xb0, (byte) 0x61, (byte) 0xb0, (byte) 0x7a, (byte) 0xb0, (byte) 0x81,
                                        (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0x41, (byte) 0xb1, (byte) 0x5a, (byte) 0xb1, (byte) 0x61, (byte) 0xb1, (byte) 0x7a, (byte) 0xb1, (byte) 0x81,
                                        (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0x41, (byte) 0xb2, (byte) 0x5a, (byte) 0xb2, (byte) 0x61, (byte) 0xb2, (byte) 0x7a, (byte) 0xb2, (byte) 0x81,
                                        (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0x41, (byte) 0xb3, (byte) 0x5a, (byte) 0xb3, (byte) 0x61, (byte) 0xb3, (byte) 0x7a, (byte) 0xb3, (byte) 0x81,
                                        (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0x41, (byte) 0xb4, (byte) 0x5a, (byte) 0xb4, (byte) 0x61, (byte) 0xb4, (byte) 0x7a, (byte) 0xb4, (byte) 0x81,
                                        (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0x41, (byte) 0xb5, (byte) 0x5a, (byte) 0xb5, (byte) 0x61, (byte) 0xb5, (byte) 0x7a, (byte) 0xb5, (byte) 0x81,
                                        (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0x41, (byte) 0xb6, (byte) 0x5a, (byte) 0xb6, (byte) 0x61, (byte) 0xb6, (byte) 0x7a, (byte) 0xb6, (byte) 0x81,
                                        (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0x41, (byte) 0xb7, (byte) 0x5a, (byte) 0xb7, (byte) 0x61, (byte) 0xb7, (byte) 0x7a, (byte) 0xb7, (byte) 0x81,
                                        (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0x41, (byte) 0xb8, (byte) 0x5a, (byte) 0xb8, (byte) 0x61, (byte) 0xb8, (byte) 0x7a, (byte) 0xb8, (byte) 0x81,
                                        (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0x41, (byte) 0xb9, (byte) 0x5a, (byte) 0xb9, (byte) 0x61, (byte) 0xb9, (byte) 0x7a, (byte) 0xb9, (byte) 0x81,
                                        (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0x41, (byte) 0xba, (byte) 0x5a, (byte) 0xba, (byte) 0x61, (byte) 0xba, (byte) 0x7a, (byte) 0xba, (byte) 0x81,
                                        (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0x41, (byte) 0xbb, (byte) 0x5a, (byte) 0xbb, (byte) 0x61, (byte) 0xbb, (byte) 0x7a, (byte) 0xbb, (byte) 0x81,
                                        (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0x41, (byte) 0xbc, (byte) 0x5a, (byte) 0xbc, (byte) 0x61, (byte) 0xbc, (byte) 0x7a, (byte) 0xbc, (byte) 0x81,
                                        (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0x41, (byte) 0xbd, (byte) 0x5a, (byte) 0xbd, (byte) 0x61, (byte) 0xbd, (byte) 0x7a, (byte) 0xbd, (byte) 0x81,
                                        (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0x41, (byte) 0xbe, (byte) 0x5a, (byte) 0xbe, (byte) 0x61, (byte) 0xbe, (byte) 0x7a, (byte) 0xbe, (byte) 0x81,
                                        (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0x41, (byte) 0xbf, (byte) 0x5a, (byte) 0xbf, (byte) 0x61, (byte) 0xbf, (byte) 0x7a, (byte) 0xbf, (byte) 0x81,
                                        (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0x41, (byte) 0xc0, (byte) 0x5a, (byte) 0xc0, (byte) 0x61, (byte) 0xc0, (byte) 0x7a, (byte) 0xc0, (byte) 0x81,
                                        (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0x41, (byte) 0xc1, (byte) 0x5a, (byte) 0xc1, (byte) 0x61, (byte) 0xc1, (byte) 0x7a, (byte) 0xc1, (byte) 0x81,
                                        (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0x41, (byte) 0xc2, (byte) 0x5a, (byte) 0xc2, (byte) 0x61, (byte) 0xc2, (byte) 0x7a, (byte) 0xc2, (byte) 0x81,
                                        (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0x41, (byte) 0xc3, (byte) 0x5a, (byte) 0xc3, (byte) 0x61, (byte) 0xc3, (byte) 0x7a, (byte) 0xc3, (byte) 0x81,
                                        (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0x41, (byte) 0xc4, (byte) 0x5a, (byte) 0xc4, (byte) 0x61, (byte) 0xc4, (byte) 0x7a, (byte) 0xc4, (byte) 0x81,
                                        (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0x41, (byte) 0xc5, (byte) 0x5a, (byte) 0xc5, (byte) 0x61, (byte) 0xc5, (byte) 0x7a, (byte) 0xc5, (byte) 0x81,
                                        (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0x41, (byte) 0xc6, (byte) 0x5a, (byte) 0xc6, (byte) 0x61, (byte) 0xc6, (byte) 0x7a, (byte) 0xc6, (byte) 0x81,
                                        (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0x41, (byte) 0xc7, (byte) 0x5a, (byte) 0xc7, (byte) 0x61, (byte) 0xc7, (byte) 0x7a, (byte) 0xc7, (byte) 0x81,
                                        (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0x41, (byte) 0xc8, (byte) 0x5a, (byte) 0xc8, (byte) 0x61, (byte) 0xc8, (byte) 0x7a, (byte) 0xc8, (byte) 0x81,
                                        (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0x41, (byte) 0xc9, (byte) 0x5a, (byte) 0xc9, (byte) 0x61, (byte) 0xc9, (byte) 0x7a, (byte) 0xc9, (byte) 0x81,
                                        (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0x41, (byte) 0xca, (byte) 0x5a, (byte) 0xca, (byte) 0x61, (byte) 0xca, (byte) 0x7a, (byte) 0xca, (byte) 0x81,
                                        (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0x41, (byte) 0xcb, (byte) 0x5a, (byte) 0xcb, (byte) 0x61, (byte) 0xcb, (byte) 0x7a, (byte) 0xcb, (byte) 0x81,
                                        (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0x41, (byte) 0xcc, (byte) 0x5a, (byte) 0xcc, (byte) 0x61, (byte) 0xcc, (byte) 0x7a, (byte) 0xcc, (byte) 0x81,
                                        (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0x41, (byte) 0xcd, (byte) 0x5a, (byte) 0xcd, (byte) 0x61, (byte) 0xcd, (byte) 0x7a, (byte) 0xcd, (byte) 0x81,
                                        (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0x41, (byte) 0xce, (byte) 0x5a, (byte) 0xce, (byte) 0x61, (byte) 0xce, (byte) 0x7a, (byte) 0xce, (byte) 0x81,
                                        (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0x41, (byte) 0xcf, (byte) 0x5a, (byte) 0xcf, (byte) 0x61, (byte) 0xcf, (byte) 0x7a, (byte) 0xcf, (byte) 0x81,
                                        (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0x41, (byte) 0xd0, (byte) 0x5a, (byte) 0xd0, (byte) 0x61, (byte) 0xd0, (byte) 0x7a, (byte) 0xd0, (byte) 0x81,
                                        (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0x41, (byte) 0xd1, (byte) 0x5a, (byte) 0xd1, (byte) 0x61, (byte) 0xd1, (byte) 0x7a, (byte) 0xd1, (byte) 0x81,
                                        (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0x41, (byte) 0xd2, (byte) 0x5a, (byte) 0xd2, (byte) 0x61, (byte) 0xd2, (byte) 0x7a, (byte) 0xd2, (byte) 0x81,
                                        (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0x41, (byte) 0xd3, (byte) 0x5a, (byte) 0xd3, (byte) 0x61, (byte) 0xd3, (byte) 0x7a, (byte) 0xd3, (byte) 0x81,
                                        (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0x41, (byte) 0xd4, (byte) 0x5a, (byte) 0xd4, (byte) 0x61, (byte) 0xd4, (byte) 0x7a, (byte) 0xd4, (byte) 0x81,
                                        (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0x41, (byte) 0xd5, (byte) 0x5a, (byte) 0xd5, (byte) 0x61, (byte) 0xd5, (byte) 0x7a, (byte) 0xd5, (byte) 0x81,
                                        (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0x41, (byte) 0xd6, (byte) 0x5a, (byte) 0xd6, (byte) 0x61, (byte) 0xd6, (byte) 0x7a, (byte) 0xd6, (byte) 0x81,
                                        (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0x41, (byte) 0xd7, (byte) 0x5a, (byte) 0xd7, (byte) 0x61, (byte) 0xd7, (byte) 0x7a, (byte) 0xd7, (byte) 0x81,
                                        (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0x41, (byte) 0xd8, (byte) 0x5a, (byte) 0xd8, (byte) 0x61, (byte) 0xd8, (byte) 0x7a, (byte) 0xd8, (byte) 0x81,
                                        (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0x41, (byte) 0xd9, (byte) 0x5a, (byte) 0xd9, (byte) 0x61, (byte) 0xd9, (byte) 0x7a, (byte) 0xd9, (byte) 0x81,
                                        (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0x41, (byte) 0xda, (byte) 0x5a, (byte) 0xda, (byte) 0x61, (byte) 0xda, (byte) 0x7a, (byte) 0xda, (byte) 0x81,
                                        (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0x41, (byte) 0xdb, (byte) 0x5a, (byte) 0xdb, (byte) 0x61, (byte) 0xdb, (byte) 0x7a, (byte) 0xdb, (byte) 0x81,
                                        (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0x41, (byte) 0xdc, (byte) 0x5a, (byte) 0xdc, (byte) 0x61, (byte) 0xdc, (byte) 0x7a, (byte) 0xdc, (byte) 0x81,
                                        (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0x41, (byte) 0xdd, (byte) 0x5a, (byte) 0xdd, (byte) 0x61, (byte) 0xdd, (byte) 0x7a, (byte) 0xdd, (byte) 0x81,
                                        (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0x41, (byte) 0xde, (byte) 0x5a, (byte) 0xde, (byte) 0x61, (byte) 0xde, (byte) 0x7a, (byte) 0xde, (byte) 0x81,
                                        (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0x41, (byte) 0xdf, (byte) 0x5a, (byte) 0xdf, (byte) 0x61, (byte) 0xdf, (byte) 0x7a, (byte) 0xdf, (byte) 0x81,
                                        (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0x41, (byte) 0xe0, (byte) 0x5a, (byte) 0xe0, (byte) 0x61, (byte) 0xe0, (byte) 0x7a, (byte) 0xe0, (byte) 0x81,
                                        (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0x41, (byte) 0xe1, (byte) 0x5a, (byte) 0xe1, (byte) 0x61, (byte) 0xe1, (byte) 0x7a, (byte) 0xe1, (byte) 0x81,
                                        (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0x41, (byte) 0xe2, (byte) 0x5a, (byte) 0xe2, (byte) 0x61, (byte) 0xe2, (byte) 0x7a, (byte) 0xe2, (byte) 0x81,
                                        (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0x41, (byte) 0xe3, (byte) 0x5a, (byte) 0xe3, (byte) 0x61, (byte) 0xe3, (byte) 0x7a, (byte) 0xe3, (byte) 0x81,
                                        (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0x41, (byte) 0xe4, (byte) 0x5a, (byte) 0xe4, (byte) 0x61, (byte) 0xe4, (byte) 0x7a, (byte) 0xe4, (byte) 0x81,
                                        (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0x41, (byte) 0xe5, (byte) 0x5a, (byte) 0xe5, (byte) 0x61, (byte) 0xe5, (byte) 0x7a, (byte) 0xe5, (byte) 0x81,
                                        (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0x41, (byte) 0xe6, (byte) 0x5a, (byte) 0xe6, (byte) 0x61, (byte) 0xe6, (byte) 0x7a, (byte) 0xe6, (byte) 0x81,
                                        (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0x41, (byte) 0xe7, (byte) 0x5a, (byte) 0xe7, (byte) 0x61, (byte) 0xe7, (byte) 0x7a, (byte) 0xe7, (byte) 0x81,
                                        (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0x41, (byte) 0xe8, (byte) 0x5a, (byte) 0xe8, (byte) 0x61, (byte) 0xe8, (byte) 0x7a, (byte) 0xe8, (byte) 0x81,
                                        (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0x41, (byte) 0xe9, (byte) 0x5a, (byte) 0xe9, (byte) 0x61, (byte) 0xe9, (byte) 0x7a, (byte) 0xe9, (byte) 0x81,
                                        (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0x41, (byte) 0xea, (byte) 0x5a, (byte) 0xea, (byte) 0x61, (byte) 0xea, (byte) 0x7a, (byte) 0xea, (byte) 0x81,
                                        (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0x41, (byte) 0xeb, (byte) 0x5a, (byte) 0xeb, (byte) 0x61, (byte) 0xeb, (byte) 0x7a, (byte) 0xeb, (byte) 0x81,
                                        (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0x41, (byte) 0xec, (byte) 0x5a, (byte) 0xec, (byte) 0x61, (byte) 0xec, (byte) 0x7a, (byte) 0xec, (byte) 0x81,
                                        (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0x41, (byte) 0xed, (byte) 0x5a, (byte) 0xed, (byte) 0x61, (byte) 0xed, (byte) 0x7a, (byte) 0xed, (byte) 0x81,
                                        (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0x41, (byte) 0xee, (byte) 0x5a, (byte) 0xee, (byte) 0x61, (byte) 0xee, (byte) 0x7a, (byte) 0xee, (byte) 0x81,
                                        (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0x41, (byte) 0xef, (byte) 0x5a, (byte) 0xef, (byte) 0x61, (byte) 0xef, (byte) 0x7a, (byte) 0xef, (byte) 0x81,
                                        (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0x41, (byte) 0xf0, (byte) 0x5a, (byte) 0xf0, (byte) 0x61, (byte) 0xf0, (byte) 0x7a, (byte) 0xf0, (byte) 0x81,
                                        (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0x41, (byte) 0xf1, (byte) 0x5a, (byte) 0xf1, (byte) 0x61, (byte) 0xf1, (byte) 0x7a, (byte) 0xf1, (byte) 0x81,
                                        (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0x41, (byte) 0xf2, (byte) 0x5a, (byte) 0xf2, (byte) 0x61, (byte) 0xf2, (byte) 0x7a, (byte) 0xf2, (byte) 0x81,
                                        (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0x41, (byte) 0xf3, (byte) 0x5a, (byte) 0xf3, (byte) 0x61, (byte) 0xf3, (byte) 0x7a, (byte) 0xf3, (byte) 0x81,
                                        (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0x41, (byte) 0xf4, (byte) 0x5a, (byte) 0xf4, (byte) 0x61, (byte) 0xf4, (byte) 0x7a, (byte) 0xf4, (byte) 0x81,
                                        (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0x41, (byte) 0xf5, (byte) 0x5a, (byte) 0xf5, (byte) 0x61, (byte) 0xf5, (byte) 0x7a, (byte) 0xf5, (byte) 0x81,
                                        (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0x41, (byte) 0xf6, (byte) 0x5a, (byte) 0xf6, (byte) 0x61, (byte) 0xf6, (byte) 0x7a, (byte) 0xf6, (byte) 0x81,
                                        (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0x41, (byte) 0xf7, (byte) 0x5a, (byte) 0xf7, (byte) 0x61, (byte) 0xf7, (byte) 0x7a, (byte) 0xf7, (byte) 0x81,
                                        (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0x41, (byte) 0xf8, (byte) 0x5a, (byte) 0xf8, (byte) 0x61, (byte) 0xf8, (byte) 0x7a, (byte) 0xf8, (byte) 0x81,
                                        (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0x41, (byte) 0xf9, (byte) 0x5a, (byte) 0xf9, (byte) 0x61, (byte) 0xf9, (byte) 0x7a, (byte) 0xf9, (byte) 0x81,
                                        (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0x41, (byte) 0xfa, (byte) 0x5a, (byte) 0xfa, (byte) 0x61, (byte) 0xfa, (byte) 0x7a, (byte) 0xfa, (byte) 0x81,
                                        (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0x41, (byte) 0xfb, (byte) 0x5a, (byte) 0xfb, (byte) 0x61, (byte) 0xfb, (byte) 0x7a, (byte) 0xfb, (byte) 0x81,
                                        (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0x41, (byte) 0xfc, (byte) 0x5a, (byte) 0xfc, (byte) 0x61, (byte) 0xfc, (byte) 0x7a, (byte) 0xfc, (byte) 0x81,
                                        (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0x41, (byte) 0xfd, (byte) 0x5a, (byte) 0xfd, (byte) 0x61, (byte) 0xfd, (byte) 0x7a, (byte) 0xfd, (byte) 0x81,
                                        (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0x41, (byte) 0xfe, (byte) 0x5a, (byte) 0xfe, (byte) 0x61, (byte) 0xfe, (byte) 0x7a, (byte) 0xfe, (byte) 0x81,
                                        (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataCP950() {
        assert TruffleString.Encoding.CP950.ordinal() == 17;
        return new TestData(new int[]{0x0000, 0x007f, 0xa140, 0xa17e, 0xa1a1, 0xa1fe, 0xa240, 0xa27e, 0xa2a1, 0xa2fe, 0xa340, 0xa37e, 0xa3a1, 0xa3fe, 0xa440, 0xa47e, 0xa4a1, 0xa4fe, 0xa540, 0xa57e,
                        0xa5a1, 0xa5fe, 0xa640, 0xa67e, 0xa6a1, 0xa6fe, 0xa740, 0xa77e, 0xa7a1, 0xa7fe, 0xa840, 0xa87e, 0xa8a1, 0xa8fe, 0xa940, 0xa97e, 0xa9a1, 0xa9fe, 0xaa40, 0xaa7e, 0xaaa1, 0xaafe,
                        0xab40, 0xab7e, 0xaba1, 0xabfe, 0xac40, 0xac7e, 0xaca1, 0xacfe, 0xad40, 0xad7e, 0xada1, 0xadfe, 0xae40, 0xae7e, 0xaea1, 0xaefe, 0xaf40, 0xaf7e, 0xafa1, 0xaffe, 0xb040, 0xb07e,
                        0xb0a1, 0xb0fe, 0xb140, 0xb17e, 0xb1a1, 0xb1fe, 0xb240, 0xb27e, 0xb2a1, 0xb2fe, 0xb340, 0xb37e, 0xb3a1, 0xb3fe, 0xb440, 0xb47e, 0xb4a1, 0xb4fe, 0xb540, 0xb57e, 0xb5a1, 0xb5fe,
                        0xb640, 0xb67e, 0xb6a1, 0xb6fe, 0xb740, 0xb77e, 0xb7a1, 0xb7fe, 0xb840, 0xb87e, 0xb8a1, 0xb8fe, 0xb940, 0xb97e, 0xb9a1, 0xb9fe, 0xba40, 0xba7e, 0xbaa1, 0xbafe, 0xbb40, 0xbb7e,
                        0xbba1, 0xbbfe, 0xbc40, 0xbc7e, 0xbca1, 0xbcfe, 0xbd40, 0xbd7e, 0xbda1, 0xbdfe, 0xbe40, 0xbe7e, 0xbea1, 0xbefe, 0xbf40, 0xbf7e, 0xbfa1, 0xbffe, 0xc040, 0xc07e, 0xc0a1, 0xc0fe,
                        0xc140, 0xc17e, 0xc1a1, 0xc1fe, 0xc240, 0xc27e, 0xc2a1, 0xc2fe, 0xc340, 0xc37e, 0xc3a1, 0xc3fe, 0xc440, 0xc47e, 0xc4a1, 0xc4fe, 0xc540, 0xc57e, 0xc5a1, 0xc5fe, 0xc640, 0xc67e,
                        0xc6a1, 0xc6fe, 0xc740, 0xc77e, 0xc7a1, 0xc7fe, 0xc840, 0xc87e, 0xc8a1, 0xc8fe, 0xc940, 0xc97e, 0xc9a1, 0xc9fe, 0xca40, 0xca7e, 0xcaa1, 0xcafe, 0xcb40, 0xcb7e, 0xcba1, 0xcbfe,
                        0xcc40, 0xcc7e, 0xcca1, 0xccfe, 0xcd40, 0xcd7e, 0xcda1, 0xcdfe, 0xce40, 0xce7e, 0xcea1, 0xcefe, 0xcf40, 0xcf7e, 0xcfa1, 0xcffe, 0xd040, 0xd07e, 0xd0a1, 0xd0fe, 0xd140, 0xd17e,
                        0xd1a1, 0xd1fe, 0xd240, 0xd27e, 0xd2a1, 0xd2fe, 0xd340, 0xd37e, 0xd3a1, 0xd3fe, 0xd440, 0xd47e, 0xd4a1, 0xd4fe, 0xd540, 0xd57e, 0xd5a1, 0xd5fe, 0xd640, 0xd67e, 0xd6a1, 0xd6fe,
                        0xd740, 0xd77e, 0xd7a1, 0xd7fe, 0xd840, 0xd87e, 0xd8a1, 0xd8fe, 0xd940, 0xd97e, 0xd9a1, 0xd9fe, 0xda40, 0xda7e, 0xdaa1, 0xdafe, 0xdb40, 0xdb7e, 0xdba1, 0xdbfe, 0xdc40, 0xdc7e,
                        0xdca1, 0xdcfe, 0xdd40, 0xdd7e, 0xdda1, 0xddfe, 0xde40, 0xde7e, 0xdea1, 0xdefe, 0xdf40, 0xdf7e, 0xdfa1, 0xdffe, 0xe040, 0xe07e, 0xe0a1, 0xe0fe, 0xe140, 0xe17e, 0xe1a1, 0xe1fe,
                        0xe240, 0xe27e, 0xe2a1, 0xe2fe, 0xe340, 0xe37e, 0xe3a1, 0xe3fe, 0xe440, 0xe47e, 0xe4a1, 0xe4fe, 0xe540, 0xe57e, 0xe5a1, 0xe5fe, 0xe640, 0xe67e, 0xe6a1, 0xe6fe, 0xe740, 0xe77e,
                        0xe7a1, 0xe7fe, 0xe840, 0xe87e, 0xe8a1, 0xe8fe, 0xe940, 0xe97e, 0xe9a1, 0xe9fe, 0xea40, 0xea7e, 0xeaa1, 0xeafe, 0xeb40, 0xeb7e, 0xeba1, 0xebfe, 0xec40, 0xec7e, 0xeca1, 0xecfe,
                        0xed40, 0xed7e, 0xeda1, 0xedfe, 0xee40, 0xee7e, 0xeea1, 0xeefe, 0xef40, 0xef7e, 0xefa1, 0xeffe, 0xf040, 0xf07e, 0xf0a1, 0xf0fe, 0xf140, 0xf17e, 0xf1a1, 0xf1fe, 0xf240, 0xf27e,
                        0xf2a1, 0xf2fe, 0xf340, 0xf37e, 0xf3a1, 0xf3fe, 0xf440, 0xf47e, 0xf4a1, 0xf4fe, 0xf540, 0xf57e, 0xf5a1, 0xf5fe, 0xf640, 0xf67e, 0xf6a1, 0xf6fe, 0xf740, 0xf77e, 0xf7a1, 0xf7fe,
                        0xf840, 0xf87e, 0xf8a1, 0xf8fe, 0xf940, 0xf97e, 0xf9a1, 0xf9fe, 0xfa40, 0xfa7e, 0xfaa1, 0xfafe, 0xfb40, 0xfb7e, 0xfba1, 0xfbfe, 0xfc40, 0xfc7e, 0xfca1, 0xfcfe, 0xfd40, 0xfd7e,
                        0xfda1, 0xfdfe, 0xfe40, 0xfe7e, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198, 0x19a,
                                        0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6, 0x1c8,
                                        0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2, 0x1e4, 0x1e6, 0x1e8, 0x1ea, 0x1ec, 0x1ee, 0x1f0, 0x1f2, 0x1f4, 0x1f6,
                                        0x1f8, 0x1fa, 0x1fc, 0x1fe, 0x200, 0x202, 0x204, 0x206, 0x208, 0x20a, 0x20c, 0x20e, 0x210, 0x212, 0x214, 0x216, 0x218, 0x21a, 0x21c, 0x21e, 0x220, 0x222, 0x224,
                                        0x226, 0x228, 0x22a, 0x22c, 0x22e, 0x230, 0x232, 0x234, 0x236, 0x238, 0x23a, 0x23c, 0x23e, 0x240, 0x242, 0x244, 0x246, 0x248, 0x24a, 0x24c, 0x24e, 0x250, 0x252,
                                        0x254, 0x256, 0x258, 0x25a, 0x25c, 0x25e, 0x260, 0x262, 0x264, 0x266, 0x268, 0x26a, 0x26c, 0x26e, 0x270, 0x272, 0x274, 0x276, 0x278, 0x27a, 0x27c, 0x27e, 0x280,
                                        0x282, 0x284, 0x286, 0x288, 0x28a, 0x28c, 0x28e, 0x290, 0x292, 0x294, 0x296, 0x298, 0x29a, 0x29c, 0x29e, 0x2a0, 0x2a2, 0x2a4, 0x2a6, 0x2a8, 0x2aa, 0x2ac, 0x2ae,
                                        0x2b0, 0x2b2, 0x2b4, 0x2b6, 0x2b8, 0x2ba, 0x2bc, 0x2be, 0x2c0, 0x2c2, 0x2c4, 0x2c6, 0x2c8, 0x2ca, 0x2cc, 0x2ce, 0x2d0, 0x2d2, 0x2d4, 0x2d6, 0x2d8, 0x2da, 0x2dc,
                                        0x2de, 0x2e0, 0x2e2, 0x2e4, 0x2e6, 0x2e8, 0x2ea, 0x2ec, 0x2ee, 0x2f0},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0x40, (byte) 0xa1, (byte) 0x7e, (byte) 0xa1, (byte) 0xa1, (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0x40,
                                        (byte) 0xa2, (byte) 0x7e, (byte) 0xa2, (byte) 0xa1, (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0x40, (byte) 0xa3, (byte) 0x7e, (byte) 0xa3, (byte) 0xa1,
                                        (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0x40, (byte) 0xa4, (byte) 0x7e, (byte) 0xa4, (byte) 0xa1, (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0x40,
                                        (byte) 0xa5, (byte) 0x7e, (byte) 0xa5, (byte) 0xa1, (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0x40, (byte) 0xa6, (byte) 0x7e, (byte) 0xa6, (byte) 0xa1,
                                        (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0x40, (byte) 0xa7, (byte) 0x7e, (byte) 0xa7, (byte) 0xa1, (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0x40,
                                        (byte) 0xa8, (byte) 0x7e, (byte) 0xa8, (byte) 0xa1, (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0x40, (byte) 0xa9, (byte) 0x7e, (byte) 0xa9, (byte) 0xa1,
                                        (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0x40, (byte) 0xaa, (byte) 0x7e, (byte) 0xaa, (byte) 0xa1, (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0x40,
                                        (byte) 0xab, (byte) 0x7e, (byte) 0xab, (byte) 0xa1, (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0x40, (byte) 0xac, (byte) 0x7e, (byte) 0xac, (byte) 0xa1,
                                        (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0x40, (byte) 0xad, (byte) 0x7e, (byte) 0xad, (byte) 0xa1, (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0x40,
                                        (byte) 0xae, (byte) 0x7e, (byte) 0xae, (byte) 0xa1, (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0x40, (byte) 0xaf, (byte) 0x7e, (byte) 0xaf, (byte) 0xa1,
                                        (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0x40, (byte) 0xb0, (byte) 0x7e, (byte) 0xb0, (byte) 0xa1, (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0x40,
                                        (byte) 0xb1, (byte) 0x7e, (byte) 0xb1, (byte) 0xa1, (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0x40, (byte) 0xb2, (byte) 0x7e, (byte) 0xb2, (byte) 0xa1,
                                        (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0x40, (byte) 0xb3, (byte) 0x7e, (byte) 0xb3, (byte) 0xa1, (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0x40,
                                        (byte) 0xb4, (byte) 0x7e, (byte) 0xb4, (byte) 0xa1, (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0x40, (byte) 0xb5, (byte) 0x7e, (byte) 0xb5, (byte) 0xa1,
                                        (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0x40, (byte) 0xb6, (byte) 0x7e, (byte) 0xb6, (byte) 0xa1, (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0x40,
                                        (byte) 0xb7, (byte) 0x7e, (byte) 0xb7, (byte) 0xa1, (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0x40, (byte) 0xb8, (byte) 0x7e, (byte) 0xb8, (byte) 0xa1,
                                        (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0x40, (byte) 0xb9, (byte) 0x7e, (byte) 0xb9, (byte) 0xa1, (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0x40,
                                        (byte) 0xba, (byte) 0x7e, (byte) 0xba, (byte) 0xa1, (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0x40, (byte) 0xbb, (byte) 0x7e, (byte) 0xbb, (byte) 0xa1,
                                        (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0x40, (byte) 0xbc, (byte) 0x7e, (byte) 0xbc, (byte) 0xa1, (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0x40,
                                        (byte) 0xbd, (byte) 0x7e, (byte) 0xbd, (byte) 0xa1, (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0x40, (byte) 0xbe, (byte) 0x7e, (byte) 0xbe, (byte) 0xa1,
                                        (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0x40, (byte) 0xbf, (byte) 0x7e, (byte) 0xbf, (byte) 0xa1, (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0x40,
                                        (byte) 0xc0, (byte) 0x7e, (byte) 0xc0, (byte) 0xa1, (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0x40, (byte) 0xc1, (byte) 0x7e, (byte) 0xc1, (byte) 0xa1,
                                        (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0x40, (byte) 0xc2, (byte) 0x7e, (byte) 0xc2, (byte) 0xa1, (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0x40,
                                        (byte) 0xc3, (byte) 0x7e, (byte) 0xc3, (byte) 0xa1, (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0x40, (byte) 0xc4, (byte) 0x7e, (byte) 0xc4, (byte) 0xa1,
                                        (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0x40, (byte) 0xc5, (byte) 0x7e, (byte) 0xc5, (byte) 0xa1, (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0x40,
                                        (byte) 0xc6, (byte) 0x7e, (byte) 0xc6, (byte) 0xa1, (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0x40, (byte) 0xc7, (byte) 0x7e, (byte) 0xc7, (byte) 0xa1,
                                        (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0x40, (byte) 0xc8, (byte) 0x7e, (byte) 0xc8, (byte) 0xa1, (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0x40,
                                        (byte) 0xc9, (byte) 0x7e, (byte) 0xc9, (byte) 0xa1, (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0x40, (byte) 0xca, (byte) 0x7e, (byte) 0xca, (byte) 0xa1,
                                        (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0x40, (byte) 0xcb, (byte) 0x7e, (byte) 0xcb, (byte) 0xa1, (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0x40,
                                        (byte) 0xcc, (byte) 0x7e, (byte) 0xcc, (byte) 0xa1, (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0x40, (byte) 0xcd, (byte) 0x7e, (byte) 0xcd, (byte) 0xa1,
                                        (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0x40, (byte) 0xce, (byte) 0x7e, (byte) 0xce, (byte) 0xa1, (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0x40,
                                        (byte) 0xcf, (byte) 0x7e, (byte) 0xcf, (byte) 0xa1, (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0x40, (byte) 0xd0, (byte) 0x7e, (byte) 0xd0, (byte) 0xa1,
                                        (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0x40, (byte) 0xd1, (byte) 0x7e, (byte) 0xd1, (byte) 0xa1, (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0x40,
                                        (byte) 0xd2, (byte) 0x7e, (byte) 0xd2, (byte) 0xa1, (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0x40, (byte) 0xd3, (byte) 0x7e, (byte) 0xd3, (byte) 0xa1,
                                        (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0x40, (byte) 0xd4, (byte) 0x7e, (byte) 0xd4, (byte) 0xa1, (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0x40,
                                        (byte) 0xd5, (byte) 0x7e, (byte) 0xd5, (byte) 0xa1, (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0x40, (byte) 0xd6, (byte) 0x7e, (byte) 0xd6, (byte) 0xa1,
                                        (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0x40, (byte) 0xd7, (byte) 0x7e, (byte) 0xd7, (byte) 0xa1, (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0x40,
                                        (byte) 0xd8, (byte) 0x7e, (byte) 0xd8, (byte) 0xa1, (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0x40, (byte) 0xd9, (byte) 0x7e, (byte) 0xd9, (byte) 0xa1,
                                        (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0x40, (byte) 0xda, (byte) 0x7e, (byte) 0xda, (byte) 0xa1, (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0x40,
                                        (byte) 0xdb, (byte) 0x7e, (byte) 0xdb, (byte) 0xa1, (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0x40, (byte) 0xdc, (byte) 0x7e, (byte) 0xdc, (byte) 0xa1,
                                        (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0x40, (byte) 0xdd, (byte) 0x7e, (byte) 0xdd, (byte) 0xa1, (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0x40,
                                        (byte) 0xde, (byte) 0x7e, (byte) 0xde, (byte) 0xa1, (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0x40, (byte) 0xdf, (byte) 0x7e, (byte) 0xdf, (byte) 0xa1,
                                        (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0xa1, (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0x40,
                                        (byte) 0xe1, (byte) 0x7e, (byte) 0xe1, (byte) 0xa1, (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0x40, (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0xa1,
                                        (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0xa1, (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0x40,
                                        (byte) 0xe4, (byte) 0x7e, (byte) 0xe4, (byte) 0xa1, (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0x40, (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0xa1,
                                        (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0xa1, (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0x40,
                                        (byte) 0xe7, (byte) 0x7e, (byte) 0xe7, (byte) 0xa1, (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0x40, (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0xa1,
                                        (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0xa1, (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0x40,
                                        (byte) 0xea, (byte) 0x7e, (byte) 0xea, (byte) 0xa1, (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0x40, (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0xa1,
                                        (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0xa1, (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0x40,
                                        (byte) 0xed, (byte) 0x7e, (byte) 0xed, (byte) 0xa1, (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0x40, (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0xa1,
                                        (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0xa1, (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0x40,
                                        (byte) 0xf0, (byte) 0x7e, (byte) 0xf0, (byte) 0xa1, (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0x40, (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0xa1,
                                        (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0xa1, (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0x40,
                                        (byte) 0xf3, (byte) 0x7e, (byte) 0xf3, (byte) 0xa1, (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0x40, (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0xa1,
                                        (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0xa1, (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0x40,
                                        (byte) 0xf6, (byte) 0x7e, (byte) 0xf6, (byte) 0xa1, (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0x40, (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0xa1,
                                        (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0xa1, (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0x40,
                                        (byte) 0xf9, (byte) 0x7e, (byte) 0xf9, (byte) 0xa1, (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0x40, (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0xa1,
                                        (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0xa1, (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0x40,
                                        (byte) 0xfc, (byte) 0x7e, (byte) 0xfc, (byte) 0xa1, (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0x40, (byte) 0xfd, (byte) 0x7e, (byte) 0xfd, (byte) 0xa1,
                                        (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0x40, (byte) 0xfe, (byte) 0x7e, (byte) 0xfe, (byte) 0xa1, (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataCP951() {
        assert TruffleString.Encoding.CP951.ordinal() == 18;
        return new TestData(new int[]{0x0000, 0x007f, 0x8740, 0x877e, 0x87a1, 0x87fe, 0x8840, 0x887e, 0x88a1, 0x88fe, 0x8940, 0x897e, 0x89a1, 0x89fe, 0x8a40, 0x8a7e, 0x8aa1, 0x8afe, 0x8b40, 0x8b7e,
                        0x8ba1, 0x8bfe, 0x8c40, 0x8c7e, 0x8ca1, 0x8cfe, 0x8d40, 0x8d7e, 0x8da1, 0x8dfe, 0x8e40, 0x8e7e, 0x8ea1, 0x8efe, 0x8f40, 0x8f7e, 0x8fa1, 0x8ffe, 0x9040, 0x907e, 0x90a1, 0x90fe,
                        0x9140, 0x917e, 0x91a1, 0x91fe, 0x9240, 0x927e, 0x92a1, 0x92fe, 0x9340, 0x937e, 0x93a1, 0x93fe, 0x9440, 0x947e, 0x94a1, 0x94fe, 0x9540, 0x957e, 0x95a1, 0x95fe, 0x9640, 0x967e,
                        0x96a1, 0x96fe, 0x9740, 0x977e, 0x97a1, 0x97fe, 0x9840, 0x987e, 0x98a1, 0x98fe, 0x9940, 0x997e, 0x99a1, 0x99fe, 0x9a40, 0x9a7e, 0x9aa1, 0x9afe, 0x9b40, 0x9b7e, 0x9ba1, 0x9bfe,
                        0x9c40, 0x9c7e, 0x9ca1, 0x9cfe, 0x9d40, 0x9d7e, 0x9da1, 0x9dfe, 0x9e40, 0x9e7e, 0x9ea1, 0x9efe, 0x9f40, 0x9f7e, 0x9fa1, 0x9ffe, 0xa040, 0xa07e, 0xa0a1, 0xa0fe, 0xa140, 0xa17e,
                        0xa1a1, 0xa1fe, 0xa240, 0xa27e, 0xa2a1, 0xa2fe, 0xa340, 0xa37e, 0xa3a1, 0xa3fe, 0xa440, 0xa47e, 0xa4a1, 0xa4fe, 0xa540, 0xa57e, 0xa5a1, 0xa5fe, 0xa640, 0xa67e, 0xa6a1, 0xa6fe,
                        0xa740, 0xa77e, 0xa7a1, 0xa7fe, 0xa840, 0xa87e, 0xa8a1, 0xa8fe, 0xa940, 0xa97e, 0xa9a1, 0xa9fe, 0xaa40, 0xaa7e, 0xaaa1, 0xaafe, 0xab40, 0xab7e, 0xaba1, 0xabfe, 0xac40, 0xac7e,
                        0xaca1, 0xacfe, 0xad40, 0xad7e, 0xada1, 0xadfe, 0xae40, 0xae7e, 0xaea1, 0xaefe, 0xaf40, 0xaf7e, 0xafa1, 0xaffe, 0xb040, 0xb07e, 0xb0a1, 0xb0fe, 0xb140, 0xb17e, 0xb1a1, 0xb1fe,
                        0xb240, 0xb27e, 0xb2a1, 0xb2fe, 0xb340, 0xb37e, 0xb3a1, 0xb3fe, 0xb440, 0xb47e, 0xb4a1, 0xb4fe, 0xb540, 0xb57e, 0xb5a1, 0xb5fe, 0xb640, 0xb67e, 0xb6a1, 0xb6fe, 0xb740, 0xb77e,
                        0xb7a1, 0xb7fe, 0xb840, 0xb87e, 0xb8a1, 0xb8fe, 0xb940, 0xb97e, 0xb9a1, 0xb9fe, 0xba40, 0xba7e, 0xbaa1, 0xbafe, 0xbb40, 0xbb7e, 0xbba1, 0xbbfe, 0xbc40, 0xbc7e, 0xbca1, 0xbcfe,
                        0xbd40, 0xbd7e, 0xbda1, 0xbdfe, 0xbe40, 0xbe7e, 0xbea1, 0xbefe, 0xbf40, 0xbf7e, 0xbfa1, 0xbffe, 0xc040, 0xc07e, 0xc0a1, 0xc0fe, 0xc140, 0xc17e, 0xc1a1, 0xc1fe, 0xc240, 0xc27e,
                        0xc2a1, 0xc2fe, 0xc340, 0xc37e, 0xc3a1, 0xc3fe, 0xc440, 0xc47e, 0xc4a1, 0xc4fe, 0xc540, 0xc57e, 0xc5a1, 0xc5fe, 0xc640, 0xc67e, 0xc6a1, 0xc6fe, 0xc740, 0xc77e, 0xc7a1, 0xc7fe,
                        0xc840, 0xc87e, 0xc8a1, 0xc8fe, 0xc940, 0xc97e, 0xc9a1, 0xc9fe, 0xca40, 0xca7e, 0xcaa1, 0xcafe, 0xcb40, 0xcb7e, 0xcba1, 0xcbfe, 0xcc40, 0xcc7e, 0xcca1, 0xccfe, 0xcd40, 0xcd7e,
                        0xcda1, 0xcdfe, 0xce40, 0xce7e, 0xcea1, 0xcefe, 0xcf40, 0xcf7e, 0xcfa1, 0xcffe, 0xd040, 0xd07e, 0xd0a1, 0xd0fe, 0xd140, 0xd17e, 0xd1a1, 0xd1fe, 0xd240, 0xd27e, 0xd2a1, 0xd2fe,
                        0xd340, 0xd37e, 0xd3a1, 0xd3fe, 0xd440, 0xd47e, 0xd4a1, 0xd4fe, 0xd540, 0xd57e, 0xd5a1, 0xd5fe, 0xd640, 0xd67e, 0xd6a1, 0xd6fe, 0xd740, 0xd77e, 0xd7a1, 0xd7fe, 0xd840, 0xd87e,
                        0xd8a1, 0xd8fe, 0xd940, 0xd97e, 0xd9a1, 0xd9fe, 0xda40, 0xda7e, 0xdaa1, 0xdafe, 0xdb40, 0xdb7e, 0xdba1, 0xdbfe, 0xdc40, 0xdc7e, 0xdca1, 0xdcfe, 0xdd40, 0xdd7e, 0xdda1, 0xddfe,
                        0xde40, 0xde7e, 0xdea1, 0xdefe, 0xdf40, 0xdf7e, 0xdfa1, 0xdffe, 0xe040, 0xe07e, 0xe0a1, 0xe0fe, 0xe140, 0xe17e, 0xe1a1, 0xe1fe, 0xe240, 0xe27e, 0xe2a1, 0xe2fe, 0xe340, 0xe37e,
                        0xe3a1, 0xe3fe, 0xe440, 0xe47e, 0xe4a1, 0xe4fe, 0xe540, 0xe57e, 0xe5a1, 0xe5fe, 0xe640, 0xe67e, 0xe6a1, 0xe6fe, 0xe740, 0xe77e, 0xe7a1, 0xe7fe, 0xe840, 0xe87e, 0xe8a1, 0xe8fe,
                        0xe940, 0xe97e, 0xe9a1, 0xe9fe, 0xea40, 0xea7e, 0xeaa1, 0xeafe, 0xeb40, 0xeb7e, 0xeba1, 0xebfe, 0xec40, 0xec7e, 0xeca1, 0xecfe, 0xed40, 0xed7e, 0xeda1, 0xedfe, 0xee40, 0xee7e,
                        0xeea1, 0xeefe, 0xef40, 0xef7e, 0xefa1, 0xeffe, 0xf040, 0xf07e, 0xf0a1, 0xf0fe, 0xf140, 0xf17e, 0xf1a1, 0xf1fe, 0xf240, 0xf27e, 0xf2a1, 0xf2fe, 0xf340, 0xf37e, 0xf3a1, 0xf3fe,
                        0xf440, 0xf47e, 0xf4a1, 0xf4fe, 0xf540, 0xf57e, 0xf5a1, 0xf5fe, 0xf640, 0xf67e, 0xf6a1, 0xf6fe, 0xf740, 0xf77e, 0xf7a1, 0xf7fe, 0xf840, 0xf87e, 0xf8a1, 0xf8fe, 0xf940, 0xf97e,
                        0xf9a1, 0xf9fe, 0xfa40, 0xfa7e, 0xfaa1, 0xfafe, 0xfb40, 0xfb7e, 0xfba1, 0xfbfe, 0xfc40, 0xfc7e, 0xfca1, 0xfcfe, 0xfd40, 0xfd7e, 0xfda1, 0xfdfe, 0xfe40, 0xfe7e, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198, 0x19a,
                                        0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6, 0x1c8,
                                        0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2, 0x1e4, 0x1e6, 0x1e8, 0x1ea, 0x1ec, 0x1ee, 0x1f0, 0x1f2, 0x1f4, 0x1f6,
                                        0x1f8, 0x1fa, 0x1fc, 0x1fe, 0x200, 0x202, 0x204, 0x206, 0x208, 0x20a, 0x20c, 0x20e, 0x210, 0x212, 0x214, 0x216, 0x218, 0x21a, 0x21c, 0x21e, 0x220, 0x222, 0x224,
                                        0x226, 0x228, 0x22a, 0x22c, 0x22e, 0x230, 0x232, 0x234, 0x236, 0x238, 0x23a, 0x23c, 0x23e, 0x240, 0x242, 0x244, 0x246, 0x248, 0x24a, 0x24c, 0x24e, 0x250, 0x252,
                                        0x254, 0x256, 0x258, 0x25a, 0x25c, 0x25e, 0x260, 0x262, 0x264, 0x266, 0x268, 0x26a, 0x26c, 0x26e, 0x270, 0x272, 0x274, 0x276, 0x278, 0x27a, 0x27c, 0x27e, 0x280,
                                        0x282, 0x284, 0x286, 0x288, 0x28a, 0x28c, 0x28e, 0x290, 0x292, 0x294, 0x296, 0x298, 0x29a, 0x29c, 0x29e, 0x2a0, 0x2a2, 0x2a4, 0x2a6, 0x2a8, 0x2aa, 0x2ac, 0x2ae,
                                        0x2b0, 0x2b2, 0x2b4, 0x2b6, 0x2b8, 0x2ba, 0x2bc, 0x2be, 0x2c0, 0x2c2, 0x2c4, 0x2c6, 0x2c8, 0x2ca, 0x2cc, 0x2ce, 0x2d0, 0x2d2, 0x2d4, 0x2d6, 0x2d8, 0x2da, 0x2dc,
                                        0x2de, 0x2e0, 0x2e2, 0x2e4, 0x2e6, 0x2e8, 0x2ea, 0x2ec, 0x2ee, 0x2f0, 0x2f2, 0x2f4, 0x2f6, 0x2f8, 0x2fa, 0x2fc, 0x2fe, 0x300, 0x302, 0x304, 0x306, 0x308, 0x30a,
                                        0x30c, 0x30e, 0x310, 0x312, 0x314, 0x316, 0x318, 0x31a, 0x31c, 0x31e, 0x320, 0x322, 0x324, 0x326, 0x328, 0x32a, 0x32c, 0x32e, 0x330, 0x332, 0x334, 0x336, 0x338,
                                        0x33a, 0x33c, 0x33e, 0x340, 0x342, 0x344, 0x346, 0x348, 0x34a, 0x34c, 0x34e, 0x350, 0x352, 0x354, 0x356, 0x358, 0x35a, 0x35c, 0x35e, 0x360, 0x362, 0x364, 0x366,
                                        0x368, 0x36a, 0x36c, 0x36e, 0x370, 0x372, 0x374, 0x376, 0x378, 0x37a, 0x37c, 0x37e, 0x380, 0x382, 0x384, 0x386, 0x388, 0x38a, 0x38c, 0x38e, 0x390, 0x392, 0x394,
                                        0x396, 0x398, 0x39a, 0x39c, 0x39e, 0x3a0, 0x3a2, 0x3a4, 0x3a6, 0x3a8, 0x3aa, 0x3ac, 0x3ae, 0x3b0, 0x3b2, 0x3b4, 0x3b6, 0x3b8, 0x3ba, 0x3bc, 0x3be, 0x3c0},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0x87, (byte) 0x40, (byte) 0x87, (byte) 0x7e, (byte) 0x87, (byte) 0xa1, (byte) 0x87, (byte) 0xfe, (byte) 0x88, (byte) 0x40,
                                        (byte) 0x88, (byte) 0x7e, (byte) 0x88, (byte) 0xa1, (byte) 0x88, (byte) 0xfe, (byte) 0x89, (byte) 0x40, (byte) 0x89, (byte) 0x7e, (byte) 0x89, (byte) 0xa1,
                                        (byte) 0x89, (byte) 0xfe, (byte) 0x8a, (byte) 0x40, (byte) 0x8a, (byte) 0x7e, (byte) 0x8a, (byte) 0xa1, (byte) 0x8a, (byte) 0xfe, (byte) 0x8b, (byte) 0x40,
                                        (byte) 0x8b, (byte) 0x7e, (byte) 0x8b, (byte) 0xa1, (byte) 0x8b, (byte) 0xfe, (byte) 0x8c, (byte) 0x40, (byte) 0x8c, (byte) 0x7e, (byte) 0x8c, (byte) 0xa1,
                                        (byte) 0x8c, (byte) 0xfe, (byte) 0x8d, (byte) 0x40, (byte) 0x8d, (byte) 0x7e, (byte) 0x8d, (byte) 0xa1, (byte) 0x8d, (byte) 0xfe, (byte) 0x8e, (byte) 0x40,
                                        (byte) 0x8e, (byte) 0x7e, (byte) 0x8e, (byte) 0xa1, (byte) 0x8e, (byte) 0xfe, (byte) 0x8f, (byte) 0x40, (byte) 0x8f, (byte) 0x7e, (byte) 0x8f, (byte) 0xa1,
                                        (byte) 0x8f, (byte) 0xfe, (byte) 0x90, (byte) 0x40, (byte) 0x90, (byte) 0x7e, (byte) 0x90, (byte) 0xa1, (byte) 0x90, (byte) 0xfe, (byte) 0x91, (byte) 0x40,
                                        (byte) 0x91, (byte) 0x7e, (byte) 0x91, (byte) 0xa1, (byte) 0x91, (byte) 0xfe, (byte) 0x92, (byte) 0x40, (byte) 0x92, (byte) 0x7e, (byte) 0x92, (byte) 0xa1,
                                        (byte) 0x92, (byte) 0xfe, (byte) 0x93, (byte) 0x40, (byte) 0x93, (byte) 0x7e, (byte) 0x93, (byte) 0xa1, (byte) 0x93, (byte) 0xfe, (byte) 0x94, (byte) 0x40,
                                        (byte) 0x94, (byte) 0x7e, (byte) 0x94, (byte) 0xa1, (byte) 0x94, (byte) 0xfe, (byte) 0x95, (byte) 0x40, (byte) 0x95, (byte) 0x7e, (byte) 0x95, (byte) 0xa1,
                                        (byte) 0x95, (byte) 0xfe, (byte) 0x96, (byte) 0x40, (byte) 0x96, (byte) 0x7e, (byte) 0x96, (byte) 0xa1, (byte) 0x96, (byte) 0xfe, (byte) 0x97, (byte) 0x40,
                                        (byte) 0x97, (byte) 0x7e, (byte) 0x97, (byte) 0xa1, (byte) 0x97, (byte) 0xfe, (byte) 0x98, (byte) 0x40, (byte) 0x98, (byte) 0x7e, (byte) 0x98, (byte) 0xa1,
                                        (byte) 0x98, (byte) 0xfe, (byte) 0x99, (byte) 0x40, (byte) 0x99, (byte) 0x7e, (byte) 0x99, (byte) 0xa1, (byte) 0x99, (byte) 0xfe, (byte) 0x9a, (byte) 0x40,
                                        (byte) 0x9a, (byte) 0x7e, (byte) 0x9a, (byte) 0xa1, (byte) 0x9a, (byte) 0xfe, (byte) 0x9b, (byte) 0x40, (byte) 0x9b, (byte) 0x7e, (byte) 0x9b, (byte) 0xa1,
                                        (byte) 0x9b, (byte) 0xfe, (byte) 0x9c, (byte) 0x40, (byte) 0x9c, (byte) 0x7e, (byte) 0x9c, (byte) 0xa1, (byte) 0x9c, (byte) 0xfe, (byte) 0x9d, (byte) 0x40,
                                        (byte) 0x9d, (byte) 0x7e, (byte) 0x9d, (byte) 0xa1, (byte) 0x9d, (byte) 0xfe, (byte) 0x9e, (byte) 0x40, (byte) 0x9e, (byte) 0x7e, (byte) 0x9e, (byte) 0xa1,
                                        (byte) 0x9e, (byte) 0xfe, (byte) 0x9f, (byte) 0x40, (byte) 0x9f, (byte) 0x7e, (byte) 0x9f, (byte) 0xa1, (byte) 0x9f, (byte) 0xfe, (byte) 0xa0, (byte) 0x40,
                                        (byte) 0xa0, (byte) 0x7e, (byte) 0xa0, (byte) 0xa1, (byte) 0xa0, (byte) 0xfe, (byte) 0xa1, (byte) 0x40, (byte) 0xa1, (byte) 0x7e, (byte) 0xa1, (byte) 0xa1,
                                        (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0x40, (byte) 0xa2, (byte) 0x7e, (byte) 0xa2, (byte) 0xa1, (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0x40,
                                        (byte) 0xa3, (byte) 0x7e, (byte) 0xa3, (byte) 0xa1, (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0x40, (byte) 0xa4, (byte) 0x7e, (byte) 0xa4, (byte) 0xa1,
                                        (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0x40, (byte) 0xa5, (byte) 0x7e, (byte) 0xa5, (byte) 0xa1, (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0x40,
                                        (byte) 0xa6, (byte) 0x7e, (byte) 0xa6, (byte) 0xa1, (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0x40, (byte) 0xa7, (byte) 0x7e, (byte) 0xa7, (byte) 0xa1,
                                        (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0x40, (byte) 0xa8, (byte) 0x7e, (byte) 0xa8, (byte) 0xa1, (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0x40,
                                        (byte) 0xa9, (byte) 0x7e, (byte) 0xa9, (byte) 0xa1, (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0x40, (byte) 0xaa, (byte) 0x7e, (byte) 0xaa, (byte) 0xa1,
                                        (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0x40, (byte) 0xab, (byte) 0x7e, (byte) 0xab, (byte) 0xa1, (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0x40,
                                        (byte) 0xac, (byte) 0x7e, (byte) 0xac, (byte) 0xa1, (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0x40, (byte) 0xad, (byte) 0x7e, (byte) 0xad, (byte) 0xa1,
                                        (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0x40, (byte) 0xae, (byte) 0x7e, (byte) 0xae, (byte) 0xa1, (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0x40,
                                        (byte) 0xaf, (byte) 0x7e, (byte) 0xaf, (byte) 0xa1, (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0x40, (byte) 0xb0, (byte) 0x7e, (byte) 0xb0, (byte) 0xa1,
                                        (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0x40, (byte) 0xb1, (byte) 0x7e, (byte) 0xb1, (byte) 0xa1, (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0x40,
                                        (byte) 0xb2, (byte) 0x7e, (byte) 0xb2, (byte) 0xa1, (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0x40, (byte) 0xb3, (byte) 0x7e, (byte) 0xb3, (byte) 0xa1,
                                        (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0x40, (byte) 0xb4, (byte) 0x7e, (byte) 0xb4, (byte) 0xa1, (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0x40,
                                        (byte) 0xb5, (byte) 0x7e, (byte) 0xb5, (byte) 0xa1, (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0x40, (byte) 0xb6, (byte) 0x7e, (byte) 0xb6, (byte) 0xa1,
                                        (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0x40, (byte) 0xb7, (byte) 0x7e, (byte) 0xb7, (byte) 0xa1, (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0x40,
                                        (byte) 0xb8, (byte) 0x7e, (byte) 0xb8, (byte) 0xa1, (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0x40, (byte) 0xb9, (byte) 0x7e, (byte) 0xb9, (byte) 0xa1,
                                        (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0x40, (byte) 0xba, (byte) 0x7e, (byte) 0xba, (byte) 0xa1, (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0x40,
                                        (byte) 0xbb, (byte) 0x7e, (byte) 0xbb, (byte) 0xa1, (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0x40, (byte) 0xbc, (byte) 0x7e, (byte) 0xbc, (byte) 0xa1,
                                        (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0x40, (byte) 0xbd, (byte) 0x7e, (byte) 0xbd, (byte) 0xa1, (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0x40,
                                        (byte) 0xbe, (byte) 0x7e, (byte) 0xbe, (byte) 0xa1, (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0x40, (byte) 0xbf, (byte) 0x7e, (byte) 0xbf, (byte) 0xa1,
                                        (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0x40, (byte) 0xc0, (byte) 0x7e, (byte) 0xc0, (byte) 0xa1, (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0x40,
                                        (byte) 0xc1, (byte) 0x7e, (byte) 0xc1, (byte) 0xa1, (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0x40, (byte) 0xc2, (byte) 0x7e, (byte) 0xc2, (byte) 0xa1,
                                        (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0x40, (byte) 0xc3, (byte) 0x7e, (byte) 0xc3, (byte) 0xa1, (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0x40,
                                        (byte) 0xc4, (byte) 0x7e, (byte) 0xc4, (byte) 0xa1, (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0x40, (byte) 0xc5, (byte) 0x7e, (byte) 0xc5, (byte) 0xa1,
                                        (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0x40, (byte) 0xc6, (byte) 0x7e, (byte) 0xc6, (byte) 0xa1, (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0x40,
                                        (byte) 0xc7, (byte) 0x7e, (byte) 0xc7, (byte) 0xa1, (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0x40, (byte) 0xc8, (byte) 0x7e, (byte) 0xc8, (byte) 0xa1,
                                        (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0x40, (byte) 0xc9, (byte) 0x7e, (byte) 0xc9, (byte) 0xa1, (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0x40,
                                        (byte) 0xca, (byte) 0x7e, (byte) 0xca, (byte) 0xa1, (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0x40, (byte) 0xcb, (byte) 0x7e, (byte) 0xcb, (byte) 0xa1,
                                        (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0x40, (byte) 0xcc, (byte) 0x7e, (byte) 0xcc, (byte) 0xa1, (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0x40,
                                        (byte) 0xcd, (byte) 0x7e, (byte) 0xcd, (byte) 0xa1, (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0x40, (byte) 0xce, (byte) 0x7e, (byte) 0xce, (byte) 0xa1,
                                        (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0x40, (byte) 0xcf, (byte) 0x7e, (byte) 0xcf, (byte) 0xa1, (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0x40,
                                        (byte) 0xd0, (byte) 0x7e, (byte) 0xd0, (byte) 0xa1, (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0x40, (byte) 0xd1, (byte) 0x7e, (byte) 0xd1, (byte) 0xa1,
                                        (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0x40, (byte) 0xd2, (byte) 0x7e, (byte) 0xd2, (byte) 0xa1, (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0x40,
                                        (byte) 0xd3, (byte) 0x7e, (byte) 0xd3, (byte) 0xa1, (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0x40, (byte) 0xd4, (byte) 0x7e, (byte) 0xd4, (byte) 0xa1,
                                        (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0x40, (byte) 0xd5, (byte) 0x7e, (byte) 0xd5, (byte) 0xa1, (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0x40,
                                        (byte) 0xd6, (byte) 0x7e, (byte) 0xd6, (byte) 0xa1, (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0x40, (byte) 0xd7, (byte) 0x7e, (byte) 0xd7, (byte) 0xa1,
                                        (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0x40, (byte) 0xd8, (byte) 0x7e, (byte) 0xd8, (byte) 0xa1, (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0x40,
                                        (byte) 0xd9, (byte) 0x7e, (byte) 0xd9, (byte) 0xa1, (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0x40, (byte) 0xda, (byte) 0x7e, (byte) 0xda, (byte) 0xa1,
                                        (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0x40, (byte) 0xdb, (byte) 0x7e, (byte) 0xdb, (byte) 0xa1, (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0x40,
                                        (byte) 0xdc, (byte) 0x7e, (byte) 0xdc, (byte) 0xa1, (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0x40, (byte) 0xdd, (byte) 0x7e, (byte) 0xdd, (byte) 0xa1,
                                        (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0x40, (byte) 0xde, (byte) 0x7e, (byte) 0xde, (byte) 0xa1, (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0x40,
                                        (byte) 0xdf, (byte) 0x7e, (byte) 0xdf, (byte) 0xa1, (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0xa1,
                                        (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0x40, (byte) 0xe1, (byte) 0x7e, (byte) 0xe1, (byte) 0xa1, (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0x40,
                                        (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0xa1, (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0xa1,
                                        (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0x40, (byte) 0xe4, (byte) 0x7e, (byte) 0xe4, (byte) 0xa1, (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0x40,
                                        (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0xa1, (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0xa1,
                                        (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0x40, (byte) 0xe7, (byte) 0x7e, (byte) 0xe7, (byte) 0xa1, (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0x40,
                                        (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0xa1, (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0xa1,
                                        (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0x40, (byte) 0xea, (byte) 0x7e, (byte) 0xea, (byte) 0xa1, (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0x40,
                                        (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0xa1, (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0xa1,
                                        (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0x40, (byte) 0xed, (byte) 0x7e, (byte) 0xed, (byte) 0xa1, (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0x40,
                                        (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0xa1, (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0xa1,
                                        (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0x40, (byte) 0xf0, (byte) 0x7e, (byte) 0xf0, (byte) 0xa1, (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0x40,
                                        (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0xa1, (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0xa1,
                                        (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0x40, (byte) 0xf3, (byte) 0x7e, (byte) 0xf3, (byte) 0xa1, (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0x40,
                                        (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0xa1, (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0xa1,
                                        (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0x40, (byte) 0xf6, (byte) 0x7e, (byte) 0xf6, (byte) 0xa1, (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0x40,
                                        (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0xa1, (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0xa1,
                                        (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0x40, (byte) 0xf9, (byte) 0x7e, (byte) 0xf9, (byte) 0xa1, (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0x40,
                                        (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0xa1, (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0xa1,
                                        (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0x40, (byte) 0xfc, (byte) 0x7e, (byte) 0xfc, (byte) 0xa1, (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0x40,
                                        (byte) 0xfd, (byte) 0x7e, (byte) 0xfd, (byte) 0xa1, (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0x40, (byte) 0xfe, (byte) 0x7e, (byte) 0xfe, (byte) 0xa1,
                                        (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataEUCJIS2004() {
        assert TruffleString.Encoding.EUC_JIS_2004.ordinal() == 19;
        return new TestData(new int[]{0x0000, 0x007f, 0x8ea1, 0x8efe, 0xa1a1, 0xa1fe, 0xa2a1, 0xa2fe, 0xa3a1, 0xa3fe, 0xa4a1, 0xa4fe, 0xa5a1, 0xa5fe, 0xa6a1, 0xa6fe, 0xa7a1, 0xa7fe, 0xa8a1, 0xa8fe,
                        0xa9a1, 0xa9fe, 0xaaa1, 0xaafe, 0xaba1, 0xabfe, 0xaca1, 0xacfe, 0xada1, 0xadfe, 0xaea1, 0xaefe, 0xafa1, 0xaffe, 0xb0a1, 0xb0fe, 0xb1a1, 0xb1fe, 0xb2a1, 0xb2fe, 0xb3a1, 0xb3fe,
                        0xb4a1, 0xb4fe, 0xb5a1, 0xb5fe, 0xb6a1, 0xb6fe, 0xb7a1, 0xb7fe, 0xb8a1, 0xb8fe, 0xb9a1, 0xb9fe, 0xbaa1, 0xbafe, 0xbba1, 0xbbfe, 0xbca1, 0xbcfe, 0xbda1, 0xbdfe, 0xbea1, 0xbefe,
                        0xbfa1, 0xbffe, 0xc0a1, 0xc0fe, 0xc1a1, 0xc1fe, 0xc2a1, 0xc2fe, 0xc3a1, 0xc3fe, 0xc4a1, 0xc4fe, 0xc5a1, 0xc5fe, 0xc6a1, 0xc6fe, 0xc7a1, 0xc7fe, 0xc8a1, 0xc8fe, 0xc9a1, 0xc9fe,
                        0xcaa1, 0xcafe, 0xcba1, 0xcbfe, 0xcca1, 0xccfe, 0xcda1, 0xcdfe, 0xcea1, 0xcefe, 0xcfa1, 0xcffe, 0xd0a1, 0xd0fe, 0xd1a1, 0xd1fe, 0xd2a1, 0xd2fe, 0xd3a1, 0xd3fe, 0xd4a1, 0xd4fe,
                        0xd5a1, 0xd5fe, 0xd6a1, 0xd6fe, 0xd7a1, 0xd7fe, 0xd8a1, 0xd8fe, 0xd9a1, 0xd9fe, 0xdaa1, 0xdafe, 0xdba1, 0xdbfe, 0xdca1, 0xdcfe, 0xdda1, 0xddfe, 0xdea1, 0xdefe, 0xdfa1, 0xdffe,
                        0xe0a1, 0xe0fe, 0xe1a1, 0xe1fe, 0xe2a1, 0xe2fe, 0xe3a1, 0xe3fe, 0xe4a1, 0xe4fe, 0xe5a1, 0xe5fe, 0xe6a1, 0xe6fe, 0xe7a1, 0xe7fe, 0xe8a1, 0xe8fe, 0xe9a1, 0xe9fe, 0xeaa1, 0xeafe,
                        0xeba1, 0xebfe, 0xeca1, 0xecfe, 0xeda1, 0xedfe, 0xeea1, 0xeefe, 0xefa1, 0xeffe, 0xf0a1, 0xf0fe, 0xf1a1, 0xf1fe, 0xf2a1, 0xf2fe, 0xf3a1, 0xf3fe, 0xf4a1, 0xf4fe, 0xf5a1, 0xf5fe,
                        0xf6a1, 0xf6fe, 0xf7a1, 0xf7fe, 0xf8a1, 0xf8fe, 0xf9a1, 0xf9fe, 0xfaa1, 0xfafe, 0xfba1, 0xfbfe, 0xfca1, 0xfcfe, 0xfda1, 0xfdfe, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0x8e, (byte) 0xa1, (byte) 0x8e, (byte) 0xfe, (byte) 0xa1, (byte) 0xa1, (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0xa1,
                                        (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0xa1, (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0xa1, (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0xa1,
                                        (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0xa1, (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0xa1, (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0xa1,
                                        (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0xa1, (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0xa1, (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0xa1,
                                        (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0xa1, (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0xa1, (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0xa1,
                                        (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0xa1, (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0xa1, (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0xa1,
                                        (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0xa1, (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0xa1, (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0xa1,
                                        (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0xa1, (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0xa1, (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0xa1,
                                        (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0xa1, (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0xa1, (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0xa1,
                                        (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0xa1, (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0xa1, (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0xa1,
                                        (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0xa1, (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0xa1, (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0xa1,
                                        (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0xa1, (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0xa1, (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0xa1,
                                        (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0xa1, (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0xa1, (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0xa1,
                                        (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0xa1, (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0xa1, (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0xa1,
                                        (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0xa1, (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0xa1, (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0xa1,
                                        (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0xa1, (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0xa1, (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0xa1,
                                        (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0xa1, (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0xa1, (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0xa1,
                                        (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0xa1, (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0xa1, (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0xa1,
                                        (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0xa1, (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0xa1, (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0xa1,
                                        (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0xa1, (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0xa1, (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0xa1,
                                        (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0xa1, (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0xa1, (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0xa1,
                                        (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0xa1, (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0xa1, (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0xa1,
                                        (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0xa1, (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0xa1, (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0xa1,
                                        (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0xa1, (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0xa1, (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0xa1,
                                        (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0xa1, (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0xa1, (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0xa1,
                                        (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0xa1, (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0xa1, (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0xa1,
                                        (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0xa1, (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0xa1, (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0xa1,
                                        (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0xa1, (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0xa1, (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0xa1,
                                        (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0xa1, (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0xa1, (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0xa1,
                                        (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0xa1, (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0xa1, (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0xa1,
                                        (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0xa1, (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0xa1, (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0xa1,
                                        (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0xa1, (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0xa1, (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataEUCJP() {
        assert TruffleString.Encoding.EUC_JP.ordinal() == 20;
        return new TestData(new int[]{0x0000, 0x007f, 0x8ea1, 0x8efe, 0xa1a1, 0xa1fe, 0xa2a1, 0xa2fe, 0xa3a1, 0xa3fe, 0xa4a1, 0xa4fe, 0xa5a1, 0xa5fe, 0xa6a1, 0xa6fe, 0xa7a1, 0xa7fe, 0xa8a1, 0xa8fe,
                        0xa9a1, 0xa9fe, 0xaaa1, 0xaafe, 0xaba1, 0xabfe, 0xaca1, 0xacfe, 0xada1, 0xadfe, 0xaea1, 0xaefe, 0xafa1, 0xaffe, 0xb0a1, 0xb0fe, 0xb1a1, 0xb1fe, 0xb2a1, 0xb2fe, 0xb3a1, 0xb3fe,
                        0xb4a1, 0xb4fe, 0xb5a1, 0xb5fe, 0xb6a1, 0xb6fe, 0xb7a1, 0xb7fe, 0xb8a1, 0xb8fe, 0xb9a1, 0xb9fe, 0xbaa1, 0xbafe, 0xbba1, 0xbbfe, 0xbca1, 0xbcfe, 0xbda1, 0xbdfe, 0xbea1, 0xbefe,
                        0xbfa1, 0xbffe, 0xc0a1, 0xc0fe, 0xc1a1, 0xc1fe, 0xc2a1, 0xc2fe, 0xc3a1, 0xc3fe, 0xc4a1, 0xc4fe, 0xc5a1, 0xc5fe, 0xc6a1, 0xc6fe, 0xc7a1, 0xc7fe, 0xc8a1, 0xc8fe, 0xc9a1, 0xc9fe,
                        0xcaa1, 0xcafe, 0xcba1, 0xcbfe, 0xcca1, 0xccfe, 0xcda1, 0xcdfe, 0xcea1, 0xcefe, 0xcfa1, 0xcffe, 0xd0a1, 0xd0fe, 0xd1a1, 0xd1fe, 0xd2a1, 0xd2fe, 0xd3a1, 0xd3fe, 0xd4a1, 0xd4fe,
                        0xd5a1, 0xd5fe, 0xd6a1, 0xd6fe, 0xd7a1, 0xd7fe, 0xd8a1, 0xd8fe, 0xd9a1, 0xd9fe, 0xdaa1, 0xdafe, 0xdba1, 0xdbfe, 0xdca1, 0xdcfe, 0xdda1, 0xddfe, 0xdea1, 0xdefe, 0xdfa1, 0xdffe,
                        0xe0a1, 0xe0fe, 0xe1a1, 0xe1fe, 0xe2a1, 0xe2fe, 0xe3a1, 0xe3fe, 0xe4a1, 0xe4fe, 0xe5a1, 0xe5fe, 0xe6a1, 0xe6fe, 0xe7a1, 0xe7fe, 0xe8a1, 0xe8fe, 0xe9a1, 0xe9fe, 0xeaa1, 0xeafe,
                        0xeba1, 0xebfe, 0xeca1, 0xecfe, 0xeda1, 0xedfe, 0xeea1, 0xeefe, 0xefa1, 0xeffe, 0xf0a1, 0xf0fe, 0xf1a1, 0xf1fe, 0xf2a1, 0xf2fe, 0xf3a1, 0xf3fe, 0xf4a1, 0xf4fe, 0xf5a1, 0xf5fe,
                        0xf6a1, 0xf6fe, 0xf7a1, 0xf7fe, 0xf8a1, 0xf8fe, 0xf9a1, 0xf9fe, 0xfaa1, 0xfafe, 0xfba1, 0xfbfe, 0xfca1, 0xfcfe, 0xfda1, 0xfdfe, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0x8e, (byte) 0xa1, (byte) 0x8e, (byte) 0xfe, (byte) 0xa1, (byte) 0xa1, (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0xa1,
                                        (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0xa1, (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0xa1, (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0xa1,
                                        (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0xa1, (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0xa1, (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0xa1,
                                        (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0xa1, (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0xa1, (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0xa1,
                                        (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0xa1, (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0xa1, (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0xa1,
                                        (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0xa1, (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0xa1, (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0xa1,
                                        (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0xa1, (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0xa1, (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0xa1,
                                        (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0xa1, (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0xa1, (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0xa1,
                                        (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0xa1, (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0xa1, (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0xa1,
                                        (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0xa1, (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0xa1, (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0xa1,
                                        (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0xa1, (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0xa1, (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0xa1,
                                        (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0xa1, (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0xa1, (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0xa1,
                                        (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0xa1, (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0xa1, (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0xa1,
                                        (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0xa1, (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0xa1, (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0xa1,
                                        (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0xa1, (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0xa1, (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0xa1,
                                        (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0xa1, (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0xa1, (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0xa1,
                                        (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0xa1, (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0xa1, (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0xa1,
                                        (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0xa1, (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0xa1, (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0xa1,
                                        (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0xa1, (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0xa1, (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0xa1,
                                        (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0xa1, (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0xa1, (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0xa1,
                                        (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0xa1, (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0xa1, (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0xa1,
                                        (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0xa1, (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0xa1, (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0xa1,
                                        (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0xa1, (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0xa1, (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0xa1,
                                        (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0xa1, (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0xa1, (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0xa1,
                                        (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0xa1, (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0xa1, (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0xa1,
                                        (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0xa1, (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0xa1, (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0xa1,
                                        (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0xa1, (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0xa1, (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0xa1,
                                        (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0xa1, (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0xa1, (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0xa1,
                                        (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0xa1, (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0xa1, (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0xa1,
                                        (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0xa1, (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0xa1, (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0xa1,
                                        (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0xa1, (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0xa1, (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0xa1,
                                        (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0xa1, (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0xa1, (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataEUCKR() {
        assert TruffleString.Encoding.EUC_KR.ordinal() == 21;
        return new TestData(new int[]{0x0000, 0x007f, 0xa1a1, 0xa1fe, 0xa2a1, 0xa2fe, 0xa3a1, 0xa3fe, 0xa4a1, 0xa4fe, 0xa5a1, 0xa5fe, 0xa6a1, 0xa6fe, 0xa7a1, 0xa7fe, 0xa8a1, 0xa8fe, 0xa9a1, 0xa9fe,
                        0xaaa1, 0xaafe, 0xaba1, 0xabfe, 0xaca1, 0xacfe, 0xada1, 0xadfe, 0xaea1, 0xaefe, 0xafa1, 0xaffe, 0xb0a1, 0xb0fe, 0xb1a1, 0xb1fe, 0xb2a1, 0xb2fe, 0xb3a1, 0xb3fe, 0xb4a1, 0xb4fe,
                        0xb5a1, 0xb5fe, 0xb6a1, 0xb6fe, 0xb7a1, 0xb7fe, 0xb8a1, 0xb8fe, 0xb9a1, 0xb9fe, 0xbaa1, 0xbafe, 0xbba1, 0xbbfe, 0xbca1, 0xbcfe, 0xbda1, 0xbdfe, 0xbea1, 0xbefe, 0xbfa1, 0xbffe,
                        0xc0a1, 0xc0fe, 0xc1a1, 0xc1fe, 0xc2a1, 0xc2fe, 0xc3a1, 0xc3fe, 0xc4a1, 0xc4fe, 0xc5a1, 0xc5fe, 0xc6a1, 0xc6fe, 0xc7a1, 0xc7fe, 0xc8a1, 0xc8fe, 0xc9a1, 0xc9fe, 0xcaa1, 0xcafe,
                        0xcba1, 0xcbfe, 0xcca1, 0xccfe, 0xcda1, 0xcdfe, 0xcea1, 0xcefe, 0xcfa1, 0xcffe, 0xd0a1, 0xd0fe, 0xd1a1, 0xd1fe, 0xd2a1, 0xd2fe, 0xd3a1, 0xd3fe, 0xd4a1, 0xd4fe, 0xd5a1, 0xd5fe,
                        0xd6a1, 0xd6fe, 0xd7a1, 0xd7fe, 0xd8a1, 0xd8fe, 0xd9a1, 0xd9fe, 0xdaa1, 0xdafe, 0xdba1, 0xdbfe, 0xdca1, 0xdcfe, 0xdda1, 0xddfe, 0xdea1, 0xdefe, 0xdfa1, 0xdffe, 0xe0a1, 0xe0fe,
                        0xe1a1, 0xe1fe, 0xe2a1, 0xe2fe, 0xe3a1, 0xe3fe, 0xe4a1, 0xe4fe, 0xe5a1, 0xe5fe, 0xe6a1, 0xe6fe, 0xe7a1, 0xe7fe, 0xe8a1, 0xe8fe, 0xe9a1, 0xe9fe, 0xeaa1, 0xeafe, 0xeba1, 0xebfe,
                        0xeca1, 0xecfe, 0xeda1, 0xedfe, 0xeea1, 0xeefe, 0xefa1, 0xeffe, 0xf0a1, 0xf0fe, 0xf1a1, 0xf1fe, 0xf2a1, 0xf2fe, 0xf3a1, 0xf3fe, 0xf4a1, 0xf4fe, 0xf5a1, 0xf5fe, 0xf6a1, 0xf6fe,
                        0xf7a1, 0xf7fe, 0xf8a1, 0xf8fe, 0xf9a1, 0xf9fe, 0xfaa1, 0xfafe, 0xfba1, 0xfbfe, 0xfca1, 0xfcfe, 0xfda1, 0xfdfe, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0xa1, (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0xa1, (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0xa1,
                                        (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0xa1, (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0xa1, (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0xa1,
                                        (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0xa1, (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0xa1, (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0xa1,
                                        (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0xa1, (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0xa1, (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0xa1,
                                        (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0xa1, (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0xa1, (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0xa1,
                                        (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0xa1, (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0xa1, (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0xa1,
                                        (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0xa1, (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0xa1, (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0xa1,
                                        (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0xa1, (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0xa1, (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0xa1,
                                        (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0xa1, (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0xa1, (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0xa1,
                                        (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0xa1, (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0xa1, (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0xa1,
                                        (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0xa1, (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0xa1, (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0xa1,
                                        (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0xa1, (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0xa1, (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0xa1,
                                        (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0xa1, (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0xa1, (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0xa1,
                                        (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0xa1, (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0xa1, (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0xa1,
                                        (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0xa1, (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0xa1, (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0xa1,
                                        (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0xa1, (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0xa1, (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0xa1,
                                        (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0xa1, (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0xa1, (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0xa1,
                                        (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0xa1, (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0xa1, (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0xa1,
                                        (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0xa1, (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0xa1, (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0xa1,
                                        (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0xa1, (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0xa1, (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0xa1,
                                        (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0xa1, (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0xa1, (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0xa1,
                                        (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0xa1, (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0xa1, (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0xa1,
                                        (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0xa1, (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0xa1, (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0xa1,
                                        (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0xa1, (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0xa1, (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0xa1,
                                        (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0xa1, (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0xa1, (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0xa1,
                                        (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0xa1, (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0xa1, (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0xa1,
                                        (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0xa1, (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0xa1, (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0xa1,
                                        (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0xa1, (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0xa1, (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0xa1,
                                        (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0xa1, (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0xa1, (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0xa1,
                                        (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0xa1, (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0xa1, (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0xa1,
                                        (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0xa1, (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0xa1, (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0xa1,
                                        (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0xa1, (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataEUCTW() {
        assert TruffleString.Encoding.EUC_TW.ordinal() == 22;
        return new TestData(new int[]{0x0000, 0x007f, 0xa1a1, 0xa1fe, 0xa2a1, 0xa2fe, 0xa3a1, 0xa3fe, 0xa4a1, 0xa4fe, 0xa5a1, 0xa5fe, 0xa6a1, 0xa6fe, 0xa7a1, 0xa7fe, 0xa8a1, 0xa8fe, 0xa9a1, 0xa9fe,
                        0xaaa1, 0xaafe, 0xaba1, 0xabfe, 0xaca1, 0xacfe, 0xada1, 0xadfe, 0xaea1, 0xaefe, 0xafa1, 0xaffe, 0xb0a1, 0xb0fe, 0xb1a1, 0xb1fe, 0xb2a1, 0xb2fe, 0xb3a1, 0xb3fe, 0xb4a1, 0xb4fe,
                        0xb5a1, 0xb5fe, 0xb6a1, 0xb6fe, 0xb7a1, 0xb7fe, 0xb8a1, 0xb8fe, 0xb9a1, 0xb9fe, 0xbaa1, 0xbafe, 0xbba1, 0xbbfe, 0xbca1, 0xbcfe, 0xbda1, 0xbdfe, 0xbea1, 0xbefe, 0xbfa1, 0xbffe,
                        0xc0a1, 0xc0fe, 0xc1a1, 0xc1fe, 0xc2a1, 0xc2fe, 0xc3a1, 0xc3fe, 0xc4a1, 0xc4fe, 0xc5a1, 0xc5fe, 0xc6a1, 0xc6fe, 0xc7a1, 0xc7fe, 0xc8a1, 0xc8fe, 0xc9a1, 0xc9fe, 0xcaa1, 0xcafe,
                        0xcba1, 0xcbfe, 0xcca1, 0xccfe, 0xcda1, 0xcdfe, 0xcea1, 0xcefe, 0xcfa1, 0xcffe, 0xd0a1, 0xd0fe, 0xd1a1, 0xd1fe, 0xd2a1, 0xd2fe, 0xd3a1, 0xd3fe, 0xd4a1, 0xd4fe, 0xd5a1, 0xd5fe,
                        0xd6a1, 0xd6fe, 0xd7a1, 0xd7fe, 0xd8a1, 0xd8fe, 0xd9a1, 0xd9fe, 0xdaa1, 0xdafe, 0xdba1, 0xdbfe, 0xdca1, 0xdcfe, 0xdda1, 0xddfe, 0xdea1, 0xdefe, 0xdfa1, 0xdffe, 0xe0a1, 0xe0fe,
                        0xe1a1, 0xe1fe, 0xe2a1, 0xe2fe, 0xe3a1, 0xe3fe, 0xe4a1, 0xe4fe, 0xe5a1, 0xe5fe, 0xe6a1, 0xe6fe, 0xe7a1, 0xe7fe, 0xe8a1, 0xe8fe, 0xe9a1, 0xe9fe, 0xeaa1, 0xeafe, 0xeba1, 0xebfe,
                        0xeca1, 0xecfe, 0xeda1, 0xedfe, 0xeea1, 0xeefe, 0xefa1, 0xeffe, 0xf0a1, 0xf0fe, 0xf1a1, 0xf1fe, 0xf2a1, 0xf2fe, 0xf3a1, 0xf3fe, 0xf4a1, 0xf4fe, 0xf5a1, 0xf5fe, 0xf6a1, 0xf6fe,
                        0xf7a1, 0xf7fe, 0xf8a1, 0xf8fe, 0xf9a1, 0xf9fe, 0xfaa1, 0xfafe, 0xfba1, 0xfbfe, 0xfca1, 0xfcfe, 0xfda1, 0xfdfe, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0xa1, (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0xa1, (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0xa1,
                                        (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0xa1, (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0xa1, (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0xa1,
                                        (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0xa1, (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0xa1, (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0xa1,
                                        (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0xa1, (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0xa1, (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0xa1,
                                        (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0xa1, (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0xa1, (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0xa1,
                                        (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0xa1, (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0xa1, (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0xa1,
                                        (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0xa1, (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0xa1, (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0xa1,
                                        (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0xa1, (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0xa1, (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0xa1,
                                        (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0xa1, (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0xa1, (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0xa1,
                                        (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0xa1, (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0xa1, (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0xa1,
                                        (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0xa1, (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0xa1, (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0xa1,
                                        (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0xa1, (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0xa1, (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0xa1,
                                        (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0xa1, (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0xa1, (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0xa1,
                                        (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0xa1, (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0xa1, (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0xa1,
                                        (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0xa1, (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0xa1, (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0xa1,
                                        (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0xa1, (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0xa1, (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0xa1,
                                        (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0xa1, (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0xa1, (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0xa1,
                                        (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0xa1, (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0xa1, (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0xa1,
                                        (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0xa1, (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0xa1, (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0xa1,
                                        (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0xa1, (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0xa1, (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0xa1,
                                        (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0xa1, (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0xa1, (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0xa1,
                                        (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0xa1, (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0xa1, (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0xa1,
                                        (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0xa1, (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0xa1, (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0xa1,
                                        (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0xa1, (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0xa1, (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0xa1,
                                        (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0xa1, (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0xa1, (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0xa1,
                                        (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0xa1, (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0xa1, (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0xa1,
                                        (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0xa1, (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0xa1, (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0xa1,
                                        (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0xa1, (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0xa1, (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0xa1,
                                        (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0xa1, (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0xa1, (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0xa1,
                                        (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0xa1, (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0xa1, (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0xa1,
                                        (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0xa1, (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0xa1, (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0xa1,
                                        (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0xa1, (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataEmacsMule() {
        assert TruffleString.Encoding.Emacs_Mule.ordinal() == 23;
        return new TestData(new int[]{0x00, 0x7f}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null);
    }

    static TestData dataEucJPms() {
        assert TruffleString.Encoding.EucJP_ms.ordinal() == 24;
        return new TestData(new int[]{0x0000, 0x007f, 0x8ea1, 0x8efe, 0xa1a1, 0xa1fe, 0xa2a1, 0xa2fe, 0xa3a1, 0xa3fe, 0xa4a1, 0xa4fe, 0xa5a1, 0xa5fe, 0xa6a1, 0xa6fe, 0xa7a1, 0xa7fe, 0xa8a1, 0xa8fe,
                        0xa9a1, 0xa9fe, 0xaaa1, 0xaafe, 0xaba1, 0xabfe, 0xaca1, 0xacfe, 0xada1, 0xadfe, 0xaea1, 0xaefe, 0xafa1, 0xaffe, 0xb0a1, 0xb0fe, 0xb1a1, 0xb1fe, 0xb2a1, 0xb2fe, 0xb3a1, 0xb3fe,
                        0xb4a1, 0xb4fe, 0xb5a1, 0xb5fe, 0xb6a1, 0xb6fe, 0xb7a1, 0xb7fe, 0xb8a1, 0xb8fe, 0xb9a1, 0xb9fe, 0xbaa1, 0xbafe, 0xbba1, 0xbbfe, 0xbca1, 0xbcfe, 0xbda1, 0xbdfe, 0xbea1, 0xbefe,
                        0xbfa1, 0xbffe, 0xc0a1, 0xc0fe, 0xc1a1, 0xc1fe, 0xc2a1, 0xc2fe, 0xc3a1, 0xc3fe, 0xc4a1, 0xc4fe, 0xc5a1, 0xc5fe, 0xc6a1, 0xc6fe, 0xc7a1, 0xc7fe, 0xc8a1, 0xc8fe, 0xc9a1, 0xc9fe,
                        0xcaa1, 0xcafe, 0xcba1, 0xcbfe, 0xcca1, 0xccfe, 0xcda1, 0xcdfe, 0xcea1, 0xcefe, 0xcfa1, 0xcffe, 0xd0a1, 0xd0fe, 0xd1a1, 0xd1fe, 0xd2a1, 0xd2fe, 0xd3a1, 0xd3fe, 0xd4a1, 0xd4fe,
                        0xd5a1, 0xd5fe, 0xd6a1, 0xd6fe, 0xd7a1, 0xd7fe, 0xd8a1, 0xd8fe, 0xd9a1, 0xd9fe, 0xdaa1, 0xdafe, 0xdba1, 0xdbfe, 0xdca1, 0xdcfe, 0xdda1, 0xddfe, 0xdea1, 0xdefe, 0xdfa1, 0xdffe,
                        0xe0a1, 0xe0fe, 0xe1a1, 0xe1fe, 0xe2a1, 0xe2fe, 0xe3a1, 0xe3fe, 0xe4a1, 0xe4fe, 0xe5a1, 0xe5fe, 0xe6a1, 0xe6fe, 0xe7a1, 0xe7fe, 0xe8a1, 0xe8fe, 0xe9a1, 0xe9fe, 0xeaa1, 0xeafe,
                        0xeba1, 0xebfe, 0xeca1, 0xecfe, 0xeda1, 0xedfe, 0xeea1, 0xeefe, 0xefa1, 0xeffe, 0xf0a1, 0xf0fe, 0xf1a1, 0xf1fe, 0xf2a1, 0xf2fe, 0xf3a1, 0xf3fe, 0xf4a1, 0xf4fe, 0xf5a1, 0xf5fe,
                        0xf6a1, 0xf6fe, 0xf7a1, 0xf7fe, 0xf8a1, 0xf8fe, 0xf9a1, 0xf9fe, 0xfaa1, 0xfafe, 0xfba1, 0xfbfe, 0xfca1, 0xfcfe, 0xfda1, 0xfdfe, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0x8e, (byte) 0xa1, (byte) 0x8e, (byte) 0xfe, (byte) 0xa1, (byte) 0xa1, (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0xa1,
                                        (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0xa1, (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0xa1, (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0xa1,
                                        (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0xa1, (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0xa1, (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0xa1,
                                        (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0xa1, (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0xa1, (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0xa1,
                                        (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0xa1, (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0xa1, (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0xa1,
                                        (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0xa1, (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0xa1, (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0xa1,
                                        (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0xa1, (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0xa1, (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0xa1,
                                        (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0xa1, (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0xa1, (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0xa1,
                                        (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0xa1, (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0xa1, (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0xa1,
                                        (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0xa1, (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0xa1, (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0xa1,
                                        (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0xa1, (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0xa1, (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0xa1,
                                        (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0xa1, (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0xa1, (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0xa1,
                                        (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0xa1, (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0xa1, (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0xa1,
                                        (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0xa1, (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0xa1, (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0xa1,
                                        (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0xa1, (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0xa1, (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0xa1,
                                        (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0xa1, (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0xa1, (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0xa1,
                                        (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0xa1, (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0xa1, (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0xa1,
                                        (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0xa1, (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0xa1, (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0xa1,
                                        (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0xa1, (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0xa1, (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0xa1,
                                        (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0xa1, (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0xa1, (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0xa1,
                                        (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0xa1, (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0xa1, (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0xa1,
                                        (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0xa1, (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0xa1, (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0xa1,
                                        (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0xa1, (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0xa1, (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0xa1,
                                        (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0xa1, (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0xa1, (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0xa1,
                                        (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0xa1, (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0xa1, (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0xa1,
                                        (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0xa1, (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0xa1, (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0xa1,
                                        (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0xa1, (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0xa1, (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0xa1,
                                        (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0xa1, (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0xa1, (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0xa1,
                                        (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0xa1, (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0xa1, (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0xa1,
                                        (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0xa1, (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0xa1, (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0xa1,
                                        (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0xa1, (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0xa1, (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0xa1,
                                        (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0xa1, (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0xa1, (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataGB12345() {
        assert TruffleString.Encoding.GB12345.ordinal() == 25;
        return new TestData(new int[]{0x0000, 0x007f, 0xa1a1, 0xa1fe, 0xa2a1, 0xa2fe, 0xa3a1, 0xa3fe, 0xa4a1, 0xa4fe, 0xa5a1, 0xa5fe, 0xa6a1, 0xa6fe, 0xa7a1, 0xa7fe, 0xa8a1, 0xa8fe, 0xa9a1, 0xa9fe,
                        0xaaa1, 0xaafe, 0xaba1, 0xabfe, 0xaca1, 0xacfe, 0xada1, 0xadfe, 0xaea1, 0xaefe, 0xafa1, 0xaffe, 0xb0a1, 0xb0fe, 0xb1a1, 0xb1fe, 0xb2a1, 0xb2fe, 0xb3a1, 0xb3fe, 0xb4a1, 0xb4fe,
                        0xb5a1, 0xb5fe, 0xb6a1, 0xb6fe, 0xb7a1, 0xb7fe, 0xb8a1, 0xb8fe, 0xb9a1, 0xb9fe, 0xbaa1, 0xbafe, 0xbba1, 0xbbfe, 0xbca1, 0xbcfe, 0xbda1, 0xbdfe, 0xbea1, 0xbefe, 0xbfa1, 0xbffe,
                        0xc0a1, 0xc0fe, 0xc1a1, 0xc1fe, 0xc2a1, 0xc2fe, 0xc3a1, 0xc3fe, 0xc4a1, 0xc4fe, 0xc5a1, 0xc5fe, 0xc6a1, 0xc6fe, 0xc7a1, 0xc7fe, 0xc8a1, 0xc8fe, 0xc9a1, 0xc9fe, 0xcaa1, 0xcafe,
                        0xcba1, 0xcbfe, 0xcca1, 0xccfe, 0xcda1, 0xcdfe, 0xcea1, 0xcefe, 0xcfa1, 0xcffe, 0xd0a1, 0xd0fe, 0xd1a1, 0xd1fe, 0xd2a1, 0xd2fe, 0xd3a1, 0xd3fe, 0xd4a1, 0xd4fe, 0xd5a1, 0xd5fe,
                        0xd6a1, 0xd6fe, 0xd7a1, 0xd7fe, 0xd8a1, 0xd8fe, 0xd9a1, 0xd9fe, 0xdaa1, 0xdafe, 0xdba1, 0xdbfe, 0xdca1, 0xdcfe, 0xdda1, 0xddfe, 0xdea1, 0xdefe, 0xdfa1, 0xdffe, 0xe0a1, 0xe0fe,
                        0xe1a1, 0xe1fe, 0xe2a1, 0xe2fe, 0xe3a1, 0xe3fe, 0xe4a1, 0xe4fe, 0xe5a1, 0xe5fe, 0xe6a1, 0xe6fe, 0xe7a1, 0xe7fe, 0xe8a1, 0xe8fe, 0xe9a1, 0xe9fe, 0xeaa1, 0xeafe, 0xeba1, 0xebfe,
                        0xeca1, 0xecfe, 0xeda1, 0xedfe, 0xeea1, 0xeefe, 0xefa1, 0xeffe, 0xf0a1, 0xf0fe, 0xf1a1, 0xf1fe, 0xf2a1, 0xf2fe, 0xf3a1, 0xf3fe, 0xf4a1, 0xf4fe, 0xf5a1, 0xf5fe, 0xf6a1, 0xf6fe,
                        0xf7a1, 0xf7fe, 0xf8a1, 0xf8fe, 0xf9a1, 0xf9fe, 0xfaa1, 0xfafe, 0xfba1, 0xfbfe, 0xfca1, 0xfcfe, 0xfda1, 0xfdfe, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0xa1, (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0xa1, (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0xa1,
                                        (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0xa1, (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0xa1, (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0xa1,
                                        (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0xa1, (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0xa1, (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0xa1,
                                        (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0xa1, (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0xa1, (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0xa1,
                                        (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0xa1, (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0xa1, (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0xa1,
                                        (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0xa1, (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0xa1, (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0xa1,
                                        (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0xa1, (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0xa1, (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0xa1,
                                        (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0xa1, (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0xa1, (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0xa1,
                                        (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0xa1, (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0xa1, (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0xa1,
                                        (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0xa1, (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0xa1, (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0xa1,
                                        (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0xa1, (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0xa1, (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0xa1,
                                        (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0xa1, (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0xa1, (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0xa1,
                                        (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0xa1, (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0xa1, (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0xa1,
                                        (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0xa1, (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0xa1, (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0xa1,
                                        (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0xa1, (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0xa1, (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0xa1,
                                        (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0xa1, (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0xa1, (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0xa1,
                                        (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0xa1, (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0xa1, (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0xa1,
                                        (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0xa1, (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0xa1, (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0xa1,
                                        (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0xa1, (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0xa1, (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0xa1,
                                        (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0xa1, (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0xa1, (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0xa1,
                                        (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0xa1, (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0xa1, (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0xa1,
                                        (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0xa1, (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0xa1, (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0xa1,
                                        (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0xa1, (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0xa1, (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0xa1,
                                        (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0xa1, (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0xa1, (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0xa1,
                                        (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0xa1, (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0xa1, (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0xa1,
                                        (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0xa1, (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0xa1, (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0xa1,
                                        (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0xa1, (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0xa1, (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0xa1,
                                        (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0xa1, (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0xa1, (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0xa1,
                                        (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0xa1, (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0xa1, (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0xa1,
                                        (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0xa1, (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0xa1, (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0xa1,
                                        (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0xa1, (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0xa1, (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0xa1,
                                        (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0xa1, (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataGB18030() {
        assert TruffleString.Encoding.GB18030.ordinal() == 26;
        return new TestData(new int[]{0x0000, 0x007f, 0x8140, 0x817e, 0x8180, 0x81fe, 0x8240, 0x827e, 0x8280, 0x82fe, 0x8340, 0x837e, 0x8380, 0x83fe, 0x8440, 0x847e, 0x8480, 0x84fe, 0x8540, 0x857e,
                        0x8580, 0x85fe, 0x8640, 0x867e, 0x8680, 0x86fe, 0x8740, 0x877e, 0x8780, 0x87fe, 0x8840, 0x887e, 0x8880, 0x88fe, 0x8940, 0x897e, 0x8980, 0x89fe, 0x8a40, 0x8a7e, 0x8a80, 0x8afe,
                        0x8b40, 0x8b7e, 0x8b80, 0x8bfe, 0x8c40, 0x8c7e, 0x8c80, 0x8cfe, 0x8d40, 0x8d7e, 0x8d80, 0x8dfe, 0x8e40, 0x8e7e, 0x8e80, 0x8efe, 0x8f40, 0x8f7e, 0x8f80, 0x8ffe, 0x9040, 0x907e,
                        0x9080, 0x90fe, 0x9140, 0x917e, 0x9180, 0x91fe, 0x9240, 0x927e, 0x9280, 0x92fe, 0x9340, 0x937e, 0x9380, 0x93fe, 0x9440, 0x947e, 0x9480, 0x94fe, 0x9540, 0x957e, 0x9580, 0x95fe,
                        0x9640, 0x967e, 0x9680, 0x96fe, 0x9740, 0x977e, 0x9780, 0x97fe, 0x9840, 0x987e, 0x9880, 0x98fe, 0x9940, 0x997e, 0x9980, 0x99fe, 0x9a40, 0x9a7e, 0x9a80, 0x9afe, 0x9b40, 0x9b7e,
                        0x9b80, 0x9bfe, 0x9c40, 0x9c7e, 0x9c80, 0x9cfe, 0x9d40, 0x9d7e, 0x9d80, 0x9dfe, 0x9e40, 0x9e7e, 0x9e80, 0x9efe, 0x9f40, 0x9f7e, 0x9f80, 0x9ffe, 0xa040, 0xa07e, 0xa080, 0xa0fe,
                        0xa140, 0xa17e, 0xa180, 0xa1fe, 0xa240, 0xa27e, 0xa280, 0xa2fe, 0xa340, 0xa37e, 0xa380, 0xa3fe, 0xa440, 0xa47e, 0xa480, 0xa4fe, 0xa540, 0xa57e, 0xa580, 0xa5fe, 0xa640, 0xa67e,
                        0xa680, 0xa6fe, 0xa740, 0xa77e, 0xa780, 0xa7fe, 0xa840, 0xa87e, 0xa880, 0xa8fe, 0xa940, 0xa97e, 0xa980, 0xa9fe, 0xaa40, 0xaa7e, 0xaa80, 0xaafe, 0xab40, 0xab7e, 0xab80, 0xabfe,
                        0xac40, 0xac7e, 0xac80, 0xacfe, 0xad40, 0xad7e, 0xad80, 0xadfe, 0xae40, 0xae7e, 0xae80, 0xaefe, 0xaf40, 0xaf7e, 0xaf80, 0xaffe, 0xb040, 0xb07e, 0xb080, 0xb0fe, 0xb140, 0xb17e,
                        0xb180, 0xb1fe, 0xb240, 0xb27e, 0xb280, 0xb2fe, 0xb340, 0xb37e, 0xb380, 0xb3fe, 0xb440, 0xb47e, 0xb480, 0xb4fe, 0xb540, 0xb57e, 0xb580, 0xb5fe, 0xb640, 0xb67e, 0xb680, 0xb6fe,
                        0xb740, 0xb77e, 0xb780, 0xb7fe, 0xb840, 0xb87e, 0xb880, 0xb8fe, 0xb940, 0xb97e, 0xb980, 0xb9fe, 0xba40, 0xba7e, 0xba80, 0xbafe, 0xbb40, 0xbb7e, 0xbb80, 0xbbfe, 0xbc40, 0xbc7e,
                        0xbc80, 0xbcfe, 0xbd40, 0xbd7e, 0xbd80, 0xbdfe, 0xbe40, 0xbe7e, 0xbe80, 0xbefe, 0xbf40, 0xbf7e, 0xbf80, 0xbffe, 0xc040, 0xc07e, 0xc080, 0xc0fe, 0xc140, 0xc17e, 0xc180, 0xc1fe,
                        0xc240, 0xc27e, 0xc280, 0xc2fe, 0xc340, 0xc37e, 0xc380, 0xc3fe, 0xc440, 0xc47e, 0xc480, 0xc4fe, 0xc540, 0xc57e, 0xc580, 0xc5fe, 0xc640, 0xc67e, 0xc680, 0xc6fe, 0xc740, 0xc77e,
                        0xc780, 0xc7fe, 0xc840, 0xc87e, 0xc880, 0xc8fe, 0xc940, 0xc97e, 0xc980, 0xc9fe, 0xca40, 0xca7e, 0xca80, 0xcafe, 0xcb40, 0xcb7e, 0xcb80, 0xcbfe, 0xcc40, 0xcc7e, 0xcc80, 0xccfe,
                        0xcd40, 0xcd7e, 0xcd80, 0xcdfe, 0xce40, 0xce7e, 0xce80, 0xcefe, 0xcf40, 0xcf7e, 0xcf80, 0xcffe, 0xd040, 0xd07e, 0xd080, 0xd0fe, 0xd140, 0xd17e, 0xd180, 0xd1fe, 0xd240, 0xd27e,
                        0xd280, 0xd2fe, 0xd340, 0xd37e, 0xd380, 0xd3fe, 0xd440, 0xd47e, 0xd480, 0xd4fe, 0xd540, 0xd57e, 0xd580, 0xd5fe, 0xd640, 0xd67e, 0xd680, 0xd6fe, 0xd740, 0xd77e, 0xd780, 0xd7fe,
                        0xd840, 0xd87e, 0xd880, 0xd8fe, 0xd940, 0xd97e, 0xd980, 0xd9fe, 0xda40, 0xda7e, 0xda80, 0xdafe, 0xdb40, 0xdb7e, 0xdb80, 0xdbfe, 0xdc40, 0xdc7e, 0xdc80, 0xdcfe, 0xdd40, 0xdd7e,
                        0xdd80, 0xddfe, 0xde40, 0xde7e, 0xde80, 0xdefe, 0xdf40, 0xdf7e, 0xdf80, 0xdffe, 0xe040, 0xe07e, 0xe080, 0xe0fe, 0xe140, 0xe17e, 0xe180, 0xe1fe, 0xe240, 0xe27e, 0xe280, 0xe2fe,
                        0xe340, 0xe37e, 0xe380, 0xe3fe, 0xe440, 0xe47e, 0xe480, 0xe4fe, 0xe540, 0xe57e, 0xe580, 0xe5fe, 0xe640, 0xe67e, 0xe680, 0xe6fe, 0xe740, 0xe77e, 0xe780, 0xe7fe, 0xe840, 0xe87e,
                        0xe880, 0xe8fe, 0xe940, 0xe97e, 0xe980, 0xe9fe, 0xea40, 0xea7e, 0xea80, 0xeafe, 0xeb40, 0xeb7e, 0xeb80, 0xebfe, 0xec40, 0xec7e, 0xec80, 0xecfe, 0xed40, 0xed7e, 0xed80, 0xedfe,
                        0xee40, 0xee7e, 0xee80, 0xeefe, 0xef40, 0xef7e, 0xef80, 0xeffe, 0xf040, 0xf07e, 0xf080, 0xf0fe, 0xf140, 0xf17e, 0xf180, 0xf1fe, 0xf240, 0xf27e, 0xf280, 0xf2fe, 0xf340, 0xf37e,
                        0xf380, 0xf3fe, 0xf440, 0xf47e, 0xf480, 0xf4fe, 0xf540, 0xf57e, 0xf580, 0xf5fe, 0xf640, 0xf67e, 0xf680, 0xf6fe, 0xf740, 0xf77e, 0xf780, 0xf7fe, 0xf840, 0xf87e, 0xf880, 0xf8fe,
                        0xf940, 0xf97e, 0xf980, 0xf9fe, 0xfa40, 0xfa7e, 0xfa80, 0xfafe, 0xfb40, 0xfb7e, 0xfb80, 0xfbfe, 0xfc40, 0xfc7e, 0xfc80, 0xfcfe, 0xfd40, 0xfd7e, 0xfd80, 0xfdfe, 0xfe40, 0xfe7e,
                        0xfe80, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198, 0x19a,
                                        0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6, 0x1c8,
                                        0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2, 0x1e4, 0x1e6, 0x1e8, 0x1ea, 0x1ec, 0x1ee, 0x1f0, 0x1f2, 0x1f4, 0x1f6,
                                        0x1f8, 0x1fa, 0x1fc, 0x1fe, 0x200, 0x202, 0x204, 0x206, 0x208, 0x20a, 0x20c, 0x20e, 0x210, 0x212, 0x214, 0x216, 0x218, 0x21a, 0x21c, 0x21e, 0x220, 0x222, 0x224,
                                        0x226, 0x228, 0x22a, 0x22c, 0x22e, 0x230, 0x232, 0x234, 0x236, 0x238, 0x23a, 0x23c, 0x23e, 0x240, 0x242, 0x244, 0x246, 0x248, 0x24a, 0x24c, 0x24e, 0x250, 0x252,
                                        0x254, 0x256, 0x258, 0x25a, 0x25c, 0x25e, 0x260, 0x262, 0x264, 0x266, 0x268, 0x26a, 0x26c, 0x26e, 0x270, 0x272, 0x274, 0x276, 0x278, 0x27a, 0x27c, 0x27e, 0x280,
                                        0x282, 0x284, 0x286, 0x288, 0x28a, 0x28c, 0x28e, 0x290, 0x292, 0x294, 0x296, 0x298, 0x29a, 0x29c, 0x29e, 0x2a0, 0x2a2, 0x2a4, 0x2a6, 0x2a8, 0x2aa, 0x2ac, 0x2ae,
                                        0x2b0, 0x2b2, 0x2b4, 0x2b6, 0x2b8, 0x2ba, 0x2bc, 0x2be, 0x2c0, 0x2c2, 0x2c4, 0x2c6, 0x2c8, 0x2ca, 0x2cc, 0x2ce, 0x2d0, 0x2d2, 0x2d4, 0x2d6, 0x2d8, 0x2da, 0x2dc,
                                        0x2de, 0x2e0, 0x2e2, 0x2e4, 0x2e6, 0x2e8, 0x2ea, 0x2ec, 0x2ee, 0x2f0, 0x2f2, 0x2f4, 0x2f6, 0x2f8, 0x2fa, 0x2fc, 0x2fe, 0x300, 0x302, 0x304, 0x306, 0x308, 0x30a,
                                        0x30c, 0x30e, 0x310, 0x312, 0x314, 0x316, 0x318, 0x31a, 0x31c, 0x31e, 0x320, 0x322, 0x324, 0x326, 0x328, 0x32a, 0x32c, 0x32e, 0x330, 0x332, 0x334, 0x336, 0x338,
                                        0x33a, 0x33c, 0x33e, 0x340, 0x342, 0x344, 0x346, 0x348, 0x34a, 0x34c, 0x34e, 0x350, 0x352, 0x354, 0x356, 0x358, 0x35a, 0x35c, 0x35e, 0x360, 0x362, 0x364, 0x366,
                                        0x368, 0x36a, 0x36c, 0x36e, 0x370, 0x372, 0x374, 0x376, 0x378, 0x37a, 0x37c, 0x37e, 0x380, 0x382, 0x384, 0x386, 0x388, 0x38a, 0x38c, 0x38e, 0x390, 0x392, 0x394,
                                        0x396, 0x398, 0x39a, 0x39c, 0x39e, 0x3a0, 0x3a2, 0x3a4, 0x3a6, 0x3a8, 0x3aa, 0x3ac, 0x3ae, 0x3b0, 0x3b2, 0x3b4, 0x3b6, 0x3b8, 0x3ba, 0x3bc, 0x3be, 0x3c0, 0x3c2,
                                        0x3c4, 0x3c6, 0x3c8, 0x3ca, 0x3cc, 0x3ce, 0x3d0, 0x3d2, 0x3d4, 0x3d6, 0x3d8, 0x3da, 0x3dc, 0x3de, 0x3e0, 0x3e2, 0x3e4, 0x3e6, 0x3e8, 0x3ea, 0x3ec, 0x3ee,
                                        0x3f0},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0x81, (byte) 0x40, (byte) 0x81, (byte) 0x7e, (byte) 0x81, (byte) 0x80, (byte) 0x81, (byte) 0xfe, (byte) 0x82, (byte) 0x40,
                                        (byte) 0x82, (byte) 0x7e, (byte) 0x82, (byte) 0x80, (byte) 0x82, (byte) 0xfe, (byte) 0x83, (byte) 0x40, (byte) 0x83, (byte) 0x7e, (byte) 0x83, (byte) 0x80,
                                        (byte) 0x83, (byte) 0xfe, (byte) 0x84, (byte) 0x40, (byte) 0x84, (byte) 0x7e, (byte) 0x84, (byte) 0x80, (byte) 0x84, (byte) 0xfe, (byte) 0x85, (byte) 0x40,
                                        (byte) 0x85, (byte) 0x7e, (byte) 0x85, (byte) 0x80, (byte) 0x85, (byte) 0xfe, (byte) 0x86, (byte) 0x40, (byte) 0x86, (byte) 0x7e, (byte) 0x86, (byte) 0x80,
                                        (byte) 0x86, (byte) 0xfe, (byte) 0x87, (byte) 0x40, (byte) 0x87, (byte) 0x7e, (byte) 0x87, (byte) 0x80, (byte) 0x87, (byte) 0xfe, (byte) 0x88, (byte) 0x40,
                                        (byte) 0x88, (byte) 0x7e, (byte) 0x88, (byte) 0x80, (byte) 0x88, (byte) 0xfe, (byte) 0x89, (byte) 0x40, (byte) 0x89, (byte) 0x7e, (byte) 0x89, (byte) 0x80,
                                        (byte) 0x89, (byte) 0xfe, (byte) 0x8a, (byte) 0x40, (byte) 0x8a, (byte) 0x7e, (byte) 0x8a, (byte) 0x80, (byte) 0x8a, (byte) 0xfe, (byte) 0x8b, (byte) 0x40,
                                        (byte) 0x8b, (byte) 0x7e, (byte) 0x8b, (byte) 0x80, (byte) 0x8b, (byte) 0xfe, (byte) 0x8c, (byte) 0x40, (byte) 0x8c, (byte) 0x7e, (byte) 0x8c, (byte) 0x80,
                                        (byte) 0x8c, (byte) 0xfe, (byte) 0x8d, (byte) 0x40, (byte) 0x8d, (byte) 0x7e, (byte) 0x8d, (byte) 0x80, (byte) 0x8d, (byte) 0xfe, (byte) 0x8e, (byte) 0x40,
                                        (byte) 0x8e, (byte) 0x7e, (byte) 0x8e, (byte) 0x80, (byte) 0x8e, (byte) 0xfe, (byte) 0x8f, (byte) 0x40, (byte) 0x8f, (byte) 0x7e, (byte) 0x8f, (byte) 0x80,
                                        (byte) 0x8f, (byte) 0xfe, (byte) 0x90, (byte) 0x40, (byte) 0x90, (byte) 0x7e, (byte) 0x90, (byte) 0x80, (byte) 0x90, (byte) 0xfe, (byte) 0x91, (byte) 0x40,
                                        (byte) 0x91, (byte) 0x7e, (byte) 0x91, (byte) 0x80, (byte) 0x91, (byte) 0xfe, (byte) 0x92, (byte) 0x40, (byte) 0x92, (byte) 0x7e, (byte) 0x92, (byte) 0x80,
                                        (byte) 0x92, (byte) 0xfe, (byte) 0x93, (byte) 0x40, (byte) 0x93, (byte) 0x7e, (byte) 0x93, (byte) 0x80, (byte) 0x93, (byte) 0xfe, (byte) 0x94, (byte) 0x40,
                                        (byte) 0x94, (byte) 0x7e, (byte) 0x94, (byte) 0x80, (byte) 0x94, (byte) 0xfe, (byte) 0x95, (byte) 0x40, (byte) 0x95, (byte) 0x7e, (byte) 0x95, (byte) 0x80,
                                        (byte) 0x95, (byte) 0xfe, (byte) 0x96, (byte) 0x40, (byte) 0x96, (byte) 0x7e, (byte) 0x96, (byte) 0x80, (byte) 0x96, (byte) 0xfe, (byte) 0x97, (byte) 0x40,
                                        (byte) 0x97, (byte) 0x7e, (byte) 0x97, (byte) 0x80, (byte) 0x97, (byte) 0xfe, (byte) 0x98, (byte) 0x40, (byte) 0x98, (byte) 0x7e, (byte) 0x98, (byte) 0x80,
                                        (byte) 0x98, (byte) 0xfe, (byte) 0x99, (byte) 0x40, (byte) 0x99, (byte) 0x7e, (byte) 0x99, (byte) 0x80, (byte) 0x99, (byte) 0xfe, (byte) 0x9a, (byte) 0x40,
                                        (byte) 0x9a, (byte) 0x7e, (byte) 0x9a, (byte) 0x80, (byte) 0x9a, (byte) 0xfe, (byte) 0x9b, (byte) 0x40, (byte) 0x9b, (byte) 0x7e, (byte) 0x9b, (byte) 0x80,
                                        (byte) 0x9b, (byte) 0xfe, (byte) 0x9c, (byte) 0x40, (byte) 0x9c, (byte) 0x7e, (byte) 0x9c, (byte) 0x80, (byte) 0x9c, (byte) 0xfe, (byte) 0x9d, (byte) 0x40,
                                        (byte) 0x9d, (byte) 0x7e, (byte) 0x9d, (byte) 0x80, (byte) 0x9d, (byte) 0xfe, (byte) 0x9e, (byte) 0x40, (byte) 0x9e, (byte) 0x7e, (byte) 0x9e, (byte) 0x80,
                                        (byte) 0x9e, (byte) 0xfe, (byte) 0x9f, (byte) 0x40, (byte) 0x9f, (byte) 0x7e, (byte) 0x9f, (byte) 0x80, (byte) 0x9f, (byte) 0xfe, (byte) 0xa0, (byte) 0x40,
                                        (byte) 0xa0, (byte) 0x7e, (byte) 0xa0, (byte) 0x80, (byte) 0xa0, (byte) 0xfe, (byte) 0xa1, (byte) 0x40, (byte) 0xa1, (byte) 0x7e, (byte) 0xa1, (byte) 0x80,
                                        (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0x40, (byte) 0xa2, (byte) 0x7e, (byte) 0xa2, (byte) 0x80, (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0x40,
                                        (byte) 0xa3, (byte) 0x7e, (byte) 0xa3, (byte) 0x80, (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0x40, (byte) 0xa4, (byte) 0x7e, (byte) 0xa4, (byte) 0x80,
                                        (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0x40, (byte) 0xa5, (byte) 0x7e, (byte) 0xa5, (byte) 0x80, (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0x40,
                                        (byte) 0xa6, (byte) 0x7e, (byte) 0xa6, (byte) 0x80, (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0x40, (byte) 0xa7, (byte) 0x7e, (byte) 0xa7, (byte) 0x80,
                                        (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0x40, (byte) 0xa8, (byte) 0x7e, (byte) 0xa8, (byte) 0x80, (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0x40,
                                        (byte) 0xa9, (byte) 0x7e, (byte) 0xa9, (byte) 0x80, (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0x40, (byte) 0xaa, (byte) 0x7e, (byte) 0xaa, (byte) 0x80,
                                        (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0x40, (byte) 0xab, (byte) 0x7e, (byte) 0xab, (byte) 0x80, (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0x40,
                                        (byte) 0xac, (byte) 0x7e, (byte) 0xac, (byte) 0x80, (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0x40, (byte) 0xad, (byte) 0x7e, (byte) 0xad, (byte) 0x80,
                                        (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0x40, (byte) 0xae, (byte) 0x7e, (byte) 0xae, (byte) 0x80, (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0x40,
                                        (byte) 0xaf, (byte) 0x7e, (byte) 0xaf, (byte) 0x80, (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0x40, (byte) 0xb0, (byte) 0x7e, (byte) 0xb0, (byte) 0x80,
                                        (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0x40, (byte) 0xb1, (byte) 0x7e, (byte) 0xb1, (byte) 0x80, (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0x40,
                                        (byte) 0xb2, (byte) 0x7e, (byte) 0xb2, (byte) 0x80, (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0x40, (byte) 0xb3, (byte) 0x7e, (byte) 0xb3, (byte) 0x80,
                                        (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0x40, (byte) 0xb4, (byte) 0x7e, (byte) 0xb4, (byte) 0x80, (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0x40,
                                        (byte) 0xb5, (byte) 0x7e, (byte) 0xb5, (byte) 0x80, (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0x40, (byte) 0xb6, (byte) 0x7e, (byte) 0xb6, (byte) 0x80,
                                        (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0x40, (byte) 0xb7, (byte) 0x7e, (byte) 0xb7, (byte) 0x80, (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0x40,
                                        (byte) 0xb8, (byte) 0x7e, (byte) 0xb8, (byte) 0x80, (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0x40, (byte) 0xb9, (byte) 0x7e, (byte) 0xb9, (byte) 0x80,
                                        (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0x40, (byte) 0xba, (byte) 0x7e, (byte) 0xba, (byte) 0x80, (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0x40,
                                        (byte) 0xbb, (byte) 0x7e, (byte) 0xbb, (byte) 0x80, (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0x40, (byte) 0xbc, (byte) 0x7e, (byte) 0xbc, (byte) 0x80,
                                        (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0x40, (byte) 0xbd, (byte) 0x7e, (byte) 0xbd, (byte) 0x80, (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0x40,
                                        (byte) 0xbe, (byte) 0x7e, (byte) 0xbe, (byte) 0x80, (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0x40, (byte) 0xbf, (byte) 0x7e, (byte) 0xbf, (byte) 0x80,
                                        (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0x40, (byte) 0xc0, (byte) 0x7e, (byte) 0xc0, (byte) 0x80, (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0x40,
                                        (byte) 0xc1, (byte) 0x7e, (byte) 0xc1, (byte) 0x80, (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0x40, (byte) 0xc2, (byte) 0x7e, (byte) 0xc2, (byte) 0x80,
                                        (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0x40, (byte) 0xc3, (byte) 0x7e, (byte) 0xc3, (byte) 0x80, (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0x40,
                                        (byte) 0xc4, (byte) 0x7e, (byte) 0xc4, (byte) 0x80, (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0x40, (byte) 0xc5, (byte) 0x7e, (byte) 0xc5, (byte) 0x80,
                                        (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0x40, (byte) 0xc6, (byte) 0x7e, (byte) 0xc6, (byte) 0x80, (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0x40,
                                        (byte) 0xc7, (byte) 0x7e, (byte) 0xc7, (byte) 0x80, (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0x40, (byte) 0xc8, (byte) 0x7e, (byte) 0xc8, (byte) 0x80,
                                        (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0x40, (byte) 0xc9, (byte) 0x7e, (byte) 0xc9, (byte) 0x80, (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0x40,
                                        (byte) 0xca, (byte) 0x7e, (byte) 0xca, (byte) 0x80, (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0x40, (byte) 0xcb, (byte) 0x7e, (byte) 0xcb, (byte) 0x80,
                                        (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0x40, (byte) 0xcc, (byte) 0x7e, (byte) 0xcc, (byte) 0x80, (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0x40,
                                        (byte) 0xcd, (byte) 0x7e, (byte) 0xcd, (byte) 0x80, (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0x40, (byte) 0xce, (byte) 0x7e, (byte) 0xce, (byte) 0x80,
                                        (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0x40, (byte) 0xcf, (byte) 0x7e, (byte) 0xcf, (byte) 0x80, (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0x40,
                                        (byte) 0xd0, (byte) 0x7e, (byte) 0xd0, (byte) 0x80, (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0x40, (byte) 0xd1, (byte) 0x7e, (byte) 0xd1, (byte) 0x80,
                                        (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0x40, (byte) 0xd2, (byte) 0x7e, (byte) 0xd2, (byte) 0x80, (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0x40,
                                        (byte) 0xd3, (byte) 0x7e, (byte) 0xd3, (byte) 0x80, (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0x40, (byte) 0xd4, (byte) 0x7e, (byte) 0xd4, (byte) 0x80,
                                        (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0x40, (byte) 0xd5, (byte) 0x7e, (byte) 0xd5, (byte) 0x80, (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0x40,
                                        (byte) 0xd6, (byte) 0x7e, (byte) 0xd6, (byte) 0x80, (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0x40, (byte) 0xd7, (byte) 0x7e, (byte) 0xd7, (byte) 0x80,
                                        (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0x40, (byte) 0xd8, (byte) 0x7e, (byte) 0xd8, (byte) 0x80, (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0x40,
                                        (byte) 0xd9, (byte) 0x7e, (byte) 0xd9, (byte) 0x80, (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0x40, (byte) 0xda, (byte) 0x7e, (byte) 0xda, (byte) 0x80,
                                        (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0x40, (byte) 0xdb, (byte) 0x7e, (byte) 0xdb, (byte) 0x80, (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0x40,
                                        (byte) 0xdc, (byte) 0x7e, (byte) 0xdc, (byte) 0x80, (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0x40, (byte) 0xdd, (byte) 0x7e, (byte) 0xdd, (byte) 0x80,
                                        (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0x40, (byte) 0xde, (byte) 0x7e, (byte) 0xde, (byte) 0x80, (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0x40,
                                        (byte) 0xdf, (byte) 0x7e, (byte) 0xdf, (byte) 0x80, (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0x80,
                                        (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0x40, (byte) 0xe1, (byte) 0x7e, (byte) 0xe1, (byte) 0x80, (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0x40,
                                        (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0x80, (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0x80,
                                        (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0x40, (byte) 0xe4, (byte) 0x7e, (byte) 0xe4, (byte) 0x80, (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0x40,
                                        (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0x80, (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0x80,
                                        (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0x40, (byte) 0xe7, (byte) 0x7e, (byte) 0xe7, (byte) 0x80, (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0x40,
                                        (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0x80, (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0x80,
                                        (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0x40, (byte) 0xea, (byte) 0x7e, (byte) 0xea, (byte) 0x80, (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0x40,
                                        (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0x80, (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0x80,
                                        (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0x40, (byte) 0xed, (byte) 0x7e, (byte) 0xed, (byte) 0x80, (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0x40,
                                        (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0x80, (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0x80,
                                        (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0x40, (byte) 0xf0, (byte) 0x7e, (byte) 0xf0, (byte) 0x80, (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0x40,
                                        (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0x80, (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0x80,
                                        (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0x40, (byte) 0xf3, (byte) 0x7e, (byte) 0xf3, (byte) 0x80, (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0x40,
                                        (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0x80, (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0x80,
                                        (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0x40, (byte) 0xf6, (byte) 0x7e, (byte) 0xf6, (byte) 0x80, (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0x40,
                                        (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0x80, (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0x80,
                                        (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0x40, (byte) 0xf9, (byte) 0x7e, (byte) 0xf9, (byte) 0x80, (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0x40,
                                        (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0x80, (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0x80,
                                        (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0x40, (byte) 0xfc, (byte) 0x7e, (byte) 0xfc, (byte) 0x80, (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0x40,
                                        (byte) 0xfd, (byte) 0x7e, (byte) 0xfd, (byte) 0x80, (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0x40, (byte) 0xfe, (byte) 0x7e, (byte) 0xfe, (byte) 0x80,
                                        (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataGB1988() {
        assert TruffleString.Encoding.GB1988.ordinal() == 27;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataGB2312() {
        assert TruffleString.Encoding.GB2312.ordinal() == 28;
        return new TestData(new int[]{0x0000, 0x007f, 0xa1a1, 0xa1fe, 0xa2a1, 0xa2fe, 0xa3a1, 0xa3fe, 0xa4a1, 0xa4fe, 0xa5a1, 0xa5fe, 0xa6a1, 0xa6fe, 0xa7a1, 0xa7fe, 0xa8a1, 0xa8fe, 0xa9a1, 0xa9fe,
                        0xaaa1, 0xaafe, 0xaba1, 0xabfe, 0xaca1, 0xacfe, 0xada1, 0xadfe, 0xaea1, 0xaefe, 0xafa1, 0xaffe, 0xb0a1, 0xb0fe, 0xb1a1, 0xb1fe, 0xb2a1, 0xb2fe, 0xb3a1, 0xb3fe, 0xb4a1, 0xb4fe,
                        0xb5a1, 0xb5fe, 0xb6a1, 0xb6fe, 0xb7a1, 0xb7fe, 0xb8a1, 0xb8fe, 0xb9a1, 0xb9fe, 0xbaa1, 0xbafe, 0xbba1, 0xbbfe, 0xbca1, 0xbcfe, 0xbda1, 0xbdfe, 0xbea1, 0xbefe, 0xbfa1, 0xbffe,
                        0xc0a1, 0xc0fe, 0xc1a1, 0xc1fe, 0xc2a1, 0xc2fe, 0xc3a1, 0xc3fe, 0xc4a1, 0xc4fe, 0xc5a1, 0xc5fe, 0xc6a1, 0xc6fe, 0xc7a1, 0xc7fe, 0xc8a1, 0xc8fe, 0xc9a1, 0xc9fe, 0xcaa1, 0xcafe,
                        0xcba1, 0xcbfe, 0xcca1, 0xccfe, 0xcda1, 0xcdfe, 0xcea1, 0xcefe, 0xcfa1, 0xcffe, 0xd0a1, 0xd0fe, 0xd1a1, 0xd1fe, 0xd2a1, 0xd2fe, 0xd3a1, 0xd3fe, 0xd4a1, 0xd4fe, 0xd5a1, 0xd5fe,
                        0xd6a1, 0xd6fe, 0xd7a1, 0xd7fe, 0xd8a1, 0xd8fe, 0xd9a1, 0xd9fe, 0xdaa1, 0xdafe, 0xdba1, 0xdbfe, 0xdca1, 0xdcfe, 0xdda1, 0xddfe, 0xdea1, 0xdefe, 0xdfa1, 0xdffe, 0xe0a1, 0xe0fe,
                        0xe1a1, 0xe1fe, 0xe2a1, 0xe2fe, 0xe3a1, 0xe3fe, 0xe4a1, 0xe4fe, 0xe5a1, 0xe5fe, 0xe6a1, 0xe6fe, 0xe7a1, 0xe7fe, 0xe8a1, 0xe8fe, 0xe9a1, 0xe9fe, 0xeaa1, 0xeafe, 0xeba1, 0xebfe,
                        0xeca1, 0xecfe, 0xeda1, 0xedfe, 0xeea1, 0xeefe, 0xefa1, 0xeffe, 0xf0a1, 0xf0fe, 0xf1a1, 0xf1fe, 0xf2a1, 0xf2fe, 0xf3a1, 0xf3fe, 0xf4a1, 0xf4fe, 0xf5a1, 0xf5fe, 0xf6a1, 0xf6fe,
                        0xf7a1, 0xf7fe, 0xf8a1, 0xf8fe, 0xf9a1, 0xf9fe, 0xfaa1, 0xfafe, 0xfba1, 0xfbfe, 0xfca1, 0xfcfe, 0xfda1, 0xfdfe, 0xfea1, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0xa1, (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0xa1, (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0xa1,
                                        (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0xa1, (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0xa1, (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0xa1,
                                        (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0xa1, (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0xa1, (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0xa1,
                                        (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0xa1, (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0xa1, (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0xa1,
                                        (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0xa1, (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0xa1, (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0xa1,
                                        (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0xa1, (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0xa1, (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0xa1,
                                        (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0xa1, (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0xa1, (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0xa1,
                                        (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0xa1, (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0xa1, (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0xa1,
                                        (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0xa1, (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0xa1, (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0xa1,
                                        (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0xa1, (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0xa1, (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0xa1,
                                        (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0xa1, (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0xa1, (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0xa1,
                                        (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0xa1, (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0xa1, (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0xa1,
                                        (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0xa1, (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0xa1, (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0xa1,
                                        (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0xa1, (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0xa1, (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0xa1,
                                        (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0xa1, (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0xa1, (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0xa1,
                                        (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0xa1, (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0xa1, (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0xa1,
                                        (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0xa1, (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0xa1, (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0xa1,
                                        (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0xa1, (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0xa1, (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0xa1,
                                        (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0xa1, (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0xa1, (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0xa1,
                                        (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0xa1, (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0xa1, (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0xa1,
                                        (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0xa1, (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0xa1, (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0xa1,
                                        (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0xa1, (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0xa1, (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0xa1,
                                        (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0xa1, (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0xa1, (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0xa1,
                                        (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0xa1, (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0xa1, (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0xa1,
                                        (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0xa1, (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0xa1, (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0xa1,
                                        (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0xa1, (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0xa1, (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0xa1,
                                        (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0xa1, (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0xa1, (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0xa1,
                                        (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0xa1, (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0xa1, (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0xa1,
                                        (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0xa1, (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0xa1, (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0xa1,
                                        (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0xa1, (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0xa1, (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0xa1,
                                        (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0xa1, (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0xa1, (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0xa1,
                                        (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0xa1, (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataGBK() {
        assert TruffleString.Encoding.GBK.ordinal() == 29;
        return new TestData(new int[]{0x0000, 0x0080, 0x8140, 0x817e, 0x8180, 0x81fe, 0x8240, 0x827e, 0x8280, 0x82fe, 0x8340, 0x837e, 0x8380, 0x83fe, 0x8440, 0x847e, 0x8480, 0x84fe, 0x8540, 0x857e,
                        0x8580, 0x85fe, 0x8640, 0x867e, 0x8680, 0x86fe, 0x8740, 0x877e, 0x8780, 0x87fe, 0x8840, 0x887e, 0x8880, 0x88fe, 0x8940, 0x897e, 0x8980, 0x89fe, 0x8a40, 0x8a7e, 0x8a80, 0x8afe,
                        0x8b40, 0x8b7e, 0x8b80, 0x8bfe, 0x8c40, 0x8c7e, 0x8c80, 0x8cfe, 0x8d40, 0x8d7e, 0x8d80, 0x8dfe, 0x8e40, 0x8e7e, 0x8e80, 0x8efe, 0x8f40, 0x8f7e, 0x8f80, 0x8ffe, 0x9040, 0x907e,
                        0x9080, 0x90fe, 0x9140, 0x917e, 0x9180, 0x91fe, 0x9240, 0x927e, 0x9280, 0x92fe, 0x9340, 0x937e, 0x9380, 0x93fe, 0x9440, 0x947e, 0x9480, 0x94fe, 0x9540, 0x957e, 0x9580, 0x95fe,
                        0x9640, 0x967e, 0x9680, 0x96fe, 0x9740, 0x977e, 0x9780, 0x97fe, 0x9840, 0x987e, 0x9880, 0x98fe, 0x9940, 0x997e, 0x9980, 0x99fe, 0x9a40, 0x9a7e, 0x9a80, 0x9afe, 0x9b40, 0x9b7e,
                        0x9b80, 0x9bfe, 0x9c40, 0x9c7e, 0x9c80, 0x9cfe, 0x9d40, 0x9d7e, 0x9d80, 0x9dfe, 0x9e40, 0x9e7e, 0x9e80, 0x9efe, 0x9f40, 0x9f7e, 0x9f80, 0x9ffe, 0xa040, 0xa07e, 0xa080, 0xa0fe,
                        0xa140, 0xa17e, 0xa180, 0xa1fe, 0xa240, 0xa27e, 0xa280, 0xa2fe, 0xa340, 0xa37e, 0xa380, 0xa3fe, 0xa440, 0xa47e, 0xa480, 0xa4fe, 0xa540, 0xa57e, 0xa580, 0xa5fe, 0xa640, 0xa67e,
                        0xa680, 0xa6fe, 0xa740, 0xa77e, 0xa780, 0xa7fe, 0xa840, 0xa87e, 0xa880, 0xa8fe, 0xa940, 0xa97e, 0xa980, 0xa9fe, 0xaa40, 0xaa7e, 0xaa80, 0xaafe, 0xab40, 0xab7e, 0xab80, 0xabfe,
                        0xac40, 0xac7e, 0xac80, 0xacfe, 0xad40, 0xad7e, 0xad80, 0xadfe, 0xae40, 0xae7e, 0xae80, 0xaefe, 0xaf40, 0xaf7e, 0xaf80, 0xaffe, 0xb040, 0xb07e, 0xb080, 0xb0fe, 0xb140, 0xb17e,
                        0xb180, 0xb1fe, 0xb240, 0xb27e, 0xb280, 0xb2fe, 0xb340, 0xb37e, 0xb380, 0xb3fe, 0xb440, 0xb47e, 0xb480, 0xb4fe, 0xb540, 0xb57e, 0xb580, 0xb5fe, 0xb640, 0xb67e, 0xb680, 0xb6fe,
                        0xb740, 0xb77e, 0xb780, 0xb7fe, 0xb840, 0xb87e, 0xb880, 0xb8fe, 0xb940, 0xb97e, 0xb980, 0xb9fe, 0xba40, 0xba7e, 0xba80, 0xbafe, 0xbb40, 0xbb7e, 0xbb80, 0xbbfe, 0xbc40, 0xbc7e,
                        0xbc80, 0xbcfe, 0xbd40, 0xbd7e, 0xbd80, 0xbdfe, 0xbe40, 0xbe7e, 0xbe80, 0xbefe, 0xbf40, 0xbf7e, 0xbf80, 0xbffe, 0xc040, 0xc07e, 0xc080, 0xc0fe, 0xc140, 0xc17e, 0xc180, 0xc1fe,
                        0xc240, 0xc27e, 0xc280, 0xc2fe, 0xc340, 0xc37e, 0xc380, 0xc3fe, 0xc440, 0xc47e, 0xc480, 0xc4fe, 0xc540, 0xc57e, 0xc580, 0xc5fe, 0xc640, 0xc67e, 0xc680, 0xc6fe, 0xc740, 0xc77e,
                        0xc780, 0xc7fe, 0xc840, 0xc87e, 0xc880, 0xc8fe, 0xc940, 0xc97e, 0xc980, 0xc9fe, 0xca40, 0xca7e, 0xca80, 0xcafe, 0xcb40, 0xcb7e, 0xcb80, 0xcbfe, 0xcc40, 0xcc7e, 0xcc80, 0xccfe,
                        0xcd40, 0xcd7e, 0xcd80, 0xcdfe, 0xce40, 0xce7e, 0xce80, 0xcefe, 0xcf40, 0xcf7e, 0xcf80, 0xcffe, 0xd040, 0xd07e, 0xd080, 0xd0fe, 0xd140, 0xd17e, 0xd180, 0xd1fe, 0xd240, 0xd27e,
                        0xd280, 0xd2fe, 0xd340, 0xd37e, 0xd380, 0xd3fe, 0xd440, 0xd47e, 0xd480, 0xd4fe, 0xd540, 0xd57e, 0xd580, 0xd5fe, 0xd640, 0xd67e, 0xd680, 0xd6fe, 0xd740, 0xd77e, 0xd780, 0xd7fe,
                        0xd840, 0xd87e, 0xd880, 0xd8fe, 0xd940, 0xd97e, 0xd980, 0xd9fe, 0xda40, 0xda7e, 0xda80, 0xdafe, 0xdb40, 0xdb7e, 0xdb80, 0xdbfe, 0xdc40, 0xdc7e, 0xdc80, 0xdcfe, 0xdd40, 0xdd7e,
                        0xdd80, 0xddfe, 0xde40, 0xde7e, 0xde80, 0xdefe, 0xdf40, 0xdf7e, 0xdf80, 0xdffe, 0xe040, 0xe07e, 0xe080, 0xe0fe, 0xe140, 0xe17e, 0xe180, 0xe1fe, 0xe240, 0xe27e, 0xe280, 0xe2fe,
                        0xe340, 0xe37e, 0xe380, 0xe3fe, 0xe440, 0xe47e, 0xe480, 0xe4fe, 0xe540, 0xe57e, 0xe580, 0xe5fe, 0xe640, 0xe67e, 0xe680, 0xe6fe, 0xe740, 0xe77e, 0xe780, 0xe7fe, 0xe840, 0xe87e,
                        0xe880, 0xe8fe, 0xe940, 0xe97e, 0xe980, 0xe9fe, 0xea40, 0xea7e, 0xea80, 0xeafe, 0xeb40, 0xeb7e, 0xeb80, 0xebfe, 0xec40, 0xec7e, 0xec80, 0xecfe, 0xed40, 0xed7e, 0xed80, 0xedfe,
                        0xee40, 0xee7e, 0xee80, 0xeefe, 0xef40, 0xef7e, 0xef80, 0xeffe, 0xf040, 0xf07e, 0xf080, 0xf0fe, 0xf140, 0xf17e, 0xf180, 0xf1fe, 0xf240, 0xf27e, 0xf280, 0xf2fe, 0xf340, 0xf37e,
                        0xf380, 0xf3fe, 0xf440, 0xf47e, 0xf480, 0xf4fe, 0xf540, 0xf57e, 0xf580, 0xf5fe, 0xf640, 0xf67e, 0xf680, 0xf6fe, 0xf740, 0xf77e, 0xf780, 0xf7fe, 0xf840, 0xf87e, 0xf880, 0xf8fe,
                        0xf940, 0xf97e, 0xf980, 0xf9fe, 0xfa40, 0xfa7e, 0xfa80, 0xfafe, 0xfb40, 0xfb7e, 0xfb80, 0xfbfe, 0xfc40, 0xfc7e, 0xfc80, 0xfcfe, 0xfd40, 0xfd7e, 0xfd80, 0xfdfe, 0xfe40, 0xfe7e,
                        0xfe80, 0xfefe},
                        new int[]{0x000, 0x001, 0x002, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028, 0x02a,
                                        0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056, 0x058,
                                        0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084, 0x086,
                                        0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2, 0x0b4,
                                        0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0, 0x0e2,
                                        0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e, 0x110,
                                        0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c, 0x13e,
                                        0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a, 0x16c,
                                        0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198, 0x19a,
                                        0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6, 0x1c8,
                                        0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2, 0x1e4, 0x1e6, 0x1e8, 0x1ea, 0x1ec, 0x1ee, 0x1f0, 0x1f2, 0x1f4, 0x1f6,
                                        0x1f8, 0x1fa, 0x1fc, 0x1fe, 0x200, 0x202, 0x204, 0x206, 0x208, 0x20a, 0x20c, 0x20e, 0x210, 0x212, 0x214, 0x216, 0x218, 0x21a, 0x21c, 0x21e, 0x220, 0x222, 0x224,
                                        0x226, 0x228, 0x22a, 0x22c, 0x22e, 0x230, 0x232, 0x234, 0x236, 0x238, 0x23a, 0x23c, 0x23e, 0x240, 0x242, 0x244, 0x246, 0x248, 0x24a, 0x24c, 0x24e, 0x250, 0x252,
                                        0x254, 0x256, 0x258, 0x25a, 0x25c, 0x25e, 0x260, 0x262, 0x264, 0x266, 0x268, 0x26a, 0x26c, 0x26e, 0x270, 0x272, 0x274, 0x276, 0x278, 0x27a, 0x27c, 0x27e, 0x280,
                                        0x282, 0x284, 0x286, 0x288, 0x28a, 0x28c, 0x28e, 0x290, 0x292, 0x294, 0x296, 0x298, 0x29a, 0x29c, 0x29e, 0x2a0, 0x2a2, 0x2a4, 0x2a6, 0x2a8, 0x2aa, 0x2ac, 0x2ae,
                                        0x2b0, 0x2b2, 0x2b4, 0x2b6, 0x2b8, 0x2ba, 0x2bc, 0x2be, 0x2c0, 0x2c2, 0x2c4, 0x2c6, 0x2c8, 0x2ca, 0x2cc, 0x2ce, 0x2d0, 0x2d2, 0x2d4, 0x2d6, 0x2d8, 0x2da, 0x2dc,
                                        0x2de, 0x2e0, 0x2e2, 0x2e4, 0x2e6, 0x2e8, 0x2ea, 0x2ec, 0x2ee, 0x2f0, 0x2f2, 0x2f4, 0x2f6, 0x2f8, 0x2fa, 0x2fc, 0x2fe, 0x300, 0x302, 0x304, 0x306, 0x308, 0x30a,
                                        0x30c, 0x30e, 0x310, 0x312, 0x314, 0x316, 0x318, 0x31a, 0x31c, 0x31e, 0x320, 0x322, 0x324, 0x326, 0x328, 0x32a, 0x32c, 0x32e, 0x330, 0x332, 0x334, 0x336, 0x338,
                                        0x33a, 0x33c, 0x33e, 0x340, 0x342, 0x344, 0x346, 0x348, 0x34a, 0x34c, 0x34e, 0x350, 0x352, 0x354, 0x356, 0x358, 0x35a, 0x35c, 0x35e, 0x360, 0x362, 0x364, 0x366,
                                        0x368, 0x36a, 0x36c, 0x36e, 0x370, 0x372, 0x374, 0x376, 0x378, 0x37a, 0x37c, 0x37e, 0x380, 0x382, 0x384, 0x386, 0x388, 0x38a, 0x38c, 0x38e, 0x390, 0x392, 0x394,
                                        0x396, 0x398, 0x39a, 0x39c, 0x39e, 0x3a0, 0x3a2, 0x3a4, 0x3a6, 0x3a8, 0x3aa, 0x3ac, 0x3ae, 0x3b0, 0x3b2, 0x3b4, 0x3b6, 0x3b8, 0x3ba, 0x3bc, 0x3be, 0x3c0, 0x3c2,
                                        0x3c4, 0x3c6, 0x3c8, 0x3ca, 0x3cc, 0x3ce, 0x3d0, 0x3d2, 0x3d4, 0x3d6, 0x3d8, 0x3da, 0x3dc, 0x3de, 0x3e0, 0x3e2, 0x3e4, 0x3e6, 0x3e8, 0x3ea, 0x3ec, 0x3ee,
                                        0x3f0},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x80, (byte) 0x81, (byte) 0x40, (byte) 0x81, (byte) 0x7e, (byte) 0x81, (byte) 0x80, (byte) 0x81, (byte) 0xfe, (byte) 0x82, (byte) 0x40,
                                        (byte) 0x82, (byte) 0x7e, (byte) 0x82, (byte) 0x80, (byte) 0x82, (byte) 0xfe, (byte) 0x83, (byte) 0x40, (byte) 0x83, (byte) 0x7e, (byte) 0x83, (byte) 0x80,
                                        (byte) 0x83, (byte) 0xfe, (byte) 0x84, (byte) 0x40, (byte) 0x84, (byte) 0x7e, (byte) 0x84, (byte) 0x80, (byte) 0x84, (byte) 0xfe, (byte) 0x85, (byte) 0x40,
                                        (byte) 0x85, (byte) 0x7e, (byte) 0x85, (byte) 0x80, (byte) 0x85, (byte) 0xfe, (byte) 0x86, (byte) 0x40, (byte) 0x86, (byte) 0x7e, (byte) 0x86, (byte) 0x80,
                                        (byte) 0x86, (byte) 0xfe, (byte) 0x87, (byte) 0x40, (byte) 0x87, (byte) 0x7e, (byte) 0x87, (byte) 0x80, (byte) 0x87, (byte) 0xfe, (byte) 0x88, (byte) 0x40,
                                        (byte) 0x88, (byte) 0x7e, (byte) 0x88, (byte) 0x80, (byte) 0x88, (byte) 0xfe, (byte) 0x89, (byte) 0x40, (byte) 0x89, (byte) 0x7e, (byte) 0x89, (byte) 0x80,
                                        (byte) 0x89, (byte) 0xfe, (byte) 0x8a, (byte) 0x40, (byte) 0x8a, (byte) 0x7e, (byte) 0x8a, (byte) 0x80, (byte) 0x8a, (byte) 0xfe, (byte) 0x8b, (byte) 0x40,
                                        (byte) 0x8b, (byte) 0x7e, (byte) 0x8b, (byte) 0x80, (byte) 0x8b, (byte) 0xfe, (byte) 0x8c, (byte) 0x40, (byte) 0x8c, (byte) 0x7e, (byte) 0x8c, (byte) 0x80,
                                        (byte) 0x8c, (byte) 0xfe, (byte) 0x8d, (byte) 0x40, (byte) 0x8d, (byte) 0x7e, (byte) 0x8d, (byte) 0x80, (byte) 0x8d, (byte) 0xfe, (byte) 0x8e, (byte) 0x40,
                                        (byte) 0x8e, (byte) 0x7e, (byte) 0x8e, (byte) 0x80, (byte) 0x8e, (byte) 0xfe, (byte) 0x8f, (byte) 0x40, (byte) 0x8f, (byte) 0x7e, (byte) 0x8f, (byte) 0x80,
                                        (byte) 0x8f, (byte) 0xfe, (byte) 0x90, (byte) 0x40, (byte) 0x90, (byte) 0x7e, (byte) 0x90, (byte) 0x80, (byte) 0x90, (byte) 0xfe, (byte) 0x91, (byte) 0x40,
                                        (byte) 0x91, (byte) 0x7e, (byte) 0x91, (byte) 0x80, (byte) 0x91, (byte) 0xfe, (byte) 0x92, (byte) 0x40, (byte) 0x92, (byte) 0x7e, (byte) 0x92, (byte) 0x80,
                                        (byte) 0x92, (byte) 0xfe, (byte) 0x93, (byte) 0x40, (byte) 0x93, (byte) 0x7e, (byte) 0x93, (byte) 0x80, (byte) 0x93, (byte) 0xfe, (byte) 0x94, (byte) 0x40,
                                        (byte) 0x94, (byte) 0x7e, (byte) 0x94, (byte) 0x80, (byte) 0x94, (byte) 0xfe, (byte) 0x95, (byte) 0x40, (byte) 0x95, (byte) 0x7e, (byte) 0x95, (byte) 0x80,
                                        (byte) 0x95, (byte) 0xfe, (byte) 0x96, (byte) 0x40, (byte) 0x96, (byte) 0x7e, (byte) 0x96, (byte) 0x80, (byte) 0x96, (byte) 0xfe, (byte) 0x97, (byte) 0x40,
                                        (byte) 0x97, (byte) 0x7e, (byte) 0x97, (byte) 0x80, (byte) 0x97, (byte) 0xfe, (byte) 0x98, (byte) 0x40, (byte) 0x98, (byte) 0x7e, (byte) 0x98, (byte) 0x80,
                                        (byte) 0x98, (byte) 0xfe, (byte) 0x99, (byte) 0x40, (byte) 0x99, (byte) 0x7e, (byte) 0x99, (byte) 0x80, (byte) 0x99, (byte) 0xfe, (byte) 0x9a, (byte) 0x40,
                                        (byte) 0x9a, (byte) 0x7e, (byte) 0x9a, (byte) 0x80, (byte) 0x9a, (byte) 0xfe, (byte) 0x9b, (byte) 0x40, (byte) 0x9b, (byte) 0x7e, (byte) 0x9b, (byte) 0x80,
                                        (byte) 0x9b, (byte) 0xfe, (byte) 0x9c, (byte) 0x40, (byte) 0x9c, (byte) 0x7e, (byte) 0x9c, (byte) 0x80, (byte) 0x9c, (byte) 0xfe, (byte) 0x9d, (byte) 0x40,
                                        (byte) 0x9d, (byte) 0x7e, (byte) 0x9d, (byte) 0x80, (byte) 0x9d, (byte) 0xfe, (byte) 0x9e, (byte) 0x40, (byte) 0x9e, (byte) 0x7e, (byte) 0x9e, (byte) 0x80,
                                        (byte) 0x9e, (byte) 0xfe, (byte) 0x9f, (byte) 0x40, (byte) 0x9f, (byte) 0x7e, (byte) 0x9f, (byte) 0x80, (byte) 0x9f, (byte) 0xfe, (byte) 0xa0, (byte) 0x40,
                                        (byte) 0xa0, (byte) 0x7e, (byte) 0xa0, (byte) 0x80, (byte) 0xa0, (byte) 0xfe, (byte) 0xa1, (byte) 0x40, (byte) 0xa1, (byte) 0x7e, (byte) 0xa1, (byte) 0x80,
                                        (byte) 0xa1, (byte) 0xfe, (byte) 0xa2, (byte) 0x40, (byte) 0xa2, (byte) 0x7e, (byte) 0xa2, (byte) 0x80, (byte) 0xa2, (byte) 0xfe, (byte) 0xa3, (byte) 0x40,
                                        (byte) 0xa3, (byte) 0x7e, (byte) 0xa3, (byte) 0x80, (byte) 0xa3, (byte) 0xfe, (byte) 0xa4, (byte) 0x40, (byte) 0xa4, (byte) 0x7e, (byte) 0xa4, (byte) 0x80,
                                        (byte) 0xa4, (byte) 0xfe, (byte) 0xa5, (byte) 0x40, (byte) 0xa5, (byte) 0x7e, (byte) 0xa5, (byte) 0x80, (byte) 0xa5, (byte) 0xfe, (byte) 0xa6, (byte) 0x40,
                                        (byte) 0xa6, (byte) 0x7e, (byte) 0xa6, (byte) 0x80, (byte) 0xa6, (byte) 0xfe, (byte) 0xa7, (byte) 0x40, (byte) 0xa7, (byte) 0x7e, (byte) 0xa7, (byte) 0x80,
                                        (byte) 0xa7, (byte) 0xfe, (byte) 0xa8, (byte) 0x40, (byte) 0xa8, (byte) 0x7e, (byte) 0xa8, (byte) 0x80, (byte) 0xa8, (byte) 0xfe, (byte) 0xa9, (byte) 0x40,
                                        (byte) 0xa9, (byte) 0x7e, (byte) 0xa9, (byte) 0x80, (byte) 0xa9, (byte) 0xfe, (byte) 0xaa, (byte) 0x40, (byte) 0xaa, (byte) 0x7e, (byte) 0xaa, (byte) 0x80,
                                        (byte) 0xaa, (byte) 0xfe, (byte) 0xab, (byte) 0x40, (byte) 0xab, (byte) 0x7e, (byte) 0xab, (byte) 0x80, (byte) 0xab, (byte) 0xfe, (byte) 0xac, (byte) 0x40,
                                        (byte) 0xac, (byte) 0x7e, (byte) 0xac, (byte) 0x80, (byte) 0xac, (byte) 0xfe, (byte) 0xad, (byte) 0x40, (byte) 0xad, (byte) 0x7e, (byte) 0xad, (byte) 0x80,
                                        (byte) 0xad, (byte) 0xfe, (byte) 0xae, (byte) 0x40, (byte) 0xae, (byte) 0x7e, (byte) 0xae, (byte) 0x80, (byte) 0xae, (byte) 0xfe, (byte) 0xaf, (byte) 0x40,
                                        (byte) 0xaf, (byte) 0x7e, (byte) 0xaf, (byte) 0x80, (byte) 0xaf, (byte) 0xfe, (byte) 0xb0, (byte) 0x40, (byte) 0xb0, (byte) 0x7e, (byte) 0xb0, (byte) 0x80,
                                        (byte) 0xb0, (byte) 0xfe, (byte) 0xb1, (byte) 0x40, (byte) 0xb1, (byte) 0x7e, (byte) 0xb1, (byte) 0x80, (byte) 0xb1, (byte) 0xfe, (byte) 0xb2, (byte) 0x40,
                                        (byte) 0xb2, (byte) 0x7e, (byte) 0xb2, (byte) 0x80, (byte) 0xb2, (byte) 0xfe, (byte) 0xb3, (byte) 0x40, (byte) 0xb3, (byte) 0x7e, (byte) 0xb3, (byte) 0x80,
                                        (byte) 0xb3, (byte) 0xfe, (byte) 0xb4, (byte) 0x40, (byte) 0xb4, (byte) 0x7e, (byte) 0xb4, (byte) 0x80, (byte) 0xb4, (byte) 0xfe, (byte) 0xb5, (byte) 0x40,
                                        (byte) 0xb5, (byte) 0x7e, (byte) 0xb5, (byte) 0x80, (byte) 0xb5, (byte) 0xfe, (byte) 0xb6, (byte) 0x40, (byte) 0xb6, (byte) 0x7e, (byte) 0xb6, (byte) 0x80,
                                        (byte) 0xb6, (byte) 0xfe, (byte) 0xb7, (byte) 0x40, (byte) 0xb7, (byte) 0x7e, (byte) 0xb7, (byte) 0x80, (byte) 0xb7, (byte) 0xfe, (byte) 0xb8, (byte) 0x40,
                                        (byte) 0xb8, (byte) 0x7e, (byte) 0xb8, (byte) 0x80, (byte) 0xb8, (byte) 0xfe, (byte) 0xb9, (byte) 0x40, (byte) 0xb9, (byte) 0x7e, (byte) 0xb9, (byte) 0x80,
                                        (byte) 0xb9, (byte) 0xfe, (byte) 0xba, (byte) 0x40, (byte) 0xba, (byte) 0x7e, (byte) 0xba, (byte) 0x80, (byte) 0xba, (byte) 0xfe, (byte) 0xbb, (byte) 0x40,
                                        (byte) 0xbb, (byte) 0x7e, (byte) 0xbb, (byte) 0x80, (byte) 0xbb, (byte) 0xfe, (byte) 0xbc, (byte) 0x40, (byte) 0xbc, (byte) 0x7e, (byte) 0xbc, (byte) 0x80,
                                        (byte) 0xbc, (byte) 0xfe, (byte) 0xbd, (byte) 0x40, (byte) 0xbd, (byte) 0x7e, (byte) 0xbd, (byte) 0x80, (byte) 0xbd, (byte) 0xfe, (byte) 0xbe, (byte) 0x40,
                                        (byte) 0xbe, (byte) 0x7e, (byte) 0xbe, (byte) 0x80, (byte) 0xbe, (byte) 0xfe, (byte) 0xbf, (byte) 0x40, (byte) 0xbf, (byte) 0x7e, (byte) 0xbf, (byte) 0x80,
                                        (byte) 0xbf, (byte) 0xfe, (byte) 0xc0, (byte) 0x40, (byte) 0xc0, (byte) 0x7e, (byte) 0xc0, (byte) 0x80, (byte) 0xc0, (byte) 0xfe, (byte) 0xc1, (byte) 0x40,
                                        (byte) 0xc1, (byte) 0x7e, (byte) 0xc1, (byte) 0x80, (byte) 0xc1, (byte) 0xfe, (byte) 0xc2, (byte) 0x40, (byte) 0xc2, (byte) 0x7e, (byte) 0xc2, (byte) 0x80,
                                        (byte) 0xc2, (byte) 0xfe, (byte) 0xc3, (byte) 0x40, (byte) 0xc3, (byte) 0x7e, (byte) 0xc3, (byte) 0x80, (byte) 0xc3, (byte) 0xfe, (byte) 0xc4, (byte) 0x40,
                                        (byte) 0xc4, (byte) 0x7e, (byte) 0xc4, (byte) 0x80, (byte) 0xc4, (byte) 0xfe, (byte) 0xc5, (byte) 0x40, (byte) 0xc5, (byte) 0x7e, (byte) 0xc5, (byte) 0x80,
                                        (byte) 0xc5, (byte) 0xfe, (byte) 0xc6, (byte) 0x40, (byte) 0xc6, (byte) 0x7e, (byte) 0xc6, (byte) 0x80, (byte) 0xc6, (byte) 0xfe, (byte) 0xc7, (byte) 0x40,
                                        (byte) 0xc7, (byte) 0x7e, (byte) 0xc7, (byte) 0x80, (byte) 0xc7, (byte) 0xfe, (byte) 0xc8, (byte) 0x40, (byte) 0xc8, (byte) 0x7e, (byte) 0xc8, (byte) 0x80,
                                        (byte) 0xc8, (byte) 0xfe, (byte) 0xc9, (byte) 0x40, (byte) 0xc9, (byte) 0x7e, (byte) 0xc9, (byte) 0x80, (byte) 0xc9, (byte) 0xfe, (byte) 0xca, (byte) 0x40,
                                        (byte) 0xca, (byte) 0x7e, (byte) 0xca, (byte) 0x80, (byte) 0xca, (byte) 0xfe, (byte) 0xcb, (byte) 0x40, (byte) 0xcb, (byte) 0x7e, (byte) 0xcb, (byte) 0x80,
                                        (byte) 0xcb, (byte) 0xfe, (byte) 0xcc, (byte) 0x40, (byte) 0xcc, (byte) 0x7e, (byte) 0xcc, (byte) 0x80, (byte) 0xcc, (byte) 0xfe, (byte) 0xcd, (byte) 0x40,
                                        (byte) 0xcd, (byte) 0x7e, (byte) 0xcd, (byte) 0x80, (byte) 0xcd, (byte) 0xfe, (byte) 0xce, (byte) 0x40, (byte) 0xce, (byte) 0x7e, (byte) 0xce, (byte) 0x80,
                                        (byte) 0xce, (byte) 0xfe, (byte) 0xcf, (byte) 0x40, (byte) 0xcf, (byte) 0x7e, (byte) 0xcf, (byte) 0x80, (byte) 0xcf, (byte) 0xfe, (byte) 0xd0, (byte) 0x40,
                                        (byte) 0xd0, (byte) 0x7e, (byte) 0xd0, (byte) 0x80, (byte) 0xd0, (byte) 0xfe, (byte) 0xd1, (byte) 0x40, (byte) 0xd1, (byte) 0x7e, (byte) 0xd1, (byte) 0x80,
                                        (byte) 0xd1, (byte) 0xfe, (byte) 0xd2, (byte) 0x40, (byte) 0xd2, (byte) 0x7e, (byte) 0xd2, (byte) 0x80, (byte) 0xd2, (byte) 0xfe, (byte) 0xd3, (byte) 0x40,
                                        (byte) 0xd3, (byte) 0x7e, (byte) 0xd3, (byte) 0x80, (byte) 0xd3, (byte) 0xfe, (byte) 0xd4, (byte) 0x40, (byte) 0xd4, (byte) 0x7e, (byte) 0xd4, (byte) 0x80,
                                        (byte) 0xd4, (byte) 0xfe, (byte) 0xd5, (byte) 0x40, (byte) 0xd5, (byte) 0x7e, (byte) 0xd5, (byte) 0x80, (byte) 0xd5, (byte) 0xfe, (byte) 0xd6, (byte) 0x40,
                                        (byte) 0xd6, (byte) 0x7e, (byte) 0xd6, (byte) 0x80, (byte) 0xd6, (byte) 0xfe, (byte) 0xd7, (byte) 0x40, (byte) 0xd7, (byte) 0x7e, (byte) 0xd7, (byte) 0x80,
                                        (byte) 0xd7, (byte) 0xfe, (byte) 0xd8, (byte) 0x40, (byte) 0xd8, (byte) 0x7e, (byte) 0xd8, (byte) 0x80, (byte) 0xd8, (byte) 0xfe, (byte) 0xd9, (byte) 0x40,
                                        (byte) 0xd9, (byte) 0x7e, (byte) 0xd9, (byte) 0x80, (byte) 0xd9, (byte) 0xfe, (byte) 0xda, (byte) 0x40, (byte) 0xda, (byte) 0x7e, (byte) 0xda, (byte) 0x80,
                                        (byte) 0xda, (byte) 0xfe, (byte) 0xdb, (byte) 0x40, (byte) 0xdb, (byte) 0x7e, (byte) 0xdb, (byte) 0x80, (byte) 0xdb, (byte) 0xfe, (byte) 0xdc, (byte) 0x40,
                                        (byte) 0xdc, (byte) 0x7e, (byte) 0xdc, (byte) 0x80, (byte) 0xdc, (byte) 0xfe, (byte) 0xdd, (byte) 0x40, (byte) 0xdd, (byte) 0x7e, (byte) 0xdd, (byte) 0x80,
                                        (byte) 0xdd, (byte) 0xfe, (byte) 0xde, (byte) 0x40, (byte) 0xde, (byte) 0x7e, (byte) 0xde, (byte) 0x80, (byte) 0xde, (byte) 0xfe, (byte) 0xdf, (byte) 0x40,
                                        (byte) 0xdf, (byte) 0x7e, (byte) 0xdf, (byte) 0x80, (byte) 0xdf, (byte) 0xfe, (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0x80,
                                        (byte) 0xe0, (byte) 0xfe, (byte) 0xe1, (byte) 0x40, (byte) 0xe1, (byte) 0x7e, (byte) 0xe1, (byte) 0x80, (byte) 0xe1, (byte) 0xfe, (byte) 0xe2, (byte) 0x40,
                                        (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0x80, (byte) 0xe2, (byte) 0xfe, (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0x80,
                                        (byte) 0xe3, (byte) 0xfe, (byte) 0xe4, (byte) 0x40, (byte) 0xe4, (byte) 0x7e, (byte) 0xe4, (byte) 0x80, (byte) 0xe4, (byte) 0xfe, (byte) 0xe5, (byte) 0x40,
                                        (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0x80, (byte) 0xe5, (byte) 0xfe, (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0x80,
                                        (byte) 0xe6, (byte) 0xfe, (byte) 0xe7, (byte) 0x40, (byte) 0xe7, (byte) 0x7e, (byte) 0xe7, (byte) 0x80, (byte) 0xe7, (byte) 0xfe, (byte) 0xe8, (byte) 0x40,
                                        (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0x80, (byte) 0xe8, (byte) 0xfe, (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0x80,
                                        (byte) 0xe9, (byte) 0xfe, (byte) 0xea, (byte) 0x40, (byte) 0xea, (byte) 0x7e, (byte) 0xea, (byte) 0x80, (byte) 0xea, (byte) 0xfe, (byte) 0xeb, (byte) 0x40,
                                        (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0x80, (byte) 0xeb, (byte) 0xfe, (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0x80,
                                        (byte) 0xec, (byte) 0xfe, (byte) 0xed, (byte) 0x40, (byte) 0xed, (byte) 0x7e, (byte) 0xed, (byte) 0x80, (byte) 0xed, (byte) 0xfe, (byte) 0xee, (byte) 0x40,
                                        (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0x80, (byte) 0xee, (byte) 0xfe, (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0x80,
                                        (byte) 0xef, (byte) 0xfe, (byte) 0xf0, (byte) 0x40, (byte) 0xf0, (byte) 0x7e, (byte) 0xf0, (byte) 0x80, (byte) 0xf0, (byte) 0xfe, (byte) 0xf1, (byte) 0x40,
                                        (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0x80, (byte) 0xf1, (byte) 0xfe, (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0x80,
                                        (byte) 0xf2, (byte) 0xfe, (byte) 0xf3, (byte) 0x40, (byte) 0xf3, (byte) 0x7e, (byte) 0xf3, (byte) 0x80, (byte) 0xf3, (byte) 0xfe, (byte) 0xf4, (byte) 0x40,
                                        (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0x80, (byte) 0xf4, (byte) 0xfe, (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0x80,
                                        (byte) 0xf5, (byte) 0xfe, (byte) 0xf6, (byte) 0x40, (byte) 0xf6, (byte) 0x7e, (byte) 0xf6, (byte) 0x80, (byte) 0xf6, (byte) 0xfe, (byte) 0xf7, (byte) 0x40,
                                        (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0x80, (byte) 0xf7, (byte) 0xfe, (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0x80,
                                        (byte) 0xf8, (byte) 0xfe, (byte) 0xf9, (byte) 0x40, (byte) 0xf9, (byte) 0x7e, (byte) 0xf9, (byte) 0x80, (byte) 0xf9, (byte) 0xfe, (byte) 0xfa, (byte) 0x40,
                                        (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0x80, (byte) 0xfa, (byte) 0xfe, (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0x80,
                                        (byte) 0xfb, (byte) 0xfe, (byte) 0xfc, (byte) 0x40, (byte) 0xfc, (byte) 0x7e, (byte) 0xfc, (byte) 0x80, (byte) 0xfc, (byte) 0xfe, (byte) 0xfd, (byte) 0x40,
                                        (byte) 0xfd, (byte) 0x7e, (byte) 0xfd, (byte) 0x80, (byte) 0xfd, (byte) 0xfe, (byte) 0xfe, (byte) 0x40, (byte) 0xfe, (byte) 0x7e, (byte) 0xfe, (byte) 0x80,
                                        (byte) 0xfe, (byte) 0xfe},
                        null, null);
    }

    static TestData dataIBM437() {
        assert TruffleString.Encoding.IBM437.ordinal() == 30;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM720() {
        assert TruffleString.Encoding.IBM720.ordinal() == 31;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM737() {
        assert TruffleString.Encoding.IBM737.ordinal() == 32;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM775() {
        assert TruffleString.Encoding.IBM775.ordinal() == 33;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM852() {
        assert TruffleString.Encoding.IBM852.ordinal() == 34;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM855() {
        assert TruffleString.Encoding.IBM855.ordinal() == 35;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM857() {
        assert TruffleString.Encoding.IBM857.ordinal() == 36;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM860() {
        assert TruffleString.Encoding.IBM860.ordinal() == 37;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM861() {
        assert TruffleString.Encoding.IBM861.ordinal() == 38;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM862() {
        assert TruffleString.Encoding.IBM862.ordinal() == 39;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM863() {
        assert TruffleString.Encoding.IBM863.ordinal() == 40;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM864() {
        assert TruffleString.Encoding.IBM864.ordinal() == 41;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM865() {
        assert TruffleString.Encoding.IBM865.ordinal() == 42;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM866() {
        assert TruffleString.Encoding.IBM866.ordinal() == 43;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM869() {
        assert TruffleString.Encoding.IBM869.ordinal() == 44;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO885910() {
        assert TruffleString.Encoding.ISO_8859_10.ordinal() == 45;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO885911() {
        assert TruffleString.Encoding.ISO_8859_11.ordinal() == 46;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO885913() {
        assert TruffleString.Encoding.ISO_8859_13.ordinal() == 47;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO885914() {
        assert TruffleString.Encoding.ISO_8859_14.ordinal() == 48;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO885915() {
        assert TruffleString.Encoding.ISO_8859_15.ordinal() == 49;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO885916() {
        assert TruffleString.Encoding.ISO_8859_16.ordinal() == 50;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO88592() {
        assert TruffleString.Encoding.ISO_8859_2.ordinal() == 51;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO88593() {
        assert TruffleString.Encoding.ISO_8859_3.ordinal() == 52;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO88594() {
        assert TruffleString.Encoding.ISO_8859_4.ordinal() == 53;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO88595() {
        assert TruffleString.Encoding.ISO_8859_5.ordinal() == 54;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO88596() {
        assert TruffleString.Encoding.ISO_8859_6.ordinal() == 55;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO88597() {
        assert TruffleString.Encoding.ISO_8859_7.ordinal() == 56;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO88598() {
        assert TruffleString.Encoding.ISO_8859_8.ordinal() == 57;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO88599() {
        assert TruffleString.Encoding.ISO_8859_9.ordinal() == 58;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataKOI8R() {
        assert TruffleString.Encoding.KOI8_R.ordinal() == 59;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataKOI8U() {
        assert TruffleString.Encoding.KOI8_U.ordinal() == 60;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataMacCentEuro() {
        assert TruffleString.Encoding.MacCentEuro.ordinal() == 61;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataMacCroatian() {
        assert TruffleString.Encoding.MacCroatian.ordinal() == 62;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataMacCyrillic() {
        assert TruffleString.Encoding.MacCyrillic.ordinal() == 63;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataMacGreek() {
        assert TruffleString.Encoding.MacGreek.ordinal() == 64;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataMacIceland() {
        assert TruffleString.Encoding.MacIceland.ordinal() == 65;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataMacJapanese() {
        assert TruffleString.Encoding.MacJapanese.ordinal() == 66;
        return new TestData(new int[]{0x0000, 0x007f, 0x00a1, 0x00df, 0x8140, 0x817e, 0x8180, 0x81fc, 0x8240, 0x827e, 0x8280, 0x82fc, 0x8340, 0x837e, 0x8380, 0x83fc, 0x8440, 0x847e, 0x8480, 0x84fc,
                        0x8540, 0x857e, 0x8580, 0x85fc, 0x8640, 0x867e, 0x8680, 0x86fc, 0x8740, 0x877e, 0x8780, 0x87fc, 0x8840, 0x887e, 0x8880, 0x88fc, 0x8940, 0x897e, 0x8980, 0x89fc, 0x8a40, 0x8a7e,
                        0x8a80, 0x8afc, 0x8b40, 0x8b7e, 0x8b80, 0x8bfc, 0x8c40, 0x8c7e, 0x8c80, 0x8cfc, 0x8d40, 0x8d7e, 0x8d80, 0x8dfc, 0x8e40, 0x8e7e, 0x8e80, 0x8efc, 0x8f40, 0x8f7e, 0x8f80, 0x8ffc,
                        0x9040, 0x907e, 0x9080, 0x90fc, 0x9140, 0x917e, 0x9180, 0x91fc, 0x9240, 0x927e, 0x9280, 0x92fc, 0x9340, 0x937e, 0x9380, 0x93fc, 0x9440, 0x947e, 0x9480, 0x94fc, 0x9540, 0x957e,
                        0x9580, 0x95fc, 0x9640, 0x967e, 0x9680, 0x96fc, 0x9740, 0x977e, 0x9780, 0x97fc, 0x9840, 0x987e, 0x9880, 0x98fc, 0x9940, 0x997e, 0x9980, 0x99fc, 0x9a40, 0x9a7e, 0x9a80, 0x9afc,
                        0x9b40, 0x9b7e, 0x9b80, 0x9bfc, 0x9c40, 0x9c7e, 0x9c80, 0x9cfc, 0x9d40, 0x9d7e, 0x9d80, 0x9dfc, 0x9e40, 0x9e7e, 0x9e80, 0x9efc, 0x9f40, 0x9f7e, 0x9f80, 0x9ffc, 0xe040, 0xe07e,
                        0xe080, 0xe0fc, 0xe140, 0xe17e, 0xe180, 0xe1fc, 0xe240, 0xe27e, 0xe280, 0xe2fc, 0xe340, 0xe37e, 0xe380, 0xe3fc, 0xe440, 0xe47e, 0xe480, 0xe4fc, 0xe540, 0xe57e, 0xe580, 0xe5fc,
                        0xe640, 0xe67e, 0xe680, 0xe6fc, 0xe740, 0xe77e, 0xe780, 0xe7fc, 0xe840, 0xe87e, 0xe880, 0xe8fc, 0xe940, 0xe97e, 0xe980, 0xe9fc, 0xea40, 0xea7e, 0xea80, 0xeafc, 0xeb40, 0xeb7e,
                        0xeb80, 0xebfc, 0xec40, 0xec7e, 0xec80, 0xecfc, 0xed40, 0xed7e, 0xed80, 0xedfc, 0xee40, 0xee7e, 0xee80, 0xeefc, 0xef40, 0xef7e, 0xef80, 0xeffc, 0xf040, 0xf07e, 0xf080, 0xf0fc,
                        0xf140, 0xf17e, 0xf180, 0xf1fc, 0xf240, 0xf27e, 0xf280, 0xf2fc, 0xf340, 0xf37e, 0xf380, 0xf3fc, 0xf440, 0xf47e, 0xf480, 0xf4fc, 0xf540, 0xf57e, 0xf580, 0xf5fc, 0xf640, 0xf67e,
                        0xf680, 0xf6fc, 0xf740, 0xf77e, 0xf780, 0xf7fc, 0xf840, 0xf87e, 0xf880, 0xf8fc, 0xf940, 0xf97e, 0xf980, 0xf9fc, 0xfa40, 0xfa7e, 0xfa80, 0xfafc, 0xfb40, 0xfb7e, 0xfb80, 0xfbfc,
                        0xfc40, 0xfc7e, 0xfc80, 0xfcfc},
                        new int[]{0x000, 0x001, 0x002, 0x003, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028,
                                        0x02a, 0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056,
                                        0x058, 0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084,
                                        0x086, 0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2,
                                        0x0b4, 0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0,
                                        0x0e2, 0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e,
                                        0x110, 0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c,
                                        0x13e, 0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a,
                                        0x16c, 0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198,
                                        0x19a, 0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6,
                                        0x1c8, 0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0xdf, (byte) 0x81, (byte) 0x40, (byte) 0x81, (byte) 0x7e, (byte) 0x81, (byte) 0x80, (byte) 0x81, (byte) 0xfc,
                                        (byte) 0x82, (byte) 0x40, (byte) 0x82, (byte) 0x7e, (byte) 0x82, (byte) 0x80, (byte) 0x82, (byte) 0xfc, (byte) 0x83, (byte) 0x40, (byte) 0x83, (byte) 0x7e,
                                        (byte) 0x83, (byte) 0x80, (byte) 0x83, (byte) 0xfc, (byte) 0x84, (byte) 0x40, (byte) 0x84, (byte) 0x7e, (byte) 0x84, (byte) 0x80, (byte) 0x84, (byte) 0xfc,
                                        (byte) 0x85, (byte) 0x40, (byte) 0x85, (byte) 0x7e, (byte) 0x85, (byte) 0x80, (byte) 0x85, (byte) 0xfc, (byte) 0x86, (byte) 0x40, (byte) 0x86, (byte) 0x7e,
                                        (byte) 0x86, (byte) 0x80, (byte) 0x86, (byte) 0xfc, (byte) 0x87, (byte) 0x40, (byte) 0x87, (byte) 0x7e, (byte) 0x87, (byte) 0x80, (byte) 0x87, (byte) 0xfc,
                                        (byte) 0x88, (byte) 0x40, (byte) 0x88, (byte) 0x7e, (byte) 0x88, (byte) 0x80, (byte) 0x88, (byte) 0xfc, (byte) 0x89, (byte) 0x40, (byte) 0x89, (byte) 0x7e,
                                        (byte) 0x89, (byte) 0x80, (byte) 0x89, (byte) 0xfc, (byte) 0x8a, (byte) 0x40, (byte) 0x8a, (byte) 0x7e, (byte) 0x8a, (byte) 0x80, (byte) 0x8a, (byte) 0xfc,
                                        (byte) 0x8b, (byte) 0x40, (byte) 0x8b, (byte) 0x7e, (byte) 0x8b, (byte) 0x80, (byte) 0x8b, (byte) 0xfc, (byte) 0x8c, (byte) 0x40, (byte) 0x8c, (byte) 0x7e,
                                        (byte) 0x8c, (byte) 0x80, (byte) 0x8c, (byte) 0xfc, (byte) 0x8d, (byte) 0x40, (byte) 0x8d, (byte) 0x7e, (byte) 0x8d, (byte) 0x80, (byte) 0x8d, (byte) 0xfc,
                                        (byte) 0x8e, (byte) 0x40, (byte) 0x8e, (byte) 0x7e, (byte) 0x8e, (byte) 0x80, (byte) 0x8e, (byte) 0xfc, (byte) 0x8f, (byte) 0x40, (byte) 0x8f, (byte) 0x7e,
                                        (byte) 0x8f, (byte) 0x80, (byte) 0x8f, (byte) 0xfc, (byte) 0x90, (byte) 0x40, (byte) 0x90, (byte) 0x7e, (byte) 0x90, (byte) 0x80, (byte) 0x90, (byte) 0xfc,
                                        (byte) 0x91, (byte) 0x40, (byte) 0x91, (byte) 0x7e, (byte) 0x91, (byte) 0x80, (byte) 0x91, (byte) 0xfc, (byte) 0x92, (byte) 0x40, (byte) 0x92, (byte) 0x7e,
                                        (byte) 0x92, (byte) 0x80, (byte) 0x92, (byte) 0xfc, (byte) 0x93, (byte) 0x40, (byte) 0x93, (byte) 0x7e, (byte) 0x93, (byte) 0x80, (byte) 0x93, (byte) 0xfc,
                                        (byte) 0x94, (byte) 0x40, (byte) 0x94, (byte) 0x7e, (byte) 0x94, (byte) 0x80, (byte) 0x94, (byte) 0xfc, (byte) 0x95, (byte) 0x40, (byte) 0x95, (byte) 0x7e,
                                        (byte) 0x95, (byte) 0x80, (byte) 0x95, (byte) 0xfc, (byte) 0x96, (byte) 0x40, (byte) 0x96, (byte) 0x7e, (byte) 0x96, (byte) 0x80, (byte) 0x96, (byte) 0xfc,
                                        (byte) 0x97, (byte) 0x40, (byte) 0x97, (byte) 0x7e, (byte) 0x97, (byte) 0x80, (byte) 0x97, (byte) 0xfc, (byte) 0x98, (byte) 0x40, (byte) 0x98, (byte) 0x7e,
                                        (byte) 0x98, (byte) 0x80, (byte) 0x98, (byte) 0xfc, (byte) 0x99, (byte) 0x40, (byte) 0x99, (byte) 0x7e, (byte) 0x99, (byte) 0x80, (byte) 0x99, (byte) 0xfc,
                                        (byte) 0x9a, (byte) 0x40, (byte) 0x9a, (byte) 0x7e, (byte) 0x9a, (byte) 0x80, (byte) 0x9a, (byte) 0xfc, (byte) 0x9b, (byte) 0x40, (byte) 0x9b, (byte) 0x7e,
                                        (byte) 0x9b, (byte) 0x80, (byte) 0x9b, (byte) 0xfc, (byte) 0x9c, (byte) 0x40, (byte) 0x9c, (byte) 0x7e, (byte) 0x9c, (byte) 0x80, (byte) 0x9c, (byte) 0xfc,
                                        (byte) 0x9d, (byte) 0x40, (byte) 0x9d, (byte) 0x7e, (byte) 0x9d, (byte) 0x80, (byte) 0x9d, (byte) 0xfc, (byte) 0x9e, (byte) 0x40, (byte) 0x9e, (byte) 0x7e,
                                        (byte) 0x9e, (byte) 0x80, (byte) 0x9e, (byte) 0xfc, (byte) 0x9f, (byte) 0x40, (byte) 0x9f, (byte) 0x7e, (byte) 0x9f, (byte) 0x80, (byte) 0x9f, (byte) 0xfc,
                                        (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0x80, (byte) 0xe0, (byte) 0xfc, (byte) 0xe1, (byte) 0x40, (byte) 0xe1, (byte) 0x7e,
                                        (byte) 0xe1, (byte) 0x80, (byte) 0xe1, (byte) 0xfc, (byte) 0xe2, (byte) 0x40, (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0x80, (byte) 0xe2, (byte) 0xfc,
                                        (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0x80, (byte) 0xe3, (byte) 0xfc, (byte) 0xe4, (byte) 0x40, (byte) 0xe4, (byte) 0x7e,
                                        (byte) 0xe4, (byte) 0x80, (byte) 0xe4, (byte) 0xfc, (byte) 0xe5, (byte) 0x40, (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0x80, (byte) 0xe5, (byte) 0xfc,
                                        (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0x80, (byte) 0xe6, (byte) 0xfc, (byte) 0xe7, (byte) 0x40, (byte) 0xe7, (byte) 0x7e,
                                        (byte) 0xe7, (byte) 0x80, (byte) 0xe7, (byte) 0xfc, (byte) 0xe8, (byte) 0x40, (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0x80, (byte) 0xe8, (byte) 0xfc,
                                        (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0x80, (byte) 0xe9, (byte) 0xfc, (byte) 0xea, (byte) 0x40, (byte) 0xea, (byte) 0x7e,
                                        (byte) 0xea, (byte) 0x80, (byte) 0xea, (byte) 0xfc, (byte) 0xeb, (byte) 0x40, (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0x80, (byte) 0xeb, (byte) 0xfc,
                                        (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0x80, (byte) 0xec, (byte) 0xfc, (byte) 0xed, (byte) 0x40, (byte) 0xed, (byte) 0x7e,
                                        (byte) 0xed, (byte) 0x80, (byte) 0xed, (byte) 0xfc, (byte) 0xee, (byte) 0x40, (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0x80, (byte) 0xee, (byte) 0xfc,
                                        (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0x80, (byte) 0xef, (byte) 0xfc, (byte) 0xf0, (byte) 0x40, (byte) 0xf0, (byte) 0x7e,
                                        (byte) 0xf0, (byte) 0x80, (byte) 0xf0, (byte) 0xfc, (byte) 0xf1, (byte) 0x40, (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0x80, (byte) 0xf1, (byte) 0xfc,
                                        (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0x80, (byte) 0xf2, (byte) 0xfc, (byte) 0xf3, (byte) 0x40, (byte) 0xf3, (byte) 0x7e,
                                        (byte) 0xf3, (byte) 0x80, (byte) 0xf3, (byte) 0xfc, (byte) 0xf4, (byte) 0x40, (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0x80, (byte) 0xf4, (byte) 0xfc,
                                        (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0x80, (byte) 0xf5, (byte) 0xfc, (byte) 0xf6, (byte) 0x40, (byte) 0xf6, (byte) 0x7e,
                                        (byte) 0xf6, (byte) 0x80, (byte) 0xf6, (byte) 0xfc, (byte) 0xf7, (byte) 0x40, (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0x80, (byte) 0xf7, (byte) 0xfc,
                                        (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0x80, (byte) 0xf8, (byte) 0xfc, (byte) 0xf9, (byte) 0x40, (byte) 0xf9, (byte) 0x7e,
                                        (byte) 0xf9, (byte) 0x80, (byte) 0xf9, (byte) 0xfc, (byte) 0xfa, (byte) 0x40, (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0x80, (byte) 0xfa, (byte) 0xfc,
                                        (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0x80, (byte) 0xfb, (byte) 0xfc, (byte) 0xfc, (byte) 0x40, (byte) 0xfc, (byte) 0x7e,
                                        (byte) 0xfc, (byte) 0x80, (byte) 0xfc, (byte) 0xfc},
                        null, null);
    }

    static TestData dataMacRoman() {
        assert TruffleString.Encoding.MacRoman.ordinal() == 67;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataMacRomania() {
        assert TruffleString.Encoding.MacRomania.ordinal() == 68;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataMacThai() {
        assert TruffleString.Encoding.MacThai.ordinal() == 69;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataMacTurkish() {
        assert TruffleString.Encoding.MacTurkish.ordinal() == 70;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataMacUkraine() {
        assert TruffleString.Encoding.MacUkraine.ordinal() == 71;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataSJISDoCoMo() {
        assert TruffleString.Encoding.SJIS_DoCoMo.ordinal() == 72;
        return new TestData(new int[]{0x0000, 0x007f, 0x00a1, 0x00df, 0x8140, 0x817e, 0x8180, 0x81fc, 0x8240, 0x827e, 0x8280, 0x82fc, 0x8340, 0x837e, 0x8380, 0x83fc, 0x8440, 0x847e, 0x8480, 0x84fc,
                        0x8540, 0x857e, 0x8580, 0x85fc, 0x8640, 0x867e, 0x8680, 0x86fc, 0x8740, 0x877e, 0x8780, 0x87fc, 0x8840, 0x887e, 0x8880, 0x88fc, 0x8940, 0x897e, 0x8980, 0x89fc, 0x8a40, 0x8a7e,
                        0x8a80, 0x8afc, 0x8b40, 0x8b7e, 0x8b80, 0x8bfc, 0x8c40, 0x8c7e, 0x8c80, 0x8cfc, 0x8d40, 0x8d7e, 0x8d80, 0x8dfc, 0x8e40, 0x8e7e, 0x8e80, 0x8efc, 0x8f40, 0x8f7e, 0x8f80, 0x8ffc,
                        0x9040, 0x907e, 0x9080, 0x90fc, 0x9140, 0x917e, 0x9180, 0x91fc, 0x9240, 0x927e, 0x9280, 0x92fc, 0x9340, 0x937e, 0x9380, 0x93fc, 0x9440, 0x947e, 0x9480, 0x94fc, 0x9540, 0x957e,
                        0x9580, 0x95fc, 0x9640, 0x967e, 0x9680, 0x96fc, 0x9740, 0x977e, 0x9780, 0x97fc, 0x9840, 0x987e, 0x9880, 0x98fc, 0x9940, 0x997e, 0x9980, 0x99fc, 0x9a40, 0x9a7e, 0x9a80, 0x9afc,
                        0x9b40, 0x9b7e, 0x9b80, 0x9bfc, 0x9c40, 0x9c7e, 0x9c80, 0x9cfc, 0x9d40, 0x9d7e, 0x9d80, 0x9dfc, 0x9e40, 0x9e7e, 0x9e80, 0x9efc, 0x9f40, 0x9f7e, 0x9f80, 0x9ffc, 0xe040, 0xe07e,
                        0xe080, 0xe0fc, 0xe140, 0xe17e, 0xe180, 0xe1fc, 0xe240, 0xe27e, 0xe280, 0xe2fc, 0xe340, 0xe37e, 0xe380, 0xe3fc, 0xe440, 0xe47e, 0xe480, 0xe4fc, 0xe540, 0xe57e, 0xe580, 0xe5fc,
                        0xe640, 0xe67e, 0xe680, 0xe6fc, 0xe740, 0xe77e, 0xe780, 0xe7fc, 0xe840, 0xe87e, 0xe880, 0xe8fc, 0xe940, 0xe97e, 0xe980, 0xe9fc, 0xea40, 0xea7e, 0xea80, 0xeafc, 0xeb40, 0xeb7e,
                        0xeb80, 0xebfc, 0xec40, 0xec7e, 0xec80, 0xecfc, 0xed40, 0xed7e, 0xed80, 0xedfc, 0xee40, 0xee7e, 0xee80, 0xeefc, 0xef40, 0xef7e, 0xef80, 0xeffc, 0xf040, 0xf07e, 0xf080, 0xf0fc,
                        0xf140, 0xf17e, 0xf180, 0xf1fc, 0xf240, 0xf27e, 0xf280, 0xf2fc, 0xf340, 0xf37e, 0xf380, 0xf3fc, 0xf440, 0xf47e, 0xf480, 0xf4fc, 0xf540, 0xf57e, 0xf580, 0xf5fc, 0xf640, 0xf67e,
                        0xf680, 0xf6fc, 0xf740, 0xf77e, 0xf780, 0xf7fc, 0xf840, 0xf87e, 0xf880, 0xf8fc, 0xf940, 0xf97e, 0xf980, 0xf9fc, 0xfa40, 0xfa7e, 0xfa80, 0xfafc, 0xfb40, 0xfb7e, 0xfb80, 0xfbfc,
                        0xfc40, 0xfc7e, 0xfc80, 0xfcfc},
                        new int[]{0x000, 0x001, 0x002, 0x003, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028,
                                        0x02a, 0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056,
                                        0x058, 0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084,
                                        0x086, 0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2,
                                        0x0b4, 0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0,
                                        0x0e2, 0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e,
                                        0x110, 0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c,
                                        0x13e, 0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a,
                                        0x16c, 0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198,
                                        0x19a, 0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6,
                                        0x1c8, 0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0xdf, (byte) 0x81, (byte) 0x40, (byte) 0x81, (byte) 0x7e, (byte) 0x81, (byte) 0x80, (byte) 0x81, (byte) 0xfc,
                                        (byte) 0x82, (byte) 0x40, (byte) 0x82, (byte) 0x7e, (byte) 0x82, (byte) 0x80, (byte) 0x82, (byte) 0xfc, (byte) 0x83, (byte) 0x40, (byte) 0x83, (byte) 0x7e,
                                        (byte) 0x83, (byte) 0x80, (byte) 0x83, (byte) 0xfc, (byte) 0x84, (byte) 0x40, (byte) 0x84, (byte) 0x7e, (byte) 0x84, (byte) 0x80, (byte) 0x84, (byte) 0xfc,
                                        (byte) 0x85, (byte) 0x40, (byte) 0x85, (byte) 0x7e, (byte) 0x85, (byte) 0x80, (byte) 0x85, (byte) 0xfc, (byte) 0x86, (byte) 0x40, (byte) 0x86, (byte) 0x7e,
                                        (byte) 0x86, (byte) 0x80, (byte) 0x86, (byte) 0xfc, (byte) 0x87, (byte) 0x40, (byte) 0x87, (byte) 0x7e, (byte) 0x87, (byte) 0x80, (byte) 0x87, (byte) 0xfc,
                                        (byte) 0x88, (byte) 0x40, (byte) 0x88, (byte) 0x7e, (byte) 0x88, (byte) 0x80, (byte) 0x88, (byte) 0xfc, (byte) 0x89, (byte) 0x40, (byte) 0x89, (byte) 0x7e,
                                        (byte) 0x89, (byte) 0x80, (byte) 0x89, (byte) 0xfc, (byte) 0x8a, (byte) 0x40, (byte) 0x8a, (byte) 0x7e, (byte) 0x8a, (byte) 0x80, (byte) 0x8a, (byte) 0xfc,
                                        (byte) 0x8b, (byte) 0x40, (byte) 0x8b, (byte) 0x7e, (byte) 0x8b, (byte) 0x80, (byte) 0x8b, (byte) 0xfc, (byte) 0x8c, (byte) 0x40, (byte) 0x8c, (byte) 0x7e,
                                        (byte) 0x8c, (byte) 0x80, (byte) 0x8c, (byte) 0xfc, (byte) 0x8d, (byte) 0x40, (byte) 0x8d, (byte) 0x7e, (byte) 0x8d, (byte) 0x80, (byte) 0x8d, (byte) 0xfc,
                                        (byte) 0x8e, (byte) 0x40, (byte) 0x8e, (byte) 0x7e, (byte) 0x8e, (byte) 0x80, (byte) 0x8e, (byte) 0xfc, (byte) 0x8f, (byte) 0x40, (byte) 0x8f, (byte) 0x7e,
                                        (byte) 0x8f, (byte) 0x80, (byte) 0x8f, (byte) 0xfc, (byte) 0x90, (byte) 0x40, (byte) 0x90, (byte) 0x7e, (byte) 0x90, (byte) 0x80, (byte) 0x90, (byte) 0xfc,
                                        (byte) 0x91, (byte) 0x40, (byte) 0x91, (byte) 0x7e, (byte) 0x91, (byte) 0x80, (byte) 0x91, (byte) 0xfc, (byte) 0x92, (byte) 0x40, (byte) 0x92, (byte) 0x7e,
                                        (byte) 0x92, (byte) 0x80, (byte) 0x92, (byte) 0xfc, (byte) 0x93, (byte) 0x40, (byte) 0x93, (byte) 0x7e, (byte) 0x93, (byte) 0x80, (byte) 0x93, (byte) 0xfc,
                                        (byte) 0x94, (byte) 0x40, (byte) 0x94, (byte) 0x7e, (byte) 0x94, (byte) 0x80, (byte) 0x94, (byte) 0xfc, (byte) 0x95, (byte) 0x40, (byte) 0x95, (byte) 0x7e,
                                        (byte) 0x95, (byte) 0x80, (byte) 0x95, (byte) 0xfc, (byte) 0x96, (byte) 0x40, (byte) 0x96, (byte) 0x7e, (byte) 0x96, (byte) 0x80, (byte) 0x96, (byte) 0xfc,
                                        (byte) 0x97, (byte) 0x40, (byte) 0x97, (byte) 0x7e, (byte) 0x97, (byte) 0x80, (byte) 0x97, (byte) 0xfc, (byte) 0x98, (byte) 0x40, (byte) 0x98, (byte) 0x7e,
                                        (byte) 0x98, (byte) 0x80, (byte) 0x98, (byte) 0xfc, (byte) 0x99, (byte) 0x40, (byte) 0x99, (byte) 0x7e, (byte) 0x99, (byte) 0x80, (byte) 0x99, (byte) 0xfc,
                                        (byte) 0x9a, (byte) 0x40, (byte) 0x9a, (byte) 0x7e, (byte) 0x9a, (byte) 0x80, (byte) 0x9a, (byte) 0xfc, (byte) 0x9b, (byte) 0x40, (byte) 0x9b, (byte) 0x7e,
                                        (byte) 0x9b, (byte) 0x80, (byte) 0x9b, (byte) 0xfc, (byte) 0x9c, (byte) 0x40, (byte) 0x9c, (byte) 0x7e, (byte) 0x9c, (byte) 0x80, (byte) 0x9c, (byte) 0xfc,
                                        (byte) 0x9d, (byte) 0x40, (byte) 0x9d, (byte) 0x7e, (byte) 0x9d, (byte) 0x80, (byte) 0x9d, (byte) 0xfc, (byte) 0x9e, (byte) 0x40, (byte) 0x9e, (byte) 0x7e,
                                        (byte) 0x9e, (byte) 0x80, (byte) 0x9e, (byte) 0xfc, (byte) 0x9f, (byte) 0x40, (byte) 0x9f, (byte) 0x7e, (byte) 0x9f, (byte) 0x80, (byte) 0x9f, (byte) 0xfc,
                                        (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0x80, (byte) 0xe0, (byte) 0xfc, (byte) 0xe1, (byte) 0x40, (byte) 0xe1, (byte) 0x7e,
                                        (byte) 0xe1, (byte) 0x80, (byte) 0xe1, (byte) 0xfc, (byte) 0xe2, (byte) 0x40, (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0x80, (byte) 0xe2, (byte) 0xfc,
                                        (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0x80, (byte) 0xe3, (byte) 0xfc, (byte) 0xe4, (byte) 0x40, (byte) 0xe4, (byte) 0x7e,
                                        (byte) 0xe4, (byte) 0x80, (byte) 0xe4, (byte) 0xfc, (byte) 0xe5, (byte) 0x40, (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0x80, (byte) 0xe5, (byte) 0xfc,
                                        (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0x80, (byte) 0xe6, (byte) 0xfc, (byte) 0xe7, (byte) 0x40, (byte) 0xe7, (byte) 0x7e,
                                        (byte) 0xe7, (byte) 0x80, (byte) 0xe7, (byte) 0xfc, (byte) 0xe8, (byte) 0x40, (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0x80, (byte) 0xe8, (byte) 0xfc,
                                        (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0x80, (byte) 0xe9, (byte) 0xfc, (byte) 0xea, (byte) 0x40, (byte) 0xea, (byte) 0x7e,
                                        (byte) 0xea, (byte) 0x80, (byte) 0xea, (byte) 0xfc, (byte) 0xeb, (byte) 0x40, (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0x80, (byte) 0xeb, (byte) 0xfc,
                                        (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0x80, (byte) 0xec, (byte) 0xfc, (byte) 0xed, (byte) 0x40, (byte) 0xed, (byte) 0x7e,
                                        (byte) 0xed, (byte) 0x80, (byte) 0xed, (byte) 0xfc, (byte) 0xee, (byte) 0x40, (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0x80, (byte) 0xee, (byte) 0xfc,
                                        (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0x80, (byte) 0xef, (byte) 0xfc, (byte) 0xf0, (byte) 0x40, (byte) 0xf0, (byte) 0x7e,
                                        (byte) 0xf0, (byte) 0x80, (byte) 0xf0, (byte) 0xfc, (byte) 0xf1, (byte) 0x40, (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0x80, (byte) 0xf1, (byte) 0xfc,
                                        (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0x80, (byte) 0xf2, (byte) 0xfc, (byte) 0xf3, (byte) 0x40, (byte) 0xf3, (byte) 0x7e,
                                        (byte) 0xf3, (byte) 0x80, (byte) 0xf3, (byte) 0xfc, (byte) 0xf4, (byte) 0x40, (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0x80, (byte) 0xf4, (byte) 0xfc,
                                        (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0x80, (byte) 0xf5, (byte) 0xfc, (byte) 0xf6, (byte) 0x40, (byte) 0xf6, (byte) 0x7e,
                                        (byte) 0xf6, (byte) 0x80, (byte) 0xf6, (byte) 0xfc, (byte) 0xf7, (byte) 0x40, (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0x80, (byte) 0xf7, (byte) 0xfc,
                                        (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0x80, (byte) 0xf8, (byte) 0xfc, (byte) 0xf9, (byte) 0x40, (byte) 0xf9, (byte) 0x7e,
                                        (byte) 0xf9, (byte) 0x80, (byte) 0xf9, (byte) 0xfc, (byte) 0xfa, (byte) 0x40, (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0x80, (byte) 0xfa, (byte) 0xfc,
                                        (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0x80, (byte) 0xfb, (byte) 0xfc, (byte) 0xfc, (byte) 0x40, (byte) 0xfc, (byte) 0x7e,
                                        (byte) 0xfc, (byte) 0x80, (byte) 0xfc, (byte) 0xfc},
                        null, null);
    }

    static TestData dataSJISKDDI() {
        assert TruffleString.Encoding.SJIS_KDDI.ordinal() == 73;
        return new TestData(new int[]{0x0000, 0x007f, 0x00a1, 0x00df, 0x8140, 0x817e, 0x8180, 0x81fc, 0x8240, 0x827e, 0x8280, 0x82fc, 0x8340, 0x837e, 0x8380, 0x83fc, 0x8440, 0x847e, 0x8480, 0x84fc,
                        0x8540, 0x857e, 0x8580, 0x85fc, 0x8640, 0x867e, 0x8680, 0x86fc, 0x8740, 0x877e, 0x8780, 0x87fc, 0x8840, 0x887e, 0x8880, 0x88fc, 0x8940, 0x897e, 0x8980, 0x89fc, 0x8a40, 0x8a7e,
                        0x8a80, 0x8afc, 0x8b40, 0x8b7e, 0x8b80, 0x8bfc, 0x8c40, 0x8c7e, 0x8c80, 0x8cfc, 0x8d40, 0x8d7e, 0x8d80, 0x8dfc, 0x8e40, 0x8e7e, 0x8e80, 0x8efc, 0x8f40, 0x8f7e, 0x8f80, 0x8ffc,
                        0x9040, 0x907e, 0x9080, 0x90fc, 0x9140, 0x917e, 0x9180, 0x91fc, 0x9240, 0x927e, 0x9280, 0x92fc, 0x9340, 0x937e, 0x9380, 0x93fc, 0x9440, 0x947e, 0x9480, 0x94fc, 0x9540, 0x957e,
                        0x9580, 0x95fc, 0x9640, 0x967e, 0x9680, 0x96fc, 0x9740, 0x977e, 0x9780, 0x97fc, 0x9840, 0x987e, 0x9880, 0x98fc, 0x9940, 0x997e, 0x9980, 0x99fc, 0x9a40, 0x9a7e, 0x9a80, 0x9afc,
                        0x9b40, 0x9b7e, 0x9b80, 0x9bfc, 0x9c40, 0x9c7e, 0x9c80, 0x9cfc, 0x9d40, 0x9d7e, 0x9d80, 0x9dfc, 0x9e40, 0x9e7e, 0x9e80, 0x9efc, 0x9f40, 0x9f7e, 0x9f80, 0x9ffc, 0xe040, 0xe07e,
                        0xe080, 0xe0fc, 0xe140, 0xe17e, 0xe180, 0xe1fc, 0xe240, 0xe27e, 0xe280, 0xe2fc, 0xe340, 0xe37e, 0xe380, 0xe3fc, 0xe440, 0xe47e, 0xe480, 0xe4fc, 0xe540, 0xe57e, 0xe580, 0xe5fc,
                        0xe640, 0xe67e, 0xe680, 0xe6fc, 0xe740, 0xe77e, 0xe780, 0xe7fc, 0xe840, 0xe87e, 0xe880, 0xe8fc, 0xe940, 0xe97e, 0xe980, 0xe9fc, 0xea40, 0xea7e, 0xea80, 0xeafc, 0xeb40, 0xeb7e,
                        0xeb80, 0xebfc, 0xec40, 0xec7e, 0xec80, 0xecfc, 0xed40, 0xed7e, 0xed80, 0xedfc, 0xee40, 0xee7e, 0xee80, 0xeefc, 0xef40, 0xef7e, 0xef80, 0xeffc, 0xf040, 0xf07e, 0xf080, 0xf0fc,
                        0xf140, 0xf17e, 0xf180, 0xf1fc, 0xf240, 0xf27e, 0xf280, 0xf2fc, 0xf340, 0xf37e, 0xf380, 0xf3fc, 0xf440, 0xf47e, 0xf480, 0xf4fc, 0xf540, 0xf57e, 0xf580, 0xf5fc, 0xf640, 0xf67e,
                        0xf680, 0xf6fc, 0xf740, 0xf77e, 0xf780, 0xf7fc, 0xf840, 0xf87e, 0xf880, 0xf8fc, 0xf940, 0xf97e, 0xf980, 0xf9fc, 0xfa40, 0xfa7e, 0xfa80, 0xfafc, 0xfb40, 0xfb7e, 0xfb80, 0xfbfc,
                        0xfc40, 0xfc7e, 0xfc80, 0xfcfc},
                        new int[]{0x000, 0x001, 0x002, 0x003, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028,
                                        0x02a, 0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056,
                                        0x058, 0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084,
                                        0x086, 0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2,
                                        0x0b4, 0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0,
                                        0x0e2, 0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e,
                                        0x110, 0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c,
                                        0x13e, 0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a,
                                        0x16c, 0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198,
                                        0x19a, 0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6,
                                        0x1c8, 0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0xdf, (byte) 0x81, (byte) 0x40, (byte) 0x81, (byte) 0x7e, (byte) 0x81, (byte) 0x80, (byte) 0x81, (byte) 0xfc,
                                        (byte) 0x82, (byte) 0x40, (byte) 0x82, (byte) 0x7e, (byte) 0x82, (byte) 0x80, (byte) 0x82, (byte) 0xfc, (byte) 0x83, (byte) 0x40, (byte) 0x83, (byte) 0x7e,
                                        (byte) 0x83, (byte) 0x80, (byte) 0x83, (byte) 0xfc, (byte) 0x84, (byte) 0x40, (byte) 0x84, (byte) 0x7e, (byte) 0x84, (byte) 0x80, (byte) 0x84, (byte) 0xfc,
                                        (byte) 0x85, (byte) 0x40, (byte) 0x85, (byte) 0x7e, (byte) 0x85, (byte) 0x80, (byte) 0x85, (byte) 0xfc, (byte) 0x86, (byte) 0x40, (byte) 0x86, (byte) 0x7e,
                                        (byte) 0x86, (byte) 0x80, (byte) 0x86, (byte) 0xfc, (byte) 0x87, (byte) 0x40, (byte) 0x87, (byte) 0x7e, (byte) 0x87, (byte) 0x80, (byte) 0x87, (byte) 0xfc,
                                        (byte) 0x88, (byte) 0x40, (byte) 0x88, (byte) 0x7e, (byte) 0x88, (byte) 0x80, (byte) 0x88, (byte) 0xfc, (byte) 0x89, (byte) 0x40, (byte) 0x89, (byte) 0x7e,
                                        (byte) 0x89, (byte) 0x80, (byte) 0x89, (byte) 0xfc, (byte) 0x8a, (byte) 0x40, (byte) 0x8a, (byte) 0x7e, (byte) 0x8a, (byte) 0x80, (byte) 0x8a, (byte) 0xfc,
                                        (byte) 0x8b, (byte) 0x40, (byte) 0x8b, (byte) 0x7e, (byte) 0x8b, (byte) 0x80, (byte) 0x8b, (byte) 0xfc, (byte) 0x8c, (byte) 0x40, (byte) 0x8c, (byte) 0x7e,
                                        (byte) 0x8c, (byte) 0x80, (byte) 0x8c, (byte) 0xfc, (byte) 0x8d, (byte) 0x40, (byte) 0x8d, (byte) 0x7e, (byte) 0x8d, (byte) 0x80, (byte) 0x8d, (byte) 0xfc,
                                        (byte) 0x8e, (byte) 0x40, (byte) 0x8e, (byte) 0x7e, (byte) 0x8e, (byte) 0x80, (byte) 0x8e, (byte) 0xfc, (byte) 0x8f, (byte) 0x40, (byte) 0x8f, (byte) 0x7e,
                                        (byte) 0x8f, (byte) 0x80, (byte) 0x8f, (byte) 0xfc, (byte) 0x90, (byte) 0x40, (byte) 0x90, (byte) 0x7e, (byte) 0x90, (byte) 0x80, (byte) 0x90, (byte) 0xfc,
                                        (byte) 0x91, (byte) 0x40, (byte) 0x91, (byte) 0x7e, (byte) 0x91, (byte) 0x80, (byte) 0x91, (byte) 0xfc, (byte) 0x92, (byte) 0x40, (byte) 0x92, (byte) 0x7e,
                                        (byte) 0x92, (byte) 0x80, (byte) 0x92, (byte) 0xfc, (byte) 0x93, (byte) 0x40, (byte) 0x93, (byte) 0x7e, (byte) 0x93, (byte) 0x80, (byte) 0x93, (byte) 0xfc,
                                        (byte) 0x94, (byte) 0x40, (byte) 0x94, (byte) 0x7e, (byte) 0x94, (byte) 0x80, (byte) 0x94, (byte) 0xfc, (byte) 0x95, (byte) 0x40, (byte) 0x95, (byte) 0x7e,
                                        (byte) 0x95, (byte) 0x80, (byte) 0x95, (byte) 0xfc, (byte) 0x96, (byte) 0x40, (byte) 0x96, (byte) 0x7e, (byte) 0x96, (byte) 0x80, (byte) 0x96, (byte) 0xfc,
                                        (byte) 0x97, (byte) 0x40, (byte) 0x97, (byte) 0x7e, (byte) 0x97, (byte) 0x80, (byte) 0x97, (byte) 0xfc, (byte) 0x98, (byte) 0x40, (byte) 0x98, (byte) 0x7e,
                                        (byte) 0x98, (byte) 0x80, (byte) 0x98, (byte) 0xfc, (byte) 0x99, (byte) 0x40, (byte) 0x99, (byte) 0x7e, (byte) 0x99, (byte) 0x80, (byte) 0x99, (byte) 0xfc,
                                        (byte) 0x9a, (byte) 0x40, (byte) 0x9a, (byte) 0x7e, (byte) 0x9a, (byte) 0x80, (byte) 0x9a, (byte) 0xfc, (byte) 0x9b, (byte) 0x40, (byte) 0x9b, (byte) 0x7e,
                                        (byte) 0x9b, (byte) 0x80, (byte) 0x9b, (byte) 0xfc, (byte) 0x9c, (byte) 0x40, (byte) 0x9c, (byte) 0x7e, (byte) 0x9c, (byte) 0x80, (byte) 0x9c, (byte) 0xfc,
                                        (byte) 0x9d, (byte) 0x40, (byte) 0x9d, (byte) 0x7e, (byte) 0x9d, (byte) 0x80, (byte) 0x9d, (byte) 0xfc, (byte) 0x9e, (byte) 0x40, (byte) 0x9e, (byte) 0x7e,
                                        (byte) 0x9e, (byte) 0x80, (byte) 0x9e, (byte) 0xfc, (byte) 0x9f, (byte) 0x40, (byte) 0x9f, (byte) 0x7e, (byte) 0x9f, (byte) 0x80, (byte) 0x9f, (byte) 0xfc,
                                        (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0x80, (byte) 0xe0, (byte) 0xfc, (byte) 0xe1, (byte) 0x40, (byte) 0xe1, (byte) 0x7e,
                                        (byte) 0xe1, (byte) 0x80, (byte) 0xe1, (byte) 0xfc, (byte) 0xe2, (byte) 0x40, (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0x80, (byte) 0xe2, (byte) 0xfc,
                                        (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0x80, (byte) 0xe3, (byte) 0xfc, (byte) 0xe4, (byte) 0x40, (byte) 0xe4, (byte) 0x7e,
                                        (byte) 0xe4, (byte) 0x80, (byte) 0xe4, (byte) 0xfc, (byte) 0xe5, (byte) 0x40, (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0x80, (byte) 0xe5, (byte) 0xfc,
                                        (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0x80, (byte) 0xe6, (byte) 0xfc, (byte) 0xe7, (byte) 0x40, (byte) 0xe7, (byte) 0x7e,
                                        (byte) 0xe7, (byte) 0x80, (byte) 0xe7, (byte) 0xfc, (byte) 0xe8, (byte) 0x40, (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0x80, (byte) 0xe8, (byte) 0xfc,
                                        (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0x80, (byte) 0xe9, (byte) 0xfc, (byte) 0xea, (byte) 0x40, (byte) 0xea, (byte) 0x7e,
                                        (byte) 0xea, (byte) 0x80, (byte) 0xea, (byte) 0xfc, (byte) 0xeb, (byte) 0x40, (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0x80, (byte) 0xeb, (byte) 0xfc,
                                        (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0x80, (byte) 0xec, (byte) 0xfc, (byte) 0xed, (byte) 0x40, (byte) 0xed, (byte) 0x7e,
                                        (byte) 0xed, (byte) 0x80, (byte) 0xed, (byte) 0xfc, (byte) 0xee, (byte) 0x40, (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0x80, (byte) 0xee, (byte) 0xfc,
                                        (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0x80, (byte) 0xef, (byte) 0xfc, (byte) 0xf0, (byte) 0x40, (byte) 0xf0, (byte) 0x7e,
                                        (byte) 0xf0, (byte) 0x80, (byte) 0xf0, (byte) 0xfc, (byte) 0xf1, (byte) 0x40, (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0x80, (byte) 0xf1, (byte) 0xfc,
                                        (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0x80, (byte) 0xf2, (byte) 0xfc, (byte) 0xf3, (byte) 0x40, (byte) 0xf3, (byte) 0x7e,
                                        (byte) 0xf3, (byte) 0x80, (byte) 0xf3, (byte) 0xfc, (byte) 0xf4, (byte) 0x40, (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0x80, (byte) 0xf4, (byte) 0xfc,
                                        (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0x80, (byte) 0xf5, (byte) 0xfc, (byte) 0xf6, (byte) 0x40, (byte) 0xf6, (byte) 0x7e,
                                        (byte) 0xf6, (byte) 0x80, (byte) 0xf6, (byte) 0xfc, (byte) 0xf7, (byte) 0x40, (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0x80, (byte) 0xf7, (byte) 0xfc,
                                        (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0x80, (byte) 0xf8, (byte) 0xfc, (byte) 0xf9, (byte) 0x40, (byte) 0xf9, (byte) 0x7e,
                                        (byte) 0xf9, (byte) 0x80, (byte) 0xf9, (byte) 0xfc, (byte) 0xfa, (byte) 0x40, (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0x80, (byte) 0xfa, (byte) 0xfc,
                                        (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0x80, (byte) 0xfb, (byte) 0xfc, (byte) 0xfc, (byte) 0x40, (byte) 0xfc, (byte) 0x7e,
                                        (byte) 0xfc, (byte) 0x80, (byte) 0xfc, (byte) 0xfc},
                        null, null);
    }

    static TestData dataSJISSoftBank() {
        assert TruffleString.Encoding.SJIS_SoftBank.ordinal() == 74;
        return new TestData(new int[]{0x0000, 0x007f, 0x00a1, 0x00df, 0x8140, 0x817e, 0x8180, 0x81fc, 0x8240, 0x827e, 0x8280, 0x82fc, 0x8340, 0x837e, 0x8380, 0x83fc, 0x8440, 0x847e, 0x8480, 0x84fc,
                        0x8540, 0x857e, 0x8580, 0x85fc, 0x8640, 0x867e, 0x8680, 0x86fc, 0x8740, 0x877e, 0x8780, 0x87fc, 0x8840, 0x887e, 0x8880, 0x88fc, 0x8940, 0x897e, 0x8980, 0x89fc, 0x8a40, 0x8a7e,
                        0x8a80, 0x8afc, 0x8b40, 0x8b7e, 0x8b80, 0x8bfc, 0x8c40, 0x8c7e, 0x8c80, 0x8cfc, 0x8d40, 0x8d7e, 0x8d80, 0x8dfc, 0x8e40, 0x8e7e, 0x8e80, 0x8efc, 0x8f40, 0x8f7e, 0x8f80, 0x8ffc,
                        0x9040, 0x907e, 0x9080, 0x90fc, 0x9140, 0x917e, 0x9180, 0x91fc, 0x9240, 0x927e, 0x9280, 0x92fc, 0x9340, 0x937e, 0x9380, 0x93fc, 0x9440, 0x947e, 0x9480, 0x94fc, 0x9540, 0x957e,
                        0x9580, 0x95fc, 0x9640, 0x967e, 0x9680, 0x96fc, 0x9740, 0x977e, 0x9780, 0x97fc, 0x9840, 0x987e, 0x9880, 0x98fc, 0x9940, 0x997e, 0x9980, 0x99fc, 0x9a40, 0x9a7e, 0x9a80, 0x9afc,
                        0x9b40, 0x9b7e, 0x9b80, 0x9bfc, 0x9c40, 0x9c7e, 0x9c80, 0x9cfc, 0x9d40, 0x9d7e, 0x9d80, 0x9dfc, 0x9e40, 0x9e7e, 0x9e80, 0x9efc, 0x9f40, 0x9f7e, 0x9f80, 0x9ffc, 0xe040, 0xe07e,
                        0xe080, 0xe0fc, 0xe140, 0xe17e, 0xe180, 0xe1fc, 0xe240, 0xe27e, 0xe280, 0xe2fc, 0xe340, 0xe37e, 0xe380, 0xe3fc, 0xe440, 0xe47e, 0xe480, 0xe4fc, 0xe540, 0xe57e, 0xe580, 0xe5fc,
                        0xe640, 0xe67e, 0xe680, 0xe6fc, 0xe740, 0xe77e, 0xe780, 0xe7fc, 0xe840, 0xe87e, 0xe880, 0xe8fc, 0xe940, 0xe97e, 0xe980, 0xe9fc, 0xea40, 0xea7e, 0xea80, 0xeafc, 0xeb40, 0xeb7e,
                        0xeb80, 0xebfc, 0xec40, 0xec7e, 0xec80, 0xecfc, 0xed40, 0xed7e, 0xed80, 0xedfc, 0xee40, 0xee7e, 0xee80, 0xeefc, 0xef40, 0xef7e, 0xef80, 0xeffc, 0xf040, 0xf07e, 0xf080, 0xf0fc,
                        0xf140, 0xf17e, 0xf180, 0xf1fc, 0xf240, 0xf27e, 0xf280, 0xf2fc, 0xf340, 0xf37e, 0xf380, 0xf3fc, 0xf440, 0xf47e, 0xf480, 0xf4fc, 0xf540, 0xf57e, 0xf580, 0xf5fc, 0xf640, 0xf67e,
                        0xf680, 0xf6fc, 0xf740, 0xf77e, 0xf780, 0xf7fc, 0xf840, 0xf87e, 0xf880, 0xf8fc, 0xf940, 0xf97e, 0xf980, 0xf9fc, 0xfa40, 0xfa7e, 0xfa80, 0xfafc, 0xfb40, 0xfb7e, 0xfb80, 0xfbfc,
                        0xfc40, 0xfc7e, 0xfc80, 0xfcfc},
                        new int[]{0x000, 0x001, 0x002, 0x003, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028,
                                        0x02a, 0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056,
                                        0x058, 0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084,
                                        0x086, 0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2,
                                        0x0b4, 0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0,
                                        0x0e2, 0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e,
                                        0x110, 0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c,
                                        0x13e, 0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a,
                                        0x16c, 0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198,
                                        0x19a, 0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6,
                                        0x1c8, 0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0xdf, (byte) 0x81, (byte) 0x40, (byte) 0x81, (byte) 0x7e, (byte) 0x81, (byte) 0x80, (byte) 0x81, (byte) 0xfc,
                                        (byte) 0x82, (byte) 0x40, (byte) 0x82, (byte) 0x7e, (byte) 0x82, (byte) 0x80, (byte) 0x82, (byte) 0xfc, (byte) 0x83, (byte) 0x40, (byte) 0x83, (byte) 0x7e,
                                        (byte) 0x83, (byte) 0x80, (byte) 0x83, (byte) 0xfc, (byte) 0x84, (byte) 0x40, (byte) 0x84, (byte) 0x7e, (byte) 0x84, (byte) 0x80, (byte) 0x84, (byte) 0xfc,
                                        (byte) 0x85, (byte) 0x40, (byte) 0x85, (byte) 0x7e, (byte) 0x85, (byte) 0x80, (byte) 0x85, (byte) 0xfc, (byte) 0x86, (byte) 0x40, (byte) 0x86, (byte) 0x7e,
                                        (byte) 0x86, (byte) 0x80, (byte) 0x86, (byte) 0xfc, (byte) 0x87, (byte) 0x40, (byte) 0x87, (byte) 0x7e, (byte) 0x87, (byte) 0x80, (byte) 0x87, (byte) 0xfc,
                                        (byte) 0x88, (byte) 0x40, (byte) 0x88, (byte) 0x7e, (byte) 0x88, (byte) 0x80, (byte) 0x88, (byte) 0xfc, (byte) 0x89, (byte) 0x40, (byte) 0x89, (byte) 0x7e,
                                        (byte) 0x89, (byte) 0x80, (byte) 0x89, (byte) 0xfc, (byte) 0x8a, (byte) 0x40, (byte) 0x8a, (byte) 0x7e, (byte) 0x8a, (byte) 0x80, (byte) 0x8a, (byte) 0xfc,
                                        (byte) 0x8b, (byte) 0x40, (byte) 0x8b, (byte) 0x7e, (byte) 0x8b, (byte) 0x80, (byte) 0x8b, (byte) 0xfc, (byte) 0x8c, (byte) 0x40, (byte) 0x8c, (byte) 0x7e,
                                        (byte) 0x8c, (byte) 0x80, (byte) 0x8c, (byte) 0xfc, (byte) 0x8d, (byte) 0x40, (byte) 0x8d, (byte) 0x7e, (byte) 0x8d, (byte) 0x80, (byte) 0x8d, (byte) 0xfc,
                                        (byte) 0x8e, (byte) 0x40, (byte) 0x8e, (byte) 0x7e, (byte) 0x8e, (byte) 0x80, (byte) 0x8e, (byte) 0xfc, (byte) 0x8f, (byte) 0x40, (byte) 0x8f, (byte) 0x7e,
                                        (byte) 0x8f, (byte) 0x80, (byte) 0x8f, (byte) 0xfc, (byte) 0x90, (byte) 0x40, (byte) 0x90, (byte) 0x7e, (byte) 0x90, (byte) 0x80, (byte) 0x90, (byte) 0xfc,
                                        (byte) 0x91, (byte) 0x40, (byte) 0x91, (byte) 0x7e, (byte) 0x91, (byte) 0x80, (byte) 0x91, (byte) 0xfc, (byte) 0x92, (byte) 0x40, (byte) 0x92, (byte) 0x7e,
                                        (byte) 0x92, (byte) 0x80, (byte) 0x92, (byte) 0xfc, (byte) 0x93, (byte) 0x40, (byte) 0x93, (byte) 0x7e, (byte) 0x93, (byte) 0x80, (byte) 0x93, (byte) 0xfc,
                                        (byte) 0x94, (byte) 0x40, (byte) 0x94, (byte) 0x7e, (byte) 0x94, (byte) 0x80, (byte) 0x94, (byte) 0xfc, (byte) 0x95, (byte) 0x40, (byte) 0x95, (byte) 0x7e,
                                        (byte) 0x95, (byte) 0x80, (byte) 0x95, (byte) 0xfc, (byte) 0x96, (byte) 0x40, (byte) 0x96, (byte) 0x7e, (byte) 0x96, (byte) 0x80, (byte) 0x96, (byte) 0xfc,
                                        (byte) 0x97, (byte) 0x40, (byte) 0x97, (byte) 0x7e, (byte) 0x97, (byte) 0x80, (byte) 0x97, (byte) 0xfc, (byte) 0x98, (byte) 0x40, (byte) 0x98, (byte) 0x7e,
                                        (byte) 0x98, (byte) 0x80, (byte) 0x98, (byte) 0xfc, (byte) 0x99, (byte) 0x40, (byte) 0x99, (byte) 0x7e, (byte) 0x99, (byte) 0x80, (byte) 0x99, (byte) 0xfc,
                                        (byte) 0x9a, (byte) 0x40, (byte) 0x9a, (byte) 0x7e, (byte) 0x9a, (byte) 0x80, (byte) 0x9a, (byte) 0xfc, (byte) 0x9b, (byte) 0x40, (byte) 0x9b, (byte) 0x7e,
                                        (byte) 0x9b, (byte) 0x80, (byte) 0x9b, (byte) 0xfc, (byte) 0x9c, (byte) 0x40, (byte) 0x9c, (byte) 0x7e, (byte) 0x9c, (byte) 0x80, (byte) 0x9c, (byte) 0xfc,
                                        (byte) 0x9d, (byte) 0x40, (byte) 0x9d, (byte) 0x7e, (byte) 0x9d, (byte) 0x80, (byte) 0x9d, (byte) 0xfc, (byte) 0x9e, (byte) 0x40, (byte) 0x9e, (byte) 0x7e,
                                        (byte) 0x9e, (byte) 0x80, (byte) 0x9e, (byte) 0xfc, (byte) 0x9f, (byte) 0x40, (byte) 0x9f, (byte) 0x7e, (byte) 0x9f, (byte) 0x80, (byte) 0x9f, (byte) 0xfc,
                                        (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0x80, (byte) 0xe0, (byte) 0xfc, (byte) 0xe1, (byte) 0x40, (byte) 0xe1, (byte) 0x7e,
                                        (byte) 0xe1, (byte) 0x80, (byte) 0xe1, (byte) 0xfc, (byte) 0xe2, (byte) 0x40, (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0x80, (byte) 0xe2, (byte) 0xfc,
                                        (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0x80, (byte) 0xe3, (byte) 0xfc, (byte) 0xe4, (byte) 0x40, (byte) 0xe4, (byte) 0x7e,
                                        (byte) 0xe4, (byte) 0x80, (byte) 0xe4, (byte) 0xfc, (byte) 0xe5, (byte) 0x40, (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0x80, (byte) 0xe5, (byte) 0xfc,
                                        (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0x80, (byte) 0xe6, (byte) 0xfc, (byte) 0xe7, (byte) 0x40, (byte) 0xe7, (byte) 0x7e,
                                        (byte) 0xe7, (byte) 0x80, (byte) 0xe7, (byte) 0xfc, (byte) 0xe8, (byte) 0x40, (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0x80, (byte) 0xe8, (byte) 0xfc,
                                        (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0x80, (byte) 0xe9, (byte) 0xfc, (byte) 0xea, (byte) 0x40, (byte) 0xea, (byte) 0x7e,
                                        (byte) 0xea, (byte) 0x80, (byte) 0xea, (byte) 0xfc, (byte) 0xeb, (byte) 0x40, (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0x80, (byte) 0xeb, (byte) 0xfc,
                                        (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0x80, (byte) 0xec, (byte) 0xfc, (byte) 0xed, (byte) 0x40, (byte) 0xed, (byte) 0x7e,
                                        (byte) 0xed, (byte) 0x80, (byte) 0xed, (byte) 0xfc, (byte) 0xee, (byte) 0x40, (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0x80, (byte) 0xee, (byte) 0xfc,
                                        (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0x80, (byte) 0xef, (byte) 0xfc, (byte) 0xf0, (byte) 0x40, (byte) 0xf0, (byte) 0x7e,
                                        (byte) 0xf0, (byte) 0x80, (byte) 0xf0, (byte) 0xfc, (byte) 0xf1, (byte) 0x40, (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0x80, (byte) 0xf1, (byte) 0xfc,
                                        (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0x80, (byte) 0xf2, (byte) 0xfc, (byte) 0xf3, (byte) 0x40, (byte) 0xf3, (byte) 0x7e,
                                        (byte) 0xf3, (byte) 0x80, (byte) 0xf3, (byte) 0xfc, (byte) 0xf4, (byte) 0x40, (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0x80, (byte) 0xf4, (byte) 0xfc,
                                        (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0x80, (byte) 0xf5, (byte) 0xfc, (byte) 0xf6, (byte) 0x40, (byte) 0xf6, (byte) 0x7e,
                                        (byte) 0xf6, (byte) 0x80, (byte) 0xf6, (byte) 0xfc, (byte) 0xf7, (byte) 0x40, (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0x80, (byte) 0xf7, (byte) 0xfc,
                                        (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0x80, (byte) 0xf8, (byte) 0xfc, (byte) 0xf9, (byte) 0x40, (byte) 0xf9, (byte) 0x7e,
                                        (byte) 0xf9, (byte) 0x80, (byte) 0xf9, (byte) 0xfc, (byte) 0xfa, (byte) 0x40, (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0x80, (byte) 0xfa, (byte) 0xfc,
                                        (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0x80, (byte) 0xfb, (byte) 0xfc, (byte) 0xfc, (byte) 0x40, (byte) 0xfc, (byte) 0x7e,
                                        (byte) 0xfc, (byte) 0x80, (byte) 0xfc, (byte) 0xfc},
                        null, null);
    }

    static TestData dataShiftJIS() {
        assert TruffleString.Encoding.Shift_JIS.ordinal() == 75;
        return new TestData(new int[]{0x0000, 0x007f, 0x00a1, 0x00df, 0x8140, 0x817e, 0x8180, 0x81fc, 0x8240, 0x827e, 0x8280, 0x82fc, 0x8340, 0x837e, 0x8380, 0x83fc, 0x8440, 0x847e, 0x8480, 0x84fc,
                        0x8540, 0x857e, 0x8580, 0x85fc, 0x8640, 0x867e, 0x8680, 0x86fc, 0x8740, 0x877e, 0x8780, 0x87fc, 0x8840, 0x887e, 0x8880, 0x88fc, 0x8940, 0x897e, 0x8980, 0x89fc, 0x8a40, 0x8a7e,
                        0x8a80, 0x8afc, 0x8b40, 0x8b7e, 0x8b80, 0x8bfc, 0x8c40, 0x8c7e, 0x8c80, 0x8cfc, 0x8d40, 0x8d7e, 0x8d80, 0x8dfc, 0x8e40, 0x8e7e, 0x8e80, 0x8efc, 0x8f40, 0x8f7e, 0x8f80, 0x8ffc,
                        0x9040, 0x907e, 0x9080, 0x90fc, 0x9140, 0x917e, 0x9180, 0x91fc, 0x9240, 0x927e, 0x9280, 0x92fc, 0x9340, 0x937e, 0x9380, 0x93fc, 0x9440, 0x947e, 0x9480, 0x94fc, 0x9540, 0x957e,
                        0x9580, 0x95fc, 0x9640, 0x967e, 0x9680, 0x96fc, 0x9740, 0x977e, 0x9780, 0x97fc, 0x9840, 0x987e, 0x9880, 0x98fc, 0x9940, 0x997e, 0x9980, 0x99fc, 0x9a40, 0x9a7e, 0x9a80, 0x9afc,
                        0x9b40, 0x9b7e, 0x9b80, 0x9bfc, 0x9c40, 0x9c7e, 0x9c80, 0x9cfc, 0x9d40, 0x9d7e, 0x9d80, 0x9dfc, 0x9e40, 0x9e7e, 0x9e80, 0x9efc, 0x9f40, 0x9f7e, 0x9f80, 0x9ffc, 0xe040, 0xe07e,
                        0xe080, 0xe0fc, 0xe140, 0xe17e, 0xe180, 0xe1fc, 0xe240, 0xe27e, 0xe280, 0xe2fc, 0xe340, 0xe37e, 0xe380, 0xe3fc, 0xe440, 0xe47e, 0xe480, 0xe4fc, 0xe540, 0xe57e, 0xe580, 0xe5fc,
                        0xe640, 0xe67e, 0xe680, 0xe6fc, 0xe740, 0xe77e, 0xe780, 0xe7fc, 0xe840, 0xe87e, 0xe880, 0xe8fc, 0xe940, 0xe97e, 0xe980, 0xe9fc, 0xea40, 0xea7e, 0xea80, 0xeafc, 0xeb40, 0xeb7e,
                        0xeb80, 0xebfc, 0xec40, 0xec7e, 0xec80, 0xecfc, 0xed40, 0xed7e, 0xed80, 0xedfc, 0xee40, 0xee7e, 0xee80, 0xeefc, 0xef40, 0xef7e, 0xef80, 0xeffc, 0xf040, 0xf07e, 0xf080, 0xf0fc,
                        0xf140, 0xf17e, 0xf180, 0xf1fc, 0xf240, 0xf27e, 0xf280, 0xf2fc, 0xf340, 0xf37e, 0xf380, 0xf3fc, 0xf440, 0xf47e, 0xf480, 0xf4fc, 0xf540, 0xf57e, 0xf580, 0xf5fc, 0xf640, 0xf67e,
                        0xf680, 0xf6fc, 0xf740, 0xf77e, 0xf780, 0xf7fc, 0xf840, 0xf87e, 0xf880, 0xf8fc, 0xf940, 0xf97e, 0xf980, 0xf9fc, 0xfa40, 0xfa7e, 0xfa80, 0xfafc, 0xfb40, 0xfb7e, 0xfb80, 0xfbfc,
                        0xfc40, 0xfc7e, 0xfc80, 0xfcfc},
                        new int[]{0x000, 0x001, 0x002, 0x003, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028,
                                        0x02a, 0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056,
                                        0x058, 0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084,
                                        0x086, 0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2,
                                        0x0b4, 0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0,
                                        0x0e2, 0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e,
                                        0x110, 0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c,
                                        0x13e, 0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a,
                                        0x16c, 0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198,
                                        0x19a, 0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6,
                                        0x1c8, 0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0xdf, (byte) 0x81, (byte) 0x40, (byte) 0x81, (byte) 0x7e, (byte) 0x81, (byte) 0x80, (byte) 0x81, (byte) 0xfc,
                                        (byte) 0x82, (byte) 0x40, (byte) 0x82, (byte) 0x7e, (byte) 0x82, (byte) 0x80, (byte) 0x82, (byte) 0xfc, (byte) 0x83, (byte) 0x40, (byte) 0x83, (byte) 0x7e,
                                        (byte) 0x83, (byte) 0x80, (byte) 0x83, (byte) 0xfc, (byte) 0x84, (byte) 0x40, (byte) 0x84, (byte) 0x7e, (byte) 0x84, (byte) 0x80, (byte) 0x84, (byte) 0xfc,
                                        (byte) 0x85, (byte) 0x40, (byte) 0x85, (byte) 0x7e, (byte) 0x85, (byte) 0x80, (byte) 0x85, (byte) 0xfc, (byte) 0x86, (byte) 0x40, (byte) 0x86, (byte) 0x7e,
                                        (byte) 0x86, (byte) 0x80, (byte) 0x86, (byte) 0xfc, (byte) 0x87, (byte) 0x40, (byte) 0x87, (byte) 0x7e, (byte) 0x87, (byte) 0x80, (byte) 0x87, (byte) 0xfc,
                                        (byte) 0x88, (byte) 0x40, (byte) 0x88, (byte) 0x7e, (byte) 0x88, (byte) 0x80, (byte) 0x88, (byte) 0xfc, (byte) 0x89, (byte) 0x40, (byte) 0x89, (byte) 0x7e,
                                        (byte) 0x89, (byte) 0x80, (byte) 0x89, (byte) 0xfc, (byte) 0x8a, (byte) 0x40, (byte) 0x8a, (byte) 0x7e, (byte) 0x8a, (byte) 0x80, (byte) 0x8a, (byte) 0xfc,
                                        (byte) 0x8b, (byte) 0x40, (byte) 0x8b, (byte) 0x7e, (byte) 0x8b, (byte) 0x80, (byte) 0x8b, (byte) 0xfc, (byte) 0x8c, (byte) 0x40, (byte) 0x8c, (byte) 0x7e,
                                        (byte) 0x8c, (byte) 0x80, (byte) 0x8c, (byte) 0xfc, (byte) 0x8d, (byte) 0x40, (byte) 0x8d, (byte) 0x7e, (byte) 0x8d, (byte) 0x80, (byte) 0x8d, (byte) 0xfc,
                                        (byte) 0x8e, (byte) 0x40, (byte) 0x8e, (byte) 0x7e, (byte) 0x8e, (byte) 0x80, (byte) 0x8e, (byte) 0xfc, (byte) 0x8f, (byte) 0x40, (byte) 0x8f, (byte) 0x7e,
                                        (byte) 0x8f, (byte) 0x80, (byte) 0x8f, (byte) 0xfc, (byte) 0x90, (byte) 0x40, (byte) 0x90, (byte) 0x7e, (byte) 0x90, (byte) 0x80, (byte) 0x90, (byte) 0xfc,
                                        (byte) 0x91, (byte) 0x40, (byte) 0x91, (byte) 0x7e, (byte) 0x91, (byte) 0x80, (byte) 0x91, (byte) 0xfc, (byte) 0x92, (byte) 0x40, (byte) 0x92, (byte) 0x7e,
                                        (byte) 0x92, (byte) 0x80, (byte) 0x92, (byte) 0xfc, (byte) 0x93, (byte) 0x40, (byte) 0x93, (byte) 0x7e, (byte) 0x93, (byte) 0x80, (byte) 0x93, (byte) 0xfc,
                                        (byte) 0x94, (byte) 0x40, (byte) 0x94, (byte) 0x7e, (byte) 0x94, (byte) 0x80, (byte) 0x94, (byte) 0xfc, (byte) 0x95, (byte) 0x40, (byte) 0x95, (byte) 0x7e,
                                        (byte) 0x95, (byte) 0x80, (byte) 0x95, (byte) 0xfc, (byte) 0x96, (byte) 0x40, (byte) 0x96, (byte) 0x7e, (byte) 0x96, (byte) 0x80, (byte) 0x96, (byte) 0xfc,
                                        (byte) 0x97, (byte) 0x40, (byte) 0x97, (byte) 0x7e, (byte) 0x97, (byte) 0x80, (byte) 0x97, (byte) 0xfc, (byte) 0x98, (byte) 0x40, (byte) 0x98, (byte) 0x7e,
                                        (byte) 0x98, (byte) 0x80, (byte) 0x98, (byte) 0xfc, (byte) 0x99, (byte) 0x40, (byte) 0x99, (byte) 0x7e, (byte) 0x99, (byte) 0x80, (byte) 0x99, (byte) 0xfc,
                                        (byte) 0x9a, (byte) 0x40, (byte) 0x9a, (byte) 0x7e, (byte) 0x9a, (byte) 0x80, (byte) 0x9a, (byte) 0xfc, (byte) 0x9b, (byte) 0x40, (byte) 0x9b, (byte) 0x7e,
                                        (byte) 0x9b, (byte) 0x80, (byte) 0x9b, (byte) 0xfc, (byte) 0x9c, (byte) 0x40, (byte) 0x9c, (byte) 0x7e, (byte) 0x9c, (byte) 0x80, (byte) 0x9c, (byte) 0xfc,
                                        (byte) 0x9d, (byte) 0x40, (byte) 0x9d, (byte) 0x7e, (byte) 0x9d, (byte) 0x80, (byte) 0x9d, (byte) 0xfc, (byte) 0x9e, (byte) 0x40, (byte) 0x9e, (byte) 0x7e,
                                        (byte) 0x9e, (byte) 0x80, (byte) 0x9e, (byte) 0xfc, (byte) 0x9f, (byte) 0x40, (byte) 0x9f, (byte) 0x7e, (byte) 0x9f, (byte) 0x80, (byte) 0x9f, (byte) 0xfc,
                                        (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0x80, (byte) 0xe0, (byte) 0xfc, (byte) 0xe1, (byte) 0x40, (byte) 0xe1, (byte) 0x7e,
                                        (byte) 0xe1, (byte) 0x80, (byte) 0xe1, (byte) 0xfc, (byte) 0xe2, (byte) 0x40, (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0x80, (byte) 0xe2, (byte) 0xfc,
                                        (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0x80, (byte) 0xe3, (byte) 0xfc, (byte) 0xe4, (byte) 0x40, (byte) 0xe4, (byte) 0x7e,
                                        (byte) 0xe4, (byte) 0x80, (byte) 0xe4, (byte) 0xfc, (byte) 0xe5, (byte) 0x40, (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0x80, (byte) 0xe5, (byte) 0xfc,
                                        (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0x80, (byte) 0xe6, (byte) 0xfc, (byte) 0xe7, (byte) 0x40, (byte) 0xe7, (byte) 0x7e,
                                        (byte) 0xe7, (byte) 0x80, (byte) 0xe7, (byte) 0xfc, (byte) 0xe8, (byte) 0x40, (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0x80, (byte) 0xe8, (byte) 0xfc,
                                        (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0x80, (byte) 0xe9, (byte) 0xfc, (byte) 0xea, (byte) 0x40, (byte) 0xea, (byte) 0x7e,
                                        (byte) 0xea, (byte) 0x80, (byte) 0xea, (byte) 0xfc, (byte) 0xeb, (byte) 0x40, (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0x80, (byte) 0xeb, (byte) 0xfc,
                                        (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0x80, (byte) 0xec, (byte) 0xfc, (byte) 0xed, (byte) 0x40, (byte) 0xed, (byte) 0x7e,
                                        (byte) 0xed, (byte) 0x80, (byte) 0xed, (byte) 0xfc, (byte) 0xee, (byte) 0x40, (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0x80, (byte) 0xee, (byte) 0xfc,
                                        (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0x80, (byte) 0xef, (byte) 0xfc, (byte) 0xf0, (byte) 0x40, (byte) 0xf0, (byte) 0x7e,
                                        (byte) 0xf0, (byte) 0x80, (byte) 0xf0, (byte) 0xfc, (byte) 0xf1, (byte) 0x40, (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0x80, (byte) 0xf1, (byte) 0xfc,
                                        (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0x80, (byte) 0xf2, (byte) 0xfc, (byte) 0xf3, (byte) 0x40, (byte) 0xf3, (byte) 0x7e,
                                        (byte) 0xf3, (byte) 0x80, (byte) 0xf3, (byte) 0xfc, (byte) 0xf4, (byte) 0x40, (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0x80, (byte) 0xf4, (byte) 0xfc,
                                        (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0x80, (byte) 0xf5, (byte) 0xfc, (byte) 0xf6, (byte) 0x40, (byte) 0xf6, (byte) 0x7e,
                                        (byte) 0xf6, (byte) 0x80, (byte) 0xf6, (byte) 0xfc, (byte) 0xf7, (byte) 0x40, (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0x80, (byte) 0xf7, (byte) 0xfc,
                                        (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0x80, (byte) 0xf8, (byte) 0xfc, (byte) 0xf9, (byte) 0x40, (byte) 0xf9, (byte) 0x7e,
                                        (byte) 0xf9, (byte) 0x80, (byte) 0xf9, (byte) 0xfc, (byte) 0xfa, (byte) 0x40, (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0x80, (byte) 0xfa, (byte) 0xfc,
                                        (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0x80, (byte) 0xfb, (byte) 0xfc, (byte) 0xfc, (byte) 0x40, (byte) 0xfc, (byte) 0x7e,
                                        (byte) 0xfc, (byte) 0x80, (byte) 0xfc, (byte) 0xfc},
                        null, null);
    }

    static TestData dataStatelessISO2022JP() {
        assert TruffleString.Encoding.Stateless_ISO_2022_JP.ordinal() == 76;
        return new TestData(new int[]{0x00, 0x7f}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null);
    }

    static TestData dataStatelessISO2022JPKDDI() {
        assert TruffleString.Encoding.Stateless_ISO_2022_JP_KDDI.ordinal() == 77;
        return new TestData(new int[]{0x00, 0x7f}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null);
    }

    static TestData dataTIS620() {
        assert TruffleString.Encoding.TIS_620.ordinal() == 78;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataUTF8DoCoMo() {
        assert TruffleString.Encoding.UTF8_DoCoMo.ordinal() == 79;
        return new TestData(new int[]{0x000000, 0x00d7ff, 0x00e000, 0x10ffff}, new int[]{0x0, 0x1, 0x4, 0x7}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0xed, (byte) 0x9f, (byte) 0xbf, (byte) 0xee, (byte) 0x80, (byte) 0x80, (byte) 0xf4, (byte) 0x8f, (byte) 0xbf, (byte) 0xbf}, null, null);
    }

    static TestData dataUTF8KDDI() {
        assert TruffleString.Encoding.UTF8_KDDI.ordinal() == 80;
        return new TestData(new int[]{0x000000, 0x00d7ff, 0x00e000, 0x10ffff}, new int[]{0x0, 0x1, 0x4, 0x7}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0xed, (byte) 0x9f, (byte) 0xbf, (byte) 0xee, (byte) 0x80, (byte) 0x80, (byte) 0xf4, (byte) 0x8f, (byte) 0xbf, (byte) 0xbf}, null, null);
    }

    static TestData dataUTF8MAC() {
        assert TruffleString.Encoding.UTF8_MAC.ordinal() == 81;
        return new TestData(new int[]{0x000000, 0x00d7ff, 0x00e000, 0x10ffff}, new int[]{0x0, 0x1, 0x4, 0x7}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0xed, (byte) 0x9f, (byte) 0xbf, (byte) 0xee, (byte) 0x80, (byte) 0x80, (byte) 0xf4, (byte) 0x8f, (byte) 0xbf, (byte) 0xbf}, null, null);
    }

    static TestData dataUTF8SoftBank() {
        assert TruffleString.Encoding.UTF8_SoftBank.ordinal() == 82;
        return new TestData(new int[]{0x000000, 0x00d7ff, 0x00e000, 0x10ffff}, new int[]{0x0, 0x1, 0x4, 0x7}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0xed, (byte) 0x9f, (byte) 0xbf, (byte) 0xee, (byte) 0x80, (byte) 0x80, (byte) 0xf4, (byte) 0x8f, (byte) 0xbf, (byte) 0xbf}, null, null);
    }

    static TestData dataWindows1250() {
        assert TruffleString.Encoding.Windows_1250.ordinal() == 83;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataWindows1251() {
        assert TruffleString.Encoding.Windows_1251.ordinal() == 84;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataWindows1252() {
        assert TruffleString.Encoding.Windows_1252.ordinal() == 85;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataWindows1253() {
        assert TruffleString.Encoding.Windows_1253.ordinal() == 86;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataWindows1254() {
        assert TruffleString.Encoding.Windows_1254.ordinal() == 87;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataWindows1255() {
        assert TruffleString.Encoding.Windows_1255.ordinal() == 88;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataWindows1256() {
        assert TruffleString.Encoding.Windows_1256.ordinal() == 89;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataWindows1257() {
        assert TruffleString.Encoding.Windows_1257.ordinal() == 90;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataWindows1258() {
        assert TruffleString.Encoding.Windows_1258.ordinal() == 91;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataWindows31J() {
        assert TruffleString.Encoding.Windows_31J.ordinal() == 92;
        return new TestData(new int[]{0x0000, 0x007f, 0x00a1, 0x00df, 0x8140, 0x817e, 0x8180, 0x81fc, 0x8240, 0x827e, 0x8280, 0x82fc, 0x8340, 0x837e, 0x8380, 0x83fc, 0x8440, 0x847e, 0x8480, 0x84fc,
                        0x8540, 0x857e, 0x8580, 0x85fc, 0x8640, 0x867e, 0x8680, 0x86fc, 0x8740, 0x877e, 0x8780, 0x87fc, 0x8840, 0x887e, 0x8880, 0x88fc, 0x8940, 0x897e, 0x8980, 0x89fc, 0x8a40, 0x8a7e,
                        0x8a80, 0x8afc, 0x8b40, 0x8b7e, 0x8b80, 0x8bfc, 0x8c40, 0x8c7e, 0x8c80, 0x8cfc, 0x8d40, 0x8d7e, 0x8d80, 0x8dfc, 0x8e40, 0x8e7e, 0x8e80, 0x8efc, 0x8f40, 0x8f7e, 0x8f80, 0x8ffc,
                        0x9040, 0x907e, 0x9080, 0x90fc, 0x9140, 0x917e, 0x9180, 0x91fc, 0x9240, 0x927e, 0x9280, 0x92fc, 0x9340, 0x937e, 0x9380, 0x93fc, 0x9440, 0x947e, 0x9480, 0x94fc, 0x9540, 0x957e,
                        0x9580, 0x95fc, 0x9640, 0x967e, 0x9680, 0x96fc, 0x9740, 0x977e, 0x9780, 0x97fc, 0x9840, 0x987e, 0x9880, 0x98fc, 0x9940, 0x997e, 0x9980, 0x99fc, 0x9a40, 0x9a7e, 0x9a80, 0x9afc,
                        0x9b40, 0x9b7e, 0x9b80, 0x9bfc, 0x9c40, 0x9c7e, 0x9c80, 0x9cfc, 0x9d40, 0x9d7e, 0x9d80, 0x9dfc, 0x9e40, 0x9e7e, 0x9e80, 0x9efc, 0x9f40, 0x9f7e, 0x9f80, 0x9ffc, 0xe040, 0xe07e,
                        0xe080, 0xe0fc, 0xe140, 0xe17e, 0xe180, 0xe1fc, 0xe240, 0xe27e, 0xe280, 0xe2fc, 0xe340, 0xe37e, 0xe380, 0xe3fc, 0xe440, 0xe47e, 0xe480, 0xe4fc, 0xe540, 0xe57e, 0xe580, 0xe5fc,
                        0xe640, 0xe67e, 0xe680, 0xe6fc, 0xe740, 0xe77e, 0xe780, 0xe7fc, 0xe840, 0xe87e, 0xe880, 0xe8fc, 0xe940, 0xe97e, 0xe980, 0xe9fc, 0xea40, 0xea7e, 0xea80, 0xeafc, 0xeb40, 0xeb7e,
                        0xeb80, 0xebfc, 0xec40, 0xec7e, 0xec80, 0xecfc, 0xed40, 0xed7e, 0xed80, 0xedfc, 0xee40, 0xee7e, 0xee80, 0xeefc, 0xef40, 0xef7e, 0xef80, 0xeffc, 0xf040, 0xf07e, 0xf080, 0xf0fc,
                        0xf140, 0xf17e, 0xf180, 0xf1fc, 0xf240, 0xf27e, 0xf280, 0xf2fc, 0xf340, 0xf37e, 0xf380, 0xf3fc, 0xf440, 0xf47e, 0xf480, 0xf4fc, 0xf540, 0xf57e, 0xf580, 0xf5fc, 0xf640, 0xf67e,
                        0xf680, 0xf6fc, 0xf740, 0xf77e, 0xf780, 0xf7fc, 0xf840, 0xf87e, 0xf880, 0xf8fc, 0xf940, 0xf97e, 0xf980, 0xf9fc, 0xfa40, 0xfa7e, 0xfa80, 0xfafc, 0xfb40, 0xfb7e, 0xfb80, 0xfbfc,
                        0xfc40, 0xfc7e, 0xfc80, 0xfcfc},
                        new int[]{0x000, 0x001, 0x002, 0x003, 0x004, 0x006, 0x008, 0x00a, 0x00c, 0x00e, 0x010, 0x012, 0x014, 0x016, 0x018, 0x01a, 0x01c, 0x01e, 0x020, 0x022, 0x024, 0x026, 0x028,
                                        0x02a, 0x02c, 0x02e, 0x030, 0x032, 0x034, 0x036, 0x038, 0x03a, 0x03c, 0x03e, 0x040, 0x042, 0x044, 0x046, 0x048, 0x04a, 0x04c, 0x04e, 0x050, 0x052, 0x054, 0x056,
                                        0x058, 0x05a, 0x05c, 0x05e, 0x060, 0x062, 0x064, 0x066, 0x068, 0x06a, 0x06c, 0x06e, 0x070, 0x072, 0x074, 0x076, 0x078, 0x07a, 0x07c, 0x07e, 0x080, 0x082, 0x084,
                                        0x086, 0x088, 0x08a, 0x08c, 0x08e, 0x090, 0x092, 0x094, 0x096, 0x098, 0x09a, 0x09c, 0x09e, 0x0a0, 0x0a2, 0x0a4, 0x0a6, 0x0a8, 0x0aa, 0x0ac, 0x0ae, 0x0b0, 0x0b2,
                                        0x0b4, 0x0b6, 0x0b8, 0x0ba, 0x0bc, 0x0be, 0x0c0, 0x0c2, 0x0c4, 0x0c6, 0x0c8, 0x0ca, 0x0cc, 0x0ce, 0x0d0, 0x0d2, 0x0d4, 0x0d6, 0x0d8, 0x0da, 0x0dc, 0x0de, 0x0e0,
                                        0x0e2, 0x0e4, 0x0e6, 0x0e8, 0x0ea, 0x0ec, 0x0ee, 0x0f0, 0x0f2, 0x0f4, 0x0f6, 0x0f8, 0x0fa, 0x0fc, 0x0fe, 0x100, 0x102, 0x104, 0x106, 0x108, 0x10a, 0x10c, 0x10e,
                                        0x110, 0x112, 0x114, 0x116, 0x118, 0x11a, 0x11c, 0x11e, 0x120, 0x122, 0x124, 0x126, 0x128, 0x12a, 0x12c, 0x12e, 0x130, 0x132, 0x134, 0x136, 0x138, 0x13a, 0x13c,
                                        0x13e, 0x140, 0x142, 0x144, 0x146, 0x148, 0x14a, 0x14c, 0x14e, 0x150, 0x152, 0x154, 0x156, 0x158, 0x15a, 0x15c, 0x15e, 0x160, 0x162, 0x164, 0x166, 0x168, 0x16a,
                                        0x16c, 0x16e, 0x170, 0x172, 0x174, 0x176, 0x178, 0x17a, 0x17c, 0x17e, 0x180, 0x182, 0x184, 0x186, 0x188, 0x18a, 0x18c, 0x18e, 0x190, 0x192, 0x194, 0x196, 0x198,
                                        0x19a, 0x19c, 0x19e, 0x1a0, 0x1a2, 0x1a4, 0x1a6, 0x1a8, 0x1aa, 0x1ac, 0x1ae, 0x1b0, 0x1b2, 0x1b4, 0x1b6, 0x1b8, 0x1ba, 0x1bc, 0x1be, 0x1c0, 0x1c2, 0x1c4, 0x1c6,
                                        0x1c8, 0x1ca, 0x1cc, 0x1ce, 0x1d0, 0x1d2, 0x1d4, 0x1d6, 0x1d8, 0x1da, 0x1dc, 0x1de, 0x1e0, 0x1e2},
                        new byte[]{(byte) 0x00, (byte) 0x7f}, null, null,
                        new byte[]{(byte) 0x00, (byte) 0x7f, (byte) 0xa1, (byte) 0xdf, (byte) 0x81, (byte) 0x40, (byte) 0x81, (byte) 0x7e, (byte) 0x81, (byte) 0x80, (byte) 0x81, (byte) 0xfc,
                                        (byte) 0x82, (byte) 0x40, (byte) 0x82, (byte) 0x7e, (byte) 0x82, (byte) 0x80, (byte) 0x82, (byte) 0xfc, (byte) 0x83, (byte) 0x40, (byte) 0x83, (byte) 0x7e,
                                        (byte) 0x83, (byte) 0x80, (byte) 0x83, (byte) 0xfc, (byte) 0x84, (byte) 0x40, (byte) 0x84, (byte) 0x7e, (byte) 0x84, (byte) 0x80, (byte) 0x84, (byte) 0xfc,
                                        (byte) 0x85, (byte) 0x40, (byte) 0x85, (byte) 0x7e, (byte) 0x85, (byte) 0x80, (byte) 0x85, (byte) 0xfc, (byte) 0x86, (byte) 0x40, (byte) 0x86, (byte) 0x7e,
                                        (byte) 0x86, (byte) 0x80, (byte) 0x86, (byte) 0xfc, (byte) 0x87, (byte) 0x40, (byte) 0x87, (byte) 0x7e, (byte) 0x87, (byte) 0x80, (byte) 0x87, (byte) 0xfc,
                                        (byte) 0x88, (byte) 0x40, (byte) 0x88, (byte) 0x7e, (byte) 0x88, (byte) 0x80, (byte) 0x88, (byte) 0xfc, (byte) 0x89, (byte) 0x40, (byte) 0x89, (byte) 0x7e,
                                        (byte) 0x89, (byte) 0x80, (byte) 0x89, (byte) 0xfc, (byte) 0x8a, (byte) 0x40, (byte) 0x8a, (byte) 0x7e, (byte) 0x8a, (byte) 0x80, (byte) 0x8a, (byte) 0xfc,
                                        (byte) 0x8b, (byte) 0x40, (byte) 0x8b, (byte) 0x7e, (byte) 0x8b, (byte) 0x80, (byte) 0x8b, (byte) 0xfc, (byte) 0x8c, (byte) 0x40, (byte) 0x8c, (byte) 0x7e,
                                        (byte) 0x8c, (byte) 0x80, (byte) 0x8c, (byte) 0xfc, (byte) 0x8d, (byte) 0x40, (byte) 0x8d, (byte) 0x7e, (byte) 0x8d, (byte) 0x80, (byte) 0x8d, (byte) 0xfc,
                                        (byte) 0x8e, (byte) 0x40, (byte) 0x8e, (byte) 0x7e, (byte) 0x8e, (byte) 0x80, (byte) 0x8e, (byte) 0xfc, (byte) 0x8f, (byte) 0x40, (byte) 0x8f, (byte) 0x7e,
                                        (byte) 0x8f, (byte) 0x80, (byte) 0x8f, (byte) 0xfc, (byte) 0x90, (byte) 0x40, (byte) 0x90, (byte) 0x7e, (byte) 0x90, (byte) 0x80, (byte) 0x90, (byte) 0xfc,
                                        (byte) 0x91, (byte) 0x40, (byte) 0x91, (byte) 0x7e, (byte) 0x91, (byte) 0x80, (byte) 0x91, (byte) 0xfc, (byte) 0x92, (byte) 0x40, (byte) 0x92, (byte) 0x7e,
                                        (byte) 0x92, (byte) 0x80, (byte) 0x92, (byte) 0xfc, (byte) 0x93, (byte) 0x40, (byte) 0x93, (byte) 0x7e, (byte) 0x93, (byte) 0x80, (byte) 0x93, (byte) 0xfc,
                                        (byte) 0x94, (byte) 0x40, (byte) 0x94, (byte) 0x7e, (byte) 0x94, (byte) 0x80, (byte) 0x94, (byte) 0xfc, (byte) 0x95, (byte) 0x40, (byte) 0x95, (byte) 0x7e,
                                        (byte) 0x95, (byte) 0x80, (byte) 0x95, (byte) 0xfc, (byte) 0x96, (byte) 0x40, (byte) 0x96, (byte) 0x7e, (byte) 0x96, (byte) 0x80, (byte) 0x96, (byte) 0xfc,
                                        (byte) 0x97, (byte) 0x40, (byte) 0x97, (byte) 0x7e, (byte) 0x97, (byte) 0x80, (byte) 0x97, (byte) 0xfc, (byte) 0x98, (byte) 0x40, (byte) 0x98, (byte) 0x7e,
                                        (byte) 0x98, (byte) 0x80, (byte) 0x98, (byte) 0xfc, (byte) 0x99, (byte) 0x40, (byte) 0x99, (byte) 0x7e, (byte) 0x99, (byte) 0x80, (byte) 0x99, (byte) 0xfc,
                                        (byte) 0x9a, (byte) 0x40, (byte) 0x9a, (byte) 0x7e, (byte) 0x9a, (byte) 0x80, (byte) 0x9a, (byte) 0xfc, (byte) 0x9b, (byte) 0x40, (byte) 0x9b, (byte) 0x7e,
                                        (byte) 0x9b, (byte) 0x80, (byte) 0x9b, (byte) 0xfc, (byte) 0x9c, (byte) 0x40, (byte) 0x9c, (byte) 0x7e, (byte) 0x9c, (byte) 0x80, (byte) 0x9c, (byte) 0xfc,
                                        (byte) 0x9d, (byte) 0x40, (byte) 0x9d, (byte) 0x7e, (byte) 0x9d, (byte) 0x80, (byte) 0x9d, (byte) 0xfc, (byte) 0x9e, (byte) 0x40, (byte) 0x9e, (byte) 0x7e,
                                        (byte) 0x9e, (byte) 0x80, (byte) 0x9e, (byte) 0xfc, (byte) 0x9f, (byte) 0x40, (byte) 0x9f, (byte) 0x7e, (byte) 0x9f, (byte) 0x80, (byte) 0x9f, (byte) 0xfc,
                                        (byte) 0xe0, (byte) 0x40, (byte) 0xe0, (byte) 0x7e, (byte) 0xe0, (byte) 0x80, (byte) 0xe0, (byte) 0xfc, (byte) 0xe1, (byte) 0x40, (byte) 0xe1, (byte) 0x7e,
                                        (byte) 0xe1, (byte) 0x80, (byte) 0xe1, (byte) 0xfc, (byte) 0xe2, (byte) 0x40, (byte) 0xe2, (byte) 0x7e, (byte) 0xe2, (byte) 0x80, (byte) 0xe2, (byte) 0xfc,
                                        (byte) 0xe3, (byte) 0x40, (byte) 0xe3, (byte) 0x7e, (byte) 0xe3, (byte) 0x80, (byte) 0xe3, (byte) 0xfc, (byte) 0xe4, (byte) 0x40, (byte) 0xe4, (byte) 0x7e,
                                        (byte) 0xe4, (byte) 0x80, (byte) 0xe4, (byte) 0xfc, (byte) 0xe5, (byte) 0x40, (byte) 0xe5, (byte) 0x7e, (byte) 0xe5, (byte) 0x80, (byte) 0xe5, (byte) 0xfc,
                                        (byte) 0xe6, (byte) 0x40, (byte) 0xe6, (byte) 0x7e, (byte) 0xe6, (byte) 0x80, (byte) 0xe6, (byte) 0xfc, (byte) 0xe7, (byte) 0x40, (byte) 0xe7, (byte) 0x7e,
                                        (byte) 0xe7, (byte) 0x80, (byte) 0xe7, (byte) 0xfc, (byte) 0xe8, (byte) 0x40, (byte) 0xe8, (byte) 0x7e, (byte) 0xe8, (byte) 0x80, (byte) 0xe8, (byte) 0xfc,
                                        (byte) 0xe9, (byte) 0x40, (byte) 0xe9, (byte) 0x7e, (byte) 0xe9, (byte) 0x80, (byte) 0xe9, (byte) 0xfc, (byte) 0xea, (byte) 0x40, (byte) 0xea, (byte) 0x7e,
                                        (byte) 0xea, (byte) 0x80, (byte) 0xea, (byte) 0xfc, (byte) 0xeb, (byte) 0x40, (byte) 0xeb, (byte) 0x7e, (byte) 0xeb, (byte) 0x80, (byte) 0xeb, (byte) 0xfc,
                                        (byte) 0xec, (byte) 0x40, (byte) 0xec, (byte) 0x7e, (byte) 0xec, (byte) 0x80, (byte) 0xec, (byte) 0xfc, (byte) 0xed, (byte) 0x40, (byte) 0xed, (byte) 0x7e,
                                        (byte) 0xed, (byte) 0x80, (byte) 0xed, (byte) 0xfc, (byte) 0xee, (byte) 0x40, (byte) 0xee, (byte) 0x7e, (byte) 0xee, (byte) 0x80, (byte) 0xee, (byte) 0xfc,
                                        (byte) 0xef, (byte) 0x40, (byte) 0xef, (byte) 0x7e, (byte) 0xef, (byte) 0x80, (byte) 0xef, (byte) 0xfc, (byte) 0xf0, (byte) 0x40, (byte) 0xf0, (byte) 0x7e,
                                        (byte) 0xf0, (byte) 0x80, (byte) 0xf0, (byte) 0xfc, (byte) 0xf1, (byte) 0x40, (byte) 0xf1, (byte) 0x7e, (byte) 0xf1, (byte) 0x80, (byte) 0xf1, (byte) 0xfc,
                                        (byte) 0xf2, (byte) 0x40, (byte) 0xf2, (byte) 0x7e, (byte) 0xf2, (byte) 0x80, (byte) 0xf2, (byte) 0xfc, (byte) 0xf3, (byte) 0x40, (byte) 0xf3, (byte) 0x7e,
                                        (byte) 0xf3, (byte) 0x80, (byte) 0xf3, (byte) 0xfc, (byte) 0xf4, (byte) 0x40, (byte) 0xf4, (byte) 0x7e, (byte) 0xf4, (byte) 0x80, (byte) 0xf4, (byte) 0xfc,
                                        (byte) 0xf5, (byte) 0x40, (byte) 0xf5, (byte) 0x7e, (byte) 0xf5, (byte) 0x80, (byte) 0xf5, (byte) 0xfc, (byte) 0xf6, (byte) 0x40, (byte) 0xf6, (byte) 0x7e,
                                        (byte) 0xf6, (byte) 0x80, (byte) 0xf6, (byte) 0xfc, (byte) 0xf7, (byte) 0x40, (byte) 0xf7, (byte) 0x7e, (byte) 0xf7, (byte) 0x80, (byte) 0xf7, (byte) 0xfc,
                                        (byte) 0xf8, (byte) 0x40, (byte) 0xf8, (byte) 0x7e, (byte) 0xf8, (byte) 0x80, (byte) 0xf8, (byte) 0xfc, (byte) 0xf9, (byte) 0x40, (byte) 0xf9, (byte) 0x7e,
                                        (byte) 0xf9, (byte) 0x80, (byte) 0xf9, (byte) 0xfc, (byte) 0xfa, (byte) 0x40, (byte) 0xfa, (byte) 0x7e, (byte) 0xfa, (byte) 0x80, (byte) 0xfa, (byte) 0xfc,
                                        (byte) 0xfb, (byte) 0x40, (byte) 0xfb, (byte) 0x7e, (byte) 0xfb, (byte) 0x80, (byte) 0xfb, (byte) 0xfc, (byte) 0xfc, (byte) 0x40, (byte) 0xfc, (byte) 0x7e,
                                        (byte) 0xfc, (byte) 0x80, (byte) 0xfc, (byte) 0xfc},
                        null, null);
    }

    static TestData dataWindows874() {
        assert TruffleString.Encoding.Windows_874.ordinal() == 93;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, new byte[]{(byte) 0x00, (byte) 0x7f}, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataCP50220() {
        assert TruffleString.Encoding.CP50220.ordinal() == 94;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, null, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataCP50221() {
        assert TruffleString.Encoding.CP50221.ordinal() == 95;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, null, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataIBM037() {
        assert TruffleString.Encoding.IBM037.ordinal() == 96;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, null, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO2022JP() {
        assert TruffleString.Encoding.ISO_2022_JP.ordinal() == 97;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, null, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO2022JP2() {
        assert TruffleString.Encoding.ISO_2022_JP_2.ordinal() == 98;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, null, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataISO2022JPKDDI() {
        assert TruffleString.Encoding.ISO_2022_JP_KDDI.ordinal() == 99;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, null, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }

    static TestData dataUTF7() {
        assert TruffleString.Encoding.UTF_7.ordinal() == 100;
        return new TestData(new int[]{0x00, 0xff}, new int[]{0x0, 0x1}, null, null, null, new byte[]{(byte) 0x00, (byte) 0xff}, null, null);
    }
}
