/*
 * Copyright (c) 2017, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifndef __TRUFFLE_INTERNAL_H
#define __TRUFFLE_INTERNAL_H

#if defined(__linux__)

#if !defined(_GNU_SOURCE)
#error "expected to be set via CLFAGS.  required for detection below"
#endif

/* musl does not set __USE_GNU in features.h
 * source: https://stackoverflow.com/a/70211227
 */
#include <features.h>

#ifdef __USE_GNU
/* glibc */
#define ENABLE_ISOLATED_NAMESPACE
#define ISOLATED_NAMESPACE 0x10000
#include <dlfcn.h>

#else  // !__USE_GNU
/* musl. dlmopen not available */
#endif // !__USE_GNU

#endif // __linux__

#include "native.h"
#include "trufflenfi.h"
#include <ffi.h>
#include <jni.h>

#ifdef _WIN32

#include <malloc.h>

#define __thread __declspec(thread)

#else // !_WIN32

#include <stdint.h>
#include <alloca.h>

#endif

struct __TruffleContextInternal {
    const struct __TruffleThreadAPI *functions;
    JavaVM *javaVM;
    jobject LibFFIContext;

#if defined(ENABLE_ISOLATED_NAMESPACE)
    jfieldID LibFFIContext_isolatedNamespaceId;
#endif

    jmethodID CallTarget_call;

    jfieldID LibFFISignature_cif;
    jfieldID LibFFISignature_signatureInfo;

    jfieldID CachedSignatureInfo_argTypes;

    jfieldID LibFFIType_type;
    jclass LibFFIType_EnvType;
    jclass LibFFIType_ObjectType;
    jclass LibFFIType_NullableType;
    jclass LibFFIType_StringType;

    jclass NativeString;
    jfieldID NativeString_nativePointer;

    jmethodID LibFFIContext_getNativeEnv;
    jmethodID LibFFIContext_attachThread;
    jmethodID LibFFIContext_detachThread;
    jmethodID LibFFIContext_createClosureNativePointer;
    jmethodID LibFFIContext_newClosureRef;
    jmethodID LibFFIContext_releaseClosureRef;
    jmethodID LibFFIContext_getClosureObject;

    jfieldID RetPatches_count;
    jfieldID RetPatches_patches;
    jfieldID RetPatches_objects;

    jfieldID NFIState_nfiErrnoAddress;
    jfieldID NFIState_hasPendingException;

    jclass NativeArgumentBuffer_Pointer;
    jfieldID NativeArgumentBuffer_Pointer_pointer;

    jclass Object;
    jclass String;
    jclass UnsatisfiedLinkError;

    void *__libc_errno_location;
#if !defined(_WIN32)
    void *__pthreads_errno_location;
#endif
};

struct __TruffleEnvInternal {
    const struct __TruffleNativeAPI *functions;
    struct __TruffleContextInternal *context;
    JNIEnv *jniEnv;
    jobject nfiState;
    int *nfiErrnoAddress;
};

extern const struct __TruffleNativeAPI truffleNativeAPI;
extern const struct __TruffleThreadAPI truffleThreadAPI;

// contains the "current" env from the most recent downcall, for faster lookup
extern __thread struct __TruffleEnvInternal *cachedTruffleEnv;

// keep this in sync with the code in com.oracle.truffle.nfi.NativeArgumentBuffer$TypeTag
enum TypeTag {
    OBJECT = 0,
    STRING,
    KEEPALIVE,
    ENV,

    BOOLEAN_ARRAY,
    BYTE_ARRAY,
    CHAR_ARRAY,
    SHORT_ARRAY,
    INT_ARRAY,
    LONG_ARRAY,
    FLOAT_ARRAY,
    DOUBLE_ARRAY
};

#define DECODE_OFFSET(encoded) (((unsigned int) (encoded)) >> 4)
#define DECODE_TAG(encoded) ((enum TypeTag)((encoded) & 0x0F))

void initialize_intrinsics(struct __TruffleContextInternal *);
void *check_intrinsify(struct __TruffleContextInternal *, void *);

#endif
