/*
 * Copyright (c) 2020, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package org.graalvm.truffle.benchmark;

import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Random;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;

import sun.misc.Unsafe;

/**
 * This benchmarks compares different ways to clone an array.
 */
@State(Scope.Benchmark)
public class ByteArrayAccessCloneBenchmark extends TruffleBenchmark {
    static final Unsafe UNSAFE;

    static {
        try {
            Field f = Unsafe.class.getDeclaredField("theUnsafe");
            f.setAccessible(true);
            UNSAFE = (Unsafe) f.get(null);
        } catch (Exception e) {
            throw new RuntimeException();
        }
    }

    @Param({"10", "100", "1_000_000"}) public String count;

    byte[] array;

    @Setup(Level.Trial)
    public void setup() {
        Random r = new Random();
        r.setSeed(0);
        array = new byte[Integer.parseInt(count.replaceAll("_", ""))];
        for (int i = 0; i < array.length; ++i) {
            array[i] = (byte) r.nextInt(256);
        }
    }

    @Benchmark
    public byte[] unsafeCopyMemory() {
        byte[] copy = new byte[array.length];
        UNSAFE.copyMemory(array, Unsafe.ARRAY_BYTE_BASE_OFFSET, copy, Unsafe.ARRAY_BYTE_BASE_OFFSET, array.length);
        return copy;
    }

    @Benchmark
    public byte[] systemArrayCopy() {
        byte[] copy = new byte[array.length];
        System.arraycopy(array, 0, copy, 0, array.length);
        return copy;
    }

    @Benchmark
    public byte[] arrayLoopCopy() {
        byte[] copy = new byte[array.length];
        for (int i = 0; i < array.length; ++i) {
            copy[i] = array[i];
        }
        return copy;
    }

    @Benchmark
    public byte[] arraysCopyOf() {
        return Arrays.copyOf(array, array.length);
    }

    @Benchmark
    public byte[] arrayClone() {
        return array.clone();
    }
}
