/*
 * Copyright (c) 2025, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.oracle.svm.hosted.webimage.codegen.phase;

import org.graalvm.nativeimage.AnnotationAccess;

import com.oracle.svm.webimage.annotation.StackifierVerification;

import jdk.graal.compiler.debug.GraalError;
import jdk.graal.compiler.hightiercodegen.reconstruction.ReconstructionData;
import jdk.graal.compiler.hightiercodegen.reconstruction.ScheduleWithReconstructionResult;
import jdk.graal.compiler.hightiercodegen.reconstruction.StackifierData;
import jdk.graal.compiler.nodes.StructuredGraph;
import jdk.graal.compiler.nodes.spi.CoreProviders;
import jdk.graal.compiler.phases.BasePhase;
import jdk.vm.ci.meta.ResolvedJavaMethod;

public class ReconstructionVerificationPhase extends BasePhase<CoreProviders> {
    @Override
    protected void run(StructuredGraph graph, CoreProviders providers) {
        ReconstructionData reconstructionData = ((ScheduleWithReconstructionResult) graph.getLastSchedule()).reconstructionData();
        if (AnnotationAccess.getAnnotation(graph.method(), StackifierVerification.class) != null) {
            verifyStackifier(graph, graph.method(), (StackifierData) reconstructionData, providers);
        }
    }

    /**
     * Verify that the stackifier generated the expected number of forward blocks and scopes.
     *
     * @param g graph
     * @param method hosted method
     * @param stackData data generated by stackifier
     * @param providers providers
     */
    private static void verifyStackifier(StructuredGraph g, ResolvedJavaMethod method, StackifierData stackData, CoreProviders providers) {
        StackifierVerification annot = AnnotationAccess.getAnnotation(method, StackifierVerification.class);
        if (stackData.getNrOfLabeledBlocks() != annot.expLabeledBlocks() ||
                        stackData.getNrThenScopes() != annot.expThenScopes() ||
                        stackData.getNrElseScopes() != annot.expElseScopes() ||
                        stackData.getNrLoopScopes() != annot.expLoopScopes() ||
                        stackData.getNrCatchScopes() != annot.expCatchScopes()) {
            String formatString = "Forward blocks %d %n Then scopes %d %n Else scopes %d %n Loop scopes %d %n Catch scopes %d %n";
            String expected = String.format(formatString,
                            annot.expLabeledBlocks(),
                            annot.expThenScopes(),
                            annot.expElseScopes(),
                            annot.expLoopScopes(),
                            annot.expCatchScopes());
            String actual = String.format(formatString,
                            stackData.getNrOfLabeledBlocks(),
                            stackData.getNrThenScopes(),
                            stackData.getNrElseScopes(),
                            stackData.getNrLoopScopes(),
                            stackData.getNrCatchScopes());
            throw new GraalError("Stackifier control flow reconstruction algorithm did not perform as expected for method " + g.method() + "\n" +
                            "Expected: " + expected + "\n" +
                            "Actual: " + actual);
        }
    }

}
