/*
 * Copyright (c) 2025, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.js.test.tools;

import static org.junit.Assert.assertEquals;

import org.graalvm.polyglot.Context;
import org.graalvm.polyglot.Value;
import org.junit.Test;

import com.oracle.truffle.api.frame.VirtualFrame;
import com.oracle.truffle.api.instrumentation.EventContext;
import com.oracle.truffle.api.instrumentation.ExecutionEventNode;
import com.oracle.truffle.api.instrumentation.ExecutionEventNodeFactory;
import com.oracle.truffle.api.instrumentation.SourceSectionFilter;
import com.oracle.truffle.api.instrumentation.StandardTags;
import com.oracle.truffle.api.instrumentation.TruffleInstrument;
import com.oracle.truffle.api.interop.InteropException;
import com.oracle.truffle.api.interop.InteropLibrary;
import com.oracle.truffle.api.interop.NodeLibrary;
import com.oracle.truffle.js.test.JSTest;

public class GR71433 {
    private static final String INJECT_SHORT_INSTRUMENT_NAME = "InjectShortInstrument";

    @TruffleInstrument.Registration(id = INJECT_SHORT_INSTRUMENT_NAME, services = InjectShortInstrument.class)
    static class InjectShortInstrument extends TruffleInstrument {

        @Override
        protected void onCreate(Env env) {
            env.registerService(this);

            ExecutionEventNodeFactory eenf = new ExecutionEventNodeFactory() {
                @Override
                public ExecutionEventNode create(final EventContext eventContext) {
                    return new ExecutionEventNode() {
                        @Override
                        protected void onEnter(final VirtualFrame frame) {
                            try {
                                Object scope = NodeLibrary.getUncached().getScope(eventContext.getInstrumentedNode(), frame, false);
                                InteropLibrary interop = InteropLibrary.getUncached(scope);
                                if (interop.isMemberWritable(scope, "scopeWrite")) {
                                    interop.writeMember(scope, "scopeWrite", (short) 42);
                                } else if (interop.isMemberWritable(scope, "unwind")) {
                                    throw eventContext.createUnwind(null);
                                }
                            } catch (InteropException iex) {
                                throw new RuntimeException(iex);
                            }
                        }

                        @Override
                        protected Object onUnwind(final VirtualFrame frame, final Object info) {
                            return (short) 211;
                        }

                    };
                }
            };
            SourceSectionFilter filter = SourceSectionFilter.newBuilder().tagIs(StandardTags.StatementTag.class).build();
            env.getInstrumenter().attachExecutionEventFactory(filter, eenf);
        }

    }

    @Test
    public void testShortInScopeWrite() {
        try (Context context = JSTest.newContextBuilder().build()) {
            context.getEngine().getInstruments().get(INJECT_SHORT_INSTRUMENT_NAME).lookup(InjectShortInstrument.class);
            Value result = context.eval("js", "(() => { var scopeWrite; return scopeWrite; })()");
            assertEquals(42, result.asInt());
        }
    }

    @Test
    public void testShortInUnwind() {
        try (Context context = JSTest.newContextBuilder().build()) {
            context.getEngine().getInstruments().get(INJECT_SHORT_INSTRUMENT_NAME).lookup(InjectShortInstrument.class);
            Value result = context.eval("js", "(() => { var unwind; return unwind; })()");
            assertEquals(211, result.asInt());
        }
    }

}
